// Shared minimal interactions for the template
function qs(sel, root=document){ return root.querySelector(sel); }

// Login page validation and UX
function initLogin(){
  const form = qs('#loginForm');
  if(!form) return;
  const email = qs('#loginEmail');
  const pass = qs('#loginPassword');
  const remember = qs('#loginRemember');
  const err = qs('#loginError');
  const toggle = qs('#togglePwd');

  const emailValid = (v)=> /.+@.+\..+/.test((v||'').trim());

  form.addEventListener('submit', (e)=>{
    e.preventDefault();
    const eVal = email?.value||'';
    const pVal = pass?.value||'';
    let ok = true;
    if(!emailValid(eVal)){ ok=false; email?.classList.add('ring-2','ring-red-500'); } else { email?.classList.remove('ring-2','ring-red-500'); }
    if(!pVal || pVal.length < 6){ ok=false; pass?.classList.add('ring-2','ring-red-500'); } else { pass?.classList.remove('ring-2','ring-red-500'); }
    if(!ok){
      if(err){ err.textContent = 'Enter a valid email and password (min 6 characters).'; err.classList.remove('hidden'); }
      showToast('Fix the highlighted fields', 'warning');
      return;
    }
    if(err) err.classList.add('hidden');
    const rememberMe = !!remember?.checked;
    // Stub login flow
    showToast('Signing you in…', 'info');
    setTimeout(()=>{
      showToast('Welcome back!', 'success');
      window.location.href = 'dashboard.html';
    }, 700);
  });

  // Show/hide password toggle
  if(toggle && pass){
    toggle.addEventListener('click', ()=>{
      const isPwd = pass.type === 'password';
      pass.type = isPwd ? 'text' : 'password';
      toggle.textContent = isPwd ? 'Hide' : 'Show';
    });
  }
}

// Snackbar (bottom-center notification with action button)
function showSnackbar(message, { actionText='View', href='', onAction=null, variant='success' }={}){
  const colors = {
    success: 'bg-slate-900', // subtle dark bar
    info: 'bg-slate-900',
    warning: 'bg-slate-900',
    error: 'bg-slate-900'
  };
  let wrap = qs('#snackbarWrap');
  if(!wrap){
    wrap = document.createElement('div');
    wrap.id = 'snackbarWrap';
    wrap.className = 'fixed z-[9999] bottom-4 left-1/2 -translate-x-1/2 flex flex-col gap-2 w-full max-w-xl px-4';
    document.body.appendChild(wrap);
  }
  const bar = document.createElement('div');
  bar.className = `${colors[variant]||colors.success} text-white shadow-xl rounded-full px-4 py-3 flex items-center justify-between animate-slide-in`;
  const msg = document.createElement('div');
  msg.className = 'text-sm';
  msg.textContent = message;
  const action = document.createElement('button');
  action.className = 'ml-4 text-sm font-semibold text-white/90 hover:text-white underline';
  action.textContent = actionText;
  action.addEventListener('click', ()=>{
    if(onAction) onAction();
    else if(href) window.location.href = href;
  });
  bar.appendChild(msg); bar.appendChild(action); wrap.appendChild(bar);
  setTimeout(()=>{ bar.classList.add('opacity-0','transition-opacity'); setTimeout(()=> bar.remove(), 250); }, 2600);
}

function initHotelDetail(){
  const sidebar = qs('#hotelSidebar');
  if(!sidebar) return;
  const btn = qs('#btnHotelAddToCart');
  const addons = qsa('#hotelAddonsSection input[type="checkbox"]');
  const baseName = sidebar.getAttribute('data-name') || 'Hotel stay';
  const room = parseFloat(sidebar.getAttribute('data-room')||'0');
  const taxes = parseFloat(sidebar.getAttribute('data-taxes')||'0');
  const basePrice = Number(room + taxes);

  btn?.addEventListener('click', ()=>{
    const parentId = `HOTEL-${Date.now()}`;
    cartAdd({ id: parentId, type:'Hotel', name: baseName, price: basePrice, qty:1 });
    addons.filter(i=>i.checked).forEach(i=>{
      try { const ex = JSON.parse(i.value); cartAdd({ ...ex, parentId }); }
      catch(e){ console.warn('Invalid hotel extra json', e); }
    });
    updateCartCount();
    showSnackbar('Hotel added to cart', { actionText: 'View cart', href: 'cart.html', variant: 'success' });
  });
}

function initBookings(){
  const root = qs('#bookingsPage');
  if(!root) return;
  const kw = qs('#bkKeyword');
  const type = qs('#bkType');
  const status = qs('#bkStatus');
  const from = qs('#bkFrom');
  const to = qs('#bkTo');
  const btn = qs('#btnBookingsSearch');
  const tbody = qs('#bookingsTable');

  const render = (items=[])=>{
    if(!tbody) return;
    if(!items.length){ tbody.innerHTML = `<tr><td class="py-4 text-center text-slate-500" colspan="7">No results</td></tr>`; return; }
    tbody.innerHTML = items.map(it=>`
      <tr class="border-t">
        <td class="py-2">${it.ref}</td>
        <td>${it.type}</td>
        <td>${it.guest}</td>
        <td>${it.dates}</td>
        <td><span class="badge">${it.status}</span></td>
        <td class="text-right">US$ ${it.total}</td>
        <td class="text-right">${it.voucher?`<a class="text-brand" href="confirmation.html">Voucher</a>`:'—'}</td>
      </tr>
    `).join('');
  };

  const search = async ()=>{
    const res = await apiGetBookingHistory({
      q: kw?.value||'', type: type?.value||'All', status: status?.value||'All', from: from?.value||'', to: to?.value||''
    });
    // For now, return demo items filtered superficially
    const demo = [
      { ref:'H-92451', type:'Hotel', guest:'John Doe', dates:'02 Mar → 11 Apr', status:'Confirmed', total:'622', voucher:true },
      { ref:'A-30211', type:'Activity', guest:'Jane Smith', dates:'15 Sep', status:'Confirmed', total:'120', voucher:true },
      { ref:'C-88273', type:'Taxi', guest:'Sam Patel', dates:'19 Sep', status:'Cancelled', total:'45', voucher:false },
    ];
    let items = demo;
    const qv = (kw?.value||'').toLowerCase();
    if(qv) items = items.filter(x=> Object.values(x).join(' ').toLowerCase().includes(qv));
    if(type && type.value!=='All') items = items.filter(x=> x.type===type.value);
    if(status && status.value!=='All') items = items.filter(x=> x.status===status.value);
    render(items);
  };

  btn?.addEventListener('click', search);
  search();
}
function qsa(sel, root=document){ return [...root.querySelectorAll(sel)]; }

// Tabs
function initTabs(){
  qsa('[data-tabs]').forEach(group=>{
    const tabs = qsa('[data-tab]', group);
    const panes = qsa('[data-pane]', group.parentElement);
    tabs.forEach(tab=>{
      tab.addEventListener('click', ()=>{
        const t = tab.getAttribute('data-tab');
        tabs.forEach(x=>{ x.classList.remove('bg-brand','text-white'); x.classList.add('bg-white'); });
        tab.classList.add('bg-brand','text-white');
        panes.forEach(p=>p.classList.toggle('hidden', p.getAttribute('data-pane')!==t));
      });
    });
  });
}

// -------------------
// CART UTILITIES + UI
// -------------------
function cartGet(){
  try { return JSON.parse(localStorage.getItem('voyago_cart')||'[]'); } catch { return []; }
}
function cartSet(items){
  localStorage.setItem('voyago_cart', JSON.stringify(items||[]));
  updateCartCount();
}
function cartAdd(item){
  const items = cartGet();
  items.push({
    id: item.id || `LI-${Date.now()}-${Math.random().toString(36).slice(2,6)}`,
    type: item.type||'Activity',
    name: item.name||'Item',
    price: Number(item.price||0),
    qty: Number(item.qty||1),
    parentId: item.parentId||null,
    meta: item.meta||{}
  });
  cartSet(items);
}
function cartRemove(id){ cartSet(cartGet().filter(x=>x.id!==id)); }
function cartClear(){ cartSet([]); }
function cartTotal(items){ return (items||cartGet()).reduce((s,x)=> s + (x.price*x.qty), 0); }

function updateCartCount(){
  const c = cartGet().length;
  const el = qs('#cartCount');
  if(el){
    el.textContent = String(c);
    el.classList.add('animate-pulse');
    setTimeout(()=> el.classList.remove('animate-pulse'), 500);
  }
}

function initCartUI(){
  updateCartCount();
}

function initActivityDetail(){
  const sidebar = qs('#activitySidebar');
  if(!sidebar) return;
  const btn = qs('#btnAddToCart');
  const baseName = qs('h1')?.textContent?.trim() || 'Activity';
  const basePriceText = sidebar.querySelector('.font-extrabold')?.textContent||'US$ 0';
  const basePrice = parseFloat((basePriceText.match(/[\d.]+/g)||['0'])[0]);
  const addonInputs = qsa('#addonsSection input[type="checkbox"]');

  btn?.addEventListener('click', ()=>{
    const parentId = `BASE-${Date.now()}`;
    cartAdd({ id: parentId, type:'Activity', name: baseName, price: basePrice, qty:1 });
    addonInputs.filter(i=>i.checked).forEach(i=>{
      try { const ad = JSON.parse(i.value); cartAdd({ ...ad, parentId }); }
      catch(e){ console.warn('Invalid addon json', e); }
    });
    updateCartCount();
    showSnackbar('Added to cart', { actionText: 'View cart', href: 'cart.html', variant: 'success' });
  });
}

function initCartPage(){
  const root = qs('#cartPage');
  if(!root) return;
  const tbody = qs('#cartTable');
  const totalEl = qs('#cartTotal');
  const btnClear = qs('#btnCartClear');
  const btnCheckout = qs('#btnProceedCheckout');

  const render = ()=>{
    const items = cartGet();
    if(!items.length){
      tbody.innerHTML = `<tr><td class="py-6 text-center text-slate-500" colspan="4">Your cart is empty</td></tr>`;
      totalEl.textContent = 'US$ 0.00';
      return;
    }
    tbody.innerHTML = items.map(x=>`
      <tr class="border-t">
        <td class="py-2">${x.name}</td>
        <td>${x.type}${x.parentId? ' (add-on)':''}</td>
        <td>US$ ${x.price.toFixed(2)}</td>
        <td class="text-right"><button class="text-brand" data-remove="${x.id}">Remove</button></td>
      </tr>
    `).join('');
    totalEl.textContent = `US$ ${cartTotal(items).toFixed(2)}`;
  };

  root.addEventListener('click', (e)=>{
    const btn = e.target.closest('[data-remove]');
    if(btn){
      cartRemove(btn.getAttribute('data-remove'));
      render();
      showSnackbar('Removed from cart', { variant:'info' });
    }
  });
  btnClear?.addEventListener('click', ()=>{ cartClear(); render(); showSnackbar('Cart cleared', { variant:'warning' }); });
  btnCheckout?.addEventListener('click', ()=>{ window.location.href = 'booking.html'; });
  render();
}

function initBookingCart(){
  const itemsWrap = qs('#cartReviewItems');
  const totalEl = qs('#cartReviewTotal');
  if(!itemsWrap || !totalEl) return;
  const items = cartGet();
  if(!items.length){
    itemsWrap.innerHTML = `<tr><td class="py-3 text-center text-slate-500" colspan="3">Your cart is empty</td></tr>`;
    totalEl.textContent = 'US$ 0.00';
    return;
  }
  itemsWrap.innerHTML = items.map(x=>`
    <tr class="border-t">
      <td class="py-2">${x.name}</td>
      <td>${x.type}${x.parentId? ' (add-on)':''}</td>
      <td class="text-right">US$ ${Number(x.price||0).toFixed(2)}</td>
    </tr>
  `).join('');
  totalEl.textContent = `US$ ${cartTotal(items).toFixed(2)}`;
}

// Mobile filters drawer
function initFilterDrawers(){
  qsa('[data-filter-open]').forEach(btn=>{
    btn.addEventListener('click', ()=>{
      const id = btn.getAttribute('data-filter-open');
      qs(`#${id}`)?.classList.remove('hidden');
      document.body.classList.add('overflow-hidden');
    });
  });
  qsa('[data-filter-close]').forEach(btn=>{
    btn.addEventListener('click', ()=>{
      btn.closest('[data-filter-drawer]')?.classList.add('hidden');
      document.body.classList.remove('overflow-hidden');
    });
  });
}

// Sticky header color on scroll (optional subtle effect)
function initHeader(){
  const h = qs('[data-header]'); if(!h) return;
  const onScroll = ()=>{ h.classList.toggle('shadow', window.scrollY>2); };
  window.addEventListener('scroll', onScroll, {passive:true}); onScroll();
}

// Booking page price summary (extras + total)
function initBooking(){
  const box = qs('#bookingSummary');
  if(!box) return; // only on booking page
  const room = parseFloat(box.getAttribute('data-room')||'0');
  const taxes = parseFloat(box.getAttribute('data-taxes')||'0');
  const extrasInputs = qsa('.extra-item');
  const roomEl = qs('#roomAmount');
  const taxEl = qs('#taxAmount');
  const extrasEl = qs('#extrasAmount');
  const totalEl = qs('#totalAmount');

  const recalc = ()=>{
    const extras = extrasInputs.reduce((s,cb)=> s + (cb.checked ? parseFloat(cb.getAttribute('data-price')||'0') : 0), 0);
    if(roomEl) roomEl.textContent = room.toFixed(0);
    if(taxEl) taxEl.textContent = taxes.toFixed(0);
    if(extrasEl) extrasEl.textContent = extras.toFixed(0);
    if(totalEl) totalEl.textContent = (room + taxes + extras).toFixed(0);
  };
  extrasInputs.forEach(cb=> cb.addEventListener('change', recalc));
  recalc();
}

// Stepper (Next/Back + validation for step 0)
function initStepper(){
  const stepper = qs('[data-stepper]');
  const stepsWrap = qs('[data-steps]');
  if(!stepper || !stepsWrap) return;
  const badges = qsa('[data-step]', stepper);
  const panes = qsa('[data-step-pane]', stepsWrap);
  let current = 0;

  const show = (idx)=>{
    current = Math.max(0, Math.min(idx, panes.length-1));
    panes.forEach((p,i)=> p.classList.toggle('hidden', i!==current));
    badges.forEach((b,i)=>{
      b.classList.toggle('active', i===current);
    });
  };

  const validateStep0 = ()=>{
    const fn = qs('#firstName');
    const ln = qs('#lastName');
    const em = qs('#email');
    const ph = qs('#phone');
    let ok = true;
    const mark = (el, valid)=>{
      if(!el) return;
      el.classList.toggle('ring-2', !valid);
      el.classList.toggle('ring-red-500', !valid);
    };
    const emailValid = (v)=> /.+@.+\..+/.test(v||'');
    ok = !!fn?.value?.trim() && ok; mark(fn, !!fn?.value?.trim());
    ok = !!ln?.value?.trim() && ok; mark(ln, !!ln?.value?.trim());
    ok = emailValid(em?.value) && ok; mark(em, emailValid(em?.value));
    ok = !!ph?.value?.trim() && ok; mark(ph, !!ph?.value?.trim());
    return ok;
  };

  // Next/Prev handlers
  stepsWrap.addEventListener('click', (e)=>{
    const nextBtn = e.target.closest('[data-next]');
    const prevBtn = e.target.closest('[data-prev]');
    if(nextBtn){
      if(current===0 && !validateStep0()) return; // block if invalid
      show(current+1);
    }
    if(prevBtn){ show(current-1); }
  });

  // Allow clicking badges to jump (optional)
  stepper.addEventListener('click', (e)=>{
    const b = e.target.closest('[data-step]');
    if(!b) return;
    const idx = parseInt(b.getAttribute('data-step')||'0',10);
    if(idx>current && current===0 && !validateStep0()) return; // block forward jump from step 0 if invalid
    show(idx);
  });

  show(0);
}

// Optional UI enhancers (Flatpickr, Tom Select) if CDNs are loaded
function initEnhancers(){
  // Flatpickr for date inputs
  if(window.flatpickr){
    qsa('input.js-date, input[type="date"]').forEach(el=>{
      if(el.dataset.enhanced) return; el.dataset.enhanced = '1';
      window.flatpickr(el, { dateFormat: 'Y-m-d' });
    });
  }
  // Tom Select for selects
  if(window.TomSelect){
    qsa('select.js-select').forEach(el=>{
      if(el.dataset.enhanced) return; el.dataset.enhanced = '1';
      new window.TomSelect(el, { create:false, allowEmptyOption:true });
    });
  }
}

// -------------------
// AUTH INITIALIZERS
// -------------------
function passwordStrength(pwd){
  const v = (pwd||'').trim();
  let score = 0;
  if(v.length >= 8) score++;
  if(/[0-9]/.test(v)) score++;
  if(/[A-Z]/.test(v) || /[^A-Za-z0-9]/.test(v)) score++;
  const labels = ['Weak','Fair','Good','Strong'];
  return { score, label: labels[score]||'Weak' };
}

function initSignup(){
  const form = qs('#signupForm');
  if(!form) return;
  const agency = qs('#agency');
  const contact = qs('#contact');
  const email = qs('#email');
  const phone = qs('#phone');
  const address = qs('#address');
  const pwd = qs('#pwd');
  const cpwd = qs('#cpwd');
  const terms = qs('#terms');
  const err = qs('#signupError');
  const t1 = qs('#togglePwd1');
  const t2 = qs('#togglePwd2');
  const hint = qs('#pwdHint');
  const emailRx = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;

  const toggle = (btn, input)=> btn && input && btn.addEventListener('click', ()=>{
    const isPwd = input.type === 'password';
    input.type = isPwd ? 'text' : 'password';
    btn.textContent = isPwd ? 'Hide' : 'Show';
  });
  toggle(t1, pwd); toggle(t2, cpwd);

  pwd?.addEventListener('input', ()=>{
    const { score, label } = passwordStrength(pwd.value);
    if(hint){
      hint.textContent = `Password strength: ${label}. Min 8 chars, include a number.`;
      hint.classList.remove('text-red-600','text-amber-600','text-green-600');
      hint.classList.add(score>=2 ? (score>=3 ? 'text-green-600':'text-amber-600') : 'text-red-600');
    }
  });

  form.addEventListener('submit', (e)=>{
    err?.classList.add('hidden'); if(err) err.textContent = '';
    const messages = [];
    if(!agency?.value) messages.push('Agency name is required.');
    if(!contact?.value) messages.push('Contact person is required.');
    if(!emailRx.test(email?.value||'')) messages.push('Enter a valid email.');
    if(!phone?.value || (phone.value.replace(/\D/g,'').length < 7)) messages.push('Enter a valid phone number.');
    if(!address?.value) messages.push('Address is required.');
    const strong = /^(?=.*\d).{8,}$/.test(pwd?.value||'');
    if(!strong) messages.push('Password must be 8+ chars and include a number.');
    if((pwd?.value||'') !== (cpwd?.value||'')) messages.push('Passwords do not match.');
    if(!terms?.checked) messages.push('Please accept Terms and Privacy.');
    if(messages.length){
      e.preventDefault();
      if(err){ err.textContent = messages.join(' '); err.classList.remove('hidden'); }
      showToast('Fix the highlighted errors', 'warning');
      return;
    }
    // Mock: proceed to KYC page
    e.preventDefault();
    showToast('Account created (demo). Redirecting to KYC…','success');
    setTimeout(()=> window.location.href = 'kyc.html', 700);
  });
}

function initResetPassword(){
  const form = qs('#resetForm');
  if(!form) return;
  const np = qs('#newPwd');
  const cp = qs('#confirmPwd');
  const err = qs('#rpError');
  const ok = qs('#rpSuccess');
  const tNew = qs('#toggleNew');
  const tConfirm = qs('#toggleConfirm');

  const toggle = (btn, input)=> btn && input && btn.addEventListener('click', ()=>{
    const isPwd = input.type === 'password';
    input.type = isPwd ? 'text' : 'password';
    btn.textContent = isPwd ? 'Hide' : 'Show';
  });
  toggle(tNew, np); toggle(tConfirm, cp);

  np?.addEventListener('input', ()=>{
    const info = form.querySelector('p.text-xs');
    const { score, label } = passwordStrength(np.value);
    if(info){
      info.textContent = `Minimum 8 characters, include at least 1 number. Strength: ${label}`;
      info.classList.remove('text-red-600','text-amber-600','text-green-600');
      info.classList.add(score>=2 ? (score>=3 ? 'text-green-600':'text-amber-600') : 'text-red-600');
    }
  });

  form.addEventListener('submit', (e)=>{
    e.preventDefault();
    err?.classList.add('hidden'); if(err) err.textContent = '';
    ok?.classList.add('hidden');
    const v = (np?.value||'').trim();
    const c = (cp?.value||'').trim();
    const strong = /^(?=.*\d).{8,}$/.test(v);
    if(!strong){
      if(err){ err.textContent = 'Password must be at least 8 characters and include a number.'; err.classList.remove('hidden'); }
      np?.focus();
      return;
    }
    if(v !== c){
      if(err){ err.textContent = 'Passwords do not match.'; err.classList.remove('hidden'); }
      cp?.focus();
      return;
    }
    ok?.classList.remove('hidden');
    form.reset();
    showToast('Password reset (demo).','success');
  });
}

function initForgotPassword(){
  const form = qs('#forgotForm');
  if(!form) return;
  const email = qs('#fpEmail');
  const err = qs('#fpError');
  const ok = qs('#fpSuccess');
  form.addEventListener('submit', (e)=>{
    e.preventDefault();
    err?.classList.add('hidden'); if(err) err.textContent = '';
    ok?.classList.add('hidden');
    const value = (email?.value||'').trim();
    const valid = /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value);
    if(!valid){
      if(err){ err.textContent = 'Please enter a valid email address.'; err.classList.remove('hidden'); }
      email?.focus();
      return;
    }
    ok?.classList.remove('hidden');
    form.reset();
    showToast('Reset link sent (demo).','info');
  });
}

document.addEventListener('DOMContentLoaded', ()=>{
  initTabs();
  initFilterDrawers();
  initHeader();
  initBooking();
  initStepper();
  initEnhancers();
  initWallet();
  initAdminTransfers();
  initBookings();
  initLogin();
  initSignup();
  initResetPassword();
  initForgotPassword();
  initCartUI();
  initActivityDetail();
  initHotelDetail();
  initCartPage();
  initBookingCart();
});

// -------------------
// API STUBS (to be wired to backend later)
// -------------------
async function apiSearch(module, params){
  // module: 'hotels' | 'activities' | 'cars' | 'yachts' | 'packages'
  console.log('apiSearch', module, params);
  return { items: [], total: 0 };
}

async function apiCreateBooking(payload){
  console.log('apiCreateBooking', payload);
  return { id: 'BK-' + Math.floor(Math.random()*100000), status: 'pending' };
}

async function apiCreateStripeTopupIntent(amount){
  console.log('apiCreateStripeTopupIntent', amount);
  // Expect backend to return clientSecret, redirectUrl etc.
  return { clientSecret: 'dummy_secret', redirectUrl: '#' };
}

async function apiGetBookingHistory(filters){
  console.log('apiGetBookingHistory', filters);
  return { items: [] };
}

async function apiListAgents(query){
  console.log('apiListAgents', query);
  return { items: [{ id: 'AGT-1001', name: 'Acme Travels', balance: 2450.0, status: 'Active' }] };
}

async function apiApproveTransfer(ref){
  console.log('apiApproveTransfer', ref);
  return { ok: true };
}

async function apiDeclineTransfer(ref){
  console.log('apiDeclineTransfer', ref);
  return { ok: true };
}

// -------------------
// PAGE INITIALIZERS
// -------------------
function initWallet(){
  // Detect presence by buttons and headings in wallet page
  const balanceCard = qsa('.card').find(card=>/Available balance/i.test(card?.textContent||''));
  if(!balanceCard) return;
  const [btnTopup, btnWithdraw] = qsa('button', balanceCard);
  const modal = qs('#topupModal');
  const btnCancel = qs('#btnTopupCancel');
  const btnSubmit = qs('#btnTopupSubmit');
  const amountInput = qs('#topupAmount');

  const openModal = ()=>{ if(modal){ modal.classList.remove('hidden'); modal.classList.add('flex'); amountInput && (amountInput.value = '100'); amountInput?.focus(); } };
  const closeModal = ()=>{ if(modal){ modal.classList.add('hidden'); modal.classList.remove('flex'); } };

  btnTopup?.addEventListener('click', openModal);
  btnCancel?.addEventListener('click', closeModal);
  modal?.addEventListener('click', (e)=>{ if(e.target===modal) closeModal(); });
  btnSubmit?.addEventListener('click', async ()=>{
    const amount = parseFloat(amountInput?.value||'0');
    if(!amount || amount<=0){ showToast('Please enter a valid amount', 'warning'); return; }
    const res = await apiCreateStripeTopupIntent(amount);
    console.log('Topup intent created', res);
    closeModal();
    showToast(`Top-up intent created (stub) for US$ ${amount.toFixed(2)}. Redirecting to Stripe…`, 'info');
    if(res.redirectUrl && res.redirectUrl !== '#') window.location.href = res.redirectUrl;
  });

  btnWithdraw?.addEventListener('click', ()=>{
    showToast('Withdraw flow coming soon (stub).', 'warning');
  });
}

function initAdminTransfers(){
  const root = qs('#adminTransfersPage');
  if(!root) return;
  const btnSearch = qs('#btnSearchAgents');
  const qInput = qs('#agentSearchQuery');
  const agentsTable = qs('#agentsTable');
  const pendingTable = qs('#pendingTransfersTable');

  btnSearch?.addEventListener('click', async ()=>{
    const res = await apiListAgents({ q: qInput?.value||'' });
    // Simple render
    if(agentsTable){
      agentsTable.innerHTML = res.items.map(a=>
        `<tr class="border-t"><td class="py-2">${a.name}</td><td>${a.id}</td><td>US$ ${a.balance.toFixed(2)}</td><td><span class="badge">${a.status}</span></td><td class="text-right"><button class="btn btn-outline" data-credit-agent="${a.id}">Credit</button> <button class="btn btn-outline" data-debit-agent="${a.id}">Debit</button></td></tr>`
      ).join('');
    }
  });

  root.addEventListener('click', async (e)=>{
    const approveBtn = e.target.closest('[data-approve-transfer]');
    const declineBtn = e.target.closest('[data-decline-transfer]');
    const creditBtn = e.target.closest('[data-credit-agent]');
    const debitBtn = e.target.closest('[data-debit-agent]');
    if(approveBtn){
      const ref = approveBtn.getAttribute('data-approve-transfer');
      await apiApproveTransfer(ref);
      showToast(`Approved transfer ${ref} (stub)`, 'success');
    }
    if(declineBtn){
      const ref = declineBtn.getAttribute('data-decline-transfer');
      await apiDeclineTransfer(ref);
      showToast(`Declined transfer ${ref} (stub)`, 'warning');
    }
    if(creditBtn){
      const id = creditBtn.getAttribute('data-credit-agent');
      showToast(`Credit flow for ${id} (stub)`, 'info');
    }
    if(debitBtn){
      const id = debitBtn.getAttribute('data-debit-agent');
      showToast(`Debit flow for ${id} (stub)`, 'info');
    }
  });
}

// -------------------
// TOAST UTIL
// -------------------
function showToast(message, variant='success'){
  const colors = {
    success: 'bg-green-600',
    info: 'bg-blue-600',
    warning: 'bg-amber-600',
    error: 'bg-red-600'
  };
  let wrap = qs('#toastWrap');
  if(!wrap){
    wrap = document.createElement('div');
    wrap.id = 'toastWrap';
    wrap.className = 'fixed z-[9999] bottom-4 right-4 flex flex-col gap-2';
    document.body.appendChild(wrap);
  }
  const toast = document.createElement('div');
  toast.className = `${colors[variant]||colors.info} text-white shadow-lg rounded-lg px-4 py-3 min-w-[240px] max-w-[360px] pointer-events-auto animate-slide-in`;
  toast.textContent = message;
  wrap.appendChild(toast);
  // auto hide
  setTimeout(()=>{
    toast.classList.add('opacity-0');
    toast.classList.add('transition-opacity');
    setTimeout(()=> toast.remove(), 300);
  }, 2200);
}
