<?php
// TEMPORARY database setup/apply tool. DELETE this file after running.
// URL: http(s)://<host>/tools_apply_db.php

require_once dirname(__DIR__) . '/config/config.php'; // expects $pdo (PDO MySQL)

header('Content-Type: text/plain; charset=utf-8');

function out($msg){ echo $msg, "\n"; }

try {
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Ensure tracking table exists
    $pdo->exec("CREATE TABLE IF NOT EXISTS db_patches (
        id INT AUTO_INCREMENT PRIMARY KEY,
        filename VARCHAR(255) NOT NULL UNIQUE,
        applied_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

    // Helper to check if applied
    $isApplied = function(string $filename) use ($pdo): bool {
        $stmt = $pdo->prepare('SELECT 1 FROM db_patches WHERE filename = :f LIMIT 1');
        $stmt->execute([':f' => $filename]);
        return (bool)$stmt->fetchColumn();
    };

    // Execute an SQL file (naive splitter by ;) and ignore empty statements
    $executeSqlFile = function(string $path) use ($pdo) {
        if (!file_exists($path)) throw new RuntimeException("SQL file not found: $path");
        $sql = file_get_contents($path);
        if ($sql === false) throw new RuntimeException("Failed to read: $path");
        // Remove BOM
        $sql = preg_replace('/^\xEF\xBB\xBF/', '', $sql);
        // Strip simple -- comments and /* */ blocks for splitting safety
        $sqlNoBlock = preg_replace('#/\*.*?\*/#s', '', $sql);
        $lines = preg_split('/\r\n|\r|\n/', $sqlNoBlock);
        $statements = [];
        $current = '';
        foreach ($lines as $line) {
            // Skip single-line comments
            if (preg_match('/^\s*--/', $line)) continue;
            $current .= $line . "\n";
            // naive: split on ; at end of line
            if (preg_match('/;\s*$/', trim($line))) {
                $statements[] = $current;
                $current = '';
            }
        }
        if (trim($current) !== '') $statements[] = $current;
        // Run within transaction only if file does not manage its own transactions
        $containsTxnCtl = (bool)preg_match('/\b(START\s+TRANSACTION|BEGIN\b|COMMIT\b|ROLLBACK\b)\b/i', $sqlNoBlock);
        $didBegin = false;
        if (!$containsTxnCtl) {
            try { $didBegin = $pdo->beginTransaction(); } catch (Throwable $_) { $didBegin = false; }
        }
        try {
            foreach ($statements as $stmtSql) {
                $stmtSql = trim($stmtSql);
                if ($stmtSql === '') continue;
                $pdo->exec($stmtSql);
            }
            if ($didBegin && $pdo->inTransaction()) { $pdo->commit(); }
        } catch (Throwable $e) {
            if ($didBegin && $pdo->inTransaction()) $pdo->rollBack();
            throw $e;
        }
    };

    // Preflight: ensure columns exist (idempotent)
    out('Preflight: ensuring required columns ...');
    $ensureCol = function(string $table, string $column, string $ddl) use ($pdo) {
        $sql = "SELECT COUNT(*) FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :t AND COLUMN_NAME = :c";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([':t' => $table, ':c' => $column]);
        $exists = (int)$stmt->fetchColumn() > 0;
        if (!$exists) {
            $pdo->exec($ddl);
            out("  Added column $table.$column");
        } else {
            out("  OK: $table.$column exists");
        }
    };
    try {
        // Orders contact fields
        $ensureCol('orders', 'customer_name',     "ALTER TABLE orders ADD COLUMN customer_name VARCHAR(191) NULL AFTER payment_method");
        $ensureCol('orders', 'customer_mobile',   "ALTER TABLE orders ADD COLUMN customer_mobile VARCHAR(50) NULL AFTER customer_name");
        $ensureCol('orders', 'customer_email',    "ALTER TABLE orders ADD COLUMN customer_email VARCHAR(191) NULL AFTER customer_mobile");
        $ensureCol('orders', 'customer_whatsapp', "ALTER TABLE orders ADD COLUMN customer_whatsapp VARCHAR(50) NULL AFTER customer_email");

        // Users: transaction PIN support
        $ensureCol('users', 'require_tx_password',         "ALTER TABLE users ADD COLUMN require_tx_password TINYINT(1) NOT NULL DEFAULT 0");
        $ensureCol('users', 'transaction_password_hash',   "ALTER TABLE users ADD COLUMN transaction_password_hash VARCHAR(255) NULL");
        $ensureCol('users', 'tx_pin_attempts',             "ALTER TABLE users ADD COLUMN tx_pin_attempts INT NOT NULL DEFAULT 0");
        $ensureCol('users', 'tx_pin_locked_until',         "ALTER TABLE users ADD COLUMN tx_pin_locked_until DATETIME NULL");
    } catch (Throwable $e) {
        out('  Note: ' . $e->getMessage());
    }

    $base = dirname(__DIR__);
    $schemaFile = $base . '/database/schema.sql';
    $patchDir = $base . '/database/patches';

    out('== DB Apply Tool ==');

    // Apply schema.sql once (tracked as schema.sql)
    // Only apply schema.sql if DB appears empty (no user tables)
    $stmt = $pdo->query("SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name NOT IN ('db_patches')");
    $existingTableCount = (int)$stmt->fetchColumn();
    if (!$isApplied('schema.sql')) {
        if ($existingTableCount === 0) {
            out('Applying schema.sql ...');
            $executeSqlFile($schemaFile);
            $stmt = $pdo->prepare('INSERT INTO db_patches (filename) VALUES (:f)');
            $stmt->execute([':f' => 'schema.sql']);
            out('OK: schema.sql applied');
        } else {
            out('Skip: schema.sql (database already has tables).');
        }
    } else {
        out('Skip: schema.sql already applied');
    }

    // Apply patches in lexicographical order
    if (is_dir($patchDir)) {
        $files = glob($patchDir . '/*.sql');
        sort($files, SORT_STRING);
        foreach ($files as $path) {
            $fname = basename($path);
            if ($isApplied($fname)) {
                out("Skip: $fname already applied");
                continue;
            }
            out("Applying $fname ...");
            $executeSqlFile($path);
            $stmt = $pdo->prepare('INSERT INTO db_patches (filename) VALUES (:f)');
            $stmt->execute([':f' => $fname]);
            out("OK: $fname applied");
        }
    } else {
        out('No patches directory found, skipping.');
    }

    out("\nAll done. You may now delete public/tools_apply_db.php for security.");

} catch (Throwable $e) {
    http_response_code(500);
    out('ERROR: ' . $e->getMessage());
}
