-- 2025-09-08: Yachts booking schema (agent portal)
-- This patch creates the core tables for yachts, yacht images (if missing),
-- and full booking lifecycle: bookings, events, payments.

START TRANSACTION;

-- Core yachts table (ensure required columns exist)
CREATE TABLE IF NOT EXISTS yachts (
  id INT UNSIGNED NOT NULL AUTO_INCREMENT,
  name VARCHAR(200) NOT NULL,
  subtitle VARCHAR(200) NULL,
  city VARCHAR(120) NULL,
  capacity INT UNSIGNED NULL,
  max_guests INT UNSIGNED NULL,
  from_price DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  booking_code VARCHAR(100) NULL,
  vendor_cost DECIMAL(12,2) NULL,
  agent_cost DECIMAL(12,2) NULL,
  customer_cost DECIMAL(12,2) NULL,
  visible_agent TINYINT(1) NOT NULL DEFAULT 1,
  currency CHAR(3) NOT NULL DEFAULT 'THB',
  active TINYINT(1) NOT NULL DEFAULT 1,
  description TEXT NULL,
  PRIMARY KEY (id),
  KEY idx_yachts_city (city),
  KEY idx_yachts_active (active),
  KEY idx_yachts_visible_agent (visible_agent)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Yacht images
CREATE TABLE IF NOT EXISTS yacht_images (
  id INT UNSIGNED NOT NULL AUTO_INCREMENT,
  yacht_id INT UNSIGNED NOT NULL,
  file_path VARCHAR(500) NOT NULL,
  is_cover TINYINT(1) NOT NULL DEFAULT 0,
  PRIMARY KEY (id),
  KEY idx_yi_yacht (yacht_id),
  CONSTRAINT fk_yi_yacht FOREIGN KEY (yacht_id) REFERENCES yachts(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Yacht bookings (professional, with audit & payment fields)
CREATE TABLE IF NOT EXISTS yacht_bookings (
  id INT UNSIGNED NOT NULL AUTO_INCREMENT,
  booking_code VARCHAR(100) NULL,
  agent_id INT UNSIGNED NOT NULL,
  yacht_id INT UNSIGNED NOT NULL,
  trip_date DATE NOT NULL,
  hours INT UNSIGNED NOT NULL,
  guests INT UNSIGNED NOT NULL,
  amount_total DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  currency CHAR(3) NOT NULL DEFAULT 'THB',
  status ENUM('pending','confirmed','cancelled','refunded') NOT NULL DEFAULT 'pending',
  payment_status ENUM('unpaid','paid','refunded','partial','failed') NOT NULL DEFAULT 'unpaid',
  payment_method VARCHAR(50) NULL,
  gateway_name VARCHAR(50) NULL,
  payment_txn_id VARCHAR(100) NULL,
  paid_at DATETIME NULL,
  -- Cancellation / Refunds
  cancelled_at DATETIME NULL,
  cancel_reason VARCHAR(500) NULL,
  refund_amount DECIMAL(12,2) NULL,
  refunded_at DATETIME NULL,
  refund_txn_id VARCHAR(100) NULL,
  -- Audit
  channel VARCHAR(50) NULL,
  ip_address VARCHAR(64) NULL,
  user_agent VARCHAR(255) NULL,
  created_by INT UNSIGNED NULL,
  updated_by INT UNSIGNED NULL,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NOT NULL,
  PRIMARY KEY (id),
  UNIQUE KEY uq_yacht_booking_code (booking_code),
  KEY idx_yb_agent (agent_id),
  KEY idx_yb_yacht (yacht_id),
  KEY idx_yb_status (status),
  KEY idx_yb_paid (payment_status),
  CONSTRAINT fk_yb_yacht FOREIGN KEY (yacht_id) REFERENCES yachts(id) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Yacht booking events (immutable audit trail)
CREATE TABLE IF NOT EXISTS yacht_booking_events (
  id INT UNSIGNED NOT NULL AUTO_INCREMENT,
  booking_id INT UNSIGNED NOT NULL,
  user_id INT UNSIGNED NULL,
  event_type VARCHAR(50) NOT NULL,
  note VARCHAR(255) NULL,
  data_json JSON NULL,
  created_at DATETIME NOT NULL,
  PRIMARY KEY (id),
  KEY idx_ybe_booking (booking_id),
  KEY idx_ybe_type (event_type),
  CONSTRAINT fk_ybe_booking FOREIGN KEY (booking_id) REFERENCES yacht_bookings(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Yacht payments (optionally separate from bookings)
CREATE TABLE IF NOT EXISTS yacht_payments (
  id INT UNSIGNED NOT NULL AUTO_INCREMENT,
  booking_id INT UNSIGNED NOT NULL,
  method VARCHAR(50) NOT NULL,
  gateway_name VARCHAR(50) NULL,
  amount DECIMAL(12,2) NOT NULL,
  currency CHAR(3) NOT NULL DEFAULT 'THB',
  status ENUM('pending','captured','failed','refunded','partial') NOT NULL DEFAULT 'pending',
  txn_id VARCHAR(100) NULL,
  idempotency_key VARCHAR(64) NULL,
  provider_ref VARCHAR(100) NULL,
  session_url TEXT NULL,
  raw_session JSON NULL,
  event_id VARCHAR(100) NULL,
  raw_event JSON NULL,
  meta_json JSON NULL,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NOT NULL,
  PRIMARY KEY (id),
  KEY idx_yp_booking (booking_id),
  KEY idx_yp_status (status),
  KEY idx_yp_event (event_id),
  CONSTRAINT fk_yp_booking FOREIGN KEY (booking_id) REFERENCES yacht_bookings(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Yacht payment intents (pre-booking payments)
CREATE TABLE IF NOT EXISTS yacht_payment_intents (
  id INT UNSIGNED NOT NULL AUTO_INCREMENT,
  agent_id INT UNSIGNED NOT NULL,
  yacht_id INT UNSIGNED NOT NULL,
  trip_date DATE NOT NULL,
  hours INT UNSIGNED NOT NULL,
  guests INT UNSIGNED NOT NULL,
  amount DECIMAL(12,2) NOT NULL,
  currency CHAR(3) NOT NULL DEFAULT 'THB',
  status ENUM('pending','succeeded','failed') NOT NULL DEFAULT 'pending',
  method VARCHAR(50) NULL,
  gateway_name VARCHAR(50) NULL,
  gateway_session_id VARCHAR(100) NULL,
  booking_id INT UNSIGNED NULL,
  -- Audit
  channel VARCHAR(50) NULL,
  ip_address VARCHAR(64) NULL,
  user_agent VARCHAR(255) NULL,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NOT NULL,
  PRIMARY KEY (id),
  KEY idx_yi_agent (agent_id),
  KEY idx_yi_yacht (yacht_id),
  KEY idx_yi_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

COMMIT;
