-- Seed a demo order for a B2B Agent and related wallet ledger entries
-- Safe guards: only runs if a B2B Agent exists and no confirmed/completed order exists yet

USE b2b_travel;

-- Pick an agent user (first B2B Agent)
SET @uid := (SELECT id FROM users WHERE role = 'B2B Agent' ORDER BY id ASC LIMIT 1);

-- Ensure wallet exists for that user
INSERT INTO wallets (user_id, balance)
SELECT @uid, 0.00
WHERE @uid IS NOT NULL
  AND NOT EXISTS (SELECT 1 FROM wallets WHERE user_id = @uid);

-- Create a demo confirmed booking if none exists yet for this agent
INSERT INTO bookings (user_id, module, item_id, pax, price, status)
SELECT @uid, 'activity', 1, 2, 1500.00, 'confirmed'
WHERE @uid IS NOT NULL
  AND NOT EXISTS (
    SELECT 1 FROM bookings WHERE user_id = @uid AND status IN ('confirmed','completed')
  );

-- Capture the most recent booking id for this agent (the one we just created or an existing one)
SET @bid := (SELECT id FROM bookings WHERE user_id = @uid ORDER BY id DESC LIMIT 1);

-- Wallet id for this agent
SET @wid := (SELECT id FROM wallets WHERE user_id = @uid LIMIT 1);

-- If booking and wallet exist, seed ledger: top-up credit then payment debit (net zero)
-- Credit top-up (approved)
INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta)
SELECT @wid, 'credit', 1500.00, 'manual', 'approved', JSON_OBJECT('note','Demo top-up for order','booking_id', @bid)
WHERE @wid IS NOT NULL AND @bid IS NOT NULL
  AND NOT EXISTS (
    SELECT 1 FROM wallet_ledger wl WHERE wl.wallet_id=@wid AND JSON_EXTRACT(wl.meta, '$.booking_id') = CAST(@bid AS JSON)
  );

-- Debit payment (approved)
INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta)
SELECT @wid, 'debit', 1500.00, 'manual', 'approved', JSON_OBJECT('note','Demo payment for order','booking_id', @bid)
WHERE @wid IS NOT NULL AND @bid IS NOT NULL
  AND NOT EXISTS (
    SELECT 1 FROM wallet_ledger wl WHERE wl.wallet_id=@wid AND wl.type='debit' AND JSON_EXTRACT(wl.meta, '$.booking_id') = CAST(@bid AS JSON)
  );

-- Keep wallet balance unchanged (top-up minus payment); if you want to reflect the credit, uncomment below
-- UPDATE wallets SET balance = balance + 1500.00 - 1500.00 WHERE id = @wid;
