-- ================= B2B Agent Profiles & KYC =================
CREATE TABLE IF NOT EXISTS agent_profiles (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  company VARCHAR(150) NULL,
  phone VARCHAR(50) NULL,
  commission_percent DECIMAL(5,2) NOT NULL DEFAULT 0.00,
  credit_limit DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  business_type ENUM('freelancer','sole_proprietor','partnership','company') NOT NULL DEFAULT 'freelancer',
  agency_name VARCHAR(150) NULL,
  profile_photo_path VARCHAR(255) NULL,
  title VARCHAR(20) NULL,
  country VARCHAR(100) NULL,
  state VARCHAR(100) NULL,
  city VARCHAR(100) NULL,
  pincode VARCHAR(20) NULL,
  address_line VARCHAR(255) NULL,
  gst_number VARCHAR(30) NULL,
  gst_company VARCHAR(150) NULL,
  iata_registered TINYINT(1) NOT NULL DEFAULT 0,
  iata_code VARCHAR(20) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NULL,
  UNIQUE KEY uniq_agent_profile (user_id),
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Business documents per agent (registration cert, partnership deed, incorporation cert, GST cert, selfie, address proof)
CREATE TABLE IF NOT EXISTS agent_business_docs (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  doc_type ENUM('selfie','registration_certificate','partnership_deed','incorporation_certificate','gst_certificate','address_proof') NOT NULL,
  file_path VARCHAR(255) NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS agent_kyc (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  id_type ENUM('passport','national_id','driving_license','other') NOT NULL,
  id_number VARCHAR(64) NOT NULL,
  doc_front_path VARCHAR(255) NOT NULL,
  doc_back_path VARCHAR(255) NULL,
  status ENUM('pending','approved','rejected') NOT NULL DEFAULT 'pending',
  remarks VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  verified_at TIMESTAMP NULL,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Ensure Customer role exists in users table (for environments created before Customer role)
ALTER TABLE users MODIFY role ENUM('Admin','Employee','Channel Partner','B2B Agent','Vendor','Customer') NOT NULL DEFAULT 'Employee';

-- Database: b2b_travel
CREATE DATABASE IF NOT EXISTS b2b_travel CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE b2b_travel;

-- Users and roles
CREATE TABLE IF NOT EXISTS users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL,
  email VARCHAR(150) NOT NULL UNIQUE,
  password VARCHAR(255) NOT NULL,
  role ENUM('Admin','Employee','Channel Partner','B2B Agent','Vendor') NOT NULL DEFAULT 'Employee',
  status ENUM('Active','Inactive') NOT NULL DEFAULT 'Active',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Vendors
CREATE TABLE IF NOT EXISTS vendors (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(150) NOT NULL,
  contact_email VARCHAR(150)
) ENGINE=InnoDB;

-- Hotels
CREATE TABLE IF NOT EXISTS hotels (
  id INT AUTO_INCREMENT PRIMARY KEY,
  vendor_id INT NOT NULL,
  name VARCHAR(200) NOT NULL,
  city VARCHAR(100) NOT NULL,
  country VARCHAR(100) NOT NULL,
  base_price DECIMAL(10,2) NOT NULL DEFAULT 0,
  active TINYINT(1) NOT NULL DEFAULT 1,
  stars TINYINT UNSIGNED NOT NULL DEFAULT 0,
  visible_customer TINYINT(1) NOT NULL DEFAULT 1,
  visible_agent TINYINT(1) NOT NULL DEFAULT 1,
  featured_customer TINYINT(1) NOT NULL DEFAULT 0,
  featured_agent TINYINT(1) NOT NULL DEFAULT 0,
  FOREIGN KEY (vendor_id) REFERENCES vendors(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Backfill for existing databases where hotels table already exists without stars
ALTER TABLE hotels
  ADD COLUMN IF NOT EXISTS stars TINYINT UNSIGNED NOT NULL DEFAULT 0 AFTER base_price;

-- Backfill for existing databases where hotels table already exists without active
ALTER TABLE hotels
  ADD COLUMN IF NOT EXISTS active TINYINT(1) NOT NULL DEFAULT 1 AFTER base_price;

-- Backfill for per-audience visibility/featured flags
ALTER TABLE hotels
  ADD COLUMN IF NOT EXISTS visible_customer TINYINT(1) NOT NULL DEFAULT 1 AFTER stars,
  ADD COLUMN IF NOT EXISTS visible_agent TINYINT(1) NOT NULL DEFAULT 1 AFTER visible_customer,
  ADD COLUMN IF NOT EXISTS featured_customer TINYINT(1) NOT NULL DEFAULT 0 AFTER visible_agent,
  ADD COLUMN IF NOT EXISTS featured_agent TINYINT(1) NOT NULL DEFAULT 0 AFTER featured_customer;

-- Hotel Rooms (room types)
CREATE TABLE IF NOT EXISTS hotel_rooms (
  id INT AUTO_INCREMENT PRIMARY KEY,
  hotel_id INT NOT NULL,
  name VARCHAR(150) NOT NULL,
  occupancy_adults INT NOT NULL DEFAULT 2,
  occupancy_children INT NOT NULL DEFAULT 0,
  size_sqm INT NULL,
  bed_info VARCHAR(120) NULL,
  active TINYINT(1) NOT NULL DEFAULT 1,
  FOREIGN KEY (hotel_id) REFERENCES hotels(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Rate Plans
CREATE TABLE IF NOT EXISTS rate_plans (
  id INT AUTO_INCREMENT PRIMARY KEY,
  hotel_id INT NOT NULL,
  name VARCHAR(120) NOT NULL,
  meal_plan ENUM('RO','BB','HB','FB') NOT NULL DEFAULT 'RO',
  refundable TINYINT(1) NOT NULL DEFAULT 1,
  refund_window_days INT NULL,
  advance_purchase_days INT NULL,
  active TINYINT(1) NOT NULL DEFAULT 1,
  FOREIGN KEY (hotel_id) REFERENCES hotels(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Daily Allocations (inventory and rules)
CREATE TABLE IF NOT EXISTS room_allocations (
  id INT AUTO_INCREMENT PRIMARY KEY,
  room_id INT NOT NULL,
  date DATE NOT NULL,
  allotment INT NOT NULL DEFAULT 0,
  min_stay INT NULL,
  max_stay INT NULL,
  closed TINYINT(1) NOT NULL DEFAULT 0,
  cta TINYINT(1) NOT NULL DEFAULT 0, -- closed to arrival
  ctd TINYINT(1) NOT NULL DEFAULT 0, -- closed to departure
  UNIQUE KEY uniq_alloc (room_id, date),
  FOREIGN KEY (room_id) REFERENCES hotel_rooms(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Daily Prices per Room + Rate Plan
CREATE TABLE IF NOT EXISTS room_prices (
  id INT AUTO_INCREMENT PRIMARY KEY,
  room_id INT NOT NULL,
  rate_plan_id INT NOT NULL,
  date DATE NOT NULL,
  price DECIMAL(10,2) NOT NULL,
  tax_included TINYINT(1) NOT NULL DEFAULT 1,
  UNIQUE KEY uniq_price (room_id, rate_plan_id, date),
  FOREIGN KEY (room_id) REFERENCES hotel_rooms(id) ON DELETE CASCADE,
  FOREIGN KEY (rate_plan_id) REFERENCES rate_plans(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Hotel Policies
CREATE TABLE IF NOT EXISTS hotel_policies (
  id INT AUTO_INCREMENT PRIMARY KEY,
  hotel_id INT NOT NULL,
  checkin_time VARCHAR(10) NULL,
  checkout_time VARCHAR(10) NULL,
  child_policy TEXT NULL,
  pet_policy TEXT NULL,
  terms_html TEXT NULL,
  FOREIGN KEY (hotel_id) REFERENCES hotels(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Images for hotel and rooms
CREATE TABLE IF NOT EXISTS hotel_images (
  id INT AUTO_INCREMENT PRIMARY KEY,
  hotel_id INT NULL,
  room_id INT NULL,
  file_path VARCHAR(255) NOT NULL,
  caption VARCHAR(150) NULL,
  sort_order INT NULL,
  FOREIGN KEY (hotel_id) REFERENCES hotels(id) ON DELETE CASCADE,
  FOREIGN KEY (room_id) REFERENCES hotel_rooms(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Hotel Bookings (detailed)
CREATE TABLE IF NOT EXISTS hotel_bookings (
  id INT AUTO_INCREMENT PRIMARY KEY,
  hotel_id INT NOT NULL,
  vendor_id INT NOT NULL,
  booking_code VARCHAR(40) NOT NULL,
  checkin DATE NOT NULL,
  checkout DATE NOT NULL,
  nights INT NOT NULL,
  room_id INT NOT NULL,
  rate_plan_id INT NULL,
  rooms_qty INT NOT NULL DEFAULT 1,
  guests_json JSON NULL,
  customer_name VARCHAR(150) NOT NULL,
  customer_email VARCHAR(150) NULL,
  status ENUM('pending','confirmed','cancelled') NOT NULL DEFAULT 'pending',
  total_price DECIMAL(12,2) NOT NULL,
  currency CHAR(3) NOT NULL DEFAULT 'THB',
  payment_status ENUM('unpaid','paid','refunded') NOT NULL DEFAULT 'unpaid',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  UNIQUE KEY uniq_booking_code (booking_code),
  FOREIGN KEY (hotel_id) REFERENCES hotels(id) ON DELETE CASCADE,
  FOREIGN KEY (vendor_id) REFERENCES vendors(id) ON DELETE CASCADE,
  FOREIGN KEY (room_id) REFERENCES hotel_rooms(id) ON DELETE CASCADE,
  FOREIGN KEY (rate_plan_id) REFERENCES rate_plans(id) ON DELETE SET NULL
) ENGINE=InnoDB;

-- Activities
CREATE TABLE IF NOT EXISTS activities (
  id INT AUTO_INCREMENT PRIMARY KEY,
  vendor_id INT NOT NULL,
  name VARCHAR(200) NOT NULL,
  city VARCHAR(100) NOT NULL,
  base_price DECIMAL(10,2) NOT NULL DEFAULT 0,
  FOREIGN KEY (vendor_id) REFERENCES vendors(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Taxi
CREATE TABLE IF NOT EXISTS taxis (
  id INT AUTO_INCREMENT PRIMARY KEY,
  vendor_id INT NOT NULL,
  name VARCHAR(200) NOT NULL,
  route VARCHAR(200) NOT NULL,
  base_price DECIMAL(10,2) NOT NULL DEFAULT 0,
  FOREIGN KEY (vendor_id) REFERENCES vendors(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Extend taxis with professional fields (non-destructive; keep existing cols)
ALTER TABLE taxis
  ADD COLUMN IF NOT EXISTS title VARCHAR(200) NULL,
  ADD COLUMN IF NOT EXISTS vehicle_type ENUM('sedan','suv','van','bus','luxury','other') NULL,
  ADD COLUMN IF NOT EXISTS capacity INT NULL,
  ADD COLUMN IF NOT EXISTS luggage_capacity INT NULL,
  ADD COLUMN IF NOT EXISTS description TEXT NULL,
  ADD COLUMN IF NOT EXISTS active TINYINT(1) NOT NULL DEFAULT 1;

-- Series number for taxis (unique human-friendly code)
ALTER TABLE taxis
  ADD COLUMN IF NOT EXISTS series_no VARCHAR(32) NULL,
  ADD UNIQUE KEY IF NOT EXISTS uniq_taxi_series (series_no);

-- Detailed media for taxis
CREATE TABLE IF NOT EXISTS taxi_images (
  id INT AUTO_INCREMENT PRIMARY KEY,
  taxi_id INT NOT NULL,
  file_path VARCHAR(255) NOT NULL,
  is_cover TINYINT(1) NOT NULL DEFAULT 0,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (taxi_id) REFERENCES taxis(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Routes: pickup/drop with optional coordinates
CREATE TABLE IF NOT EXISTS taxi_routes (
  id INT AUTO_INCREMENT PRIMARY KEY,
  taxi_id INT NOT NULL,
  pickup_name VARCHAR(150) NOT NULL,
  pickup_lat DECIMAL(10,7) NULL,
  pickup_lng DECIMAL(10,7) NULL,
  drop_name VARCHAR(150) NOT NULL,
  drop_lat DECIMAL(10,7) NULL,
  drop_lng DECIMAL(10,7) NULL,
  notes VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (taxi_id) REFERENCES taxis(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Pricing per taxi (supports models)
CREATE TABLE IF NOT EXISTS taxi_pricing (
  id INT AUTO_INCREMENT PRIMARY KEY,
  taxi_id INT NOT NULL,
  model ENUM('fixed','distance','hourly') NOT NULL DEFAULT 'fixed',
  currency CHAR(3) NOT NULL DEFAULT 'THB',
  base_fare DECIMAL(10,2) NOT NULL DEFAULT 0.00,
  per_km DECIMAL(10,2) NULL,
  per_hour DECIMAL(10,2) NULL,
  wait_per_hour DECIMAL(10,2) NULL,
  min_km DECIMAL(10,2) NULL,
  min_hours DECIMAL(10,2) NULL,
  tax_pct DECIMAL(5,2) NULL,
  fee_pct DECIMAL(5,2) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (taxi_id) REFERENCES taxis(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Availability controls per taxi
CREATE TABLE IF NOT EXISTS taxi_availability (
  id INT AUTO_INCREMENT PRIMARY KEY,
  taxi_id INT NOT NULL,
  dow JSON NULL,              -- e.g., {"mon":{"on":1,"start":"06:00","end":"23:00"}, ...}
  blackout_dates JSON NULL,   -- ["2025-12-31", ...]
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (taxi_id) REFERENCES taxis(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- eVisa
CREATE TABLE IF NOT EXISTS evisas (
  id INT AUTO_INCREMENT PRIMARY KEY,
  vendor_id INT NOT NULL,
  country VARCHAR(100) NOT NULL,
  base_price DECIMAL(10,2) NOT NULL DEFAULT 0,
  FOREIGN KEY (vendor_id) REFERENCES vendors(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Wallets and ledger
CREATE TABLE IF NOT EXISTS wallets (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  balance DECIMAL(12,2) NOT NULL DEFAULT 0,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS wallet_ledger (
  id INT AUTO_INCREMENT PRIMARY KEY,
  wallet_id INT NOT NULL,
  type ENUM('credit','debit') NOT NULL,
  amount DECIMAL(12,2) NOT NULL,
  method ENUM('stripe','manual') NOT NULL,
  status ENUM('pending','approved','rejected') NOT NULL DEFAULT 'pending',
  meta JSON NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (wallet_id) REFERENCES wallets(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Bookings
CREATE TABLE IF NOT EXISTS bookings (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  module ENUM('hotel','activity','taxi','evisa') NOT NULL,
  item_id INT NOT NULL,
  pax INT NOT NULL DEFAULT 1,
  price DECIMAL(12,2) NOT NULL,
  status ENUM('pending','confirmed','completed','cancelled') NOT NULL DEFAULT 'pending',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Support Tickets
CREATE TABLE IF NOT EXISTS tickets (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  subject VARCHAR(200) NOT NULL,
  message TEXT NOT NULL,
  status ENUM('open','closed') NOT NULL DEFAULT 'open',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Promo Codes
CREATE TABLE IF NOT EXISTS promo_codes (
  id INT AUTO_INCREMENT PRIMARY KEY,
  code VARCHAR(50) UNIQUE NOT NULL,
  discount_type ENUM('flat','percent') NOT NULL,
  discount_value DECIMAL(10,2) NOT NULL,
  valid_from DATE NULL,
  valid_to DATE NULL,
  usage_limit INT NULL
) ENGINE=InnoDB;

-- Corporate Rates
CREATE TABLE IF NOT EXISTS corporate_rates (
  id INT AUTO_INCREMENT PRIMARY KEY,
  company_name VARCHAR(150) NOT NULL,
  discount_percent DECIMAL(5,2) NOT NULL DEFAULT 0
) ENGINE=InnoDB;

-- Calendars and per-date pricing
CREATE TABLE IF NOT EXISTS calendars (
  id INT AUTO_INCREMENT PRIMARY KEY,
  module ENUM('hotel','activity','taxi','evisa') NOT NULL,
  item_id INT NOT NULL,
  date DATE NOT NULL,
  price DECIMAL(10,2) NULL,
  blackout TINYINT(1) NOT NULL DEFAULT 0,
  UNIQUE KEY unique_date (module, item_id, date)
) ENGINE=InnoDB;

-- Group Pricing
CREATE TABLE IF NOT EXISTS group_pricing (
  id INT AUTO_INCREMENT PRIMARY KEY,
  module ENUM('hotel','activity','taxi','evisa') NOT NULL,
  item_id INT NOT NULL,
  pax_threshold INT NOT NULL,
  group_price DECIMAL(10,2) NOT NULL
) ENGINE=InnoDB;

-- Commissions for channel partners
CREATE TABLE IF NOT EXISTS partner_commissions (
  id INT AUTO_INCREMENT PRIMARY KEY,
  booking_id INT NOT NULL,
  partner_user_id INT NOT NULL,
  amount DECIMAL(12,2) NOT NULL,
  status ENUM('pending','paid') NOT NULL DEFAULT 'pending',
  FOREIGN KEY (booking_id) REFERENCES bookings(id) ON DELETE CASCADE,
  FOREIGN KEY (partner_user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- API Tokens
CREATE TABLE IF NOT EXISTS api_tokens (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  token VARCHAR(64) NOT NULL UNIQUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Audit Logs
CREATE TABLE IF NOT EXISTS audit_logs (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NULL,
  ip VARCHAR(45) NULL,
  action VARCHAR(150) NOT NULL,
  details TEXT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Sample Data
INSERT INTO users (name, email, password, role) VALUES
('Admin', 'admin@example.com', '$2y$10$2yhashforadminreplace', 'Admin');

-- For sample: password = admin123 (generate and replace hash after import)

INSERT INTO vendors (name, contact_email) VALUES ('Sample Vendor', 'vendor@example.com');
INSERT INTO hotels (vendor_id, name, city, country, base_price) VALUES (1, 'Sample Hotel', 'Bangkok', 'Thailand', 80.00);
INSERT INTO activities (vendor_id, name, city, base_price) VALUES (1, 'Temple Tour', 'Bangkok', 30.00);
INSERT INTO taxis (vendor_id, name, route, base_price) VALUES (1, 'Airport Transfer', 'BKK -> City', 25.00);
INSERT INTO evisas (vendor_id, country, base_price) VALUES (1, 'Thailand', 50.00);

-- ================= Additional Essential Tables =================

-- User sessions (optional persistent sessions)
CREATE TABLE IF NOT EXISTS user_sessions (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  session_id VARCHAR(128) NOT NULL,
  user_agent VARCHAR(255) NULL,
  ip VARCHAR(45) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  last_seen TIMESTAMP NULL,
  UNIQUE KEY uniq_user_session (user_id, session_id),
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- IP action logs
CREATE TABLE IF NOT EXISTS ip_action_logs (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NULL,
  ip VARCHAR(45) NULL,
  action VARCHAR(150) NOT NULL,
  meta JSON NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
) ENGINE=InnoDB;

-- Channel partners
CREATE TABLE IF NOT EXISTS channel_partners (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(150) NOT NULL,
  contact_email VARCHAR(150) NULL,
  status ENUM('Active','Inactive') NOT NULL DEFAULT 'Active',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Ensure column for secure partner transaction password
ALTER TABLE channel_partners
  ADD COLUMN IF NOT EXISTS transaction_password_hash VARCHAR(255) NULL;

CREATE TABLE IF NOT EXISTS channel_partner_wallet (
  id INT AUTO_INCREMENT PRIMARY KEY,
  partner_id INT NOT NULL,
  balance DECIMAL(12,2) NOT NULL DEFAULT 0,
  FOREIGN KEY (partner_id) REFERENCES channel_partners(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS channel_partner_wallet_ledger (
  id INT AUTO_INCREMENT PRIMARY KEY,
  wallet_id INT NOT NULL,
  type ENUM('credit','debit') NOT NULL,
  amount DECIMAL(12,2) NOT NULL,
  method ENUM('manual','transfer') NOT NULL,
  status ENUM('pending','approved','rejected') NOT NULL DEFAULT 'pending',
  meta JSON NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (wallet_id) REFERENCES channel_partner_wallet(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS channel_partner_commissions (
  id INT AUTO_INCREMENT PRIMARY KEY,
  partner_id INT NOT NULL,
  booking_id INT NOT NULL,
  amount DECIMAL(12,2) NOT NULL,
  status ENUM('pending','approved','paid') NOT NULL DEFAULT 'pending',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (partner_id) REFERENCES channel_partners(id) ON DELETE CASCADE,
  FOREIGN KEY (booking_id) REFERENCES bookings(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Channel Partner Settings (granular controls per partner)
CREATE TABLE IF NOT EXISTS channel_partner_settings (
  id INT AUTO_INCREMENT PRIMARY KEY,
  partner_id INT NOT NULL,
  -- Wallet transfer controls
  allow_wallet_transfer TINYINT(1) NOT NULL DEFAULT 1,
  require_tx_password TINYINT(1) NOT NULL DEFAULT 1,
  wallet_transfer_daily_limit DECIMAL(12,2) NULL,
  -- Agent management controls
  allow_agent_creation TINYINT(1) NOT NULL DEFAULT 1,
  agent_creation_limit INT NULL,
  allow_agent_password_change TINYINT(1) NOT NULL DEFAULT 0,
  -- API access control
  api_enabled TINYINT(1) NOT NULL DEFAULT 0,
  -- Pricing controls
  allow_extra_margin TINYINT(1) NOT NULL DEFAULT 0,
  max_margin_percent DECIMAL(5,2) NULL,
  max_flat_markup DECIMAL(10,2) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NULL,
  UNIQUE KEY uniq_partner (partner_id),
  FOREIGN KEY (partner_id) REFERENCES channel_partners(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Vendor helpers
CREATE TABLE IF NOT EXISTS vendor_users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  vendor_id INT NOT NULL,
  user_id INT NOT NULL,
  role VARCHAR(50) DEFAULT 'Manager',
  UNIQUE KEY uniq_vendor_user (vendor_id, user_id),
  FOREIGN KEY (vendor_id) REFERENCES vendors(id) ON DELETE CASCADE,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS vendor_bank_accounts (
  id INT AUTO_INCREMENT PRIMARY KEY,
  vendor_id INT NOT NULL,
  bank_name VARCHAR(120) NULL,
  account_name VARCHAR(120) NULL,
  account_number VARCHAR(64) NULL,
  swift_code VARCHAR(32) NULL,
  json_details JSON NULL,
  FOREIGN KEY (vendor_id) REFERENCES vendors(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Locations and currencies
CREATE TABLE IF NOT EXISTS locations (
  id INT AUTO_INCREMENT PRIMARY KEY,
  country VARCHAR(100) NOT NULL,
  city VARCHAR(100) NULL,
  timezone VARCHAR(64) NULL
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS currencies (
  id INT AUTO_INCREMENT PRIMARY KEY,
  code CHAR(3) NOT NULL UNIQUE,
  rate_to_base DECIMAL(12,6) NOT NULL DEFAULT 1.000000,
  updated_at TIMESTAMP NULL
) ENGINE=InnoDB;

-- Activities advanced pricing
CREATE TABLE IF NOT EXISTS activity_tiers (
  id INT AUTO_INCREMENT PRIMARY KEY,
  activity_id INT NOT NULL,
  name VARCHAR(100) NOT NULL,
  description VARCHAR(255) NULL,
  FOREIGN KEY (activity_id) REFERENCES activities(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS activity_showtimes (
  id INT AUTO_INCREMENT PRIMARY KEY,
  activity_id INT NOT NULL,
  show_time TIME NOT NULL,
  FOREIGN KEY (activity_id) REFERENCES activities(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS activity_prices (
  id INT AUTO_INCREMENT PRIMARY KEY,
  activity_id INT NOT NULL,
  tier_id INT NULL,
  showtime_id INT NULL,
  pax_type ENUM('adult','child','infant') NOT NULL DEFAULT 'adult',
  is_group TINYINT(1) NOT NULL DEFAULT 0,
  pax_threshold INT NULL,
  price DECIMAL(10,2) NOT NULL,
  FOREIGN KEY (activity_id) REFERENCES activities(id) ON DELETE CASCADE,
  FOREIGN KEY (tier_id) REFERENCES activity_tiers(id) ON DELETE SET NULL,
  FOREIGN KEY (showtime_id) REFERENCES activity_showtimes(id) ON DELETE SET NULL
) ENGINE=InnoDB;

-- Hotels advanced
CREATE TABLE IF NOT EXISTS hotel_rooms (
  id INT AUTO_INCREMENT PRIMARY KEY,
  hotel_id INT NOT NULL,
  name VARCHAR(120) NOT NULL,
  capacity INT NOT NULL DEFAULT 2,
  FOREIGN KEY (hotel_id) REFERENCES hotels(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS hotel_room_prices (
  id INT AUTO_INCREMENT PRIMARY KEY,
  room_id INT NOT NULL,
  date DATE NULL,
  is_group TINYINT(1) NOT NULL DEFAULT 0,
  pax_threshold INT NULL,
  price DECIMAL(10,2) NOT NULL,
  FOREIGN KEY (room_id) REFERENCES hotel_rooms(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS hotel_gallery (
  id INT AUTO_INCREMENT PRIMARY KEY,
  hotel_id INT NOT NULL,
  image_path VARCHAR(255) NOT NULL,
  caption VARCHAR(150) NULL,
  FOREIGN KEY (hotel_id) REFERENCES hotels(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Taxi advanced
CREATE TABLE IF NOT EXISTS taxi_companies (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(150) NOT NULL,
  support_phone VARCHAR(50) NULL
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS taxi_prices (
  id INT AUTO_INCREMENT PRIMARY KEY,
  taxi_id INT NOT NULL,
  company_id INT NULL,
  is_group TINYINT(1) NOT NULL DEFAULT 0,
  pax_threshold INT NULL,
  price DECIMAL(10,2) NOT NULL,
  FOREIGN KEY (taxi_id) REFERENCES taxis(id) ON DELETE CASCADE,
  FOREIGN KEY (company_id) REFERENCES taxi_companies(id) ON DELETE SET NULL
) ENGINE=InnoDB;

-- eVisa advanced
CREATE TABLE IF NOT EXISTS evisa_types (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(120) NOT NULL,
  country VARCHAR(100) NOT NULL
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS evisa_docs (
  id INT AUTO_INCREMENT PRIMARY KEY,
  type_id INT NOT NULL,
  doc_name VARCHAR(120) NOT NULL,
  required TINYINT(1) NOT NULL DEFAULT 1,
  FOREIGN KEY (type_id) REFERENCES evisa_types(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS evisa_applications (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  type_id INT NOT NULL,
  status ENUM('draft','submitted','approved','rejected') NOT NULL DEFAULT 'draft',
  data JSON NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  FOREIGN KEY (type_id) REFERENCES evisa_types(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Packages
CREATE TABLE IF NOT EXISTS packages (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(150) NOT NULL,
  description TEXT NULL
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS package_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  package_id INT NOT NULL,
  module ENUM('hotel','activity','taxi','evisa') NOT NULL,
  item_id INT NOT NULL,
  qty INT NOT NULL DEFAULT 1,
  FOREIGN KEY (package_id) REFERENCES packages(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Wallet requests (topups/withdrawals initiation)
CREATE TABLE IF NOT EXISTS wallet_requests (
  id INT AUTO_INCREMENT PRIMARY KEY,
  wallet_id INT NOT NULL,
  type ENUM('topup','withdraw') NOT NULL,
  amount DECIMAL(12,2) NOT NULL,
  status ENUM('pending','approved','rejected') NOT NULL DEFAULT 'pending',
  meta JSON NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (wallet_id) REFERENCES wallets(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Support module (separate from simple tickets table)
CREATE TABLE IF NOT EXISTS support_tickets (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  subject VARCHAR(200) NOT NULL,
  status ENUM('open','closed') NOT NULL DEFAULT 'open',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS support_messages (
  id INT AUTO_INCREMENT PRIMARY KEY,
  ticket_id INT NOT NULL,
  user_id INT NULL,
  message TEXT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (ticket_id) REFERENCES support_tickets(id) ON DELETE CASCADE,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS attachments (
  id INT AUTO_INCREMENT PRIMARY KEY,
  ticket_id INT NULL,
  message_id INT NULL,
  file_path VARCHAR(255) NOT NULL,
  mime VARCHAR(100) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (ticket_id) REFERENCES support_tickets(id) ON DELETE CASCADE,
  FOREIGN KEY (message_id) REFERENCES support_messages(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Calendar extras
CREATE TABLE IF NOT EXISTS calendar_overrides (
  id INT AUTO_INCREMENT PRIMARY KEY,
  module ENUM('hotel','activity','taxi','evisa') NOT NULL,
  item_id INT NOT NULL,
  date DATE NOT NULL,
  price DECIMAL(10,2) NULL,
  UNIQUE KEY uniq_override (module, item_id, date)
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS blackout_windows (
  id INT AUTO_INCREMENT PRIMARY KEY,
  module ENUM('hotel','activity','taxi','evisa') NOT NULL,
  item_id INT NOT NULL,
  start_date DATE NOT NULL,
  end_date DATE NOT NULL
) ENGINE=InnoDB;

-- Promo
CREATE TABLE IF NOT EXISTS promo_redemptions (
  id INT AUTO_INCREMENT PRIMARY KEY,
  promo_id INT NOT NULL,
  user_id INT NOT NULL,
  booking_id INT NULL,
  redeemed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (promo_id) REFERENCES promo_codes(id) ON DELETE CASCADE,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  FOREIGN KEY (booking_id) REFERENCES bookings(id) ON DELETE SET NULL
) ENGINE=InnoDB;

-- Corporate
CREATE TABLE IF NOT EXISTS corporate_accounts (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(150) NOT NULL,
  status ENUM('Active','Inactive') NOT NULL DEFAULT 'Active'
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS corporate_mappings (
  id INT AUTO_INCREMENT PRIMARY KEY,
  account_id INT NOT NULL,
  user_id INT NOT NULL,
  FOREIGN KEY (account_id) REFERENCES corporate_accounts(id) ON DELETE CASCADE,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Communications logs
CREATE TABLE IF NOT EXISTS email_logs (
  id INT AUTO_INCREMENT PRIMARY KEY,
  to_email VARCHAR(150) NOT NULL,
  subject VARCHAR(200) NULL,
  body MEDIUMTEXT NULL,
  meta JSON NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS whatsapp_logs (
  id INT AUTO_INCREMENT PRIMARY KEY,
  to_number VARCHAR(30) NOT NULL,
  message TEXT NULL,
  meta JSON NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Ensure bookings table has financial columns per spec
ALTER TABLE bookings
  ADD COLUMN IF NOT EXISTS vendor_cost DECIMAL(12,2) NULL,
  ADD COLUMN IF NOT EXISTS agent_price DECIMAL(12,2) NULL,
  ADD COLUMN IF NOT EXISTS promo_applied TINYINT(1) NOT NULL DEFAULT 0,
  ADD COLUMN IF NOT EXISTS group_applied TINYINT(1) NOT NULL DEFAULT 0,
  ADD COLUMN IF NOT EXISTS profit_system DECIMAL(12,2) NULL,
  ADD COLUMN IF NOT EXISTS partner_commission DECIMAL(12,2) NULL;

-- Extend room_prices to support explicit vendor/agent/customer pricing
ALTER TABLE room_prices
  ADD COLUMN IF NOT EXISTS vendor_cost DECIMAL(10,2) NULL AFTER price,
  ADD COLUMN IF NOT EXISTS agent_price DECIMAL(10,2) NULL AFTER vendor_cost,
  ADD COLUMN IF NOT EXISTS customer_price DECIMAL(10,2) NULL AFTER agent_price;

-- ================= Vendor Packages (module-agnostic; used for activities) =================
CREATE TABLE IF NOT EXISTS vendor_packages (
  id INT AUTO_INCREMENT PRIMARY KEY,
  vendor_id INT NOT NULL,
  name VARCHAR(150) NOT NULL,
  module ENUM('activity','hotel','taxi','evisa') NOT NULL DEFAULT 'activity',
  requires_show_time TINYINT(1) NOT NULL DEFAULT 1,
  age_policy ENUM('adult_only','family') NOT NULL DEFAULT 'adult_only',
  address_override TEXT NULL,
  active TINYINT(1) NOT NULL DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NULL DEFAULT NULL,
  FOREIGN KEY (vendor_id) REFERENCES vendors(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS vendor_package_variants (
  id INT AUTO_INCREMENT PRIMARY KEY,
  package_id INT NOT NULL,
  name VARCHAR(100) NOT NULL,
  notes TEXT NULL,
  active TINYINT(1) NOT NULL DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NULL DEFAULT NULL,
  FOREIGN KEY (package_id) REFERENCES vendor_packages(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS vendor_package_prices (
  id INT AUTO_INCREMENT PRIMARY KEY,
  package_id INT NOT NULL,
  variant_id INT NOT NULL,
  price_type ENUM('base','group_tier') NOT NULL DEFAULT 'base',
  pax_type ENUM('adult','child','flat') NOT NULL DEFAULT 'adult',
  min_quantity INT NULL,
  vendor_cost DECIMAL(10,2) NOT NULL,
  agent_price DECIMAL(10,2) NOT NULL,
  customer_price DECIMAL(10,2) NULL,
  customer_promo_price DECIMAL(10,2) NULL,
  promo_start_at DATETIME NULL,
  promo_end_at DATETIME NULL,
  currency CHAR(3) NOT NULL DEFAULT 'THB',
  pickup_type ENUM('none','included','optional') NOT NULL DEFAULT 'none',
  pickup_scope ENUM('pickup_only','pickup_and_drop') NOT NULL DEFAULT 'pickup_and_drop',
  pickup_radius_km DECIMAL(4,1) NULL,
  pickup_fee DECIMAL(10,2) NULL,
  pickup_notes VARCHAR(255) NULL,
  active TINYINT(1) NOT NULL DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NULL DEFAULT NULL,
  FOREIGN KEY (package_id) REFERENCES vendor_packages(id) ON DELETE CASCADE,
  FOREIGN KEY (variant_id) REFERENCES vendor_package_variants(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS vendor_package_showtimes (
  id INT AUTO_INCREMENT PRIMARY KEY,
  package_id INT NOT NULL,
  variant_id INT NULL,
  time VARCHAR(8) NOT NULL,
  active TINYINT(1) NOT NULL DEFAULT 1,
  FOREIGN KEY (package_id) REFERENCES vendor_packages(id) ON DELETE CASCADE,
  FOREIGN KEY (variant_id) REFERENCES vendor_package_variants(id) ON DELETE SET NULL
) ENGINE=InnoDB;
