<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;

class AgentController extends Controller
{
    public function index(): void
    {
        Auth::requireRole(['B2B Agent']);
        $this->view('agent/index', ['title' => 'Agent Dashboard']);
    }

    public function activities(): void
    {
        Auth::requireRole(['B2B Agent']);
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        $q = isset($_GET['q']) ? trim((string)$_GET['q']) : '';
        $city = isset($_GET['city']) ? trim((string)$_GET['city']) : '';
        // Render Activities List (two identical versions; toggle via ?v=2)
        $v = isset($_GET['v']) ? (int)$_GET['v'] : 1;
        $tpl = ($v === 2) ? 'agent/activities_list_v2' : 'agent/activities_list_v1';
        $this->view($tpl, [
            'q' => $q,
            'city' => $city,
            'title' => 'Activities',
        ]);
    }

    public function taxis(): void
    {
        Auth::requireRole(['B2B Agent']);
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        $q = isset($_GET['q']) ? trim((string)$_GET['q']) : '';
        // Accept alternate param names from Agent landing search: pickup => from, city => to
        $from = isset($_GET['from']) ? trim((string)$_GET['from']) : '';
        if ($from === '' && isset($_GET['pickup'])) { $from = trim((string)$_GET['pickup']); }
        $to = isset($_GET['to']) ? trim((string)$_GET['to']) : '';
        if ($to === '' && isset($_GET['city'])) { $to = trim((string)$_GET['city']); }
        $vehicleType = isset($_GET['vehicle_type']) ? trim((string)$_GET['vehicle_type']) : '';
        $minCapacity = isset($_GET['min_capacity']) ? (int)$_GET['min_capacity'] : 0;

        $this->view('agent/taxis_list', [
            'q' => $q,
            'from' => $from,
            'to' => $to,
            'vehicle_type' => $vehicleType,
            'min_capacity' => $minCapacity,
            'title' => 'Taxis',
        ]);
    }

    public function hotels(): void
    {
        Auth::requireRole(['B2B Agent']);
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        $city = isset($_GET['city']) ? trim((string)$_GET['city']) : '';
        $checkin = isset($_GET['checkin']) ? trim((string)$_GET['checkin']) : '';
        $checkout = isset($_GET['checkout']) ? trim((string)$_GET['checkout']) : '';
        $adults = isset($_GET['adults']) ? (int)$_GET['adults'] : 2;
        $rooms = isset($_GET['rooms']) ? (int)$_GET['rooms'] : 1;

        $this->view('agent/hotels_list', [
            'city' => $city,
            'checkin' => $checkin,
            'checkout' => $checkout,
            'adults' => $adults,
            'rooms' => $rooms,
            'title' => 'Hotels',
        ]);
    }

    public function packages(): void
    {
        Auth::requireRole(['B2B Agent']);
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        $q = isset($_GET['q']) ? trim((string)$_GET['q']) : '';
        $city = isset($_GET['city']) ? trim((string)$_GET['city']) : '';
        // Reuse activities search list UX for packages until dedicated filters are defined
        $this->view('agent/packages_list', [
            'q' => $q,
            'city' => $city,
            'title' => 'Packages',
        ]);
    }

    public function yachts(): void
    {
        Auth::requireRole(['B2B Agent']);
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        $q = isset($_GET['q']) ? trim((string)$_GET['q']) : '';
        $city = isset($_GET['city']) ? trim((string)$_GET['city']) : '';
        $minCapacity = isset($_GET['min_capacity']) ? (int)$_GET['min_capacity'] : 0;
        $maxGuests = isset($_GET['max_guests']) ? (int)$_GET['max_guests'] : 0;

        $this->view('agent/yachts_list', [
            'q' => $q,
            'city' => $city,
            'min_capacity' => $minCapacity,
            'max_guests' => $maxGuests,
            'title' => 'Yachts',
        ]);
    }

    public function wallet(): void
    {
        Auth::requireRole(['B2B Agent']);
        $user = $_SESSION['user'] ?? null;
        $userId = (int)($user['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }

        // Balance
        $stmt = $this->pdo->prepare('SELECT balance FROM wallets WHERE user_id = :u LIMIT 1');
        $stmt->execute(['u' => $userId]);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);
        $balance = $row ? (float)$row['balance'] : 0.0;

        // Ledger (last 50)
        $ledger = [];
        $stmtL = $this->pdo->prepare('SELECT wl.id, wl.type, wl.amount, wl.method, wl.status, wl.meta, wl.created_at
                                       FROM wallet_ledger wl
                                       JOIN wallets w ON w.id = wl.wallet_id
                                       WHERE w.user_id = :u
                                       ORDER BY wl.id DESC
                                       LIMIT 50');
        $stmtL->execute(['u' => $userId]);
        $ledger = $stmtL->fetchAll();

        // Bookings (last 50)
        $stmtB = $this->pdo->prepare('SELECT id, module, item_id, pax, price, status, created_at
                                       FROM bookings
                                       WHERE user_id = :u
                                       ORDER BY id DESC
                                       LIMIT 50');
        $stmtB->execute(['u' => $userId]);
        $bookings = $stmtB->fetchAll();

        $this->view('agent/wallet', compact('balance','ledger','bookings'));
    }

    public function bookings(): void
    {
        Auth::requireRole(['B2B Agent']);
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        // Pending (unpaid) = status 'pending'
        $stmtP = $this->pdo->prepare("SELECT 
                                             id, module, item_id, pax, price, status, created_at,
                                             CASE module
                                               WHEN 'activity' THEN (SELECT p.name FROM vendor_packages p WHERE p.id = item_id)
                                               WHEN 'hotel'    THEN (SELECT h.name FROM hotels h WHERE h.id = item_id)
                                               WHEN 'taxi'     THEN (SELECT t.name FROM taxis t WHERE t.id = item_id)
                                               WHEN 'evisa'    THEN (SELECT CONCAT('eVisa ', ev.country) FROM evisas ev WHERE ev.id = item_id)
                                               ELSE NULL
                                             END AS item_name
                                           FROM bookings
                                           WHERE user_id = :u AND status = 'pending'
                                           ORDER BY id DESC
                                           LIMIT 100");
        $stmtP->execute(['u' => $userId]);
        $pending = $stmtP->fetchAll();

        // Paid/Orders = status confirmed or completed
        $stmtPaid = $this->pdo->prepare("SELECT 
                                              id, module, item_id, pax, price, status, created_at,
                                              CASE module
                                                WHEN 'activity' THEN (SELECT p.name FROM vendor_packages p WHERE p.id = item_id)
                                                WHEN 'hotel'    THEN (SELECT h.name FROM hotels h WHERE h.id = item_id)
                                                WHEN 'taxi'     THEN (SELECT t.name FROM taxis t WHERE t.id = item_id)
                                                WHEN 'evisa'    THEN (SELECT CONCAT('eVisa ', ev.country) FROM evisas ev WHERE ev.id = item_id)
                                                ELSE NULL
                                              END AS item_name
                                              FROM bookings
                                              WHERE user_id = :u AND status IN ('confirmed','completed')
                                              ORDER BY id DESC
                                              LIMIT 100");
        $stmtPaid->execute(['u' => $userId]);
        $paid = $stmtPaid->fetchAll();

        $this->view('agent/bookings', [
            'pending' => $pending,
            'paid' => $paid,
        ]);
    }

    public function bookingPay(): void
    {
        Auth::requireRole(['B2B Agent']);
        $user = $_SESSION['user'] ?? null;
        $userId = (int)($user['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        $bookingId = (int)($_GET['id'] ?? 0);
        if ($bookingId <= 0) { $_SESSION['flash'] = 'Invalid booking id.'; $this->redirect('/agent/bookings'); }

        // Fetch booking and ensure it belongs to user and is pending
        $stmt = $this->pdo->prepare('SELECT * FROM bookings WHERE id = :id AND user_id = :u LIMIT 1');
        $stmt->execute(['id' => $bookingId, 'u' => $userId]);
        $booking = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$booking) { $_SESSION['flash'] = 'Booking not found.'; $this->redirect('/agent/bookings'); }
        if ((string)$booking['status'] !== 'pending') { $_SESSION['flash'] = 'This booking is not pending payment.'; $this->redirect('/agent/bookings'); }

        // If an order already exists for this booking, reuse it
        $orderId = null;
        try {
            // Try to find an existing order that has this booking linked via order_items.booking_id
            $q = $this->pdo->prepare('SELECT oi.order_id FROM order_items oi WHERE oi.booking_id = :bid LIMIT 1');
            $q->execute(['bid' => $bookingId]);
            $orderId = (int)($q->fetchColumn() ?: 0);
        } catch (\Throwable $e) {
            $orderId = 0; // table may not exist in older envs; continue to create one
        }

        if ($orderId <= 0) {
            // Create an order and order_item for this booking, mapping contact details if available
            $this->pdo->beginTransaction();
            try {
                $currency = 'THB';
                // Extract contact info from booking.details_json if present
                $contactName = null; $contactMobile = null; $contactEmail = null; $contactWhatsApp = null;
                try {
                    $det = json_decode((string)($booking['details_json'] ?? 'null'), true);
                    if (is_array($det)) {
                        if (isset($det['contact']) && is_array($det['contact'])) {
                            $contactName = trim((string)($det['contact']['name'] ?? '')) ?: null;
                            $contactMobile = trim((string)($det['contact']['mobile'] ?? '')) ?: null;
                            $contactEmail = trim((string)($det['contact']['email'] ?? '')) ?: null;
                            $contactWhatsApp = trim((string)($det['contact']['whatsapp'] ?? '')) ?: null;
                        }
                    }
                } catch (\Throwable $e) { /* ignore */ }

                // Try inserting with customer_* columns first; fallback to minimal insert if columns are absent
                try {
                    $insOrder = $this->pdo->prepare('INSERT INTO orders (user_id, status, total_amount, currency, payment_method, customer_name, customer_mobile, customer_email, customer_whatsapp) VALUES (:uid, "pending", :total, :cur, "none", :cname, :cmobile, :cemail, :cwa)');
                    $insOrder->execute([
                        'uid' => $userId,
                        'total' => (float)$booking['price'],
                        'cur' => $currency,
                        'cname' => $contactName,
                        'cmobile' => $contactMobile,
                        'cemail' => $contactEmail,
                        'cwa' => $contactWhatsApp,
                    ]);
                } catch (\Throwable $e) {
                    $insOrder = $this->pdo->prepare('INSERT INTO orders (user_id, status, total_amount, currency, payment_method) VALUES (:uid, "pending", :total, :cur, "none")');
                    $insOrder->execute([
                        'uid' => $userId,
                        'total' => (float)$booking['price'],
                        'cur' => $currency,
                    ]);
                }
                $orderId = (int)$this->pdo->lastInsertId();

                // Compute unit price from pax if possible
                $pax = max(1, (int)($booking['pax'] ?? 1));
                $lineTotal = (float)$booking['price'];
                $unitPrice = $pax > 0 ? ($lineTotal / $pax) : $lineTotal;
                $insItem = $this->pdo->prepare('INSERT INTO order_items (order_id, module, item_id, qty, unit_price, line_total, currency, booking_id) VALUES (:oid, :m, :iid, :q, :up, :lt, :cur, :bid)');
                $insItem->execute([
                    'oid' => $orderId,
                    'm' => (string)$booking['module'],
                    'iid' => (int)$booking['item_id'],
                    'q' => $pax,
                    'up' => $unitPrice,
                    'lt' => $lineTotal,
                    'cur' => $currency,
                    'bid' => $bookingId,
                ]);

                // Link booking to order (if column exists)
                try {
                    $this->pdo->prepare('UPDATE bookings SET order_id = :oid WHERE id = :id')->execute(['oid' => $orderId, 'id' => $bookingId]);
                } catch (\Throwable $e) { /* ignore if column missing */ }

                $this->pdo->commit();
            } catch (\Throwable $e) {
                if ($this->pdo->inTransaction()) { $this->pdo->rollBack(); }
                $_SESSION['flash'] = 'Failed to initiate checkout: ' . $e->getMessage();
                $this->redirect('/agent/bookings');
            }
        }

        // Generate signed URL and redirect to agent checkout
        $appKey = function_exists('env') ? (string)env('APP_KEY', '') : '';
        if ($appKey === '') { $appKey = getenv('APP_KEY') ?: 'devkey'; }
        $sig = hash_hmac('sha256', $orderId . ':' . $userId, $appKey);
        $url = '/agent/checkout/pay?id=' . urlencode((string)$orderId) . '&sig=' . urlencode($sig);
        $this->redirect($url);
    }

    public function bookingView(): void
    {
        Auth::requireRole(['B2B Agent']);
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $_SESSION['flash'] = 'Invalid booking id.'; $this->redirect('/agent/bookings'); }

        // Fetch booking ensuring it belongs to current user
        $stmt = $this->pdo->prepare('SELECT * FROM bookings WHERE id = :id AND user_id = :u LIMIT 1');
        $stmt->execute(['id' => $id, 'u' => $userId]);
        $booking = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$booking) { $_SESSION['flash'] = 'Booking not found.'; $this->redirect('/agent/bookings'); }

        // Resolve item detail by module
        $item = null; $module = (string)$booking['module'];
        if ($module === 'activity') {
            $sql = 'SELECT p.id, p.name, p.thumbnail_path, v.name AS vendor_name, v.city
                    FROM vendor_packages p JOIN vendors v ON v.id = p.vendor_id WHERE p.id = :id';
            $q = $this->pdo->prepare($sql); $q->execute(['id' => (int)$booking['item_id']]);
            $item = $q->fetch(\PDO::FETCH_ASSOC) ?: null;
        } elseif ($module === 'hotel') {
            $q = $this->pdo->prepare('SELECT id, name, city, stars FROM hotels WHERE id = :id');
            $q->execute(['id' => (int)$booking['item_id']]); $item = $q->fetch(\PDO::FETCH_ASSOC) ?: null;
        } elseif ($module === 'taxi') {
            $q = $this->pdo->prepare('SELECT id, name, route FROM taxis WHERE id = :id');
            $q->execute(['id' => (int)$booking['item_id']]); $item = $q->fetch(\PDO::FETCH_ASSOC) ?: null;
        } elseif ($module === 'evisa') {
            $q = $this->pdo->prepare('SELECT id, country FROM evisas WHERE id = :id');
            $q->execute(['id' => (int)$booking['item_id']]); $r = $q->fetch(\PDO::FETCH_ASSOC) ?: null; if ($r) { $r['name'] = 'eVisa ' . ($r['country'] ?? ''); } $item = $r;
        }

        // Variant and price meta (activities)
        $variant = null; $priceRow = null;
        if ($module === 'activity') {
            $vid = (int)($booking['variant_id'] ?? 0);
            $pid = (int)($booking['price_id'] ?? 0);
            if ($vid > 0) {
                try {
                    $s = $this->pdo->prepare('SELECT id, name FROM vendor_package_variants WHERE id = :id');
                    $s->execute(['id' => $vid]);
                    $variant = $s->fetch(\PDO::FETCH_ASSOC) ?: null;
                } catch (\Throwable $e) { /* ignore */ }
            }
            if ($pid > 0) {
                try {
                    $s2 = $this->pdo->prepare('SELECT id, price_type, pax_type, min_quantity, agent_price, currency FROM vendor_package_prices WHERE id = :id');
                    $s2->execute(['id' => $pid]);
                    $priceRow = $s2->fetch(\PDO::FETCH_ASSOC) ?: null;
                } catch (\Throwable $e) { /* ignore */ }
            }
        }

        // Compute pax breakdown and unit price
        $pax = max(1, (int)($booking['pax'] ?? 1));
        $lineTotal = (float)($booking['price'] ?? 0);
        $unitPrice = $pax > 0 ? ($lineTotal / $pax) : $lineTotal;
        $breakdown = [
            'pax' => $pax,
            'pax_type' => (string)($booking['pax_type'] ?? ''),
            'adults' => isset($booking['adults']) ? (int)$booking['adults'] : null,
            'children' => isset($booking['children']) ? (int)$booking['children'] : null,
            'infants' => isset($booking['infants']) ? (int)$booking['infants'] : null,
            'show_time' => $booking['show_time'] ?? null,
        ];

        // Find linked order (if any)
        $orderId = null;
        try {
            $qo = $this->pdo->prepare('SELECT oi.order_id FROM order_items oi WHERE oi.booking_id = :bid LIMIT 1');
            $qo->execute(['bid' => $id]);
            $orderId = (int)($qo->fetchColumn() ?: 0);
        } catch (\Throwable $e) { $orderId = 0; }
        if ($orderId <= 0) {
            try {
                $orderId = (int)($booking['order_id'] ?? 0);
            } catch (\Throwable $e) { /* ignore */ }
        }

        $this->view('agent/booking_view', [
            'booking' => $booking,
            'item' => $item,
            'orderId' => $orderId > 0 ? $orderId : null,
            'variant' => $variant,
            'priceRow' => $priceRow,
            'unitPrice' => $unitPrice,
            'breakdown' => $breakdown,
        ]);
    }

    // Create pending booking for Activities from B2B Agent activity-detail form
    public function createActivityBooking(): void
    {
        Auth::requireRole(['B2B Agent']);
        Security::requireCsrf();
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }

        $packageId = (int)($_POST['package_id'] ?? 0);
        $variantId = isset($_POST['variant_id']) && $_POST['variant_id'] !== '' ? (int)$_POST['variant_id'] : null;
        $priceId   = (int)($_POST['price_id'] ?? 0);
        $date      = trim((string)($_POST['date'] ?? ''));
        $pax       = max(1, (int)($_POST['pax'] ?? 1));
        $showTime  = trim((string)($_POST['show_time'] ?? ''));
        if ($packageId <= 0 || $priceId <= 0 || $date === '') {
            $_SESSION['flash'] = 'Please fill all required fields.';
            $this->redirect('/agent');
        }

        // Validate package and price
        $pkg = null; $priceRow = null;
        try {
            $s = $this->pdo->prepare("SELECT p.id, p.name, v.module FROM vendor_packages p JOIN vendors v ON v.id = p.vendor_id WHERE p.id = :id AND p.active = 1 LIMIT 1");
            $s->execute(['id' => $packageId]);
            $pkg = $s->fetch(\PDO::FETCH_ASSOC) ?: null;
            if (!$pkg || ($pkg['module'] ?? '') !== 'activity') { throw new \RuntimeException('Invalid activity package'); }

            $sp = $this->pdo->prepare("SELECT id, price_type, pax_type, min_quantity, agent_price, customer_price, price, currency FROM vendor_package_prices WHERE id = :pid AND package_id = :pkg AND active = 1 LIMIT 1");
            $sp->execute(['pid' => $priceId, 'pkg' => $packageId]);
            $priceRow = $sp->fetch(\PDO::FETCH_ASSOC) ?: null;
            if (!$priceRow) { throw new \RuntimeException('Invalid price option'); }
        } catch (\Throwable $e) {
            $_SESSION['flash'] = 'Selected activity or price is not available.';
            $this->redirect('/agent');
        }

        // Apply min quantity if defined
        $minQty = (int)($priceRow['min_quantity'] ?? 0);
        if ($minQty > 0 && $pax < $minQty) { $pax = $minQty; }

        // Agent-first unit price
        $unit = $priceRow['agent_price'] !== null ? (float)$priceRow['agent_price'] : (( $priceRow['customer_price'] !== null ? (float)$priceRow['customer_price'] : (float)$priceRow['price']));
        $lineTotal = $unit * max(1, $pax);

        // Insert booking as pending
        $this->pdo->beginTransaction();
        try {
            $sql = "INSERT INTO bookings (user_id, module, item_id, variant_id, price_id, pax, price, pax_type, show_time, status, created_at)
                    VALUES (:uid, 'activity', :item, :variant, :price_id, :pax, :price, :pax_type, :stime, 'pending', NOW())";
            $ins = $this->pdo->prepare($sql);
            $ins->execute([
                'uid' => $userId,
                'item' => $packageId,
                'variant' => $variantId,
                'price_id' => $priceId,
                'pax' => $pax,
                'price' => $lineTotal,
                'pax_type' => (string)($priceRow['pax_type'] ?? ''),
                'stime' => $showTime !== '' ? $showTime : null,
            ]);
            $bookingId = (int)$this->pdo->lastInsertId();
            $this->pdo->commit();
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) { $this->pdo->rollBack(); }
            $_SESSION['flash'] = 'Failed to create booking.';
            $this->redirect('/agent');
        }

        // Redirect to existing pay flow
        $this->redirect('/agent/booking/pay?id='.(string)$bookingId);
    }

    // Create pending booking for Hotels from B2B Agent hotel-detail form
    public function createHotelBooking(): void
    {
        Auth::requireRole(['B2B Agent']);
        Security::requireCsrf();
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }

        $hotelId = (int)($_POST['hotel_id'] ?? 0);
        $checkIn = trim((string)($_POST['check_in_date'] ?? ''));
        $checkOut = trim((string)($_POST['check_out_date'] ?? ''));
        $rooms = max(1, (int)($_POST['rooms'] ?? 1));
        $roomId = isset($_POST['room_id']) && $_POST['room_id'] !== '' ? (int)$_POST['room_id'] : null;
        $adults = isset($_POST['adults']) ? max(1, (int)$_POST['adults']) : null;
        $children = isset($_POST['children']) ? max(0, (int)$_POST['children']) : null;

        if ($hotelId <= 0 || $checkIn === '' || $checkOut === '') {
            $_SESSION['flash'] = 'Please fill all required fields.';
            $this->redirect('/agent');
        }

        // Validate dates and compute nights (support dd/mm/YYYY from flatpickr)
        $parseDate = function(string $s){
            $s = trim($s);
            if ($s === '') return null;
            // Try common formats
            $fmts = ['d/m/Y','d-m-Y','Y-m-d','m/d/Y'];
            foreach ($fmts as $f) { $dt = \DateTime::createFromFormat($f, $s); if ($dt instanceof \DateTime) { return $dt; } }
            // Fallback: strtotime best-effort
            $ts = strtotime($s);
            if ($ts !== false) { return (new \DateTime())->setTimestamp($ts); }
            return null;
        };
        $d1 = $parseDate($checkIn);
        $d2 = $parseDate($checkOut);
        if (!$d1 || !$d2) {
            $_SESSION['errors'][] = 'Invalid date format.';
            $this->redirect('/agent');
        }
        $diff = (int)$d1->diff($d2)->days;
        if ($diff <= 0) {
            $_SESSION['errors'][] = 'Check-out must be after check-in.';
            $this->redirect('/agent');
        }

        // Enforce policy: no same-day or past check-ins (check-in must be at least tomorrow)
        try {
            $today = new \DateTime('today');
            $tomorrowDt = (clone $today)->modify('+1 day');
            // Compare by date (ignore time)
            $ciDateOnly = new \DateTime($d1->format('Y-m-d'));
            if ($ciDateOnly < $tomorrowDt) {
                $_SESSION['errors'][] = 'Check-in must be at least tomorrow.';
                $this->redirect('/agent');
            }
        } catch (\Throwable $e) { /* ignore and continue */ }

        // Fetch hotel and base price
        $hotel = null;
        try {
            $q = $this->pdo->prepare('SELECT id, name, base_price FROM hotels WHERE id = :id LIMIT 1');
            $q->execute(['id' => $hotelId]);
            $hotel = $q->fetch(\PDO::FETCH_ASSOC) ?: null;
        } catch (\Throwable $e) { $hotel = null; }
        if (!$hotel) {
            $_SESSION['flash'] = 'Hotel not found.';
            $this->redirect('/agent');
        }

        // Try compute nightly price from room prices when roomId provided
        $perNight = [];
        if ($roomId && $diff > 0) {
            // Room Prices v2 (preferred)
            try {
                $sql = "SELECT rp.date, COALESCE(rp.agent_price, rp.price) AS price FROM room_prices rp WHERE rp.room_id = :rid AND rp.date >= :cin AND rp.date < :cout ORDER BY rp.date ASC";
                $s = $this->pdo->prepare($sql);
                $s->execute(['rid' => $roomId, 'cin' => $d1->format('Y-m-d'), 'cout' => $d2->format('Y-m-d')]);
                $rows = $s->fetchAll(\PDO::FETCH_ASSOC) ?: [];
                foreach ($rows as $r) { $perNight[$r['date']] = (float)($r['price'] ?? 0); }
            } catch (\Throwable $e) { /* ignore */ }
            // Legacy hotel_room_prices fallback
            if (count($perNight) === 0) {
                try {
                    $sql2 = "SELECT date, price FROM hotel_room_prices WHERE room_id = :rid AND date >= :cin AND date < :cout ORDER BY date ASC";
                    $s2 = $this->pdo->prepare($sql2);
                    $s2->execute(['rid' => $roomId, 'cin' => $d1->format('Y-m-d'), 'cout' => $d2->format('Y-m-d')]);
                    $rows2 = $s2->fetchAll(\PDO::FETCH_ASSOC) ?: [];
                    foreach ($rows2 as $r) { $perNight[$r['date']] = (float)($r['price'] ?? 0); }
                } catch (\Throwable $e) { /* ignore */ }
            }
        }

        // Build the date range and sum pricing
        $sum = 0.0; $dateList = [];
        for ($i = 0; $i < $diff; $i++) {
            $d = (clone $d1)->modify("+{$i} day");
            $ds = $d->format('Y-m-d');
            $dateList[] = $ds;
            if (!empty($perNight)) {
                $sum += (float)($perNight[$ds] ?? 0.0);
            }
        }

        if (empty($perNight)) {
            // Fallback: Agent-first price = base_price per night per room
            $unitNight = (float)($hotel['base_price'] ?? 0);
            if ($unitNight < 0) { $unitNight = 0; }
            $sum = $unitNight * $diff;
        }

        // Total across rooms quantity
        $lineTotal = $sum * max(1, $rooms);

        // Compute expected guest counts
        // Prefer detailed selections_json (array of {room_id, qty, cap}) to derive per-room capacities
        $defaultCap = 2; // sensible default in absence of explicit capacity
        $personsPerRoom = [];
        $roomNamesByIndex = []; // 1-based index => room name for presentation
        $selectionsJson = isset($_POST['selections_json']) ? (string)$_POST['selections_json'] : '';
        if ($selectionsJson !== '') {
            try {
                $parsed = json_decode($selectionsJson, true);
                if (is_array($parsed)) {
                    $roomIndexCursor = 1; // 1-based
                    foreach ($parsed as $entry) {
                        $qty = isset($entry['qty']) ? (int)$entry['qty'] : 0;
                        $cap = isset($entry['cap']) && (int)$entry['cap'] > 0 ? (int)$entry['cap'] : $defaultCap;
                        $selRoomId = isset($entry['room_id']) ? (int)$entry['room_id'] : 0;
                        $selRoomName = null;
                        if ($selRoomId > 0) {
                            try {
                                $qrn = $this->pdo->prepare('SELECT name FROM hotel_rooms WHERE id = :rid LIMIT 1');
                                $qrn->execute(['rid' => $selRoomId]);
                                $nm = $qrn->fetchColumn();
                                if ($nm !== false && $nm !== null) { $selRoomName = (string)$nm; }
                            } catch (\Throwable $e) { /* ignore */ }
                        }
                        for ($i = 0; $i < $qty; $i++) {
                            $personsPerRoom[] = $cap;
                            if ($selRoomName !== null && $selRoomName !== '') {
                                $roomNamesByIndex[$roomIndexCursor] = $selRoomName;
                            }
                            $roomIndexCursor++;
                        }
                    }
                }
            } catch (\Throwable $e) { $personsPerRoom = []; $roomNamesByIndex = []; }
        }
        // Fallback to single room capacity if no selections_json provided
        if (empty($personsPerRoom)) {
            $capacity = null;
            if ($roomId) {
                try {
                    $qr = $this->pdo->prepare('SELECT capacity FROM hotel_rooms WHERE id = :rid LIMIT 1');
                    $qr->execute(['rid' => $roomId]);
                    $capVal = $qr->fetchColumn();
                    if ($capVal !== false && $capVal !== null) { $capacity = (int)$capVal; }
                } catch (\Throwable $e) { $capacity = null; }
            }
            $capEach = ($capacity && $capacity > 0) ? $capacity : $defaultCap;
            for ($i = 0; $i < max(1, (int)$rooms); $i++) { $personsPerRoom[] = $capEach; }
        }
        // Adults initially equal total persons; children can be chosen later on the form
        $computedAdults = array_sum(array_map('intval', $personsPerRoom));
        if (is_int($adults) && $adults > 0) { $computedAdults = $adults; } // honor explicit adults if sent
        $computedChildren = (is_int($children) && $children > 0) ? $children : 0;

        // Set pax to total guests (adults + children). For now children defaults to 0 as selectors were removed
        $pax = max(1, (int)($computedAdults + $computedChildren));

        // Prepare details JSON
        // Build 1-based maps for persons and adults per room
        $personsPerRoomMap = [];
        foreach ($personsPerRoom as $idx => $cap) { $personsPerRoomMap[$idx+1] = (int)$cap; }
        // Initially set adults_per_room equal to persons_per_room; user can set some to child later
        $adultsPerRoom = $personsPerRoomMap;

        $details = [
            'check_in_date' => $d1->format('Y-m-d'),
            'check_out_date' => $d2->format('Y-m-d'),
            'nights' => $diff,
            'rooms' => $rooms,
            'pricing_basis' => empty($perNight) ? 'base_price_per_night_per_room' : 'room_prices_per_night',
            'room_id' => $roomId,
            // Store computed guests; adults default to total persons, children 0 by default
            'guests' => [ 'adults' => $computedAdults, 'children' => $computedChildren ],
            // Hint for guest-info UI to pre-target per-room capacities and initial adults-per-room
            'rooms_breakdown' => [
                'persons_per_room' => $personsPerRoomMap,
                'adults_per_room' => $adultsPerRoom,
            ],
            // Optional: expanded room names for each selected room index (1-based) when multiple room types were selected
            'room_names_by_index' => !empty($roomNamesByIndex) ? $roomNamesByIndex : null,
            'dates' => $dateList,
        ];

        // Persist guest names list if provided by the form (JSON array of {title, first_name, last_name, age})
        $guestsListJson = isset($_POST['guests_list_json']) ? (string)$_POST['guests_list_json'] : '';
        $parsedGuests = [];
        if ($guestsListJson !== '') {
            try {
                $gl = json_decode($guestsListJson, true);
                if (json_last_error() === JSON_ERROR_NONE && is_array($gl)) {
                    $parsedGuests = $gl;
                    $details['guests_list'] = $parsedGuests;
                }
            } catch (\Throwable $e) { /* ignore malformed */ }
        }

        // Server-side validation: require guest names present and complete matching pax
        $expectedGuests = max(1, (int)($pax ?? 0));
        $validCount = 0;
        $hasEmpty = false;
        if (is_array($parsedGuests)) {
            foreach ($parsedGuests as $g) {
                $first = trim((string)($g['first_name'] ?? ($g['fname'] ?? '')));
                $last  = trim((string)($g['last_name'] ?? ($g['lname'] ?? '')));
                if ($first === '' && $last === '') { $hasEmpty = true; }
                else { $validCount++; }
            }
        }
        if ($expectedGuests > 0 && ($validCount !== $expectedGuests || $hasEmpty)) {
            $_SESSION['errors'][] = 'Please enter all guest names ('.$expectedGuests.').';
            $this->redirect('/agent/hotel?id='.(string)$hotelId);
        }

        // Optionally persist contact details and special requests (to enrich admin view and order mapping later)
        $contactName = isset($_POST['contact_name']) ? trim((string)$_POST['contact_name']) : '';
        $contactEmail = isset($_POST['contact_email']) ? trim((string)$_POST['contact_email']) : '';
        $contactMobile = isset($_POST['contact_mobile']) ? trim((string)$_POST['contact_mobile']) : '';
        $contactWhatsapp = isset($_POST['contact_whatsapp']) ? trim((string)$_POST['contact_whatsapp']) : '';
        $specialRequests = isset($_POST['special_requests']) ? (string)$_POST['special_requests'] : '';
        if ($contactName !== '' || $contactEmail !== '' || $contactMobile !== '' || $contactWhatsapp !== '') {
            $details['contact'] = [
                'name' => $contactName !== '' ? $contactName : null,
                'email' => $contactEmail !== '' ? $contactEmail : null,
                'mobile' => $contactMobile !== '' ? $contactMobile : null,
                'whatsapp' => $contactWhatsapp !== '' ? $contactWhatsapp : null,
            ];
        }
        if ($specialRequests !== '') {
            $details['special_requests'] = $specialRequests;
        }

        $this->pdo->beginTransaction();
        try {
            // Try insert including details_json column if exists
            $sql = "INSERT INTO bookings (user_id, module, item_id, pax, price, status, details_json, created_at)
                    VALUES (:uid, 'hotel', :item, :pax, :price, 'pending', :details, NOW())";
            try {
                $ins = $this->pdo->prepare($sql);
                $ins->execute([
                    'uid' => $userId,
                    'item' => $hotelId,
                    'pax' => $pax,
                    'price' => $lineTotal,
                    'details' => json_encode($details, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
                ]);
            } catch (\Throwable $e) {
                // Fallback: insert without details_json
                $sql2 = "INSERT INTO bookings (user_id, module, item_id, pax, price, status, created_at)
                         VALUES (:uid, 'hotel', :item, :pax, :price, 'pending', NOW())";
                $ins2 = $this->pdo->prepare($sql2);
                $ins2->execute([
                    'uid' => $userId,
                    'item' => $hotelId,
                    'pax' => $pax,
                    'price' => $lineTotal,
                ]);
            }
            $bookingId = (int)$this->pdo->lastInsertId();
            // Best-effort: persist adults/children columns if they exist in bookings table
            try {
                $this->pdo->prepare('UPDATE bookings SET adults = :a, children = :c WHERE id = :id')->execute([
                    'a' => $computedAdults,
                    'c' => $computedChildren,
                    'id' => $bookingId,
                ]);
            } catch (\Throwable $e) { /* ignore if columns missing */ }
            $this->pdo->commit();
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) { $this->pdo->rollBack(); }
            $_SESSION['flash'] = 'Failed to create booking.';
            $this->redirect('/agent');
        }

        // Redirect to payment
        $this->redirect('/agent/booking/guest-info?id='.(string)$bookingId);
    }

    public function orders(): void
    {
        Auth::requireRole(['B2B Agent']);
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        // Consider orders as bookings with confirmed/completed statuses
        $stmt = $this->pdo->prepare("SELECT id, module, item_id, pax, price, status, created_at FROM bookings WHERE user_id = :u AND status IN ('confirmed','completed') ORDER BY id DESC LIMIT 100");
        $stmt->execute(['u' => $userId]);
        $orders = $stmt->fetchAll();
        $this->view('agent/orders', ['orders' => $orders]);
    }

    public function support(): void
    {
        Auth::requireRole(['B2B Agent']);
        $this->view('agent/support', []);
    }

    public function activityDetail(): void
    {
        Auth::requireRole(['B2B Agent']);
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $_SESSION['flash'] = 'Invalid activity.'; $this->redirect('/agent'); }
        $this->view('agent/activity_detail', [
            'id' => $id,
            'title' => 'Activity Details',
        ]);
    }

    public function hotelDetail(): void
    {
        Auth::requireRole(['B2B Agent']);
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $_SESSION['flash'] = 'Invalid hotel.'; $this->redirect('/agent/hotels'); }
        $checkin = isset($_GET['checkin']) ? trim((string)$_GET['checkin']) : '';
        $checkout = isset($_GET['checkout']) ? trim((string)$_GET['checkout']) : '';
        $adults = isset($_GET['adults']) ? (int)$_GET['adults'] : null;
        $rooms = isset($_GET['rooms']) ? (int)$_GET['rooms'] : null;
        $children = isset($_GET['children']) ? (int)$_GET['children'] : null;
        $roomId = isset($_GET['room_id']) && $_GET['room_id'] !== '' ? (int)$_GET['room_id'] : null;
        $this->view('agent/hotel_detail', [
            'id' => $id,
            'checkin' => $checkin,
            'checkout' => $checkout,
            'adults' => $adults,
            'rooms' => $rooms,
            'children' => $children,
            'pref_room_id' => $roomId,
            'title' => 'Hotel Details',
        ]);
    }

    public function orderView(): void
    {
        Auth::requireRole(['B2B Agent']);
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $_SESSION['flash'] = 'Invalid order id.'; $this->redirect('/agent/orders'); }

        // Fetch booking ensuring it belongs to current user
        $stmt = $this->pdo->prepare('SELECT * FROM bookings WHERE id = :id AND user_id = :u LIMIT 1');
        $stmt->execute(['id' => $id, 'u' => $userId]);
        $booking = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$booking) { $_SESSION['flash'] = 'Order not found.'; $this->redirect('/agent/orders'); }

        // Related wallet ledger entries by meta.booking_id
        $ledger = [];
        $sql = "SELECT wl.* FROM wallet_ledger wl
                JOIN wallets w ON w.id = wl.wallet_id
                WHERE w.user_id = :u
                  AND (
                        (wl.meta IS NOT NULL AND JSON_EXTRACT(wl.meta, '$.booking_id') = :bid)
                        OR (wl.meta IS NOT NULL AND JSON_EXTRACT(wl.meta, '$.bookingId') = :bid)
                      )
                ORDER BY wl.id DESC";
        try {
            $stmtL = $this->pdo->prepare($sql);
            $stmtL->execute(['u' => $userId, 'bid' => (string)$id]);
            $ledger = $stmtL->fetchAll();
        } catch (\Throwable $e) {
            // Some MySQL versions may need CAST; fallback: fetch all recent and filter in PHP
            $stmtL2 = $this->pdo->prepare('SELECT wl.* FROM wallet_ledger wl JOIN wallets w ON w.id=wl.wallet_id WHERE w.user_id=:u ORDER BY wl.id DESC LIMIT 200');
            $stmtL2->execute(['u' => $userId]);
            $rows = $stmtL2->fetchAll();
            foreach ($rows as $r) {
                $meta = json_decode($r['meta'] ?? 'null', true);
                if (is_array($meta) && ((string)($meta['booking_id'] ?? '') === (string)$id || (string)($meta['bookingId'] ?? '') === (string)$id)) {
                    $ledger[] = $r;
                }
            }
        }

        // Wallet balance
        $stmtW = $this->pdo->prepare('SELECT balance FROM wallets WHERE user_id = :u LIMIT 1');
        $stmtW->execute(['u' => $userId]);
        $balance = (float)($stmtW->fetchColumn() ?: 0);

        $this->view('agent/order_view', [
            'booking' => $booking,
            'ledger' => $ledger,
            'balance' => $balance,
        ]);
    }

    // --- Guest Info Step (Hotel and generic bookings) ---
    public function bookingGuestInfo(): void
    {
        Auth::requireRole(['B2B Agent']);
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        $bookingId = (int)($_GET['id'] ?? 0);
        if ($bookingId <= 0) { $_SESSION['flash'] = 'Invalid booking id.'; $this->redirect('/agent/bookings'); }

        $stmt = $this->pdo->prepare('SELECT * FROM bookings WHERE id = :id AND user_id = :u LIMIT 1');
        $stmt->execute(['id' => $bookingId, 'u' => $userId]);
        $booking = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$booking) { $_SESSION['flash'] = 'Booking not found.'; $this->redirect('/agent/bookings'); }
        if ((string)$booking['status'] !== 'pending') { $_SESSION['flash'] = 'Only pending bookings can be edited.'; $this->redirect('/agent/bookings'); }

        // Decode existing details_json if any
        $details = [];
        try { $details = json_decode((string)($booking['details_json'] ?? 'null'), true); if (!is_array($details)) $details = []; } catch (\Throwable $e) { $details = []; }

        // Determine pax breakdown hints
        $pax = max(1, (int)($booking['pax'] ?? 1));
        $guestCounts = [
            'adults' => (int)($details['guests']['adults'] ?? ($booking['adults'] ?? 0)),
            'children' => (int)($details['guests']['children'] ?? ($booking['children'] ?? 0)),
        ];
        if (($guestCounts['adults'] + $guestCounts['children']) <= 0) { $guestCounts['adults'] = $pax; }

        // Rooms count (for hotels this equals quantity of rooms); fallback to pax
        $roomsCnt = (int)($details['rooms'] ?? 0);
        if ($roomsCnt <= 0) { $roomsCnt = max(1, (int)($booking['pax'] ?? 1)); }

        // Load existing per-room guests from DB if table exists
        $roomGuests = [];
        for ($i = 1; $i <= $roomsCnt; $i++) { $roomGuests[$i] = []; }
        try {
            $qg = $this->pdo->prepare('SELECT room_index, guest_index, type, full_name FROM booking_guests WHERE booking_id = :bid ORDER BY room_index ASC, guest_index ASC, id ASC');
            $qg->execute(['bid' => $bookingId]);
            $rows = $qg->fetchAll(\PDO::FETCH_ASSOC) ?: [];
            foreach ($rows as $r) {
                $ri = (int)($r['room_index'] ?? 0); if ($ri <= 0) { $ri = 1; }
                if (!isset($roomGuests[$ri])) { $roomGuests[$ri] = []; }
                $roomGuests[$ri][] = [ 'name' => (string)$r['full_name'], 'type' => (string)$r['type'] ];
            }
        } catch (\Throwable $e) {
            // Fallback to flat guests_details (pre-migration)
            if (isset($details['guests_details']) && is_array($details['guests_details'])) {
                $flat = $details['guests_details'];
                $ri = 1;
                foreach ($flat as $g) {
                    if (!isset($roomGuests[$ri])) { $roomGuests[$ri] = []; }
                    $roomGuests[$ri][] = [ 'name' => (string)($g['name'] ?? ''), 'type' => (string)($g['type'] ?? 'adult') ];
                    // Round-robin assign into rooms for display if overflows
                    $ri++; if ($ri > $roomsCnt) { $ri = 1; }
                }
            }
        }

        // Resolve human-friendly item and room names for hotel bookings
        $itemName = null; $roomName = null;
        try {
            if (($booking['module'] ?? '') === 'hotel') {
                $qH = $this->pdo->prepare('SELECT name FROM hotels WHERE id = :id');
                $qH->execute(['id' => (int)$booking['item_id']]);
                $itemName = (string)($qH->fetchColumn() ?: '');
                $rid = isset($details['room_id']) ? (int)$details['room_id'] : 0;
                if ($rid > 0) {
                    $qR = $this->pdo->prepare('SELECT name FROM hotel_rooms WHERE id = :rid');
                    $qR->execute(['rid' => $rid]);
                    $roomName = (string)($qR->fetchColumn() ?: '');
                }
            }
        } catch (\Throwable $e) { /* ignore */ }

        // Adults per room target: prefer details_json breakdown if present, else even split (non-explicit)
        $adultsPerRoom = [];
        $explicitAdultsPerRoom = false;
        try {
            if (isset($details['rooms_breakdown']['adults_per_room']) && is_array($details['rooms_breakdown']['adults_per_room'])) {
                foreach ($details['rooms_breakdown']['adults_per_room'] as $idx => $val) {
                    $i = (int)$idx; if ($i <= 0) { $i = count($adultsPerRoom)+1; }
                    $adultsPerRoom[$i] = max(0, (int)$val);
                }
                if (!empty($adultsPerRoom)) { $explicitAdultsPerRoom = true; }
            }
        } catch (\Throwable $e) { /* ignore */ }
        if (empty($adultsPerRoom)) {
            $totalAdults = (int)($guestCounts['adults'] ?? 0);
            $base = intdiv(max(0, $totalAdults), max(1, $roomsCnt));
            $rem = max(0, $totalAdults - $base * max(1, $roomsCnt));
            for ($i = 1; $i <= max(1, $roomsCnt); $i++) { $adultsPerRoom[$i] = $base + ($rem > 0 ? 1 : 0); if ($rem > 0) $rem--; }
        }

        $this->view('agent/booking_guest_info', [
            'booking' => $booking,
            'details' => $details,
            'guestCounts' => $guestCounts,
            'roomsCnt' => $roomsCnt,
            'roomGuests' => $roomGuests,
            'adultsPerRoom' => $adultsPerRoom,
            'explicitAdultsPerRoom' => $explicitAdultsPerRoom,
            'itemName' => $itemName,
            'roomName' => $roomName,
        ]);
    }

    public function bookingGuestInfoSave(): void
    {
        Auth::requireRole(['B2B Agent']);
        Security::requireCsrf();
        $userId = (int)($_SESSION['user']['id'] ?? 0);
        if ($userId <= 0) { $this->redirect('/login/agent'); }
        $bookingId = (int)($_POST['booking_id'] ?? 0);
        if ($bookingId <= 0) { $_SESSION['flash'] = 'Invalid booking id.'; $this->redirect('/agent/bookings'); }

        // Fetch booking
        $stmt = $this->pdo->prepare('SELECT * FROM bookings WHERE id = :id AND user_id = :u LIMIT 1');
        $stmt->execute(['id' => $bookingId, 'u' => $userId]);
        $booking = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$booking) { $_SESSION['flash'] = 'Booking not found.'; $this->redirect('/agent/bookings'); }
        if ((string)$booking['status'] !== 'pending') { $_SESSION['flash'] = 'Only pending bookings can be edited.'; $this->redirect('/agent/bookings'); }

        // Collect posted guest info (per-room structure: guest_name[room][i], guest_type[room][i])
        $roomsPosted = isset($_POST['guest_name']) && is_array($_POST['guest_name']) ? $_POST['guest_name'] : [];
        $typesPosted = isset($_POST['guest_type']) && is_array($_POST['guest_type']) ? $_POST['guest_type'] : [];
        $guestsByRoom = [];
        $totalGuests = 0;
        foreach ($roomsPosted as $roomIdx => $nameArr) {
            if (!is_array($nameArr)) { continue; }
            $roomIdx = (int)$roomIdx; if ($roomIdx <= 0) { $roomIdx = 1; }
            $guestsByRoom[$roomIdx] = [];
            $typeArr = isset($typesPosted[$roomIdx]) && is_array($typesPosted[$roomIdx]) ? $typesPosted[$roomIdx] : [];
            $n = max(count($nameArr), count($typeArr));
            for ($i = 0; $i < $n; $i++) {
                $name = trim((string)($nameArr[$i] ?? ''));
                $type = trim((string)($typeArr[$i] ?? 'adult'));
                if ($name !== '') {
                    $guestsByRoom[$roomIdx][] = [ 'name' => $name, 'type' => ($type === 'child' ? 'child' : 'adult') ];
                    $totalGuests++;
                }
            }
        }
        // Backward compatibility: flat arrays guest_name[]/guest_type[]
        if ($totalGuests === 0) {
            $flatNames = isset($_POST['guest_name']) && is_array($_POST['guest_name']) ? $_POST['guest_name'] : [];
            $flatTypes = isset($_POST['guest_type']) && is_array($_POST['guest_type']) ? $_POST['guest_type'] : [];
            $n2 = max(count($flatNames), count($flatTypes));
            $guestsByRoom[1] = [];
            for ($i = 0; $i < $n2; $i++) {
                $name = trim((string)($flatNames[$i] ?? ''));
                $type = trim((string)($flatTypes[$i] ?? 'adult'));
                if ($name !== '') { $guestsByRoom[1][] = [ 'name' => $name, 'type' => ($type === 'child' ? 'child' : 'adult') ]; $totalGuests++; }
            }
        }
        if ($totalGuests === 0) {
            $_SESSION['errors'][] = 'Please enter at least one guest name.';
            $this->redirect('/agent/booking/guest-info?id='.(string)$bookingId);
        }

        // Merge into details_json under guests_by_room (and keep old guests_details for compatibility)
        $details = [];
        try { $details = json_decode((string)($booking['details_json'] ?? 'null'), true); if (!is_array($details)) $details = []; } catch (\Throwable $e) { $details = []; }
        $details['guests_by_room'] = $guestsByRoom;
        // Flatten for compatibility
        $flat = [];
        foreach ($guestsByRoom as $ri => $arr) { foreach ($arr as $g) { $flat[] = $g; } }
        $details['guests_details'] = $flat;

        // Persist adjustable persons/adults per room targets if provided
        // 1) persons_target[] => rooms_breakdown.persons_per_room
        if (isset($_POST['persons_target']) && is_array($_POST['persons_target'])) {
            $targetsP = [];
            $sumP = 0;
            foreach ($_POST['persons_target'] as $idx => $val) {
                $i = (int)$idx; if ($i <= 0) { continue; }
                $v = max(0, (int)$val);
                $targetsP[$i] = $v;
                $sumP += $v;
            }
            if (!isset($details['rooms_breakdown']) || !is_array($details['rooms_breakdown'])) { $details['rooms_breakdown'] = []; }
            $details['rooms_breakdown']['persons_per_room'] = $targetsP;
        }
        // 2) adults_target[] (backward compatibility) => rooms_breakdown.adults_per_room
        if (isset($_POST['adults_target']) && is_array($_POST['adults_target'])) {
            $targetsA = [];
            foreach ($_POST['adults_target'] as $idx => $val) {
                $i = (int)$idx; if ($i <= 0) { continue; }
                $targetsA[$i] = max(0, (int)$val);
            }
            if (!isset($details['rooms_breakdown']) || !is_array($details['rooms_breakdown'])) { $details['rooms_breakdown'] = []; }
            $details['rooms_breakdown']['adults_per_room'] = $targetsA;
        }

        // Recompute guest counts from submitted names/types
        $flatAdults = 0; $flatChildren = 0;
        foreach ($guestsByRoom as $ri => $arr) {
            foreach ($arr as $g) { if (($g['type'] ?? 'adult') === 'child') $flatChildren++; else $flatAdults++; }
        }
        if (!isset($details['guests']) || !is_array($details['guests'])) { $details['guests'] = []; }
        $details['guests']['adults'] = $flatAdults;
        $details['guests']['children'] = $flatChildren;

        // Optional contact fields at booking level (useful before order is created)
        $contactName = trim((string)($_POST['contact_name'] ?? ''));
        $contactMobile = trim((string)($_POST['contact_mobile'] ?? ''));
        if ($contactName !== '' || $contactMobile !== '') {
            $details['contact'] = [ 'name' => $contactName, 'mobile' => $contactMobile ];
        }

        // Persist DB rows in booking_guests within a transaction (best-effort if table exists)
        $this->pdo->beginTransaction();
        try {
            // Update details_json
            $upd = $this->pdo->prepare('UPDATE bookings SET details_json = :d WHERE id = :id AND user_id = :u');
            $upd->execute([
                'd' => json_encode($details, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
                'id' => $bookingId,
                'u' => $userId,
            ]);

            // Clear and insert guests
            try {
                $this->pdo->prepare('DELETE FROM booking_guests WHERE booking_id = :bid')->execute(['bid' => $bookingId]);
                $ins = $this->pdo->prepare('INSERT INTO booking_guests (booking_id, room_index, guest_index, type, full_name) VALUES (:bid, :ri, :gi, :t, :n)');
                foreach ($guestsByRoom as $ri => $arr) {
                    $gi = 1;
                    foreach ($arr as $g) {
                        $ins->execute([
                            'bid' => $bookingId,
                            'ri' => (int)$ri,
                            'gi' => $gi++,
                            't' => (string)$g['type'],
                            'n' => (string)$g['name'],
                        ]);
                    }
                }
            } catch (\Throwable $e) { /* table may be missing; ignore */ }

            $this->pdo->commit();
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) { $this->pdo->rollBack(); }
            $_SESSION['flash'] = 'Failed to save guest details.';
            $this->redirect('/agent/booking/guest-info?id='.(string)$bookingId);
        }

        // Continue to payment
        $this->redirect('/agent/booking/pay?id='.(string)$bookingId);
    }
}
