<?php
  // app/Views/agent/index.php
  // Mobile-first Agent shell with bottom tabs (Hotels, Activities, Taxi)
?>
<style>
  /* Sticky bottom nav on mobile */
  .agent-bottom-nav{position:sticky;bottom:0;z-index:1020;background:#fff;border-top:1px solid rgba(0,0,0,.08)}
  .agent-bottom-nav .nav-link{font-size:.85rem;color:#6c757d}
  .agent-bottom-nav .nav-link.active{color:#0d6efd}
  .agent-cta{position:fixed;right:16px;bottom:72px;z-index:1020}
  @media (min-width: 992px){.agent-cta{bottom:24px}}
  .input-chip{display:inline-flex;align-items:center;gap:.375rem;background:#f1f5f9;border-radius:999px;padding:.375rem .625rem;font-size:.85rem}
</style>

<div class="container-xxl py-3">
  <div class="row">
    <div class="col-12">
      <div class="d-flex align-items-center justify-content-between mb-2">
        <div>
          <div class="small text-muted">Welcome back</div>
          <h1 class="h5 mb-0">Agent Workspace</h1>
        </div>
        <div class="d-none d-md-flex align-items-center gap-2">
          <span class="input-chip" id="agentWalletChip"><i class="bi bi-wallet2"></i> Wallet <strong class="ms-1">–</strong></span>
          <span class="input-chip"><i class="bi bi-graph-up"></i> Insights</span>
        </div>
      </div>

 

      <ul class="nav nav-tabs d-none d-md-flex" id="agentTabs" role="tablist">
        <li class="nav-item" role="presentation"><button class="nav-link active" id="activities-tab" data-bs-toggle="tab" data-bs-target="#activities" type="button" role="tab" aria-controls="activities" aria-selected="true"><i class="bi bi-ticket-perforated me-1"></i> Activities</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" id="taxi-tab" data-bs-toggle="tab" data-bs-target="#taxi" type="button" role="tab" aria-controls="taxi" aria-selected="false"><i class="bi bi-taxi-front-fill me-1"></i> Taxi</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" id="hotels-tab" data-bs-toggle="tab" data-bs-target="#hotels" type="button" role="tab" aria-controls="hotels" aria-selected="false"><i class="bi bi-building me-1"></i> Hotels</button></li>
      </ul>

      <div class="tab-content pt-2" id="agentTabsContent">
        <!-- Hotels -->
        <div class="tab-pane fade" id="hotels" role="tabpanel" aria-labelledby="hotels-tab">
          <div class="card card-elevated mb-3">
            <div class="card-body">
              <h2 class="h6 mb-3">Search Hotels</h2>
              <form class="row g-2" id="hotelSearchForm">
                <div class="col-12 col-md-4">
                  <label class="form-label small">Destination</label>
                  <input type="text" id="hotelCity" name="city" class="form-control" placeholder="City, area, or hotel name" autocomplete="off">
                </div>
                <div class="col-6 col-md-2">
                  <label class="form-label small">Check-in</label>
                  <input type="date" id="hotelCheckin" name="checkin" class="form-control">
                </div>
                <div class="col-6 col-md-2">
                  <label class="form-label small">Check-out</label>
                  <input type="date" id="hotelCheckout" name="checkout" class="form-control">
                </div>
                <div class="col-6 col-md-2">
                  <label class="form-label small">Guests</label>
                  <input type="number" min="1" value="2" id="hotelGuests" name="adults" class="form-control">
                </div>
                <div class="col-6 col-md-2 d-grid">
                  <label class="form-label invisible d-none d-md-block">Search</label>
                  <button class="btn btn-primary"><i class="bi bi-search me-1"></i> Find</button>
                </div>
              </form>
            </div>
          </div>

          <div class="card card-elevated">
            <div class="card-body">
              <div class="d-flex align-items-center justify-content-between mb-2">
                <h3 class="h6 mb-0">Recent destinations</h3>
                <a class="small" href="#">View all</a>
              </div>
              <div class="d-flex gap-2 flex-wrap">
                <span class="input-chip"><i class="bi bi-geo-alt"></i> Bangkok</span>
                <span class="input-chip"><i class="bi bi-geo-alt"></i> Phuket</span>
                <span class="input-chip"><i class="bi bi-geo-alt"></i> Pattaya</span>
              </div>
              <hr>
              <div id="hotelResults" class="mt-2">
                <div class="text-muted small">Search to see results.</div>
              </div>
            </div>
          </div>
        </div>

        <!-- Activities -->
        <div class="tab-pane fade show active" id="activities" role="tabpanel" aria-labelledby="activities-tab">
          <div class="card card-elevated mb-3">
            <div class="card-body">
              <h2 class="h6 mb-3">Search Activities</h2>
              <form class="row g-2" id="activitySearchForm">
                <div class="col-12 col-md-6">
                  <label class="form-label small">Location</label>
                  <select id="activityCity" name="city" class="form-select">
                    <option value="" selected disabled>Select city…</option>
                  </select>
                </div>
                <div class="col-12 col-md-2 d-grid">
                  <label class="form-label invisible d-none d-md-block">Search</label>
                  <button class="btn btn-primary" type="submit"><i class="bi bi-search me-1"></i> Find</button>
                </div>
              </form>
            </div>
          </div>
          <div class="card card-elevated">
            <div class="card-body">
              <hr>
              <div id="activityResults" class="mt-2">
                <div class="text-muted small">Search to see activities.</div>
              </div>
            </div>
          </div>
        </div>

        <!-- Taxi -->
        <div class="tab-pane fade" id="taxi" role="tabpanel" aria-labelledby="taxi-tab">
          <div class="card card-elevated mb-3">
            <div class="card-body">
              <h2 class="h6 mb-3">Book Taxi</h2>
              <form class="row g-2" id="taxiQuoteForm">
                <div class="col-12 col-md-4">
                  <label class="form-label small">Pickup</label>
                  <input type="text" id="taxiFrom" name="from" class="form-control" placeholder="Airport, hotel, or address" autocomplete="off">
                </div>
                <div class="col-12 col-md-4">
                  <label class="form-label small">Drop</label>
                  <input type="text" id="taxiTo" name="to" class="form-control" placeholder="Destination" autocomplete="off">
                </div>
                <div class="col-6 col-md-2">
                  <label class="form-label small">Date</label>
                  <input type="datetime-local" id="taxiAt" name="at" class="form-control">
                </div>
                <div class="col-6 col-md-2 d-grid">
                  <label class="form-label invisible d-none d-md-block">Search</label>
                  <button class="btn btn-primary"><i class="bi bi-search me-1"></i> Quote</button>
                </div>
              </form>
            </div>
          </div>
          <div class="card card-elevated">
            <div class="card-body">
              <h3 class="h6 mb-2">Saved routes</h3>
              <div class="text-muted small">No saved routes yet.</div>
              <hr>
              <div id="taxiResults" class="mt-2">
                <div class="text-muted small">Enter pickup and drop to get quotes.</div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- Mobile bottom nav -->
<div class="agent-bottom-nav d-md-none">
  <ul class="nav nav-pills nav-justified px-2 py-1" role="tablist">
    <li class="nav-item" role="presentation">
      <button class="nav-link active" data-bs-toggle="tab" data-bs-target="#activities" type="button" role="tab" aria-controls="activities" aria-selected="true"><i class="bi bi-ticket-perforated"></i><div class="small">Activities</div></button>
    </li>
    <li class="nav-item" role="presentation">
      <button class="nav-link" data-bs-toggle="tab" data-bs-target="#taxi" type="button" role="tab" aria-controls="taxi" aria-selected="false"><i class="bi bi-taxi-front-fill"></i><div class="small">Taxi</div></button>
    </li>
    <li class="nav-item" role="presentation">
      <button class="nav-link" data-bs-toggle="tab" data-bs-target="#hotels" type="button" role="tab" aria-controls="hotels" aria-selected="false"><i class="bi bi-building"></i><div class="small">Hotels</div></button>
    </li>
  </ul>
</div>

<!-- Floating action: New Booking (future hook) -->
<div class="agent-cta d-md-none">
  <button class="btn btn-primary rounded-circle shadow" style="width:52px;height:52px"><i class="bi bi-plus-lg"></i></button>
  </div>

<script>
  // Wallet balance
  async function loadWallet() {
    try {
      const res = await fetch('/b2b/api/wallet', {headers: {'Accept':'application/json'}});
      if (!res.ok) return;
      const data = await res.json();
      const chip = document.getElementById('agentWalletChip');
      if (chip && data && typeof data.balance !== 'undefined') {
        chip.innerHTML = `<i class="bi bi-wallet2"></i> Wallet <strong class="ms-1">${data.balance.toFixed(2)} ${data.currency || 'THB'}</strong>`;
      }
    } catch (e) {}
  }

  // Hotel search
  const hotelForm = document.getElementById('hotelSearchForm');
  const hotelResults = document.getElementById('hotelResults');
  if (hotelForm) {
    hotelForm.addEventListener('submit', async (ev) => {
      ev.preventDefault();
      const city = document.getElementById('hotelCity').value.trim();
      const checkin = document.getElementById('hotelCheckin').value;
      const checkout = document.getElementById('hotelCheckout').value;
      const adults = document.getElementById('hotelGuests').value || 2;
      hotelResults.innerHTML = '<div class="text-muted">Loading…</div>';
      try {
        const url = `/b2b/api/hotels/search?city=${encodeURIComponent(city)}&checkin=${encodeURIComponent(checkin)}&checkout=${encodeURIComponent(checkout)}&adults=${encodeURIComponent(adults)}`;
        const res = await fetch(url, {headers:{'Accept':'application/json'}});
        const json = await res.json();
        const items = json.data || [];
        if (!items.length) { hotelResults.innerHTML = '<div class="text-muted small">No hotels found.</div>'; return; }
        hotelResults.innerHTML = items.map(h => `
          <div class="border rounded p-2 mb-2">
            <div class="d-flex justify-content-between">
              <div>
                <div class="fw-semibold">${h.name}</div>
                <div class="text-muted small">${h.city}, ${h.country} • ${'★'.repeat(h.stars || 0)}</div>
              </div>
              <div class="text-end">
                <div class="fw-semibold">THB ${Number(h.nightly_from||0).toFixed(2)}/night</div>
                <div class="small text-muted">Est. total: THB ${Number(h.total_estimate||0).toFixed(2)}</div>
                <button class="btn btn-sm btn-outline-primary mt-1" data-hotel-id="${h.id}" onclick="loadHotelDetails(${h.id})">View</button>
              </div>
            </div>
          </div>`).join('');
      } catch (e) {
        hotelResults.innerHTML = '<div class="text-danger small">Error loading results.</div>';
      }
    });
  }

  // Hotel details (basic)
  async function loadHotelDetails(id) {
    try {
      const checkin = document.getElementById('hotelCheckin').value;
      const checkout = document.getElementById('hotelCheckout').value;
      const res = await fetch(`/b2b/api/hotels/details?id=${id}&checkin=${encodeURIComponent(checkin)}&checkout=${encodeURIComponent(checkout)}`, {headers:{'Accept':'application/json'}});
      const json = await res.json();
      const rlist = (json.rooms||[]).map(r => `<li>${r.name} <span class="text-muted small">(cap: ${r.capacity||2})</span></li>`).join('');
      alert(`${json.hotel.name}\nRooms:\n- ${(json.rooms||[]).map(r=>r.name).join('\n- ')}`);
    } catch (e) { alert('Failed to load details'); }
  }
  window.loadHotelDetails = loadHotelDetails;

  // Activities search
  const actForm = document.getElementById('activitySearchForm');
  const actResults = document.getElementById('activityResults');
  const actCitySelect = document.getElementById('activityCity');
  async function fetchActivities() {
    const city = actCitySelect.value || '';
    actResults.innerHTML = '<div class="text-muted">Loading…</div>';
    try {
      const params = new URLSearchParams({ city });
      const res = await fetch(`/b2b/api/activities/search?${params.toString()}`, {headers:{'Accept':'application/json'}});
      const json = await res.json();
      const items = json.data || [];
      const countBar = `<div class="small text-muted mb-2">${items.length} activit${items.length===1?'y':'ies'} found</div>`;
      if (!items.length) { actResults.innerHTML = countBar + '<div class="text-muted small">No activities found.</div>'; return; }
      const listHtml = items.map(a => `
        <div class="border rounded p-2 mb-2 d-flex align-items-center gap-2">
          <div class="flex-shrink-0" style="width:92px">
            <div class="ratio ratio-4x3 bg-light rounded overflow-hidden">
              ${a.thumbnail ? `<img src="${a.thumbnail}" alt="${a.name}" class="w-100 h-100 object-fit-cover">` : ''}
            </div>
          </div>
          <div class="flex-grow-1">
            <div class="fw-semibold">${a.name}</div>
            <div class="text-muted small">${a.city || ''}</div>
          </div>
          <div class="text-end">
            <div class="fw-semibold">THB ${Number(a.from_price||0).toFixed(2)}</div>
            <button class="btn btn-sm btn-outline-primary mt-1" onclick="loadActivityDetails(${a.id})">View</button>
          </div>
        </div>`).join('');
      actResults.innerHTML = countBar + listHtml;
      actResults.dataset.loaded = '1';
    } catch (e) {
      actResults.innerHTML = '<div class="text-danger small">Error loading results.</div>';
    }
  }
  if (actForm) {
    actForm.addEventListener('submit', async (ev) => {
      ev.preventDefault();
      fetchActivities();
    });
  }

  // Load cities into dropdown and trigger search when changed
  async function loadCityDropdown() {
    try {
      const params = new URLSearchParams({ country: 'Thailand' });
      const res = await fetch(`/b2b/api/activities/cities?${params.toString()}`, {headers:{'Accept':'application/json'}});
      const json = await res.json();
      const items = json.data || [];
      const opts = items.map(c => `<option value="${c.city}">${c.city}</option>`).join('');
      actCitySelect.innerHTML = `<option value="" selected disabled>Select city…</option>` + opts;
      // Auto-select first city and trigger search once
      if (!actCitySelect.dataset.autoloaded) {
        const first = actCitySelect.querySelector('option[value]:not([value=""])');
        if (first) {
          actCitySelect.value = first.value;
          actCitySelect.dataset.autoloaded = '1';
          fetchActivities();
        }
      }
    } catch (e) { /* ignore */ }
  }
  if (actCitySelect) {
    actCitySelect.addEventListener('change', () => { fetchActivities(); });
  }
  // Auto-load cities when Activities tab becomes active or on initial view
  const actTabBtn = document.getElementById('activities-tab');
  if (actTabBtn) {
    actTabBtn.addEventListener('shown.bs.tab', () => {
      loadCityDropdown();
    });
  }
  // If Activities is already the active pane on load, fetch immediately
  if (document.getElementById('activities')?.classList.contains('show')) {
    loadCityDropdown();
  }

  // Activity details modal + loader
  function ensureActivityModal() {
    let modal = document.getElementById('activityDetailsModal');
    if (!modal) {
      const wrapper = document.createElement('div');
      wrapper.innerHTML = `
      <div class="modal fade" id="activityDetailsModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-dialog-scrollable">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title">Activity Details</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
              <div id="activityDetailsBody" class="small text-muted">Loading…</div>
              <div id="activitySelectionsWrap" class="mt-3" style="display:none">
                <div class="d-flex justify-content-between align-items-center mb-1">
                  <div class="fw-semibold">Selections <span class="badge bg-secondary" id="activitySelectionsCount">0</span></div>
                  <div class="small text-muted">Total: <strong id="activitySelectionsTotal">0.00</strong> <span id="activitySelectionsCurrency">THB</span></div>
                </div>
                <div id="activitySelections" class="border rounded p-2 small"></div>
              </div>
            </div>
            <div class="modal-footer">
              <button type="button" class="btn btn-outline-primary me-auto" id="btnBookAll" disabled onclick="bookAllSelections()"><i class="bi bi-cart-check"></i> Checkout</button>
              <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
          </div>
        </div>
      </div>`;
      document.body.appendChild(wrapper.firstElementChild);
    }
  }

  async function loadActivityDetails(id) {
    ensureActivityModal();
    const modalEl = document.getElementById('activityDetailsModal');
    const bodyEl = document.getElementById('activityDetailsBody');
    if (bodyEl) bodyEl.innerHTML = 'Loading…';
    const modal = new bootstrap.Modal(modalEl);
    modal.show();
    try {
      const res = await fetch(`/b2b/api/activities/details?id=${id}`, {headers:{'Accept':'application/json'}});
      const data = await res.json();
      const pkg = data.package || {};
      const variants = data.variants || [];
      const prices = data.prices || [];
      const requiresShow = (pkg.requires_show_time === 1 || pkg.requires_show_time === '1');

      // Group prices by variant
      const pricesByVariant = prices.reduce((acc, p) => {
        const vid = p.variant_id;
        (acc[vid] = acc[vid] || []).push(p);
        return acc;
      }, {});

      const variantCards = variants.map((v, vIdx) => {
        const vpricesAll = (pricesByVariant[v.id] || []);
        const nonFlat = vpricesAll.filter(p => p.pax_type !== 'flat');
        const flat = vpricesAll.find(p => p.pax_type === 'flat');
        const minPer = nonFlat.length ? Math.min(...nonFlat.map(p => Number(p.agent_price||0))) : null;
        const headerRight = `
          <div class=\"text-end small\">
            ${minPer !== null ? `<div>Normal from <strong>THB ${minPer.toFixed(2)}</strong></div>` : ''}
            ${flat ? `<div class=\"text-success\">Group <strong>THB ${Number(flat.agent_price||0).toFixed(2)}</strong> <span class=\"text-muted\">(min ${flat.min_quantity||1})</span></div>` : ''}
          </div>`;
        const rows = nonFlat.map(p => {
          const label = `${p.pax_type}`;
          const priceStr = `${Number(p.agent_price||0).toFixed(2)} ${p.currency||'THB'}`;
          return `
            <div class=\"d-flex justify-content-between align-items-center py-2 border-bottom\">
              <div>
                <div class=\"fw-semibold\">${label}</div>
                <div class=\"text-muted small\">${priceStr}${p.pickup_type?` • pickup: ${p.pickup_type}`:''}</div>
              </div>
              <div class=\"d-flex align-items-center gap-2\"> 
                <div class=\"input-group input-group-sm\" style=\"width:140px\"> 
                  <button class=\"btn btn-outline-secondary qty-minus\" type=\"button\">−</button> 
                  <input type=\"number\" class=\"form-control form-control-sm price-qty text-center\" data-variant-id=\"${v.id}\" data-unit-price=\"${Number(p.agent_price||0)}\" data-currency=\"${p.currency||'THB'}\" value=\"0\" min=\"0\"> 
                  <button class=\"btn btn-outline-secondary qty-plus\" type=\"button\">+</button> 
                </div> 
                <button class=\"btn btn-sm btn-outline-secondary\" onclick=\"addToCartFromButton(this, ${pkg.id}, ${p.variant_id}, ${p.id}, '${p.pax_type}', ${Number(p.agent_price||0)}, '${p.currency||'THB'}', '${(v.name||'').replace(/\\\"/g,'\\\\\\\"') + ' • ' + (p.pax_type||'')}', false)\"><i class=\"bi bi-plus\"></i> Add</button>
              </div>
            </div>`;
        }).join('');
        const collapseId = `variant_${v.id}_body`;
        const isOpen = vIdx < 3; // keep first 3 expanded
        return `
        <div class=\"border rounded p-2 mb-2\" id=\"variant_${v.id}_container\">
          <div class=\"d-flex justify-content-between align-items-center\"> 
            <button class=\"btn btn-link p-0 text-start fw-semibold\" data-bs-toggle=\"collapse\" data-bs-target=\"#${collapseId}\" aria-expanded=\"${isOpen?'true':'false'}\" aria-controls=\"${collapseId}\">${v.name}</button>
            ${headerRight}
          </div>
          <div id=\"${collapseId}\" class=\"collapse ${isOpen?'show':''}\">
            ${requiresShow ? `
            <div class=\"mt-2\"> 
              <label class=\"form-label small mb-1\">Showtime</label>
              <select id=\"showtime_for_variant_${v.id}\" class=\"form-select form-select-sm\"> 
                ${(v.showtimes||[]).length ? (v.showtimes||[]).map(t=>`<option value=\"${t}\">${t}</option>`).join('') : '<option value=\"\">No times available</option>'}
              </select>
            </div>` : ''}
            <div class=\"mt-2\"> 
              ${nonFlat.length ? `
                <div class=\"border rounded\"> 
                  ${rows}
                </div>` : '<div class="text-muted small">No per-person prices for this variant.</div>'}
              <div id=\"variant_${v.id}_flat_offer\" class=\"mt-2\" style=\"display:none\"></div>
              <div id=\"variant_${v.id}_flat_meta\" data-has-flat=\"${flat?1:0}\" data-min-qty=\"${flat? (flat.min_quantity||''): ''}\" data-agent-price=\"${flat? (flat.agent_price||''): ''}\" data-currency=\"${flat? (flat.currency||'THB'): ''}\" data-price-id=\"${flat? flat.id: ''}\"></div>
            </div>
          </div>
        </div>`;
      }).join('');

      const img = pkg.thumbnail_path ? `<img src="${pkg.thumbnail_path}" alt="${pkg.name}" class="rounded" style="width:140px;height:105px;object-fit:cover">` : '';

      const html = `
        <div class="d-flex gap-3 align-items-start mb-3">
          ${img}
          <div>
            <div class="h6 mb-1">${pkg.name||''}</div>
            <div class="text-muted small">${pkg.city||''}</div>
          </div>
        </div>
        <div class="mb-3">
          <div class="fw-semibold mb-1">Variants</div>
          ${variantCards || '<div class="text-muted small">No active variants.</div>'}
        </div>
        `;

      if (bodyEl) bodyEl.innerHTML = html;
      // After render, wire flat offer activation based on per-variant summed qty from per-person rows
      const alerted = new Set();
      function calcVariantSum(variantId) {
        const cont = document.getElementById(`variant_${variantId}_container`);
        if (!cont) return { qty:0, subtotal:0, currency:'THB' };
        let qty = 0;
        let subtotal = 0;
        let currency = 'THB';
        cont.querySelectorAll('.price-qty').forEach(inp => {
          const q = parseInt(inp.value||'0',10);
          const unit = parseFloat(inp.getAttribute('data-unit-price')||'0');
          const cur = inp.getAttribute('data-currency')||'THB';
          if (!isNaN(q) && q>0) {
            qty += q;
            subtotal += q * unit;
            currency = cur;
          }
        });
        return { qty, subtotal, currency };
      }
      function updateVariantFlatOffer(variantId) {
        const meta = document.getElementById(`variant_${variantId}_flat_meta`);
        const offer = document.getElementById(`variant_${variantId}_flat_offer`);
        if (!meta || !offer) return;
        const hasFlat = meta.getAttribute('data-has-flat') === '1';
        if (!hasFlat) { offer.style.display='none'; offer.innerHTML=''; return; }
        const minq = parseInt(meta.getAttribute('data-min-qty')||'0',10);
        const price = parseFloat(meta.getAttribute('data-agent-price')||'0');
        const currency = meta.getAttribute('data-currency')||'THB';
        const priceId = parseInt(meta.getAttribute('data-price-id')||'0',10);
        const vAgg = calcVariantSum(variantId);
        if (vAgg.qty >= minq && minq>0) {
          offer.style.display='block';
          const vName = (variants.find(x => Number(x.id) === Number(variantId))?.name || 'Variant').replace(/"/g,'\\"');
          offer.innerHTML = `
            <div class=\"d-flex flex-wrap justify-content-between align-items-center border rounded p-2 gap-2\">
              <div class=\"small flex-grow-1\">
                <span>Flat ${minq}+ total for </span>
                <input type=\"number\" id=\"variant_${variantId}_flat_qty\" class=\"form-control form-control-sm d-inline-block\" value=\"${vAgg.qty}\" min=\"${minq}\" style=\"width:90px\"> <span>pax:</span>
                <strong id=\"variant_${variantId}_flat_total\"></strong> <span>${currency}</span>
                <div class=\"text-muted mt-1\"><small>Per-person total: <span id=\"variant_${variantId}_pp_total\"></span> ${vAgg.currency} • You save: <span id=\"variant_${variantId}_save\"></span> ${vAgg.currency}</small></div>
                <div class=\"mt-1\"><small><strong>New cart total if applied:</strong> <span id=\"variant_${variantId}_cart_new_total\"></span> ${vAgg.currency}</small></div>
              </div>
              <div class=\"d-flex align-items-center gap-2\">
                <button class=\"btn btn-sm btn-outline-secondary\" onclick=\"addToCartFromButton(this, ${pkg.id}, ${variantId}, ${priceId}, 'flat', ${price}, '${currency}', '${vName} • flat', true)\"><i class=\"bi bi-plus\"></i> Add</button>
              </div>
            </div>
            <div class=\"alert alert-success py-2 px-3 mt-2 mb-2\">Group booking discount activated for ${vName}: group ${minq}+ applied.</div>`;
          const qtyEl = document.getElementById(`variant_${variantId}_flat_qty`);
          const totalEl = document.getElementById(`variant_${variantId}_flat_total`);
          const ppEl = document.getElementById(`variant_${variantId}_pp_total`);
          const saveEl = document.getElementById(`variant_${variantId}_save`);
          const cartNewEl = document.getElementById(`variant_${variantId}_cart_new_total`);
          const compute = () => {
            const q = Math.max(minq, parseInt(qtyEl.value||`${minq}`,10));
            const groups = Math.ceil(q / minq);
            const flatTotal = groups * price;
            totalEl.textContent = flatTotal.toFixed(2);
            ppEl.textContent = (vAgg.subtotal || 0).toFixed(2);
            const savings = Math.max(0, (vAgg.subtotal||0) - flatTotal);
            saveEl.textContent = savings.toFixed(2);
            try {
              const currentCartTotal = (selectionsState.items||[]).reduce((acc,it)=>acc + (parseFloat(it.total)||0), 0);
              const projected = Math.max(0, currentCartTotal - (vAgg.subtotal||0) + flatTotal);
              if (cartNewEl) cartNewEl.textContent = projected.toFixed(2);
            } catch(_) {}
          };
          compute();
          qtyEl.addEventListener('input', compute);
          if (!alerted.has(variantId)) { alerted.add(variantId); }
        } else {
          offer.style.display='none';
          offer.innerHTML='';
          alerted.delete(variantId);
        }
      }
      function wireQtyInputs() {
        document.querySelectorAll('#activityDetailsBody .price-qty').forEach(inp => {
          const toggleBtn = () => {
            const val = parseInt(inp.value||'0',10) || 0;
            const row = inp.closest('.d-flex');
            const addBtn = row ? row.querySelector('button.btn.btn-sm.btn-outline-secondary') : null;
            if (addBtn) addBtn.disabled = val < 1;
          };
          inp.addEventListener('input', () => {
            const vid = parseInt(inp.getAttribute('data-variant-id')||'0',10);
            toggleBtn();
            if (vid) updateVariantFlatOffer(vid);
          });
          // Plus/Minus controls
          const row = inp.closest('.d-flex');
          const minusBtn = row ? row.querySelector('.qty-minus') : null;
          const plusBtn = row ? row.querySelector('.qty-plus') : null;
          if (minusBtn) minusBtn.addEventListener('click', () => {
            const current = Math.max(0, parseInt(inp.value||'0',10) || 0);
            const next = Math.max(0, current - 1);
            inp.value = String(next);
            inp.dispatchEvent(new Event('input'));
          });
          if (plusBtn) plusBtn.addEventListener('click', () => {
            const current = Math.max(0, parseInt(inp.value||'0',10) || 0);
            const next = current + 1;
            inp.value = String(next);
            inp.dispatchEvent(new Event('input'));
          });
          // Initialize disabled/enabled state
          toggleBtn();
        });
      }
      // initial wiring and calculation for all variants
      wireQtyInputs();
      (variants||[]).forEach(v => updateVariantFlatOffer(v.id));
    } catch (e) {
      if (bodyEl) bodyEl.innerHTML = '<div class="text-danger">Failed to load details.</div>';
    }
  }
  window.loadActivityDetails = loadActivityDetails;

  // Selections cart state
  const selectionsState = { items: [], currency: 'THB' };
  function refreshSelectionsUI() {
    const wrap = document.getElementById('activitySelectionsWrap');
    const listEl = document.getElementById('activitySelections');
    const countEl = document.getElementById('activitySelectionsCount');
    const totalEl = document.getElementById('activitySelectionsTotal');
    const currEl = document.getElementById('activitySelectionsCurrency');
    const btnAll = document.getElementById('btnBookAll');
    const items = selectionsState.items;
    if (!wrap || !listEl) return;
    wrap.style.display = items.length ? 'block' : 'none';
    // Hide the header total; we show the projected total within each variant discount panel
    if (totalEl && totalEl.parentElement) totalEl.parentElement.style.display = 'none';
    if (countEl) countEl.textContent = String(items.length);
    let total = 0;
    items.forEach(it => { total += (parseFloat(it.total)||0); });
    if (totalEl) totalEl.textContent = total.toFixed(2);
    if (currEl) currEl.textContent = selectionsState.currency || 'THB';
    if (btnAll) btnAll.disabled = items.length === 0;
    listEl.innerHTML = (items.map((it, idx) => `
      <div class="d-flex justify-content-between align-items-center py-1 border-bottom">
        <div>
          <div>${it.label}</div>
          <div class="text-muted">qty ${it.qty} • ${it.pax_type} • ${Number(it.total||0).toFixed(2)} ${it.currency}</div>
        </div>
        <button class="btn btn-sm btn-link text-danger" onclick="removeSelection(${idx})"><i class="bi bi-x"></i></button>
      </div>
    `).join('')) || '<div class="text-muted">No selections yet.</div>';
    // Trigger recompute of any visible flat panels' projected totals
    try {
      document.querySelectorAll('[id$="_flat_qty"]').forEach(el => {
        el.dispatchEvent(new Event('input'));
      });
    } catch(_) {}
  }
  function removeSelection(idx) { selectionsState.items.splice(idx,1); refreshSelectionsUI(); }

  function addToCartFromButton(btnEl, packageId, variantId, priceId, paxType, agentPrice, currency, label, isFlat) {
    try {
      let qty = 0;
      if (btnEl) {
        // For per-person rows, qty input is adjacent in the same flex row
        const row = btnEl.closest('.d-flex');
        const rowQty = row ? row.querySelector('.price-qty') : null;
        if (rowQty) qty = parseInt(rowQty.value||'0',10);
      }
      if (isFlat) {
        const fq = document.getElementById(`variant_${variantId}_flat_qty`);
        if (fq) qty = parseInt(fq.value||'0',10);
      }
      // Require at least 1 to add
      if (!qty || isNaN(qty) || qty < 1) { return; }
      const meta = document.getElementById(`variant_${variantId}_flat_meta`);
      let total = 0;
      if (paxType === 'flat' && meta) {
        const minq = Math.max(1, parseInt(meta.getAttribute('data-min-qty')||'1',10));
        const groups = Math.ceil(qty / minq);
        total = groups * Number(agentPrice||0);
      } else {
        total = Number(agentPrice||0) * qty;
      }
      const stEl = document.getElementById(`showtime_for_variant_${variantId}`);
      const show_time = stEl ? (stEl.value||'') : '';
      selectionsState.currency = currency || selectionsState.currency;
      selectionsState.items.push({ package_id: packageId, variant_id: variantId, price_id: priceId, qty, pax_type: paxType, total, currency, label, show_time });
      refreshSelectionsUI();
    } catch (e) { alert('Failed to add'); }
  }
  window.addToCartFromButton = addToCartFromButton;

  // Lightweight result modal for bookings
  function ensureBookingResultModal() {
    let m = document.getElementById('bookingResultModal');
    if (!m) {
      const wrap = document.createElement('div');
      wrap.innerHTML = `
      <div class="modal fade" id="bookingResultModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title">Booking Result</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
              <div id="bookingResultBody" class="small text-muted">Processing…</div>
            </div>
            <div class="modal-footer">
              <button type="button" class="btn btn-primary" data-bs-dismiss="modal">OK</button>
            </div>
          </div>
        </div>
      </div>`;
      document.body.appendChild(wrap.firstElementChild);
    }
  }

  // Checkout modal
  function ensureCheckoutModal() {
    let m = document.getElementById('checkoutModal');
    if (!m) {
      const wrap = document.createElement('div');
      wrap.innerHTML = `
      <div class="modal fade" id="checkoutModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-dialog-scrollable">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title">Checkout</h5>
              <div class="ms-auto small text-muted" id="checkoutWalletInfo"></div>
              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
              <div id="checkoutBody" class="small text-muted">Preparing checkout…</div>
            </div>
            <div class="modal-footer" id="checkoutFooter">
              <button type="button" class="btn btn-primary" id="checkoutPayBtn" disabled>Pay with Wallet</button>
              <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
          </div>
        </div>
      </div>`;
      document.body.appendChild(wrap.firstElementChild);
    }
  }

  async function loadWalletIntoCheckout() {
    try {
      const el = document.getElementById('checkoutWalletInfo');
      if (!el) return;
      el.textContent = 'Wallet: …';
      const res = await fetch('/b2b/api/wallet', { headers: { 'Accept': 'application/json' } });
      if (!res.ok) { el.textContent = 'Wallet: –'; return; }
      const data = await res.json();
      if (typeof data.balance !== 'undefined') {
        el.textContent = `Wallet: ${Number(data.balance||0).toFixed(2)} ${data.currency||'THB'}`;
      }
    } catch (_) {
      const el = document.getElementById('checkoutWalletInfo');
      if (el) el.textContent = 'Wallet: –';
    }
  }

  async function createCheckoutOrder() {
    const cBody = document.getElementById('checkoutBody');
    const payBtn = document.getElementById('checkoutPayBtn');
    try {
      // Validate selections
      const invalid = (selectionsState.items||[]).some(it => !it || isNaN(parseInt(it.qty||'0',10)) || parseInt(it.qty,10) <= 0);
      if (invalid || !selectionsState.items.length) {
        if (cBody) cBody.innerHTML = '<div class="text-danger">Your cart is empty or has invalid quantities.</div>';
        if (payBtn) { payBtn.disabled = true; }
        return;
      }
      if (cBody) cBody.innerHTML = '<div class="text-muted">Creating order…</div>';
      if (payBtn) { payBtn.disabled = true; payBtn.textContent = 'Pay with Wallet'; payBtn.onclick = null; }
      const payload = { items: selectionsState.items.map(it => ({ package_id: it.package_id, variant_id: it.variant_id, price_id: it.price_id, qty: it.qty, show_time: it.show_time })) };
      const res = await fetch('/b2b/api/checkout/create', { method:'POST', headers:{'Content-Type':'application/json','Accept':'application/json','X-CSRF-Token': (window.CSRF_TOKEN||'')}, body: JSON.stringify(payload)});
      const json = await res.json();
      if (!res.ok || json.error) {
        if (cBody) cBody.innerHTML = `<div class="text-danger mb-2">${(json && json.error) ? json.error : 'Failed to create order.'}</div><button class="btn btn-outline-secondary btn-sm" onclick="createCheckoutOrder()">Retry</button>`;
        return;
      }
      // Render order summary
      const order = json.order || {};
      const items = json.items || order.items || [];
      const currency = json.currency || order.currency || (items[0]?.currency) || selectionsState.currency || 'THB';
      const subtotal = Number(json.subtotal || order.subtotal || items.reduce((a,b)=>a + (parseFloat(b.total||b.amount||0)),0));
      const fees = Number(json.fees || order.fees || 0);
      const total = Number(json.total || order.total || (subtotal + fees));
      const list = items.length ? items.map((it, i) => `
        <div class="d-flex justify-content-between py-1 border-bottom">
          <div>${it.label || it.name || ('Item #' + (i+1))} <span class="text-muted small">× ${it.qty || 1}</span></div>
          <div class="text-muted small">${Number(it.total || it.amount || 0).toFixed(2)} ${it.currency||currency}</div>
        </div>`).join('') : '<div class="text-muted">No items.</div>';
      if (cBody) cBody.innerHTML = `
        <div class="mb-2">Order <strong>#${order.id || json.order_id || ''}</strong></div>
        <div class="border rounded p-2 mb-2">${list}</div>
        <div class="d-flex justify-content-between small"><div>Subtotal</div><div>${subtotal.toFixed(2)} ${currency}</div></div>
        <div class="d-flex justify-content-between small"><div>Fees</div><div>${fees.toFixed(2)} ${currency}</div></div>
        <div class="d-flex justify-content-between"><div><strong>Total</strong></div><div><strong>${total.toFixed(2)} ${currency}</strong></div></div>
        <hr>
        <div class="small">Payment method:</div>
        <div class="d-flex gap-2">
          <button class="btn btn-outline-primary btn-sm" disabled>Wallet</button>
          <button class="btn btn-outline-secondary btn-sm" disabled title="Coming soon">Stripe</button>
        </div>`;
      const oid = order.id || json.order_id;
      if (payBtn && oid) { payBtn.disabled = false; payBtn.textContent = 'Pay with Wallet'; payBtn.onclick = () => payCheckoutWallet(oid); }
    } catch (e) {
      if (cBody) cBody.innerHTML = '<div class="text-danger mb-2">Unexpected error while creating order.</div><button class="btn btn-outline-secondary btn-sm" onclick="createCheckoutOrder()">Retry</button>';
      if (payBtn) { payBtn.disabled = true; }
    }
  }

  async function payCheckoutWallet(orderId) {
    const payBtn = document.getElementById('checkoutPayBtn');
    const body = document.getElementById('checkoutBody');
    try {
      if (payBtn) { payBtn.disabled = true; payBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Processing…'; }
      const res = await fetch('/b2b/api/checkout/pay-wallet', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json', 'Accept': 'application/json', 'X-CSRF-Token': (window.CSRF_TOKEN||'') },
        body: JSON.stringify({ order_id: orderId })
      });
      const json = await res.json();
      if (!res.ok || json.error) {
        if (body) body.innerHTML = `<div class="text-danger">${(json && json.error) ? json.error : 'Payment failed.'}</div>`;
        if (payBtn) { payBtn.disabled = false; payBtn.textContent = 'Pay with Wallet'; }
        return;
      }
      // Success UI
      const bookings = json.bookings || json.items || [];
      const currency = json.currency || (bookings[0]?.currency) || 'THB';
      const total = Number(json.total || bookings.reduce((a,b)=>a + (parseFloat(b.total)||0),0) || 0);
      const list = bookings.length ? bookings.map(b => `
        <div class="d-flex justify-content-between py-1 border-bottom">
          <div>#${String(b.booking_id || b.id || '')}</div>
          <div class="text-muted small">${Number(b.total||0).toFixed(2)} ${b.currency||currency}</div>
        </div>`).join('') : '<div class="text-muted">No booking details returned.</div>';
      if (body) body.innerHTML = `
        <div class="alert alert-success">Payment successful. Booking(s) confirmed.</div>
        <div class="border rounded p-2 mb-2">${list}</div>
        <div class="text-end"><strong>Total paid:</strong> ${total.toFixed(2)} ${currency}</div>`;
      // Clear cart and refresh wallet
      selectionsState.items = []; refreshSelectionsUI(); loadWallet();
      // Disable pay button after success
      if (payBtn) { payBtn.disabled = true; payBtn.textContent = 'Paid'; }
    } catch (e) {
      if (body) body.innerHTML = '<div class="text-danger">Unexpected error while processing payment.</div>';
      if (payBtn) { payBtn.disabled = false; payBtn.textContent = 'Pay with Wallet'; }
    }
  }

  async function bookAllSelections() {
    try {
      if (!selectionsState.items.length) return;
      // Immediately create order (no contact yet) and redirect to checkout page
      const payload = { items: selectionsState.items.map(it => ({ package_id: it.package_id, variant_id: it.variant_id, price_id: it.price_id, qty: it.qty, show_time: it.show_time })) };
      const res = await fetch('/b2b/api/checkout/create', { method:'POST', headers:{ 'Content-Type':'application/json', 'Accept':'application/json', 'X-CSRF-Token': (window.CSRF_TOKEN||'') }, body: JSON.stringify(payload) });
      const json = await res.json();
      if (!res.ok || json.error || !json.order_id) { alert((json && json.error) ? json.error : 'Failed to create order.'); return; }
      if (!json.checkout_url) { alert('Order created but checkout link is missing. Please try again.'); return; }
      const next = json.checkout_url;
      window.location.href = next;
    } catch (e) { alert('Error creating bookings'); }
    finally {
      const btn = document.getElementById('btnBookAll');
      if (btn) { btn.innerHTML = '<i class="bi bi-cart-check"></i> Checkout'; }
    }
  }
  window.bookAllSelections = bookAllSelections;

  // Book selected activity price
  async function bookActivity(btnEl, packageId, variantId, priceId, isFlat=false) {
    try {
      let qty = 1;
      // Prefer nearest row qty if present
      if (btnEl) {
        const tr = btnEl.closest('tr');
        const rowQty = tr ? tr.querySelector('.price-qty') : null;
        if (rowQty) qty = Math.max(1, parseInt(rowQty.value||'1',10));
      }
      // For flat offer, use the variant-specific flat qty input if present
      if (isFlat) {
        const fq = document.getElementById(`variant_${variantId}_flat_qty`);
        if (fq) qty = Math.max(1, parseInt(fq.value||'1',10));
      }
      // Safe default if nothing else found
      if (!qty || isNaN(qty)) { qty = 1; }
      let show_time = '';
      const stEl = document.getElementById(`showtime_for_variant_${variantId}`);
      if (stEl) show_time = stEl.value || '';

      const res = await fetch('/b2b/api/activities/book', {
        method: 'POST',
        headers: {'Content-Type':'application/json','Accept':'application/json'},
        body: JSON.stringify({ package_id: packageId, variant_id: variantId, price_id: priceId, qty, show_time })
      });
      const json = await res.json();
      if (!res.ok || json.error) {
        alert(json.error || 'Failed to create booking');
        return;
      }
      alert(`Booking created: #${json.booking_id}\nTotal: ${json.total} ${json.currency}`);
    } catch (e) {
      alert('Error creating booking');
    }
  }
  window.bookActivity = bookActivity;

  // Taxi quote
  const taxiForm = document.getElementById('taxiQuoteForm');
  const taxiResults = document.getElementById('taxiResults');
  if (taxiForm) {
    taxiForm.addEventListener('submit', async (ev) => {
      ev.preventDefault();
      const from = document.getElementById('taxiFrom').value.trim();
      const to = document.getElementById('taxiTo').value.trim();
      const at = document.getElementById('taxiAt').value;
      taxiResults.innerHTML = '<div class="text-muted">Loading…</div>';
      try {
        const res = await fetch(`/b2b/api/taxi/quote?from=${encodeURIComponent(from)}&to=${encodeURIComponent(to)}&at=${encodeURIComponent(at)}`, {headers:{'Accept':'application/json'}});
        const json = await res.json();
        const items = json.data || [];
        if (!items.length) { taxiResults.innerHTML = '<div class="text-muted small">No quotes found.</div>'; return; }
        taxiResults.innerHTML = items.map(t => `
          <div class="border rounded p-2 mb-2 d-flex justify-content-between">
            <div>
              <div class="fw-semibold">${t.name}</div>
              <div class="text-muted small">${t.route}</div>
            </div>
            <div class="text-end">
              <div class="fw-semibold">THB ${Number(t.price||0).toFixed(2)}</div>
              <button class="btn btn-sm btn-outline-primary mt-1">Book</button>
            </div>
          </div>`).join('');
      } catch (e) {
        taxiResults.innerHTML = '<div class="text-danger small">Error loading quotes.</div>';
      }
    });
  }

  // Contact modal submit -> create checkout order
  const contactForm = document.getElementById('contactForm');
  const contactName = document.getElementById('contactName');
  const contactMobile = document.getElementById('contactMobile');
  const contactEmail = document.getElementById('contactEmail');
  const contactWhatsapp = document.getElementById('contactWhatsapp');
  const contactError = document.getElementById('contactFormError');
  const contactSubmitBtn = document.getElementById('contactSubmitBtn');
  if (contactForm) {
    contactForm.addEventListener('submit', async (ev) => {
      ev.preventDefault();
      const name = (contactName?.value||'').trim();
      const mobile = (contactMobile?.value||'').trim();
      const email = (contactEmail?.value||'').trim();
      const whatsapp = (contactWhatsapp?.value||'').trim();
      if (!name || name.length < 2) { contactError.classList.remove('d-none'); contactError.textContent = 'Please enter a valid name (min 2 chars).'; return; }
      if (!mobile || mobile.length < 6) { contactError.classList.remove('d-none'); contactError.textContent = 'Please enter a valid mobile (min 6 chars).'; return; }
      if (email && !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) { contactError.classList.remove('d-none'); contactError.textContent = 'Please enter a valid email address.'; return; }
      contactError.classList.add('d-none');

      const items = Array.isArray(window.__pendingCheckoutItems) ? window.__pendingCheckoutItems : []; 
      if (!items.length) { contactError.classList.remove('d-none'); contactError.textContent = 'No items to checkout.'; return; }

      try {
        if (contactSubmitBtn) { contactSubmitBtn.disabled = true; contactSubmitBtn.textContent = 'Creating…'; }
        const payload = { items, customer_name: name, customer_mobile: mobile, customer_email: email, customer_whatsapp: whatsapp };
        const res = await fetch('/b2b/api/checkout/create', { method:'POST', headers:{ 'Content-Type':'application/json', 'Accept':'application/json', 'X-CSRF-Token': (window.CSRF_TOKEN||'') }, body: JSON.stringify(payload) });
        const json = await res.json();
        if (!res.ok || json.error || !json.order_id) { contactError.classList.remove('d-none'); contactError.textContent = (json && json.error) ? json.error : 'Failed to create order.'; if (contactSubmitBtn) { contactSubmitBtn.disabled = false; contactSubmitBtn.textContent = 'Continue to Checkout'; } return; }
        // Hide modal and redirect
        try { const modalEl = document.getElementById('contactModal'); if (modalEl) { const inst = bootstrap.Modal.getInstance(modalEl) || new bootstrap.Modal(modalEl); inst.hide(); } } catch(_) {}
        if (!json.checkout_url) { contactError.classList.remove('d-none'); contactError.textContent = 'Checkout link missing. Please try again.'; if (contactSubmitBtn) { contactSubmitBtn.disabled = false; contactSubmitBtn.textContent = 'Continue to Checkout'; } return; }
        const next = json.checkout_url;
        window.location.href = next;
      } catch (e) {
        contactError.classList.remove('d-none');
        contactError.textContent = 'Unexpected error while creating order.';
      } finally {
        if (contactSubmitBtn) { contactSubmitBtn.disabled = false; contactSubmitBtn.textContent = 'Continue to Checkout'; }
      }
    });
  }

  // Init
  loadWallet();
</script>
