-- Drop and recreate the function with correct schema
DROP FUNCTION IF EXISTS insert_wallet_transaction;

DELIMITER $$
CREATE FUNCTION insert_wallet_transaction(
    p_user_id INT,
    p_transaction_type ENUM('credit','debit'),
    p_category ENUM('admin_credit','booking_payment','taxi_payment','refund','transfer_in','transfer_out','adjustment','topup'),
    p_amount DECIMAL(12,2),
    p_description VARCHAR(255),
    p_reference_id VARCHAR(100),
    p_reference_type ENUM('order','taxi_booking','hotel_booking','admin_action','transfer','refund'),
    p_admin_user_id INT,
    p_metadata JSON
) RETURNS INT
READS SQL DATA
MODIFIES SQL DATA
DETERMINISTIC
BEGIN
    DECLARE current_balance DECIMAL(12,2) DEFAULT 0.00;
    DECLARE new_balance DECIMAL(12,2) DEFAULT 0.00;
    DECLARE transaction_id INT DEFAULT 0;
    
    -- Get current wallet balance
    SELECT COALESCE(balance, 0.00) INTO current_balance 
    FROM wallets 
    WHERE user_id = p_user_id 
    LIMIT 1;
    
    -- Calculate new balance
    IF p_transaction_type = 'credit' THEN
        SET new_balance = current_balance + p_amount;
    ELSE
        SET new_balance = current_balance - p_amount;
    END IF;
    
    -- Insert transaction record
    INSERT INTO wallet_transactions (
        user_id, transaction_type, category, amount, 
        balance_before, balance_after, description, 
        reference_id, reference_type, admin_user_id, metadata, 
        status, created_at
    ) VALUES (
        p_user_id, p_transaction_type, p_category, p_amount,
        current_balance, new_balance, p_description,
        p_reference_id, p_reference_type, p_admin_user_id, p_metadata,
        'completed', NOW()
    );
    
    SET transaction_id = LAST_INSERT_ID();
    
    -- Update wallet balance (without updated_at column)
    UPDATE wallets 
    SET balance = new_balance 
    WHERE user_id = p_user_id;
    
    RETURN transaction_id;
END$$
DELIMITER ;

-- Drop and recreate triggers
DROP TRIGGER IF EXISTS taxi_refund_wallet_transaction;
DROP TRIGGER IF EXISTS admin_credit_wallet_transaction;

-- Trigger for taxi refunds
DELIMITER $$
CREATE TRIGGER taxi_refund_wallet_transaction
AFTER UPDATE ON taxi_bookings
FOR EACH ROW
BEGIN
    DECLARE refund_amount DECIMAL(12,2) DEFAULT 0.00;
    DECLARE transaction_id INT DEFAULT 0;
    
    -- Check if this is a refund update
    IF (NEW.status = 'refunded' OR NEW.payment_status = 'refunded' OR NEW.refund_amount IS NOT NULL) 
       AND (OLD.status != 'refunded' AND OLD.payment_status != 'refunded' AND OLD.refund_amount IS NULL) THEN
        
        -- Determine refund amount
        SET refund_amount = COALESCE(NEW.refund_amount, NEW.amount_total);
        
        -- Only process if refund amount > 0 and payment was via wallet
        IF refund_amount > 0 AND NEW.payment_method = 'wallet' THEN
            -- Insert wallet transaction for refund
            SET transaction_id = insert_wallet_transaction(
                NEW.agent_id,
                'credit',
                'refund',
                refund_amount,
                CONCAT('Taxi Booking Refund - ', COALESCE(NEW.booking_code, NEW.id)),
                NEW.id,
                'taxi_booking',
                NULL,
                JSON_OBJECT(
                    'original_amount', NEW.amount_total,
                    'refund_amount', refund_amount,
                    'booking_code', NEW.booking_code,
                    'refund_reason', 'Taxi booking cancelled'
                )
            );
        END IF;
    END IF;
END$$
DELIMITER ;

-- Trigger for admin credits (when wallets table is updated by admin)
DELIMITER $$
CREATE TRIGGER admin_credit_wallet_transaction
AFTER UPDATE ON wallets
FOR EACH ROW
BEGIN
    DECLARE credit_amount DECIMAL(12,2) DEFAULT 0.00;
    DECLARE transaction_id INT DEFAULT 0;
    
    -- Check if balance increased (admin credit)
    IF NEW.balance > OLD.balance THEN
        SET credit_amount = NEW.balance - OLD.balance;
        
        -- Insert wallet transaction for admin credit
        SET transaction_id = insert_wallet_transaction(
            NEW.user_id,
            'credit',
            'admin_credit',
            credit_amount,
            CONCAT('Admin Credit - Balance adjustment of ฿', FORMAT(credit_amount, 2)),
            CONCAT('ADMIN_CREDIT_', UNIX_TIMESTAMP()),
            'admin_action',
            NULL, -- We don't track which admin did this yet
            JSON_OBJECT(
                'previous_balance', OLD.balance,
                'new_balance', NEW.balance,
                'credit_amount', credit_amount,
                'admin_note', 'Manual balance adjustment'
            )
        );
    END IF;
END$$
DELIMITER ;

-- =============================================================
-- Holiday Package Bookings (complete the 5th booking type)
-- and a Unified Agent Booking History View
-- =============================================================

-- Create holiday package bookings if missing
CREATE TABLE IF NOT EXISTS holiday_package_bookings (
  id                INT UNSIGNED NOT NULL AUTO_INCREMENT,
  booking_code      VARCHAR(40)      NULL,
  agent_id          INT UNSIGNED     NOT NULL,
  holiday_id        INT              NOT NULL,
  variant_id        INT              NULL,
  start_date        DATE             NOT NULL,
  end_date          DATE             NOT NULL,
  pax               INT UNSIGNED     NOT NULL DEFAULT 1,
  amount_total      DECIMAL(12,2)    NOT NULL DEFAULT 0.00,
  currency          CHAR(3)          NOT NULL DEFAULT 'THB',
  status            ENUM('pending','confirmed','cancelled','refunded') NOT NULL DEFAULT 'pending',
  payment_status    ENUM('unpaid','paid','refunded','partial')         NOT NULL DEFAULT 'unpaid',
  payment_method    VARCHAR(50)      NULL,
  gateway_name      VARCHAR(50)      NULL,
  payment_txn_id    VARCHAR(100)     NULL,
  paid_at           DATETIME         NULL,
  cancelled_at      DATETIME         NULL,
  cancel_reason     VARCHAR(500)     NULL,
  refund_amount     DECIMAL(12,2)    NULL,
  refunded_at       DATETIME         NULL,
  refund_txn_id     VARCHAR(100)     NULL,
  channel           VARCHAR(50)      NULL,
  customer_name     VARCHAR(150)     NULL,
  customer_email    VARCHAR(150)     NULL,
  customer_phone    VARCHAR(50)      NULL,
  ip_address        VARCHAR(64)      NULL,
  user_agent        VARCHAR(255)     NULL,
  created_by        INT UNSIGNED     NULL,
  updated_by        INT UNSIGNED     NULL,
  created_at        DATETIME         NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at        DATETIME         NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  UNIQUE KEY uq_holiday_booking_code (booking_code),
  KEY idx_hpb_agent (agent_id),
  KEY idx_hpb_status (status),
  KEY idx_hpb_payment (payment_status),
  KEY idx_hpb_dates (start_date, end_date)
);

-- Events table for holiday bookings (audit trail)
CREATE TABLE IF NOT EXISTS holiday_booking_events (
  id         INT UNSIGNED NOT NULL AUTO_INCREMENT,
  booking_id INT UNSIGNED NOT NULL,
  user_id    INT UNSIGNED NULL,
  event_type VARCHAR(48)  NOT NULL,
  note       VARCHAR(255) NULL,
  data_json  LONGTEXT,
  created_at DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  KEY idx_hbe_booking (booking_id),
  KEY idx_hbe_type (event_type)
);

-- ---------------------------------------------
-- Unified Agent Booking History View
-- Columns:
--   booking_type, booking_id, booking_code, agent_id, vendor_id,
--   status, payment_status, created_at, travel_date, amount_total,
--   currency, customer_name, customer_email, customer_phone, channel
-- ---------------------------------------------
DROP VIEW IF EXISTS vw_agent_booking_history;
CREATE OR REPLACE VIEW vw_agent_booking_history AS
-- Activity
SELECT
  'activity'                                   AS booking_type,
  ab.id                                        AS booking_id,
  ab.booking_code                              AS booking_code,
  ab.agent_id                                  AS agent_id,
  ab.vendor_id                                 AS vendor_id,
  ab.status                                    AS status,
  ab.payment_status                            AS payment_status,
  ab.created_at                                AS created_at,
  ab.booking_date                              AS travel_date,
  ab.amount_total                              AS amount_total,
  ab.currency                                  AS currency,
  ab.lead_name                                 AS customer_name,
  ab.lead_email                                AS customer_email,
  ab.lead_phone                                AS customer_phone,
  ab.channel                                   AS channel
FROM activity_bookings ab

UNION ALL

-- Hotel
SELECT
  'hotel'                                      AS booking_type,
  hb.id                                        AS booking_id,
  hb.booking_code                              AS booking_code,
  hb.user_id                                   AS agent_id,
  hb.vendor_id                                 AS vendor_id,
  hb.status                                    AS status,
  hb.payment_status                            AS payment_status,
  hb.created_at                                AS created_at,
  hb.checkin                                   AS travel_date,
  hb.total_price                               AS amount_total,
  hb.currency                                  AS currency,
  hb.customer_name                             AS customer_name,
  hb.customer_email                            AS customer_email,
  NULL                                         AS customer_phone,
  NULL                                         AS channel
FROM hotel_bookings hb

UNION ALL

-- Taxi
SELECT
  'taxi'                                       AS booking_type,
  tb.id                                        AS booking_id,
  tb.booking_code                              AS booking_code,
  tb.agent_id                                  AS agent_id,
  tb.vendor_id                                 AS vendor_id,
  tb.status                                    AS status,
  tb.payment_status                            AS payment_status,
  tb.created_at                                AS created_at,
  tb.trip_date                                 AS travel_date,
  COALESCE(tb.amount, tb.total_amount, 0.00)   AS amount_total,
  COALESCE(tb.currency, 'THB')                 AS currency,
  tb.customer_name                             AS customer_name,
  tb.customer_email                            AS customer_email,
  tb.customer_phone                            AS customer_phone,
  tb.channel                                   AS channel
FROM taxi_bookings tb

UNION ALL

-- Yacht
SELECT
  'yacht'                                      AS booking_type,
  yb.id                                        AS booking_id,
  yb.booking_code                              AS booking_code,
  yb.agent_id                                  AS agent_id,
  NULL                                         AS vendor_id,
  yb.status                                    AS status,
  yb.payment_status                            AS payment_status,
  yb.created_at                                AS created_at,
  yb.trip_date                                 AS travel_date,
  yb.amount_total                              AS amount_total,
  yb.currency                                  AS currency,
  NULL                                         AS customer_name,
  NULL                                         AS customer_email,
  NULL                                         AS customer_phone,
  yb.channel                                   AS channel
FROM yacht_bookings yb

UNION ALL

-- Holiday (new)
SELECT
  'holiday'                                    AS booking_type,
  hp.id                                        AS booking_id,
  hp.booking_code                              AS booking_code,
  hp.agent_id                                  AS agent_id,
  NULL                                         AS vendor_id,
  hp.status                                    AS status,
  hp.payment_status                            AS payment_status,
  hp.created_at                                AS created_at,
  hp.start_date                                AS travel_date,
  hp.amount_total                              AS amount_total,
  hp.currency                                  AS currency,
  hp.customer_name                             AS customer_name,
  hp.customer_email                            AS customer_email,
  hp.customer_phone                            AS customer_phone,
  hp.channel                                   AS channel
FROM holiday_package_bookings hp;

-- ---------------------------------------------
-- Unified Payments & Refunds View (Agent scoped via agent_id)
-- Notes:
--  - Includes module payment tables (taxi_payments, yacht_payments) joined to module bookings
--  - Includes wallet_transactions for categories likely tied to bookings
-- ---------------------------------------------
DROP VIEW IF EXISTS vw_agent_booking_payments_refunds;
CREATE OR REPLACE VIEW vw_agent_booking_payments_refunds AS
-- Taxi payments (captures and refunds)
SELECT
  'taxi'                AS booking_type,
  b.id                  AS booking_id,
  b.agent_id            AS agent_id,
  p.method              AS method,
  COALESCE(p.gateway_name,'wallet') AS gateway,
  p.amount              AS amount,
  p.currency            AS currency,
  p.status              AS status,
  p.txn_id              AS txn_id,
  p.created_at          AS created_at,
  p.updated_at          AS updated_at
FROM taxi_payments p
JOIN taxi_bookings b ON b.id = p.booking_id

UNION ALL

-- Yacht payments (captures and refunds)
SELECT
  'yacht'               AS booking_type,
  b.id                  AS booking_id,
  b.agent_id            AS agent_id,
  p.method              AS method,
  COALESCE(p.gateway_name,'wallet') AS gateway,
  p.amount              AS amount,
  p.currency            AS currency,
  p.status              AS status,
  p.txn_id              AS txn_id,
  p.created_at          AS created_at,
  p.updated_at          AS updated_at
FROM yacht_payments p
JOIN yacht_bookings b ON b.id = p.booking_id

UNION ALL

-- Wallet transactions tied to booking flow (agent scoped)
SELECT
  NULL                  AS booking_type,
  NULL                  AS booking_id,
  wt.user_id            AS agent_id,
  CASE WHEN wt.transaction_type = 'credit' THEN 'wallet_credit' ELSE 'wallet_debit' END AS method,
  'wallet'              AS gateway,
  wt.amount             AS amount,
  'THB'                 AS currency,
  wt.status             AS status,
  wt.reference_id       AS txn_id,
  wt.created_at         AS created_at,
  wt.updated_at         AS updated_at
FROM wallet_transactions wt
WHERE wt.category IN ('booking_payment','taxi_payment','refund');

-- ---------------------------------------------
-- Unified Partner Commissions View
-- Maps commissions to agent and module using the generic bookings table
-- ---------------------------------------------
DROP VIEW IF EXISTS vw_agent_commissions;
CREATE OR REPLACE VIEW vw_agent_commissions AS
SELECT
  pc.id                 AS commission_id,
  b.module              AS booking_type,
  pc.booking_id         AS booking_id,
  b.user_id             AS agent_id,
  pc.amount             AS commission_amount,
  pc.status             AS status,
  pc.created_at         AS created_at
FROM partner_commissions pc
JOIN bookings b ON b.id = pc.booking_id;

-- ---------------------------------------------
-- Basic Profit View (per booking, best-effort)
-- Calculates net_collected from module amounts (or payments), then subtracts vendor_cost and commissions
-- NOTE: For accuracy, ensure vendor_cost is snapshotted on each booking row at creation time
-- ---------------------------------------------
DROP VIEW IF EXISTS vw_agent_profit;
CREATE OR REPLACE VIEW vw_agent_profit AS
-- Activity: use activity_bookings.amount_total as collected
SELECT
  'activity'                           AS booking_type,
  ab.id                                AS booking_id,
  ab.agent_id                          AS agent_id,
  ab.amount_total                      AS collected,
  NULL                                  AS vendor_cost,
  0.00                                  AS commissions,
  ab.amount_total                       AS profit,
  ab.currency                           AS currency,
  ab.created_at                         AS created_at
FROM activity_bookings ab

UNION ALL

-- Hotel: use hotel_bookings.total_price as collected
SELECT
  'hotel'                              AS booking_type,
  hb.id                                AS booking_id,
  hb.agent_id                          AS agent_id,
  hb.total_price                       AS collected,
  NULL                                  AS vendor_cost,
  0.00                                  AS commissions,
  hb.total_price                        AS profit,
  hb.currency                           AS currency,
  hb.created_at                         AS created_at
FROM hotel_bookings hb

UNION ALL

-- Taxi: sum of captured taxi_payments per booking
SELECT
  'taxi'                               AS booking_type,
  b.id                                 AS booking_id,
  b.agent_id                           AS agent_id,
  COALESCE((SELECT SUM(tp.amount) FROM taxi_payments tp WHERE tp.booking_id = b.id AND tp.status = 'captured'),0.00) AS collected,
  NULL                                  AS vendor_cost,
  0.00                                  AS commissions,
  COALESCE((SELECT SUM(tp.amount) FROM taxi_payments tp WHERE tp.booking_id = b.id AND tp.status = 'captured'),0.00) AS profit,
  COALESCE(b.currency,'THB')            AS currency,
  b.created_at                          AS created_at
FROM taxi_bookings b

UNION ALL

-- Yacht: amount_total as collected
SELECT
  'yacht'                              AS booking_type,
  yb.id                                AS booking_id,
  yb.agent_id                          AS agent_id,
  yb.amount_total                      AS collected,
  NULL                                  AS vendor_cost,
  0.00                                  AS commissions,
  yb.amount_total                       AS profit,
  yb.currency                           AS currency,
  yb.created_at                         AS created_at
FROM yacht_bookings yb

UNION ALL

-- Holiday: amount_total as collected
SELECT
  'holiday'                            AS booking_type,
  hp.id                                AS booking_id,
  hp.agent_id                          AS agent_id,
  hp.amount_total                      AS collected,
  NULL                                  AS vendor_cost,
  0.00                                  AS commissions,
  hp.amount_total                       AS profit,
  hp.currency                           AS currency,
  hp.created_at                         AS created_at
FROM holiday_package_bookings hp;

-- ======================================================================
-- Re-define vw_agent_booking_history with safe columns and unified collation
-- ======================================================================
DROP VIEW IF EXISTS vw_agent_booking_history;
CREATE OR REPLACE VIEW vw_agent_booking_history AS
-- Activity
SELECT
  'activity' COLLATE utf8mb4_unicode_ci                  AS booking_type,
  ab.id                                                  AS booking_id,
  ab.booking_code COLLATE utf8mb4_unicode_ci             AS booking_code,
  ab.agent_id                                            AS agent_id,
  ab.vendor_id                                           AS vendor_id,
  ab.status COLLATE utf8mb4_unicode_ci                   AS status,
  ab.payment_status COLLATE utf8mb4_unicode_ci           AS payment_status,
  ab.created_at                                          AS created_at,
  ab.booking_date                                        AS travel_date,
  ab.amount_total                                        AS amount_total,
  ab.currency COLLATE utf8mb4_unicode_ci                 AS currency,
  ab.lead_name COLLATE utf8mb4_unicode_ci                AS customer_name,
  ab.lead_email COLLATE utf8mb4_unicode_ci               AS customer_email,
  ab.lead_phone COLLATE utf8mb4_unicode_ci               AS customer_phone,
  ab.channel COLLATE utf8mb4_unicode_ci                  AS channel
FROM activity_bookings ab

UNION ALL

-- Hotel (no user_id column in hotel_bookings)
SELECT
  'hotel' COLLATE utf8mb4_unicode_ci                     AS booking_type,
  hb.id                                                  AS booking_id,
  hb.booking_code COLLATE utf8mb4_unicode_ci             AS booking_code,
  hb.agent_id                                            AS agent_id,
  hb.vendor_id                                           AS vendor_id,
  hb.status COLLATE utf8mb4_unicode_ci                   AS status,
  hb.payment_status COLLATE utf8mb4_unicode_ci           AS payment_status,
  hb.created_at                                          AS created_at,
  hb.checkin                                             AS travel_date,
  hb.total_price                                         AS amount_total,
  hb.currency COLLATE utf8mb4_unicode_ci                 AS currency,
  hb.customer_name COLLATE utf8mb4_unicode_ci            AS customer_name,
  hb.customer_email COLLATE utf8mb4_unicode_ci           AS customer_email,
  NULL                                                   AS customer_phone,
  NULL                                                   AS channel
FROM hotel_bookings hb

UNION ALL

-- Taxi (derive amount from payments; safe defaults for text)
SELECT
  'taxi' COLLATE utf8mb4_unicode_ci                      AS booking_type,
  tb.id                                                  AS booking_id,
  tb.booking_code COLLATE utf8mb4_unicode_ci             AS booking_code,
  tb.agent_id                                            AS agent_id,
  tb.vendor_id                                           AS vendor_id,
  tb.status COLLATE utf8mb4_unicode_ci                   AS status,
  tb.payment_status COLLATE utf8mb4_unicode_ci           AS payment_status,
  tb.created_at                                          AS created_at,
  tb.trip_date                                           AS travel_date,
  COALESCE((SELECT SUM(tp.amount) FROM taxi_payments tp WHERE tp.booking_id = tb.id AND tp.status = 'captured'), 0.00) AS amount_total,
  'THB' COLLATE utf8mb4_unicode_ci                       AS currency,
  NULL                                                   AS customer_name,
  NULL                                                   AS customer_email,
  NULL                                                   AS customer_phone,
  NULL                                                   AS channel
FROM taxi_bookings tb

UNION ALL

-- Yacht
SELECT
  'yacht' COLLATE utf8mb4_unicode_ci                     AS booking_type,
  yb.id                                                  AS booking_id,
  yb.booking_code COLLATE utf8mb4_unicode_ci             AS booking_code,
  yb.agent_id                                            AS agent_id,
  NULL                                                   AS vendor_id,
  yb.status COLLATE utf8mb4_unicode_ci                   AS status,
  yb.payment_status COLLATE utf8mb4_unicode_ci           AS payment_status,
  yb.created_at                                          AS created_at,
  yb.trip_date                                           AS travel_date,
  yb.amount_total                                        AS amount_total,
  yb.currency COLLATE utf8mb4_unicode_ci                 AS currency,
  NULL                                                   AS customer_name,
  NULL                                                   AS customer_email,
  NULL                                                   AS customer_phone,
  yb.channel COLLATE utf8mb4_unicode_ci                  AS channel
FROM yacht_bookings yb

UNION ALL

-- Holiday
SELECT
  'holiday' COLLATE utf8mb4_unicode_ci                   AS booking_type,
  hp.id                                                  AS booking_id,
  hp.booking_code COLLATE utf8mb4_unicode_ci             AS booking_code,
  hp.agent_id                                            AS agent_id,
  NULL                                                   AS vendor_id,
  hp.status COLLATE utf8mb4_unicode_ci                   AS status,
  hp.payment_status COLLATE utf8mb4_unicode_ci           AS payment_status,
  hp.created_at                                          AS created_at,
  hp.start_date                                          AS travel_date,
  hp.amount_total                                        AS amount_total,
  hp.currency COLLATE utf8mb4_unicode_ci                 AS currency,
  hp.customer_name COLLATE utf8mb4_unicode_ci            AS customer_name,
  hp.customer_email COLLATE utf8mb4_unicode_ci           AS customer_email,
  hp.customer_phone COLLATE utf8mb4_unicode_ci           AS customer_phone,
  hp.channel COLLATE utf8mb4_unicode_ci                  AS channel
FROM holiday_package_bookings hp;
