<?php
// app/Views/agent/taxi_payment.php
$title = $title ?? 'Taxi Payment';
$booking = $booking ?? null;
$intent = $intent ?? null;
$csrf_pay = $csrf_pay ?? '';
$active = 'taxi';
require __DIR__ . '/partials/header.php';
?>

<div class="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-50">
  <!-- Professional Header -->
  <div class="bg-white border-b border-slate-200 shadow-sm">
    <div class="max-w-7xl mx-auto px-4 py-4">
      <nav class="flex items-center gap-2 text-sm text-slate-600 mb-2">
        <a href="/b2b/agent/taxi" class="hover:text-blue-600 transition-colors">Transfers</a>
        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"/>
        </svg>
        <span class="text-slate-900 font-medium">Payment</span>
      </nav>
      <h1 class="text-2xl font-bold text-slate-900">Complete Your Payment</h1>
      <p class="text-slate-600 mt-1">Review your booking details and proceed with payment</p>
    </div>
  </div>

  <?php
    // Map error codes from query string to friendly messages
    $err = (string)($_GET['err'] ?? '');
    $errMsg = '';
    $details = (string)($_GET['details'] ?? '');
    $dbError = (string)($_GET['db_error'] ?? '');
    
    switch ($err) {
      case 'pin': $errMsg = 'Invalid PIN. Please enter a 4-6 digit numeric PIN.'; break;
      case 'pin_invalid': 
        $attemptsLeft = (int)($_GET['left'] ?? 0);
        $errMsg = 'Incorrect PIN. ' . ($attemptsLeft > 0 
          ? "$attemptsLeft attempts remaining." 
          : 'Please try again later.');
        break;
      case 'pin_not_set': 
        $errMsg = 'Transaction PIN not set. Please set a PIN in your account settings before making wallet payments.'; 
        break;
      case 'wallet_locked': 
        $errMsg = 'Account temporarily locked. Too many failed PIN attempts. Please try again in 24 hours or contact support.'; 
        break;
      case 'pin_check': 
        $errMsg = 'Unable to verify PIN. Please try again or contact support if the issue persists.'; 
        break;
      case 'pin_locked': 
        $errMsg = 'Account locked. Too many failed PIN attempts. Please try again in 24 hours or contact support.'; 
        break;
      case 'insufficient': 
        $errMsg = 'Insufficient wallet balance. Please top up your wallet or choose another payment method.'; 
        break;
      case 'ledger': 
        $errMsg = 'We could not process your wallet payment. Please try again or choose another payment method.'; 
        break;
      case 'amount': 
        $errMsg = 'Payment amount validation failed. Please refresh the page and try again.'; 
        break;
      case 'verify_mismatch': 
        $errMsg = 'Amount verification mismatch. Please go back and re-check your booking details.'; 
        break;
      case 'notfound': 
        $errMsg = 'We could not find your booking. Please start a new booking.'; 
        break;
      case 'gateway_not_configured': 
        $errMsg = 'Card payment gateway is not configured. Please choose another payment method.'; 
        break;
      case 'db_insert': 
        $errMsg = 'We could not confirm your booking due to a system error. Please try again.'; 
        break;
      case 'exception': 
        $errMsg = 'An error occurred: ' . htmlspecialchars($details);
        if ($dbError) {
            $errMsg .= ' (Database: ' . htmlspecialchars($dbError) . ')';
        }
        break;
      default: 
        $errMsg = '';
    }
  ?>
  <?php if ($errMsg !== ''): ?>
    <div class="max-w-5xl mx-auto px-4 pt-4">
      <div class="bg-red-50 border border-red-200 text-red-700 rounded-lg p-3 text-sm font-mono">
        <div class="font-bold mb-2">Error Details:</div>
        <div class="whitespace-pre-wrap"><?= nl2br(htmlspecialchars($errMsg)) ?></div>
        <?php if (!empty($dbError)): ?>
          <div class="mt-2 pt-2 border-t border-red-100">
            <div class="font-semibold">Database Error:</div>
            <div class="whitespace-pre-wrap"><?= htmlspecialchars($dbError) ?></div>
          </div>
        <?php endif; ?>
      </div>
    </div>
  <?php endif; ?>

  <div class="max-w-5xl mx-auto px-4 py-8">
    <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
      <!-- Payment Summary Sidebar -->
      <div class="lg:col-span-1">
        <div class="bg-gradient-to-br from-white to-slate-50 rounded-2xl shadow-xl border border-slate-200 p-6 sticky top-6">
          <!-- Header with Icon -->
          <div class="flex items-center gap-3 mb-6">
            <div class="w-10 h-10 bg-gradient-to-r from-emerald-500 to-green-500 rounded-full flex items-center justify-center">
              <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
              </svg>
            </div>
            <div>
              <h2 class="text-lg font-bold text-slate-900">Payment Summary</h2>
              <p class="text-xs text-slate-600">Secure & Instant</p>
            </div>
          </div>
          
          <!-- Booking Details Cards -->
          <div class="space-y-3 mb-6">
            <div class="bg-white rounded-lg p-3 border border-slate-100">
              <div class="flex justify-between items-center">
                <span class="text-xs text-slate-500 uppercase tracking-wide">Booking Code</span>
                <span class="text-sm font-bold text-slate-900">
                  <?php if ($booking): ?>
                    <?= htmlspecialchars((string)($booking['booking_code'] ?? ('TAX-' . (int)($booking['id'] ?? 0)))) ?>
                  <?php else: ?>
                    <?php $intentCode = 'INT-' . date('Ymd') . '-' . (int)($intent['id'] ?? 0); ?>
                    <?= htmlspecialchars($intentCode) ?>
                  <?php endif; ?>
                </span>
              </div>
            </div>
            
            <div class="bg-white rounded-lg p-3 border border-slate-100">
              <div class="flex justify-between items-center">
                <span class="text-xs text-slate-500 uppercase tracking-wide">Transfer</span>
                <span class="text-sm font-bold text-slate-900">
                  <?= htmlspecialchars((string)($booking['ride_name'] ?? ($intent['ride_name'] ?? '—'))) ?>
                </span>
              </div>
            </div>
            
            <div class="grid grid-cols-2 gap-2">
              <div class="bg-white rounded-lg p-3 border border-slate-100">
                <div class="text-center">
                  <div class="text-xs text-slate-500 uppercase tracking-wide mb-1">Passengers</div>
                  <div class="text-lg font-bold text-slate-900"><?= (int)($booking['pax'] ?? ($intent['pax'] ?? 1)) ?></div>
                </div>
              </div>
              <div class="bg-white rounded-lg p-3 border border-slate-100">
                <div class="text-center">
                  <div class="text-xs text-slate-500 uppercase tracking-wide mb-1">Date</div>
                  <div class="text-sm font-bold text-slate-900"><?= date('M j', strtotime($booking['trip_date'] ?? ($intent['trip_date'] ?? 'now'))) ?></div>
                </div>
              </div>
            </div>
          </div>

          <?php
            $amt = $booking ? (float)($booking['amount_total'] ?? 0) : (float)($intent['amount'] ?? 0);
            $cur = (string)($booking['currency'] ?? ($intent['currency'] ?? 'THB'));
          ?>
          
          <!-- Total Amount -->
          <div class="bg-gradient-to-r from-red-500 to-rose-500 rounded-xl p-4 text-center text-white">
            <div class="text-sm opacity-90 mb-1">Total Amount</div>
            <div class="text-3xl font-black">
              ฿<?= number_format($amt, 0) ?>
            </div>
            <div class="text-xs opacity-75 mt-2">
              <span class="bg-white bg-opacity-20 px-2 py-1 rounded-full">
                Payment Due
              </span>
            </div>
          </div>
        </div>
      </div>

      <!-- Payment Details -->
      <div class="lg:col-span-2">
        <div class="bg-gradient-to-br from-white to-slate-50 rounded-2xl shadow-xl border border-slate-200 p-8">
          <!-- Header with Icon -->
          <div class="flex items-center gap-3 mb-6">
            <div class="w-10 h-10 bg-gradient-to-r from-blue-500 to-indigo-500 rounded-full flex items-center justify-center">
              <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 17a2 2 0 11-4 0 2 2 0 014 0zM19 17a2 2 0 11-4 0 2 2 0 014 0z"/>
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16V6a1 1 0 00-1-1H4a1 1 0 00-1 1v10a1 1 0 001 1h1m8-1a1 1 0 01-1 1H9m4-1V8a1 1 0 011-1h2.586a1 1 0 01.707.293l2.414 2.414a1 1 0 01.293.707V16a1 1 0 01-1 1h-1m-6-1a1 1 0 001 1h1M5 17a2 2 0 104 0M15 17a2 2 0 104 0"/>
              </svg>
            </div>
            <div>
              <h2 class="text-xl font-bold text-slate-900">Booking Details</h2>
              <p class="text-xs text-slate-600">Transfer Information</p>
            </div>
          </div>

          <!-- Trip Route -->
          <div class="bg-gradient-to-r from-emerald-50 to-green-50 rounded-xl p-5 mb-6 border border-emerald-100">
            <div class="flex items-center gap-2 mb-4">
              <div class="w-6 h-6 bg-emerald-500 rounded-full flex items-center justify-center">
                <svg class="w-3 h-3 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"/>
                  <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"/>
                </svg>
              </div>
              <h3 class="text-sm font-bold text-emerald-800">Trip Route</h3>
            </div>

            <?php
              // Derive structured pickup/drop view values from either $booking or $intent
              $getv = function($k, $def='') use ($booking,$intent){ return htmlspecialchars((string)($booking[$k] ?? ($intent[$k] ?? $def))); };
              $pickupType = strtolower($booking['pickup_type'] ?? ($intent['pickup_type'] ?? ''));
              $dropType   = strtolower($booking['drop_type']   ?? ($intent['drop_type']   ?? ''));
              // Pickup formatted line
              $pickupLine = '';
              if ($pickupType === 'airport') { $pickupLine = 'Airport — ' . $getv('pickup_airport_code'); }
              elseif ($pickupType === 'hotel') { $pickupLine = 'Hotel — ' . $getv('pickup_hotel_name') . ($getv('pickup_hotel_address')!=='' ? ', ' . $getv('pickup_hotel_address') : ''); }
              elseif ($pickupType === 'apartment') { $pickupLine = 'Apartment — ' . $getv('pickup_apartment_name') . ($getv('pickup_apartment_address')!=='' ? ', ' . $getv('pickup_apartment_address') : ''); }
              elseif ($pickupType === 'address') { $pickupLine = 'Address — ' . $getv('pickup_address'); }
              // Drop formatted line
              $dropLine = '';
              if ($dropType === 'airport') { $dropLine = 'Airport — ' . $getv('drop_airport_code'); }
              elseif ($dropType === 'hotel') { $dropLine = 'Hotel — ' . $getv('drop_hotel_name') . ($getv('drop_hotel_address')!=='' ? ', ' . $getv('drop_hotel_address') : ''); }
              elseif ($dropType === 'apartment') { $dropLine = 'Apartment — ' . $getv('drop_apartment_name') . ($getv('drop_apartment_address')!=='' ? ', ' . $getv('drop_apartment_address') : ''); }
              elseif ($dropType === 'address') { $dropLine = 'Address — ' . $getv('drop_address'); }
              $itDesc = (string)($itinerary_desc ?? '');
            ?>

            <div class="flex items-center gap-4">
              <div class="flex-1 bg-white rounded-lg p-3 border border-emerald-200">
                <div class="text-xs text-emerald-600 font-medium mb-1 uppercase tracking-wide">Pickup</div>
                <div class="text-sm font-bold text-slate-900"><?= $pickupLine !== '' ? $pickupLine : '—' ?></div>
              </div>
              <div class="flex-shrink-0">
                <div class="w-8 h-8 bg-emerald-500 rounded-full flex items-center justify-center">
                  <svg class="w-4 h-4 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 8l4 4m0 0l-4 4m4-4H3"/>
                  </svg>
                </div>
              </div>
              <div class="flex-1 bg-white rounded-lg p-3 border border-emerald-200">
                <div class="text-xs text-emerald-600 font-medium mb-1 uppercase tracking-wide">Drop-off</div>
                <div class="text-sm font-bold text-slate-900"><?= $dropLine !== '' ? $dropLine : '—' ?></div>
              </div>
            </div>
          </div>

          <?php if ($itDesc !== ''): ?>
          <div class="bg-blue-50 rounded-xl p-4 mb-6">
            <h3 class="text-sm font-semibold text-slate-700 mb-2">Itinerary Notes</h3>
            <div class="text-sm text-slate-600 whitespace-pre-line"><?= htmlspecialchars($itDesc) ?></div>
          </div>
          <?php endif; ?>

          <!-- Customer & Trip Details -->
          <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
            <div class="space-y-4">
              <h3 class="text-sm font-semibold text-slate-700">Customer Information</h3>
              <div class="space-y-3">
                <div>
                  <div class="text-xs text-slate-500">Name</div>
                  <div class="text-sm font-medium text-slate-900"><?= $getv('customer_name','—') ?></div>
                </div>
                <div>
                  <div class="text-xs text-slate-500">Email</div>
                  <div class="text-sm font-medium text-slate-900 break-all"><?= $getv('customer_email','—') ?></div>
                </div>
                <div>
                  <div class="text-xs text-slate-500">Phone</div>
                  <div class="text-sm font-medium text-slate-900"><?= $getv('customer_phone','—') ?></div>
                </div>
              </div>
            </div>
            
            <div class="space-y-4">
              <h3 class="text-sm font-semibold text-slate-700">Trip Information</h3>
              <div class="space-y-3">
                <div>
                  <div class="text-xs text-slate-500">Pickup Time</div>
                  <div class="text-sm font-medium text-slate-900"><?= htmlspecialchars((string)($booking['pickup_time'] ?? ($intent['pickup_time'] ?? '—'))) ?></div>
                </div>
                <div>
                  <div class="text-xs text-slate-500">Flight Number</div>
                  <div class="text-sm font-medium text-slate-900"><?= htmlspecialchars((string)($booking['flight_no'] ?? ($intent['flight_no'] ?? '—'))) ?></div>
                </div>
                <?php $notes = htmlspecialchars((string)($booking['notes'] ?? ($intent['notes'] ?? ''))); ?>
                <?php if ($notes && $notes !== '—'): ?>
                <div>
                  <div class="text-xs text-slate-500">Special Notes</div>
                  <div class="text-sm font-medium text-slate-900 whitespace-pre-line"><?= $notes ?></div>
                </div>
                <?php endif; ?>
              </div>
            </div>
          </div>

      <?php
        // Extract itinerary stops and compute extras subtotal
        $itJson = $booking ? (string)($booking['itinerary_json'] ?? '') : (string)($intent['itinerary_json'] ?? '');
        $stops = [];
        $extrasSubtotal = 0.0;
        if ($itJson) {
          $arr = json_decode($itJson, true);
          if (is_array($arr)) {
            $stops = $arr;
            foreach ($arr as $s) { $extrasSubtotal += (float)($s['extra_price'] ?? 0); }
          }
        }
        $amountTotal = $booking ? (float)($booking['amount_total'] ?? 0) : (float)($intent['amount'] ?? 0);
        $baseDerived = max(0.0, $amountTotal - $extrasSubtotal);
      ?>

      <?php if (!empty($stops)): ?>
      <div class="mb-4">
        <div class="text-sm font-semibold mb-2">Itinerary Stops</div>
        <div class="overflow-x-auto">
          <table class="table table-zebra w-full text-sm">
            <thead>
              <tr>
                <th style="width:48px">#</th>
                <th style="width:200px">Label</th>
                <th>Address</th>
                <th style="width:120px">Wait (min)</th>
                <th style="width:140px">Extra (THB)</th>
                <th style="width:220px">Notes</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($stops as $i => $s): ?>
                <tr>
                  <td class="text-muted"><?= (int)($s['seq'] ?? ($i+1)) ?></td>
                  <td><?= htmlspecialchars((string)($s['label'] ?? '')) ?></td>
                  <td><?= htmlspecialchars((string)($s['address'] ?? '')) ?></td>
                  <td><?= htmlspecialchars((string)($s['wait_minutes'] ?? '')) ?></td>
                  <td><?= number_format((float)($s['extra_price'] ?? 0), 2) ?></td>
                  <td><?= htmlspecialchars((string)($s['notes'] ?? '')) ?></td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
        <div class="mt-3 text-sm">
          <div>Extras Subtotal: <span class="font-semibold"><?= number_format($extrasSubtotal, 2) ?> THB</span></div>
          <div>Base (derived): <span class="font-semibold"><?= number_format($baseDerived, 2) ?> THB</span></div>
          <div>Total: <span class="font-extrabold"><?= number_format($amountTotal, 2) ?> THB</span></div>
        </div>
      </div>
      <?php endif; ?>

          <!-- Payment Method -->
          <div class="border-t border-slate-200 pt-6">
            <h3 class="text-lg font-bold text-slate-900 mb-4">Choose Payment Method</h3>
            
            <form id="taxiPayForm" method="post" action="/b2b/agent/taxi/pay" class="space-y-6">
              <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrf_pay) ?>">
              <?php if ($booking): ?>
                <input type="hidden" name="booking_id" value="<?= (int)($booking['id'] ?? 0) ?>">
              <?php endif; ?>
              <?php if ($intent): ?>
                <input type="hidden" name="intent_id" value="<?= (int)($intent['id'] ?? 0) ?>">
              <?php endif; ?>

              <div class="space-y-3">
                <label class="flex items-center p-4 border border-slate-300 rounded-lg cursor-pointer hover:bg-slate-50 transition-colors">
                  <input type="radio" name="method" value="wallet" id="wallet_payment" class="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 focus:ring-blue-500 focus:ring-2">
                  <div class="ml-3 flex-1">
                    <div class="text-sm font-medium text-slate-900">Wallet Payment</div>
                    <div class="text-xs text-slate-500">Pay using your agent wallet balance</div>
                  </div>
                  <svg class="w-5 h-5 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 003 3z"/>
                  </svg>
                </label>
                
                <label class="flex items-center p-4 border border-slate-300 rounded-lg cursor-pointer hover:bg-slate-50 transition-colors">
                  <input type="radio" name="method" value="gateway" id="card_payment" class="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 focus:ring-blue-500 focus:ring-2">
                  <div class="ml-3 flex-1">
                    <div class="text-sm font-medium text-slate-900">Credit/Debit Card</div>
                    <div class="text-xs text-slate-500">Pay securely with your card via Stripe</div>
                  </div>
                  <svg class="w-5 h-5 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 003 3z"/>
                  </svg>
                </label>
              </div>

              <div id="walletPinBlock" class="hidden"></div>

              <div id="payError" class="text-red-600 text-sm hidden bg-red-50 p-3 rounded-lg"></div>

              <div class="flex gap-3 pt-4">
                <?php if ($booking): ?>
                  <a href="/b2b/agent/taxi/detail?id=<?= (int)($booking['taxi_id'] ?? 0) ?>" class="px-6 py-3 border border-slate-300 text-slate-700 rounded-lg hover:bg-slate-50 transition-colors font-medium">
                    Back to Details
                  </a>
                <?php else: ?>
                  <a href="/b2b/agent/taxi/detail?id=<?= (int)($intent['taxi_id'] ?? 0) ?>" class="px-6 py-3 border border-slate-300 text-slate-700 rounded-lg hover:bg-slate-50 transition-colors font-medium">
                    Back to Details
                  </a>
                <?php endif; ?>
                
                <button type="button" id="smartPayButton" class="flex-1 px-8 py-3 bg-gradient-to-r from-gray-400 to-gray-500 text-white font-bold rounded-lg shadow-md transition-all duration-300 cursor-not-allowed opacity-75" disabled onclick="return handleSmartPayment()">
                  <span id="payButtonIcon">⏳</span> <span id="payButtonText">Select Payment Method</span> - ฿<?= number_format($amt, 0) ?>
                </button>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- Professional Wallet PIN Modal -->
<div id="walletPinModal" class="fixed inset-0 bg-black bg-opacity-50 backdrop-blur-sm z-50 hidden flex items-center justify-center p-4">
  <div class="bg-white rounded-2xl shadow-2xl max-w-md w-full transform transition-all duration-300 scale-95 opacity-0" id="walletPinModalContent">
    <div class="p-6">
      <!-- Modal Header -->
      <div class="flex items-center justify-between mb-6">
        <div class="flex items-center gap-3">
          <div class="w-10 h-10 bg-gradient-to-r from-emerald-500 to-green-500 rounded-full flex items-center justify-center">
            <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"/>
            </svg>
          </div>
          <div>
            <h3 class="text-lg font-bold text-slate-900">Secure Payment</h3>
            <p class="text-sm text-slate-600">Enter your wallet PIN</p>
          </div>
        </div>
        <button type="button" onclick="closeWalletPinModal()" class="text-slate-400 hover:text-slate-600 transition-colors">
          <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
          </svg>
        </button>
      </div>

      <!-- Payment Summary -->
      <div class="bg-slate-50 rounded-xl p-4 mb-6">
        <div class="flex justify-between items-center">
          <span class="text-sm text-slate-600">Payment Amount</span>
          <span class="text-xl font-bold text-slate-900">฿<?= number_format($amt, 0) ?></span>
        </div>
        <div class="flex justify-between items-center mt-2">
          <span class="text-xs text-slate-500">Transfer Booking</span>
          <span class="text-xs text-slate-700 font-medium"><?= htmlspecialchars((string)($booking['ride_name'] ?? ($intent['ride_name'] ?? 'Transfer'))) ?></span>
        </div>
      </div>

      <!-- PIN Input -->
      <div class="mb-6">
        <label class="block text-sm font-semibold text-slate-700 mb-3">Enter Your 4-6 Digit PIN</label>
        <input type="password" id="modalPinInput" pattern="\d{4,6}" placeholder="••••••" maxlength="6" class="w-full px-4 py-4 text-center text-2xl font-mono border-2 border-slate-300 rounded-xl focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500 transition-all duration-300 tracking-widest" autocomplete="off">
        <div class="text-xs text-slate-500 mt-2 text-center">Your PIN is encrypted and secure</div>
      </div>

      <!-- Action Buttons -->
      <div class="flex gap-3">
        <button type="button" onclick="closeWalletPinModal()" class="flex-1 px-4 py-3 border border-slate-300 text-slate-700 rounded-lg hover:bg-slate-50 transition-colors font-medium">
          Cancel
        </button>
        <button type="button" onclick="processWalletPayment()" class="flex-1 px-4 py-3 bg-gradient-to-r from-emerald-500 to-green-500 hover:from-emerald-600 hover:to-green-600 text-white font-bold rounded-lg shadow-md hover:shadow-lg transition-all duration-300 transform hover:scale-105" id="confirmPaymentBtn">
          Confirm Payment
        </button>
      </div>
    </div>
  </div>
</div>

<script>
  (function(){
    const radios = document.querySelectorAll('input[name=method]');
    const pinBlock = document.getElementById('walletPinBlock');
    function sync(){
      const val = Array.from(radios).find(r=>r.checked)?.value || '';
      // PIN block is now always hidden since we use modal
    }
    radios.forEach(r => {
      r.addEventListener('change', function() {
        sync();
        updatePaymentMethod();
      });
    });
    sync();
    updatePaymentMethod();

    // Professional Modal Functions
    window.showWalletPinModal = function(){
      const modal = document.getElementById('walletPinModal');
      const content = document.getElementById('walletPinModalContent');
      const pinInput = document.getElementById('modalPinInput');
      
      modal.classList.remove('hidden');
      setTimeout(() => {
        content.classList.remove('scale-95', 'opacity-0');
        content.classList.add('scale-100', 'opacity-100');
        pinInput.focus();
      }, 10);
      return false;
    }

    window.closeWalletPinModal = function(){
      const modal = document.getElementById('walletPinModal');
      const content = document.getElementById('walletPinModalContent');
      const pinInput = document.getElementById('modalPinInput');
      
      content.classList.remove('scale-100', 'opacity-100');
      content.classList.add('scale-95', 'opacity-0');
      setTimeout(() => {
        modal.classList.add('hidden');
        pinInput.value = '';
      }, 300);
    }

    window.processWalletPayment = function(){
      const pinInput = document.getElementById('modalPinInput');
      const pin = pinInput.value.trim();
      const pinRegex = /^\d{4,6}$/;
      
      // Reset any previous error states
      pinInput.classList.remove('border-red-500', 'ring-red-500');
      
      // Validate PIN format (4-6 digits)
      if (!pinRegex.test(pin)) {
        pinInput.classList.add('border-red-500', 'ring-red-500');
        pinInput.focus();
        
        // Show error message
        const errorMsg = 'Please enter a valid 4-6 digit PIN';
        let errorElement = document.getElementById('pinError');
        if (!errorElement) {
          errorElement = document.createElement('div');
          errorElement.id = 'pinError';
          errorElement.className = 'text-red-600 text-sm mt-1 text-center';
          pinInput.parentNode.insertBefore(errorElement, pinInput.nextSibling);
        }
        errorElement.textContent = errorMsg;
        
        return false;
      }
      
      // Add PIN to form and submit
      const form = document.getElementById('taxiPayForm');
      let pinField = form.querySelector('input[name="pin"]');
      if (!pinField) {
        pinField = document.createElement('input');
        pinField.type = 'hidden';
        pinField.name = 'pin';
        form.appendChild(pinField);
      }
      pinField.value = pin;
      
      // Ensure wallet method is selected
      const wallet = Array.from(radios).find(r=>r.value==='wallet');
      if (wallet) wallet.checked = true;
      
      // Show loading state
      const confirmBtn = document.getElementById('confirmPaymentBtn');
      confirmBtn.innerHTML = '<svg class="animate-spin -ml-1 mr-3 h-5 w-5 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path></svg>Processing...';
      confirmBtn.disabled = true;
      
      form.submit();
      return false;
    }

    // Close modal on escape key
    document.addEventListener('keydown', function(e) {
      if (e.key === 'Escape') {
        closeWalletPinModal();
      }
    });

    // Close modal on backdrop click
    document.getElementById('walletPinModal').addEventListener('click', function(e) {
      if (e.target === this) {
        closeWalletPinModal();
      }
    });

    // PIN input formatting
    document.getElementById('modalPinInput').addEventListener('input', function(e) {
      e.target.value = e.target.value.replace(/[^0-9]/g, '');
      e.target.classList.remove('border-red-500', 'ring-red-500');
    });

    // Enter key to submit
    document.getElementById('modalPinInput').addEventListener('keypress', function(e) {
      if (e.key === 'Enter') {
        processWalletPayment();
      }
    });

    window.submitWallet = function(){
      return showWalletPinModal();
    }

    // Smart payment handler
    let selectedPaymentMethod = '';
    
    function updatePaymentMethod() {
      const walletRadio = document.getElementById('wallet_payment');
      const cardRadio = document.getElementById('card_payment');
      const payButton = document.getElementById('smartPayButton');
      const payButtonIcon = document.getElementById('payButtonIcon');
      const payButtonText = document.getElementById('payButtonText');
      
      if (walletRadio && walletRadio.checked) {
        selectedPaymentMethod = 'wallet';
        if (payButton) {
          payButton.className = 'flex-1 px-8 py-3 bg-gradient-to-r from-emerald-500 to-green-500 hover:from-emerald-600 hover:to-green-600 text-white font-bold rounded-lg shadow-md hover:shadow-lg transition-all duration-300 transform hover:scale-105';
          payButton.disabled = false;
          payButtonIcon.textContent = '🚗';
          payButtonText.textContent = 'Pay Now';
        }
      } else if (cardRadio && cardRadio.checked) {
        selectedPaymentMethod = 'card';
        if (payButton) {
          payButton.className = 'flex-1 px-8 py-3 bg-gradient-to-r from-blue-500 to-indigo-500 hover:from-blue-600 hover:to-indigo-600 text-white font-bold rounded-lg shadow-md hover:shadow-lg transition-all duration-300 transform hover:scale-105';
          payButton.disabled = false;
          payButtonIcon.textContent = '💳';
          payButtonText.textContent = 'Pay Now';
        }
      } else {
        // No payment method selected
        if (payButton) {
          payButton.className = 'flex-1 px-8 py-3 bg-gradient-to-r from-gray-400 to-gray-500 text-white font-bold rounded-lg shadow-md transition-all duration-300 cursor-not-allowed opacity-75';
          payButton.disabled = true;
          payButtonIcon.textContent = '⏳';
          payButtonText.textContent = 'Select Payment Method';
        }
      }
    }

    function handleSmartPayment() {
      console.log('handleSmartPayment called, selectedPaymentMethod:', selectedPaymentMethod);
      if (selectedPaymentMethod === 'wallet') {
        console.log('Opening wallet PIN modal');
        return showWalletPinModal();
      } else if (selectedPaymentMethod === 'card') {
        console.log('Triggering card payment gateway');
        return payGateway();
      }
      console.log('No payment method selected');
      return false;
    }

    // Add event listeners for payment method changes
    radios.forEach(r => {
      r.addEventListener('change', function() {
        sync();
        updatePaymentMethod();
        updatePaymentMethod();
      });
    });

    window.payGateway = async function(){
      try{
        const gw = Array.from(radios).find(r=>r.value==='gateway');
        if (gw) { gw.checked = true; sync(); }
        
        // Show loading state on smart button
        const payButton = document.getElementById('smartPayButton');
        const originalText = payButton.innerHTML;
        payButton.innerHTML = '<svg class="animate-spin -ml-1 mr-3 h-5 w-5 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path></svg>Processing...';
        payButton.disabled = true;
        
        const payload = {};
        <?php if ($booking): ?>
          payload.booking_id = <?= (int)($booking['id'] ?? 0) ?>;
        <?php else: ?>
          payload.intent_id = <?= (int)($intent['id'] ?? 0) ?>;
        <?php endif; ?>
        payload.gateway = 'stripe';
        const res = await fetch('/b2b/api/taxi/pay-gateway', {
          method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(payload)
        });
        const json = await res.json();
        if (json && json.redirect_url){ window.location.href = json.redirect_url; return false; }
        
        // Reset button on error
        payButton.innerHTML = originalText;
        payButton.disabled = false;
        alert(json && json.error ? json.error : 'Gateway init failed');
      }catch(e){ 
        // Reset button on error
        const payButton = document.getElementById('smartPayButton');
        if (payButton) { payButton.innerHTML = '💳 Pay by Card'; payButton.disabled = false; }
        alert('Gateway error: '+ (e && e.message ? e.message : String(e))); 
      }
      return false;
    }

    // Make functions globally available
    window.handleSmartPayment = handleSmartPayment;
    window.updatePaymentMethod = updatePaymentMethod;
  })();
</script>

<?php require __DIR__ . '/partials/footer.php'; ?>
