<?php
// Reusable camera overlay and JS, ported from admin/management_agents_create.php
?>
<style>
  .camera-overlay {position:fixed; inset:0; background:rgba(0,0,0,.65); display:none; align-items:center; justify-content:center; z-index:1050;}
  .camera-box {background:#fff; border-radius:8px; width:min(720px, 96vw); padding:12px; box-shadow:0 10px 30px rgba(0,0,0,.3);} 
  .camera-toolbar {display:flex; gap:8px; justify-content:space-between; align-items:center; margin-bottom:8px;}
  .camera-stage {position:relative; background:#0b1220; display:flex; justify-content:center; align-items:center; border-radius:6px; overflow:hidden;}
  .camera-stage video, .camera-stage canvas {max-width:100%; width:100%;}
  .face-guide { position:absolute; inset:12px; border:3px dashed rgba(255,255,255,0.65); border-radius:20px; pointer-events:none; box-shadow: inset 0 0 0 9999px rgba(0,0,0,0.08); }
  .btn-circle {border-radius:999px; width:56px; height:56px; display:inline-flex; align-items:center; justify-content:center; background:#ffffff; border:3px solid #e5e7eb; box-shadow:0 6px 14px rgba(0,0,0,.15);} 
  #btnShutter { position:relative; }
  #btnShutter .shutter-dot { position:absolute; width:28px; height:28px; border-radius:999px; background:#6d28d9; /* brand violet */ box-shadow:0 2px 8px rgba(109,40,217,.35); transform:translate(-50%,-50%); left:50%; top:50%; }
  #btnShutter:active { transform: scale(0.98); }
  .thumb {max-height:64px; border-radius:6px; margin-top:6px; border:1px solid #e5e7eb;}
  .preview-meta {font-size:12px; color:#6c757d;}
  /* Mobile full-screen, native-like layout */
  @media (max-width: 768px) {
    .camera-box { width:100vw; height:100vh; border-radius:0; padding:0; }
    .camera-toolbar { position:absolute; top: calc(env(safe-area-inset-top, 0px) + 8px); left:12px; right:12px; z-index:5; background: rgba(255,255,255,0.9); border-radius:12px; padding:6px 8px; box-shadow:0 4px 16px rgba(0,0,0,.12); }
    .camera-stage { position:absolute; inset:0; }
    .camera-stage video, .camera-stage canvas { width:100%; height:100%; object-fit: cover; }
    #livenessPanel { position:absolute; left:12px; right:12px; bottom: calc(92px + env(safe-area-inset-bottom, 0px)); background: rgba(255,255,255,0.9); backdrop-filter: saturate(180%) blur(8px); border:1px solid #e5e7eb; }
    .camera-actions { position:absolute; left:0; right:0; bottom: calc(env(safe-area-inset-bottom, 0px) + 10px); display:flex; justify-content:space-between; align-items:center; padding:0 16px; }
    #btnShutter.btn-circle { width:72px; height:72px; border-width:4px; }
    #btnUsePhoto, #btnRetake { background:#fff; }
  }
</style>
<div class="camera-overlay" id="cameraOverlay" aria-hidden="true">
  <div class="camera-box">
    <div class="camera-toolbar">
      <div>
        <strong>Live Capture</strong>
        <small class="text-muted ms-2" id="cameraTargetLabel"></small>
      </div>
      <div>
        <button type="button" class="btn btn-outline-secondary btn-sm" id="btnSwitchCam">Switch</button>
        <button type="button" class="btn btn-outline-danger btn-sm" id="btnCloseCam">Close</button>
      </div>
    </div>
    <div class="camera-stage">
      <video id="camVideo" autoplay playsinline></video>
      <canvas id="camCanvas" style="display:none;"></canvas>
      <div class="face-guide" id="faceGuide" style="display:none;"></div>
    </div>
    <div id="livenessPanel" style="display:none; margin-top:8px; padding:8px; border:1px dashed #e5e7eb; border-radius:8px; background:#f8fafc;">
      <div id="livenessPrompt" style="font-size:14px; font-weight:600; color:#0f172a;">Center your face in the frame</div>
      <div style="display:flex; gap:10px; margin-top:6px; font-size:12px; color:#475569;">
        <div id="chk_center">◻ Center</div>
        <div id="chk_right">◻ Turn right</div>
        <div id="chk_left">◻ Turn left</div>
        <div id="chk_blink">◻ Blink</div>
      </div>
    </div>
    <div class="d-flex justify-content-between align-items-center mt-2 camera-actions" style="display:flex; justify-content:space-between; align-items:center; margin-top:8px;">
      <div>
        <button type="button" class="btn btn-secondary btn-sm" id="btnRetake" disabled>Retake</button>
        <button type="button" class="btn btn-primary btn-sm" id="btnUsePhoto" disabled>Use Photo</button>
      </div>
      <div>
        <button type="button" class="btn btn-circle" id="btnShutter" title="Capture" aria-label="Capture">
          <span class="shutter-dot" aria-hidden="true"></span>
        </button>
      </div>
    </div>
  </div>
  <input type="file" id="_hiddenCaptureInput" accept="image/*" style="display:none;">
  <input type="hidden" id="captureTargetName" value="">
</div>
<script>
(function(){
  const overlay = document.getElementById('cameraOverlay');
  if (!overlay) return;
  const video = document.getElementById('camVideo');
  const canvas = document.getElementById('camCanvas');
  const btnSwitch = document.getElementById('btnSwitchCam');
  const btnClose = document.getElementById('btnCloseCam');
  const btnShutter = document.getElementById('btnShutter');
  const btnUse = document.getElementById('btnUsePhoto');
  const btnRetake = document.getElementById('btnRetake');
  const hiddenCapture = document.getElementById('_hiddenCaptureInput');
  const captureTargetName = document.getElementById('captureTargetName');
  const targetLabel = document.getElementById('cameraTargetLabel');
  const livenessPanel = document.getElementById('livenessPanel');
  const livenessPrompt = document.getElementById('livenessPrompt');
  const chkCenter = document.getElementById('chk_center');
  const chkRight = document.getElementById('chk_right');
  const chkLeft = document.getElementById('chk_left');
  const chkBlink = document.getElementById('chk_blink');

  let currentStream = null;
  let facing = 'environment';
  let capturedBlob = null;
  let lastStampText = '';

  function fmt(ts){ const d = new Date(ts); const pad=n=>String(n).padStart(2,'0'); return `${d.getFullYear()}-${pad(d.getMonth()+1)}-${pad(d.getDate())} ${pad(d.getHours())}:${pad(d.getMinutes())}`; }
  function stopStream(){ if(currentStream){ currentStream.getTracks().forEach(t=>t.stop()); currentStream=null; } }
  async function startStream(){
    try {
      stopStream();
      const constraints = { video: { facingMode: { ideal: facing }, width: { ideal: 1280 }, height: { ideal: 720 } }, audio: false };
      currentStream = await navigator.mediaDevices.getUserMedia(constraints);
      video.srcObject = currentStream; await video.play();
    } catch(e){
      // Native fallback
      hiddenCapture.onchange = () => {
        const f = hiddenCapture.files && hiddenCapture.files[0];
        if (!f) return; useBlob(f, f.type || 'image/jpeg');
      };
      // capture hint for mobile
      hiddenCapture.setAttribute('capture', facing === 'user' ? 'user' : 'environment');
      hiddenCapture.click();
    }
  }
  function openCam(targetName){
    captureTargetName.value = targetName;
    targetLabel.textContent = '[' + targetName + ']';
    overlay.style.display = 'flex';
    capturedBlob = null; btnUse.disabled = true; btnRetake.disabled = true;
    canvas.style.display='none'; video.style.display='block';
    // Liveness visible only for selfie
    const isSelfie = (targetName === 'doc_selfie');
    livenessPanel.style.display = isSelfie ? 'block' : 'none';
    const faceGuide = document.getElementById('faceGuide');
    if (faceGuide) faceGuide.style.display = isSelfie ? 'block' : 'none';
    if (isSelfie) initLiveness(); else teardownLiveness();
    startStream();
  }
  async function drawToCanvas(){
    const w = video.videoWidth || 1280; const h = video.videoHeight || 720;
    canvas.width = w; canvas.height = h; const ctx = canvas.getContext('2d');
    ctx.drawImage(video,0,0,w,h);
    lastStampText = fmt(Date.now());
    const padX=14, padY=10; ctx.font='20px system-ui, -apple-system, Segoe UI, Roboto, Arial'; ctx.textBaseline='bottom';
    const textW = ctx.measureText(lastStampText).width; const boxW = textW + padX*2; const boxH = 28 + padY*2;
    const x=w - boxW - 12; const y = h - 12; ctx.fillStyle='rgba(0,0,0,0.55)'; ctx.fillRect(x, y - boxH, boxW, boxH);
    ctx.fillStyle='#fff'; ctx.fillText(lastStampText, x+padX, y-padY);
    return new Promise(resolve=> canvas.toBlob(b=>resolve(b),'image/jpeg',0.92));
  }
  function useBlob(blob){ capturedBlob=blob; btnUse.disabled=false; btnRetake.disabled=false; canvas.style.display='block'; video.style.display='none'; }
  function closeCam(){ stopStream(); overlay.style.display='none'; capturedBlob=null; }

  btnShutter.addEventListener('click', async ()=>{ if(!video.srcObject) return; const b = await drawToCanvas(); useBlob(b); });
  btnRetake.addEventListener('click', ()=>{ capturedBlob=null; btnUse.disabled=true; btnRetake.disabled=true; canvas.style.display='none'; video.style.display='block'; });
  btnUse.addEventListener('click', ()=>{
    if(!capturedBlob) return; const file = new File([capturedBlob], 'capture_'+Date.now()+'.jpg', {type:'image/jpeg'});
    const input = document.querySelector('#'+captureTargetName.value+', input[name="'+captureTargetName.value+'"]');
    if (input) { try { const dt = new DataTransfer(); dt.items.add(file); input.files = dt.files; input.dispatchEvent(new Event('change')); } catch(_){} }
    // Write liveness results back to host form if selfie
    if (captureTargetName.value === 'doc_selfie') {
      const fMotion = document.getElementById('liveness_motion');
      const fBlink = document.getElementById('liveness_blink');
      if (fMotion) fMotion.value = (livePassed.motion ? 'passed' : 'failed');
      if (fBlink) fBlink.value = (livePassed.blink ? 'passed' : 'failed');
    }
    closeCam();
  });
  btnClose.addEventListener('click', closeCam);
  btnSwitch.addEventListener('click', ()=>{ facing = (facing==='environment') ? 'user' : 'environment'; startStream(); });

  // Bind any .capture-btn buttons on page
  document.querySelectorAll('.capture-btn[data-target]').forEach(btn=>{
    btn.addEventListener('click', ()=>{
      const t = btn.getAttribute('data-target'); if (!t) return; openCam(t);
      // reset overlay to live view
      canvas.style.display = 'none';
      video.style.display = 'block';
    });
  });

  // ---------- Liveness (only for selfie) ----------
  let faceInterval = null;
  let faceMesh = null;
  let lState = 'center';
  const livePassed = { center:false, motion:false, blink:false };
  const frameBuf = [];
  const MAX_FRAMES = 24; // ~5s at 200ms

  function setPrompt(text){ if (livenessPrompt) livenessPrompt.textContent = text; }
  function checkMark(el, ok){ if(!el) return; el.textContent = (ok ? '✅ ' : '◻ ') + el.textContent.replace(/^✅ |^◻\s*/, ''); }

  function teardownLiveness(){
    lState = 'center'; livePassed.center=false; livePassed.motion=false; livePassed.blink=false;
    if (faceInterval) { clearInterval(faceInterval); faceInterval=null; }
    faceMesh = null;
    if (chkCenter) chkCenter.textContent='◻ Center';
    if (chkRight) chkRight.textContent='◻ Turn right';
    if (chkLeft) chkLeft.textContent='◻ Turn left';
    if (chkBlink) chkBlink.textContent='◻ Blink';
    setPrompt('Center your face in the frame');
    btnUse.disabled = true;
  }

  async function initLiveness(){
    teardownLiveness();
    // Load MediaPipe FaceMesh from CDN if not present
    if (!window.FaceMesh) {
      await new Promise((resolve)=>{
        const s = document.createElement('script'); s.src='https://cdn.jsdelivr.net/npm/@mediapipe/face_mesh/face_mesh.js'; s.onload=resolve; document.head.appendChild(s);
      });
      await new Promise((resolve)=>{
        const s = document.createElement('script'); s.src='https://cdn.jsdelivr.net/npm/@mediapipe/camera_utils/camera_utils.js'; s.onload=resolve; document.head.appendChild(s);
      });
      await new Promise((resolve)=>{
        const s = document.createElement('script'); s.src='https://cdn.jsdelivr.net/npm/@mediapipe/drawing_utils/drawing_utils.js'; s.onload=resolve; document.head.appendChild(s);
      });
    }
    // Fallback simple sampler if FaceMesh fails to construct
    try {
      faceMesh = new FaceMesh({locateFile: (file)=>`https://cdn.jsdelivr.net/npm/@mediapipe/face_mesh/${file}`});
      faceMesh.setOptions({ maxNumFaces: 1, refineLandmarks: true, minDetectionConfidence: 0.5, minTrackingConfidence: 0.5 });
    } catch(e) {
      console.warn('FaceMesh init failed, skipping liveness', e);
      // Keep panel visible so user sees prompts; allow capture without gating
      btnUse.disabled = false;
      return;
    }

    const tmpCanvas = document.createElement('canvas');
    const tctx = tmpCanvas.getContext('2d');
    const procCanvas = document.createElement('canvas');
    const pctx = procCanvas.getContext('2d');

    function estimateBrightness(cnv){
      const w = cnv.width, h = cnv.height; pctx.drawImage(cnv,0,0,64,64); const img = pctx.getImageData(0,0,64,64).data;
      let s=0; for(let i=0;i<img.length;i+=4){ s += img[i]*0.2126 + img[i+1]*0.7152 + img[i+2]*0.0722; }
      return s/(img.length/4)/255; // 0..1
    }
    function estimateSharpness(cnv){
      // very light Laplacian-like metric on downscaled image
      pctx.drawImage(cnv,0,0,64,64); const {data,width,height} = pctx.getImageData(0,0,64,64);
      function grayAt(x,y){ const idx=(y*width+x)*4; return (data[idx]*0.299+data[idx+1]*0.587+data[idx+2]*0.114); }
      let sum=0, cnt=0; for(let y=1;y<height-1;y++){ for(let x=1;x<width-1;x++){ const c=grayAt(x,y); const lap = Math.abs(4*c - grayAt(x-1,y) - grayAt(x+1,y) - grayAt(x,y-1) - grayAt(x,y+1)); sum += lap; cnt++; }}
      return sum/(cnt*255); // normalized ~0..1
    }
    async function dataURLFromCanvas(cnv){ return cnv.toDataURL('image/jpeg',0.92); }
    async function blobFromDataURL(url){ const res = await fetch(url); return await res.blob(); }

    faceInterval = setInterval(async ()=>{
      if (!video.videoWidth) return;
      const w = video.videoWidth, h = video.videoHeight;
      tmpCanvas.width=w; tmpCanvas.height=h;
      tctx.drawImage(video,0,0,w,h);
      const image = tmpCanvas; // HTMLCanvasElement works for send
      let results = null;
      try {
        results = await faceMesh.send({image});
      } catch(e) { return; }
      if (!results || !results.multiFaceLandmarks || results.multiFaceLandmarks.length===0) {
        setPrompt('Center your face in the frame');
        return;
      }
      const lm = results.multiFaceLandmarks[0];
      // Keypoints (eyes) — indices from MediaPipe
      const leftEyeI = lm[33]; // left eye outer
      const rightEyeI = lm[263]; // right eye outer
      const leftUp = lm[159], leftDown = lm[145];
      const rightUp = lm[386], rightDown = lm[374];
      if (!leftEyeI || !rightEyeI || !leftUp || !leftDown || !rightUp || !rightDown) return;

      // Center: face within central 40% width band
      const cx = (leftEyeI.x + rightEyeI.x)/2; // normalized 0..1
      const centered = (cx > 0.3 && cx < 0.7);
      if (!livePassed.center && centered) {
        livePassed.center = true; checkMark(chkCenter,true); setPrompt('Turn your head slowly to the RIGHT'); lState='right';
      }

      // Yaw: compare x of eyes; if head turns right, right eye moves closer to center (cx increases significantly)
      if (lState==='right') {
        if (cx >= 0.62) { checkMark(chkRight,true); setPrompt('Now turn your head to the LEFT'); lState='left'; }
      } else if (lState==='left') {
        if (cx <= 0.38) { checkMark(chkLeft,true); setPrompt('Blink once'); lState='blink'; livePassed.motion=true; }
      }

      // Blink: eye aspect ratio (vertical/horizontal) drop
      if (lState==='blink') {
        const leftEAR = Math.abs(leftUp.y - leftDown.y) / Math.abs(leftEyeI.x - rightEyeI.x);
        const rightEAR = Math.abs(rightUp.y - rightDown.y) / Math.abs(leftEyeI.x - rightEyeI.x);
        const ear = (leftEAR + rightEAR)/2;
        if (ear < 0.02) { checkMark(chkBlink,true); setPrompt('Liveness passed'); livePassed.blink = true; lState='done'; }
      }

      // Buffer frames for best-frame selection
      try {
        const b = estimateBrightness(tmpCanvas);
        const s = estimateSharpness(tmpCanvas);
        const url = await dataURLFromCanvas(tmpCanvas);
        frameBuf.push({url, score: s * (b>0.25 && b<0.95 ? 1 : 0.5)});
        if (frameBuf.length > MAX_FRAMES) frameBuf.shift();
      } catch(_) {}

      // When done: pick best frame automatically
      if (lState==='done') {
        if (frameBuf.length) {
          frameBuf.sort((a,b)=>b.score-a.score);
          const best = frameBuf[0];
          try {
            const blob = await blobFromDataURL(best.url);
            capturedBlob = blob; btnUse.disabled=false; btnRetake.disabled=false; canvas.style.display='block'; video.style.display='none';
            // Show preview in canvas
            const img = new Image(); img.onload = ()=>{ canvas.width=video.videoWidth||1280; canvas.height=video.videoHeight||720; const ctx=canvas.getContext('2d'); ctx.drawImage(img,0,0,canvas.width,canvas.height); }; img.src=best.url;
          } catch(e) {
            btnUse.disabled=false; // fallback allow manual use
          }
        } else {
          btnUse.disabled=false;
        }
        clearInterval(faceInterval); faceInterval=null;
      }
    }, 200);
  }
})();
</script>
