<?php
// app/Views/admin/hotels_form.php
use App\Core\Security;
$editing = !empty($hotel);
$action = $editing ? '/admin/hotels/update' : '/admin/hotels/store';
// Human-friendly labels for meal plan codes
$mealLabels = [
  'RO' => 'RO - Room Only',
  'BB' => 'BB - Bed & Breakfast',
  'HB' => 'HB - Half Board',
  'FB' => 'FB - Full Board',
];
// Featured label options (single source of truth)
$featuredLabels = [
  'top_choice' => 'Top Choice',
  'recommended' => 'Recommended',
  'trusted_partner' => 'Trusted Partner',
  'best_value' => 'Best Value',
];
?>
<div class="container py-3">
  <div class="bg-white rounded-3 shadow-sm p-3">
  <div class="d-flex justify-content-between align-items-center mb-3 flex-wrap gap-2">
    <h2 class="h5 mb-0"><i class="bi bi-building me-2"></i><?= $editing ? 'Edit Hotel' : 'Add Hotel' ?></h2>
    <div class="d-flex gap-2">
      <a href="/admin/hotels" class="btn btn-sm btn-outline-secondary"><i class="bi bi-arrow-left me-1"></i>Back</a>
      <?php if ($editing): ?>
        <form method="post" action="/admin/hotels/delete" onsubmit="return confirm('Delete this hotel?');">
          <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf ?? Security::csrfToken()) ?>">
          <input type="hidden" name="id" value="<?= (int)$hotel['id'] ?>">
          <button class="btn btn-sm btn-outline-danger" <?= empty($_SESSION['admin_hotels_unlocked']) ? 'disabled' : '' ?>><i class="bi bi-trash me-1"></i>Delete</button>
        </form>
      <?php endif; ?>
    </div>
  </div>

    <ul class="nav nav-tabs" id="hotelTabs" role="tablist" data-active-tab="<?= htmlspecialchars($tab ?? 'details') ?>">
      <li class="nav-item" role="presentation">
        <button class="nav-link active" id="tab-details" data-bs-toggle="tab" data-bs-target="#pane-details" type="button" role="tab">Details</button>
      </li>
      <li class="nav-item" role="presentation">
        <button class="nav-link" id="tab-rooms" data-bs-toggle="tab" data-bs-target="#pane-rooms" type="button" role="tab">Rooms</button>
      </li>
      <li class="nav-item" role="presentation">
        <button class="nav-link" id="tab-rateplans" data-bs-toggle="tab" data-bs-target="#pane-rateplans" type="button" role="tab">Rate Plans</button>
      </li>
      <li class="nav-item" role="presentation">
        <button class="nav-link" id="tab-calendar" data-bs-toggle="tab" data-bs-target="#pane-calendar" type="button" role="tab">Calendar</button>
      </li>
      <li class="nav-item" role="presentation">
        <button class="nav-link" id="tab-policies" data-bs-toggle="tab" data-bs-target="#pane-policies" type="button" role="tab">Policies</button>
      </li>
      <li class="nav-item" role="presentation">
        <button class="nav-link" id="tab-images" data-bs-toggle="tab" data-bs-target="#pane-images" type="button" role="tab">Images</button>
      </li>
      <li class="nav-item" role="presentation">
        <button class="nav-link" id="tab-bookings" data-bs-toggle="tab" data-bs-target="#pane-bookings" type="button" role="tab">Bookings</button>
      </li>
    </ul>

    <div class="tab-content border border-top-0 rounded-bottom p-3 bg-white">
      <div class="tab-pane fade show active" id="pane-details" role="tabpanel">
        <form method="post" action="<?= $action ?>">
          <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf ?? Security::csrfToken()) ?>">
          <?php if ($editing): ?><input type="hidden" name="id" value="<?= (int)$hotel['id'] ?>"><?php endif; ?>
        <div class="row g-3">
          <div class="col-md-6">
            <label class="form-label">Hotel Name</label>
            <input type="text" name="name" class="form-control" required value="<?= htmlspecialchars($hotel['name'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Country</label>
            <select name="country" id="countrySelect" class="form-select" required>
              <option value="">-- Select Country --</option>
              <?php foreach (($countries ?? []) as $c): ?>
                <option value="<?= htmlspecialchars($c) ?>" <?= (($hotel['country'] ?? '') === $c ? 'selected' : '') ?>><?= htmlspecialchars($c) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-3">
            <label class="form-label">City</label>
            <select name="city" id="citySelect" class="form-select" required>
              <option value="">-- Select City --</option>
              <?php $selCountry = $hotel['country'] ?? ''; $selCity = $hotel['city'] ?? ''; ?>
              <?php foreach (($cities ?? []) as $rc): ?>
                <option data-country="<?= htmlspecialchars($rc['country']) ?>" value="<?= htmlspecialchars($rc['city']) ?>" <?= ($selCountry === ($rc['country'] ?? '') && $selCity === ($rc['city'] ?? '') ? 'selected' : '') ?>>
                  <?= htmlspecialchars($rc['city']) ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-3">
            <label class="form-label">Base Price</label>
            <div class="input-group">
              <span class="input-group-text">THB</span>
              <input type="number" step="0.01" name="base_price" class="form-control" value="<?= htmlspecialchars($hotel['base_price'] ?? '0.00') ?>">
            </div>
          </div>
          <div class="col-md-2">
            <label class="form-label">Stars</label>
            <?php $starsSel = (int)($hotel['stars'] ?? 0); if ($starsSel < 0) $starsSel = 0; if ($starsSel > 5) $starsSel = 5; ?>
            <select name="stars" class="form-select">
              <?php for ($s=0; $s<=5; $s++): ?>
                <option value="<?= $s ?>" <?= ($starsSel === $s ? 'selected' : '') ?>><?= $s === 0 ? 'Unrated' : $s.' Star'.($s>1?'s':'') ?></option>
              <?php endfor; ?>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label">Vendor</label>
            <select class="form-select" name="vendor_id" id="vendorSelect" required>
              <option value="">-- Select Vendor --</option>
              <?php foreach (($vendors ?? []) as $v): ?>
                <?php $vCountry = $v['country'] ?? ($hotel['country'] ?? ''); $vCity = $v['city'] ?? ($hotel['city'] ?? ''); ?>
                <option data-country="<?= htmlspecialchars($vCountry) ?>" data-city="<?= htmlspecialchars($vCity) ?>" value="<?= (int)($v['id'] ?? 0) ?>" <?= ($hotel['vendor_id'] ?? null) == ($v['id'] ?? null) ? 'selected' : '' ?>><?= htmlspecialchars($v['name'] ?? '') ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-12">
            <hr class="my-2">
            <div class="row g-3">
              <div class="col-md-6">
                <label class="form-label d-block">Visibility</label>
                <?php
                  $visCust = isset($hotel['visible_customer']) ? (int)$hotel['visible_customer'] : 1;
                  $visAgent = isset($hotel['visible_agent']) ? (int)$hotel['visible_agent'] : 1;
                ?>
                <div class="form-check form-switch">
                  <input type="hidden" name="visible_customer" value="0">
                  <input class="form-check-input" type="checkbox" id="visibleCustomer" name="visible_customer" value="1" <?= ($visCust === 1 ? 'checked' : '') ?>>
                  <label class="form-check-label" for="visibleCustomer">Visible to Customers (B2C)</label>
                </div>
                <div class="form-check form-switch">
                  <input type="hidden" name="visible_agent" value="0">
                  <input class="form-check-input" type="checkbox" id="visibleAgent" name="visible_agent" value="1" <?= ($visAgent === 1 ? 'checked' : '') ?>>
                  <label class="form-check-label" for="visibleAgent">Visible to Agents (B2B)</label>
                </div>
              </div>
              <div class="col-md-6">
                <label class="form-label d-block">Featured</label>
                <?php
                  $featCust = isset($hotel['featured_customer']) ? (int)$hotel['featured_customer'] : 0;
                  $featAgent = isset($hotel['featured_agent']) ? (int)$hotel['featured_agent'] : 0;
                ?>
                <div class="form-check form-switch">
                  <input type="hidden" name="featured_customer" value="0">
                  <input class="form-check-input" type="checkbox" id="featuredCustomer" name="featured_customer" value="1" <?= ($featCust === 1 ? 'checked' : '') ?>>
                  <label class="form-check-label" for="featuredCustomer">Featured for Customers (B2C)</label>
                </div>
                <div class="form-check form-switch mt-3">
                  <input type="hidden" name="featured_agent" value="0">
                  <input class="form-check-input" type="checkbox" id="featuredAgent" name="featured_agent" value="1" <?= ($featAgent === 1 ? 'checked' : '') ?>>
                  <label class="form-check-label" for="featuredAgent">Featured for Agents (B2B)</label>
                </div>
                <div class="mt-2 ms-4" id="agentLabelWrap" style="<?= ($featAgent === 1 ? '' : 'display:none;') ?>">
                  <label for="featuredAgentLabel" class="form-label mb-1">Agent Featured Label</label>
                  <select class="form-select form-select-sm" id="featuredAgentLabel" name="featured_agent_label" <?= ($featAgent === 1 ? '' : 'disabled') ?> >
                    <option value="">-- Select Label --</option>
                    <?php 
                      $selAgentLabel = $hotel['featured_agent_label'] ?? '';
                    ?>
                    <?php foreach ($featuredLabels as $val => $text): ?>
                      <option value="<?= htmlspecialchars($val) ?>" <?= ($selAgentLabel === $val ? 'selected' : '') ?>><?= htmlspecialchars($text) ?></option>
                    <?php endforeach; ?>
                  </select>
                  <div class="form-text">Shown on B2B when featured is ON. Defaults to "Recommended" if left blank.</div>
                </div>
              </div>
            </div>
          </div>
        </div>
        <div class="mt-3">
          <button class="btn btn-primary"><i class="bi bi-save me-1"></i><?= $editing ? 'Save Changes' : 'Create Hotel' ?></button>
        </div>
        </form>
      </div>

      <div class="tab-pane fade" id="pane-rooms" role="tabpanel">
        <?php if (!$editing): ?>
          <div class="alert alert-info">Save the hotel first, then manage rooms.</div>
        <?php else: ?>
        <div class="row g-3 mb-3">
          <div class="col-12"><h6 class="mb-0">Add Room Type</h6></div>
          <div class="col-12">
            <form class="row g-2 align-items-end" method="post" action="/admin/hotels/rooms/store">
              <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf ?? Security::csrfToken()) ?>">
              <input type="hidden" name="hotel_id" value="<?= (int)($hotel['id'] ?? 0) ?>">
              <div class="col-md-4">
                <label class="form-label">Name</label>
                <input type="text" name="name" class="form-control" required>
              </div>
              <div class="col-md-2">
                <label class="form-label">Adults</label>
                <input type="number" name="occupancy_adults" class="form-control" value="2" min="1">
              </div>
              <div class="col-md-2">
                <label class="form-label">Children</label>
                <input type="number" name="occupancy_children" class="form-control" value="0" min="0">
              </div>
              <div class="col-md-2">
                <label class="form-label">Size (sqm)</label>
                <input type="number" name="size_sqm" class="form-control" min="0">
              </div>
              <div class="col-md-2">
                <label class="form-label">Active</label>
                <select class="form-select" name="active">
                  <option value="1">Yes</option>
                  <option value="0">No</option>
                </select>
              </div>
              <div class="col-md-6">
                <label class="form-label">Bed Info</label>
                <input type="text" name="bed_info" class="form-control" placeholder="1 King, 2 Twin, etc.">
              </div>
              <div class="col-md-12">
                <button class="btn btn-sm btn-primary"><i class="bi bi-plus-circle me-1"></i>Add Room</button>
              </div>
            </form>
          </div>
        </div>

        <div class="table-responsive">
          <table class="table table-sm align-middle">
            <thead class="table-light">
              <tr>
                <th style="width:26%">Name</th>
                <th style="width:10%">Adults</th>
                <th style="width:10%">Children</th>
                <th style="width:12%">Size</th>
                <th style="width:12%">Active</th>
                <th>Bed Info</th>
                <th style="width:16%" class="text-end">Actions</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach (($rooms ?? []) as $room): ?>
              <tr>
                <form method="post" action="/admin/hotels/rooms/update" class="row g-0">
                  <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf ?? Security::csrfToken()) ?>">
                  <input type="hidden" name="hotel_id" value="<?= (int)$hotel['id'] ?>">
                  <input type="hidden" name="id" value="<?= (int)$room['id'] ?>">
                  <td><input type="text" class="form-control form-control-sm" name="name" value="<?= htmlspecialchars($room['name']) ?>"></td>
                  <td style="max-width:110px"><input type="number" class="form-control form-control-sm" name="occupancy_adults" value="<?= (int)$room['occupancy_adults'] ?>" min="1"></td>
                  <td style="max-width:110px"><input type="number" class="form-control form-control-sm" name="occupancy_children" value="<?= (int)$room['occupancy_children'] ?>" min="0"></td>
                  <td style="max-width:140px"><input type="number" class="form-control form-control-sm" name="size_sqm" value="<?= htmlspecialchars((string)($room['size_sqm'] ?? '')) ?>" min="0"></td>
                  <td style="max-width:140px">
                    <select class="form-select form-select-sm" name="active">
                      <option value="1" <?= ((int)$room['active']===1?'selected':'') ?>>Yes</option>
                      <option value="0" <?= ((int)$room['active']===0?'selected':'') ?>>No</option>
                    </select>
                  </td>
                  <td><input type="text" class="form-control form-control-sm" name="bed_info" value="<?= htmlspecialchars($room['bed_info'] ?? '') ?>"></td>
                  <td class="text-end text-nowrap">
                    <button class="btn btn-sm btn-outline-primary"><i class="bi bi-save me-1"></i>Save</button>
                  </form>
                    <form method="post" action="/admin/hotels/rooms/delete" class="d-inline">
                      <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf ?? Security::csrfToken()) ?>">
                      <input type="hidden" name="hotel_id" value="<?= (int)$hotel['id'] ?>">
                      <input type="hidden" name="id" value="<?= (int)$room['id'] ?>">
                      <button class="btn btn-sm btn-outline-danger" onclick="return confirm('Delete room type?')"><i class="bi bi-trash"></i></button>
                    </form>
                  </td>
              </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
        <?php endif; ?>
      </div>

      <div class="tab-pane fade" id="pane-rateplans" role="tabpanel">
        <?php if (!$editing): ?>
          <div class="alert alert-info">Save the hotel first, then manage rate plans.</div>
        <?php else: ?>
        <div class="row g-3 mb-3">
          <div class="col-12"><h6 class="mb-0">Add Rate Plan</h6></div>
          <div class="col-12">
            <form class="row g-2 align-items-end" method="post" action="/admin/hotels/rateplans/store">
              <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf ?? Security::csrfToken()) ?>">
              <input type="hidden" name="hotel_id" value="<?= (int)($hotel['id'] ?? 0) ?>">
              <div class="col-md-4">
                <label class="form-label">Name</label>
                <input type="text" name="name" class="form-control" required>
              </div>
              <div class="col-md-2">
                <label class="form-label">Meal Plan</label>
                <select name="meal_plan" class="form-select">
                  <option value="RO"><?= htmlspecialchars($mealLabels['RO']) ?></option>
                  <option value="BB"><?= htmlspecialchars($mealLabels['BB']) ?></option>
                  <option value="HB"><?= htmlspecialchars($mealLabels['HB']) ?></option>
                  <option value="FB"><?= htmlspecialchars($mealLabels['FB']) ?></option>
                </select>
              </div>
              <div class="col-md-2">
                <label class="form-label">Refundable</label>
                <select name="refundable" class="form-select">
                  <option value="1">Yes</option>
                  <option value="0">No</option>
                </select>
              </div>
              <div class="col-md-2">
                <label class="form-label">Refund Window (days)</label>
                <input type="number" name="refund_window_days" class="form-control" min="0">
              </div>
              <div class="col-md-2">
                <label class="form-label">Advance Purchase (days)</label>
                <input type="number" name="advance_purchase_days" class="form-control" min="0">
              </div>
              <div class="col-md-2">
                <label class="form-label">Active</label>
                <select name="active" class="form-select">
                  <option value="1">Yes</option>
                  <option value="0">No</option>
                </select>
              </div>
              <div class="col-12">
                <button class="btn btn-sm btn-primary"><i class="bi bi-plus-circle me-1"></i>Add Rate Plan</button>
              </div>
            </form>
          </div>
        </div>

        <div class="table-responsive">
          <table class="table table-sm align-middle">
            <thead class="table-light">
              <tr>
                <th style="width:26%">Name</th>
                <th style="width:12%">Meal</th>
                <th style="width:12%">Refundable</th>
                <th style="width:18%">Refund Window</th>
                <th style="width:18%">Advance Purchase</th>
                <th style="width:10%">Active</th>
                <th style="width:16%" class="text-end">Actions</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach (($rate_plans ?? []) as $rp): ?>
              <tr>
                <form method="post" action="/admin/hotels/rateplans/update" class="row g-0">
                  <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf ?? Security::csrfToken()) ?>">
                  <input type="hidden" name="hotel_id" value="<?= (int)$hotel['id'] ?>">
                  <input type="hidden" name="id" value="<?= (int)$rp['id'] ?>">
                  <td><input type="text" class="form-control form-control-sm" name="name" value="<?= htmlspecialchars($rp['name']) ?>"></td>
                  <td style="max-width:150px">
                    <select class="form-select form-select-sm" name="meal_plan">
                      <?php $meals = ['RO','BB','HB','FB']; foreach ($meals as $m): ?>
                        <option value="<?= $m ?>" <?= ($rp['meal_plan'] === $m ? 'selected' : '') ?>><?= htmlspecialchars($mealLabels[$m] ?? $m) ?></option>
                      <?php endforeach; ?>
                    </select>
                  </td>
                  <td style="max-width:140px">
                    <select class="form-select form-select-sm" name="refundable">
                      <option value="1" <?= ((int)$rp['refundable']===1?'selected':'') ?>>Yes</option>
                      <option value="0" <?= ((int)$rp['refundable']===0?'selected':'') ?>>No</option>
                    </select>
                  </td>
                  <td style="max-width:180px"><input type="number" class="form-control form-control-sm" name="refund_window_days" value="<?= htmlspecialchars((string)($rp['refund_window_days'] ?? '')) ?>" min="0"></td>
                  <td style="max-width:180px"><input type="number" class="form-control form-control-sm" name="advance_purchase_days" value="<?= htmlspecialchars((string)($rp['advance_purchase_days'] ?? '')) ?>" min="0"></td>
                  <td style="max-width:120px">
                    <select class="form-select form-select-sm" name="active">
                      <option value="1" <?= ((int)$rp['active']===1?'selected':'') ?>>Yes</option>
                      <option value="0" <?= ((int)$rp['active']===0?'selected':'') ?>>No</option>
                    </select>
                  </td>
                  <td class="text-end text-nowrap">
                    <button class="btn btn-sm btn-outline-primary"><i class="bi bi-save me-1"></i>Save</button>
                  </form>
                    <form method="post" action="/admin/hotels/rateplans/delete" class="d-inline">
                      <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf ?? Security::csrfToken()) ?>">
                      <input type="hidden" name="hotel_id" value="<?= (int)$hotel['id'] ?>">
                      <input type="hidden" name="id" value="<?= (int)$rp['id'] ?>">
                      <button class="btn btn-sm btn-outline-danger" onclick="return confirm('Delete rate plan?')"><i class="bi bi-trash"></i></button>
                    </form>
                  </td>
              </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
        <?php endif; ?>
      </div>

      <div class="tab-pane fade" id="pane-calendar" role="tabpanel">
        <?php if (!$editing): ?>
          <div class="alert alert-info">Save the hotel first, then manage calendar.</div>
        <?php else: ?>
        <form class="row g-2 align-items-end mb-3" method="get" action="/admin/hotels/edit">
          <input type="hidden" name="id" value="<?= (int)$hotel['id'] ?>">
          <input type="hidden" name="tab" value="calendar">
          <div class="col-md-4">
            <label class="form-label">Room</label>
            <select class="form-select" name="room_id" required>
              <option value="">-- Select Room --</option>
              <?php foreach (($rooms ?? []) as $r): ?>
                <option value="<?= (int)$r['id'] ?>" <?= ((int)($cal_room_id ?? 0) === (int)$r['id'] ? 'selected' : '') ?>><?= htmlspecialchars($r['name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label">Rate Plan (optional for pricing)</label>
            <select class="form-select" name="rate_plan_id">
              <option value="0">-- None --</option>
              <?php foreach (($rate_plans ?? []) as $rp): ?>
                <option value="<?= (int)$rp['id'] ?>" <?= ((int)($cal_rate_plan_id ?? 0) === (int)$rp['id'] ? 'selected' : '') ?>><?= htmlspecialchars($rp['name']) ?> (<?= htmlspecialchars($mealLabels[$rp['meal_plan']] ?? $rp['meal_plan']) ?>)</option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-3">
            <label class="form-label">Month</label>
            <input type="month" class="form-control" name="ym" value="<?= htmlspecialchars($cal_ym ?? date('Y-m')) ?>" required>
          </div>
          <div class="col-md-1">
            <button class="btn btn-primary w-100"><i class="bi bi-arrow-repeat me-1"></i>Load</button>
          </div>
        </form>

        <?php if (!empty($cal) && (int)($cal_room_id ?? 0) > 0): ?>
        <form method="post" action="/admin/hotels/calendar/save">
          <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf ?? Security::csrfToken()) ?>">
          <input type="hidden" name="hotel_id" value="<?= (int)$hotel['id'] ?>">
          <input type="hidden" name="room_id" value="<?= (int)$cal_room_id ?>">
          <input type="hidden" name="rate_plan_id" value="<?= (int)($cal_rate_plan_id ?? 0) ?>">
          <input type="hidden" name="ym" value="<?= htmlspecialchars($cal_ym) ?>">

          <style>
            /* Scoped calendar styling */
            #pane-calendar .calendar-table{border:1px solid #dee2e6; border-radius:.25rem}
            #pane-calendar table thead th{position:sticky; top:0; z-index:2; background:#f8f9fa}
            #pane-calendar table tbody th,
            #pane-calendar table tbody td{vertical-align:middle}
            #pane-calendar table tbody tr:nth-child(odd){background-color:#fcfcfd}
            #pane-calendar table tbody tr:hover{background-color:#f5f7fa}
            #pane-calendar table th.sticky-col,
            #pane-calendar table td.sticky-col{position:sticky; left:0; z-index:1; background:inherit}
            #pane-calendar table thead th.sticky-col{z-index:3}
            #pane-calendar .num-input{ text-align:right }
            /* Compact sizing */
            #pane-calendar table.table > :not(caption) > * > *{padding:.25rem .5rem}
            #pane-calendar .form-control, 
            #pane-calendar .form-select{font-size:.875rem}
            #pane-calendar .form-control.form-control-sm{padding:.125rem .35rem; height:calc(1.5rem + var(--bs-border-width,1px)*2)}
            #pane-calendar .input-group-text{padding:.125rem .35rem; font-size:.8rem}
            #pane-calendar .form-check-label{font-size:.82rem}
            #pane-calendar .form-check-input{width:.95rem; height:.95rem}
            #pane-calendar .d-flex.gap-3{gap:.5rem !important}
            #pane-calendar .price-badge{font-size:.68rem; padding:.1rem .3rem}
            #pane-calendar .col-narrow{width:10%}
            #pane-calendar .col-wide{width:30%}
            /* Prevent column squeeze but keep layout tight */
            #pane-calendar table{min-width:1100px}
            /* Field-specific widths */
            #pane-calendar input[name="allotment[]"]{max-width:80px; width:80px}
            #pane-calendar input[name="vendor_cost[]"],
            #pane-calendar input[name="agent_price[]"],
            #pane-calendar input[name="customer_price[]"]{min-width:150px; width:150px}
          </style>
          <div class="table-responsive calendar-table">
            <table class="table table-sm align-middle">
              <thead class="table-light">
                <tr>
                  <th class="sticky-col" style="width:12%" data-bs-toggle="tooltip" title="Pricing date">Date</th>
                  <th class="col-narrow" data-bs-toggle="tooltip" title="Sellable rooms">Allotment</th>
                  <th class="col-narrow" data-bs-toggle="tooltip" title="Your vendor’s cost per night">Vendor Cost</th>
                  <th class="col-narrow" data-bs-toggle="tooltip" title="Your B2B agent price per night">Agent Price</th>
                  <th class="col-narrow" data-bs-toggle="tooltip" title="Customer-facing B2C price per night">Customer Price</th>
                  <th class="col-narrow" data-bs-toggle="tooltip" title="Is price tax-inclusive?">Tax Incl</th>
                  <th class="col-narrow" data-bs-toggle="tooltip" title="Minimum nights">Min Stay</th>
                  <th class="col-narrow" data-bs-toggle="tooltip" title="Maximum nights">Max Stay</th>
                  <th class="col-narrow" data-bs-toggle="tooltip" title="Live margin preview">Margins</th>
                  <th class="col-wide" data-bs-toggle="tooltip" title="Close, CTA, CTD flags">Flags</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($cal as $i => $d): ?>
                  <tr>
                    <td class="sticky-col">
                      <?= htmlspecialchars($d['date']) ?>
                      <input type="hidden" name="date[]" value="<?= htmlspecialchars($d['date']) ?>">
                    </td>
                    <td><input type="number" class="form-control form-control-sm num-input" name="allotment[]" value="<?= htmlspecialchars((string)$d['allotment']) ?>" min="0"></td>
                    <td>
                      <div class="input-group input-group-sm">
                        <span class="input-group-text">THB</span>
                        <input type="number" step="0.01" class="form-control num-input cal-vendor" data-row-index="<?= $i ?>" name="vendor_cost[]" value="<?= htmlspecialchars((string)($d['vendor_cost'] ?? '')) ?>" <?= ((int)($cal_rate_plan_id ?? 0) > 0 ? '' : 'disabled') ?> >
                      </div>
                    </td>
                    <td>
                      <div class="input-group input-group-sm">
                        <span class="input-group-text">THB</span>
                        <input type="number" step="0.01" class="form-control num-input cal-agent-price cal-agent" data-row-index="<?= $i ?>" name="agent_price[]" value="<?= htmlspecialchars((string)($d['agent_price'] ?? '')) ?>" <?= ((int)($cal_rate_plan_id ?? 0) > 0 ? '' : 'disabled') ?> >
                        <input type="hidden" class="cal-legacy-price" data-row-index="<?= $i ?>" name="price[]" value="<?= htmlspecialchars((string)$d['price']) ?>">
                      </div>
                    </td>
                    <td>
                      <div class="input-group input-group-sm">
                        <span class="input-group-text">THB</span>
                        <input type="number" step="0.01" class="form-control num-input cal-customer" data-row-index="<?= $i ?>" name="customer_price[]" value="<?= htmlspecialchars((string)($d['customer_price'] ?? '')) ?>" <?= ((int)($cal_rate_plan_id ?? 0) > 0 ? '' : 'disabled') ?> >
                      </div>
                    </td>
                    <td class="text-center">
                      <input class="form-check-input" type="checkbox" name="tax_included[<?= $i ?>]" value="1" <?= ((int)($d['tax_included'] ?? 1) === 1 ? 'checked' : '') ?> <?= ((int)($cal_rate_plan_id ?? 0) > 0 ? '' : 'disabled') ?>>
                    </td>
                    <td><input type="number" class="form-control form-control-sm num-input" name="min_stay[]" value="<?= htmlspecialchars((string)$d['min_stay']) ?>" min="0"></td>
                    <td><input type="number" class="form-control form-control-sm num-input" name="max_stay[]" value="<?= htmlspecialchars((string)$d['max_stay']) ?>" min="0"></td>
                    <td>
                      <div>
                        <span class="badge bg-secondary price-badge me-1" data-row="<?= $i ?>" data-role="marg-av">A−V: —</span>
                        <span class="badge bg-secondary price-badge" data-row="<?= $i ?>" data-role="marg-ca">C−A: —</span>
                      </div>
                    </td>
                    <td>
                      <div class="d-flex flex-wrap gap-3">
                        <div class="form-check form-check-inline">
                          <input class="form-check-input" type="checkbox" name="closed[<?= $i ?>]" value="1" <?= ((int)$d['closed'] === 1 ? 'checked' : '') ?>>
                          <label class="form-check-label">Closed</label>
                        </div>
                        <div class="form-check form-check-inline">
                          <input class="form-check-input" type="checkbox" name="cta[<?= $i ?>]" value="1" <?= ((int)$d['cta'] === 1 ? 'checked' : '') ?>>
                          <label class="form-check-label">CTA (Close To Arrival)</label>
                        </div>
                        <div class="form-check form-check-inline">
                          <input class="form-check-input" type="checkbox" name="ctd[<?= $i ?>]" value="1" <?= ((int)$d['ctd'] === 1 ? 'checked' : '') ?>>
                          <label class="form-check-label">CTD (Close To Departure)</label>
                        </div>
                      </div>
                    </td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>

          <div class="d-flex justify-content-end">
            <button class="btn btn-success"><i class="bi bi-save me-1"></i>Save Calendar</button>
          </div>
        </form>
        <?php endif; ?>
        <?php endif; ?>
      </div>

      <div class="tab-pane fade" id="pane-policies" role="tabpanel">
        <?php if (!$editing): ?>
          <div class="alert alert-info">Save the hotel first, then add policies.</div>
        <?php else: ?>
        <form method="post" action="/admin/hotels/policies/save" class="row g-3">
          <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf ?? Security::csrfToken()) ?>">
          <input type="hidden" name="hotel_id" value="<?= (int)$hotel['id'] ?>">
          <div class="col-md-3">
            <label class="form-label">Check-in Time</label>
            <input type="time" name="checkin_time" class="form-control" value="<?= htmlspecialchars($policy['checkin_time'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Check-out Time</label>
            <input type="time" name="checkout_time" class="form-control" value="<?= htmlspecialchars($policy['checkout_time'] ?? '') ?>">
          </div>
          <div class="col-12">
            <label class="form-label">Child Policy</label>
            <textarea name="child_policy" class="form-control" rows="3"><?= htmlspecialchars($policy['child_policy'] ?? '') ?></textarea>
          </div>
          <div class="col-12">
            <label class="form-label">Pet Policy</label>
            <textarea name="pet_policy" class="form-control" rows="3"><?= htmlspecialchars($policy['pet_policy'] ?? '') ?></textarea>
          </div>
          <div class="col-12">
            <label class="form-label">Terms & Conditions</label>
            <textarea name="terms_html" class="form-control" rows="6"><?= htmlspecialchars($policy['terms_html'] ?? '') ?></textarea>
            <div class="form-text">You can paste HTML content if needed.</div>
          </div>
          <div class="col-12 text-end">
            <button class="btn btn-primary"><i class="bi bi-save me-1"></i>Save Policies</button>
          </div>
        </form>
        <?php endif; ?>
      </div>

      <div class="tab-pane fade" id="pane-images" role="tabpanel">
        <?php if (!$editing): ?>
          <div class="alert alert-info">Save the hotel first, then manage images.</div>
        <?php else: ?>
        <div class="mb-3">
          <form method="post" action="/admin/hotels/images/upload" enctype="multipart/form-data" class="row g-2 align-items-end">
            <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf ?? Security::csrfToken()) ?>">
            <input type="hidden" name="hotel_id" value="<?= (int)$hotel['id'] ?>">
            <div class="col-md-4">
              <label class="form-label">Image</label>
              <input id="imageInput" type="file" name="image" class="form-control d-none" accept="image/*" required>
              <div id="imageDropzone" class="border border-2 border-secondary-subtle rounded-3 p-4 text-center bg-light-subtle" role="button">
                <div class="fw-semibold mb-1">Drag & drop image here, or click to browse</div>
                <div class="small text-muted">JPG, PNG, WEBP • Max size depends on server limits</div>
              </div>
              <div id="imageSelected" class="form-text text-truncate d-none"></div>
            </div>
            <div class="col-md-3">
              <label class="form-label">Caption</label>
              <input type="text" name="caption" class="form-control">
            </div>
            <div class="col-md-2">
              <label class="form-label">Sort Order</label>
              <input type="number" name="sort_order" class="form-control" min="0">
            </div>
            <div class="col-md-2">
              <label class="form-label">Room (optional)</label>
              <select name="room_id" class="form-select">
                <option value="">Hotel-level</option>
                <?php foreach (($rooms ?? []) as $r): ?>
                  <option value="<?= (int)$r['id'] ?>"><?= htmlspecialchars($r['name']) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-1">
              <button class="btn btn-primary w-100">Upload</button>
            </div>
          </form>
        </div>
        <div class="row g-3">
          <?php foreach (($images ?? []) as $img): ?>
            <div class="col-md-3">
              <div class="card h-100">
                <img src="<?= htmlspecialchars($img['file_path']) ?>" class="card-img-top" alt="">
                <div class="card-body p-2">
                  <div class="small text-muted">#<?= (int)$img['id'] ?><?= isset($img['room_id']) && $img['room_id'] ? ' • Room '.$img['room_id'] : '' ?></div>
                  <div class="fw-semibold text-truncate" title="<?= htmlspecialchars($img['caption'] ?? '') ?>"><?= htmlspecialchars($img['caption'] ?? '') ?></div>
                </div>
                <div class="card-footer p-2 d-flex justify-content-between align-items-center">
                  <span class="small">Sort: <?= htmlspecialchars((string)($img['sort_order'] ?? '—')) ?></span>
                  <form method="post" action="/admin/hotels/images/delete" onsubmit="return confirm('Delete this image?')">
                    <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf ?? Security::csrfToken()) ?>">
                    <input type="hidden" name="hotel_id" value="<?= (int)$hotel['id'] ?>">
                    <input type="hidden" name="id" value="<?= (int)$img['id'] ?>">
                    <button class="btn btn-sm btn-outline-danger">Delete</button>
                  </form>
                </div>
              </div>
            </div>
          <?php endforeach; ?>
          <?php if (empty($images)): ?>
            <div class="col-12"><div class="alert alert-secondary">No images yet.</div></div>
          <?php endif; ?>
        </div>
        <?php endif; ?>
      </div>

      <div class="tab-pane fade" id="pane-bookings" role="tabpanel">
        <div class="text-muted">Related bookings list will appear here. Coming next.</div>
      </div>
    </div>
  </div>
</div>
<script>
(function(){
  // Activate requested tab based on data-active-tab
  const tabsEl = document.getElementById('hotelTabs');
  if (!tabsEl) return;
  const active = tabsEl.getAttribute('data-active-tab') || 'details';
  const trigger = document.querySelector(`#hotelTabs button[data-bs-target="#pane-${active}"]`);
  if (trigger && window.bootstrap && bootstrap.Tab) {
    const t = new bootstrap.Tab(trigger);
    t.show();
  } else if (trigger) {
    // fallback: toggle classes
    document.querySelectorAll('#hotelTabs button').forEach(b=>b.classList.remove('active'));
    trigger.classList.add('active');
    document.querySelectorAll('.tab-pane').forEach(p=>p.classList.remove('show','active'));
    const pane = document.querySelector(`#pane-${active}`);
    if (pane) pane.classList.add('show','active');
  }
})();

// Sync Agent Featured toggle with label dropdown
(function(){
  const featAgent = document.getElementById('featuredAgent');
  const agentWrap = document.getElementById('agentLabelWrap');
  const agentSel = document.getElementById('featuredAgentLabel');
  if (!featAgent || !agentWrap || !agentSel) return;
  const sync = () => {
    const on = !!featAgent.checked;
    agentWrap.style.display = on ? '' : 'none';
    agentSel.disabled = !on;
    if (!on) agentSel.value = '';
  };
  featAgent.addEventListener('change', sync);
  sync();
})();

// Dependent selects: filter City by selected Country
(function(){
  const countrySel = document.getElementById('countrySelect');
  const citySel = document.getElementById('citySelect');
  const vendorSel = document.getElementById('vendorSelect');
  if (!countrySel || !citySel) return;

  // Keep a cache of all city options
  const allCityOptions = Array.from(citySel.querySelectorAll('option'));

  const populate = () => {
    const ctry = countrySel.value || '';
    const prev = citySel.value || '';
    // Reset options: keep the first placeholder, then matching cities
    citySel.innerHTML = '';
    const placeholder = document.createElement('option');
    placeholder.value = '';
    placeholder.textContent = '-- Select City --';
    citySel.appendChild(placeholder);
    const matches = allCityOptions.filter(opt => (opt.dataset.country || '') === ctry && (opt.value || '') !== '');
    matches.forEach(opt => citySel.appendChild(opt.cloneNode(true)));
    // Try re-select previous if still valid
    if (prev && matches.some(o => o.value === prev)) {
      citySel.value = prev;
    }
  };

  // Initial on load
  populate();
  // On change
  countrySel.addEventListener('change', populate);
  
  // Vendor filtering by selected location (module='hotel' vendors were preloaded server-side)
  const filterVendors = () => {
    if (!vendorSel) return;
    const ctry = countrySel.value || '';
    const cty = citySel.value || '';
    const keepPlaceholder = [];
    const options = Array.from(vendorSel.options);
    options.forEach((opt, idx) => {
      if (idx === 0) return; // keep placeholder
      const oc = (opt.getAttribute('data-country') || '');
      const oy = (opt.getAttribute('data-city') || '');
      const match = (cty ? (oc === ctry && oy === cty) : (oc === ctry));
      opt.hidden = !match;
      opt.disabled = !match;
    });
    // If current selection hidden, reset to placeholder
    if (vendorSel.selectedIndex > 0) {
      const selOpt = vendorSel.selectedOptions[0];
      if (selOpt && (selOpt.hidden || selOpt.disabled)) {
        vendorSel.selectedIndex = 0;
      }
    }
  };
  // Run once and on changes
  filterVendors();
  countrySel.addEventListener('change', filterVendors);
  citySel.addEventListener('change', filterVendors);
})();

// Simple drag & drop for Images tab
(function(){
  const dz = document.getElementById('imageDropzone');
  const input = document.getElementById('imageInput');
  const selected = document.getElementById('imageSelected');
  if (!dz || !input) return;

  const setFile = (fileList) => {
    if (!fileList || fileList.length === 0) return;
    // Assign dropped files to input via DataTransfer
    const dt = new DataTransfer();
    dt.items.add(fileList[0]);
    input.files = dt.files;
    dz.classList.remove('border-secondary-subtle');
    dz.classList.add('border-primary');
    if (selected) {
      selected.textContent = `Selected: ${fileList[0].name}`;
      selected.classList.remove('d-none');
    }
  };

  ['dragenter','dragover'].forEach(ev => dz.addEventListener(ev, e => {
    e.preventDefault(); e.stopPropagation();
    dz.classList.add('border-primary', 'bg-light');
  }));
  ['dragleave','drop'].forEach(ev => dz.addEventListener(ev, e => {
    e.preventDefault(); e.stopPropagation();
    if (ev !== 'drop') {
      dz.classList.remove('bg-light');
    }
  }));
  dz.addEventListener('drop', e => {
    const files = e.dataTransfer?.files;
    setFile(files);
    dz.classList.remove('bg-light');
  });
  dz.addEventListener('click', () => input.click());
  input.addEventListener('change', () => setFile(input.files));
})();

// Sync Agent Price -> legacy price[] hidden field for backward compatibility
(function(){
  const table = document.querySelector('#pane-calendar table');
  if (!table) return;
  const sync = (rowIndex, val) => {
    const hidden = table.querySelector(`input.cal-legacy-price[data-row-index="${rowIndex}"]`);
    if (hidden) hidden.value = val;
  };
  table.querySelectorAll('input.cal-agent-price').forEach(inp => {
    inp.addEventListener('input', e => {
      const idx = e.currentTarget.getAttribute('data-row-index');
      sync(idx, e.currentTarget.value || '');
    });
  });
})();

// Professional UX: live margins, validation, formatting, tooltips
(function(){
  const pane = document.getElementById('pane-calendar');
  if (!pane) return;
  // Initialize Bootstrap tooltips if available
  if (window.bootstrap && bootstrap.Tooltip) {
    [...pane.querySelectorAll('[data-bs-toggle="tooltip"]')].forEach(el => new bootstrap.Tooltip(el));
  }

  const parseNum = (v) => {
    if (v === null || v === undefined) return NaN;
    const n = parseFloat((v+'').replace(/,/g,''));
    return isNaN(n) ? NaN : n;
  };
  const fmt2 = (n) => {
    if (n === '' || isNaN(n)) return '';
    return Number(n).toFixed(2);
  };

  const recalcRow = (row) => {
    const i = row.querySelector('.cal-agent')?.getAttribute('data-row-index') ?? '0';
    const vInp = row.querySelector(`.cal-vendor[data-row-index="${i}"]`);
    const aInp = row.querySelector(`.cal-agent[data-row-index="${i}"]`);
    const cInp = row.querySelector(`.cal-customer[data-row-index="${i}"]`);
    const avBadge = row.querySelector(`[data-role="marg-av"][data-row="${i}"]`);
    const caBadge = row.querySelector(`[data-role="marg-ca"][data-row="${i}"]`);

    const v = parseNum(vInp?.value);
    const a = parseNum(aInp?.value);
    const c = parseNum(cInp?.value);

    // Reset classes
    [avBadge, caBadge].forEach(b => { if (b){ b.classList.remove('bg-success','bg-danger','bg-secondary'); b.classList.add('bg-secondary'); }});
    [vInp, aInp, cInp].forEach(inp => { if (inp){ inp.classList.remove('is-invalid'); }});

    // Margins
    if (!isNaN(a) && !isNaN(v) && avBadge){
      const av = a - v;
      avBadge.textContent = `A−V: ${fmt2(av)}`;
      avBadge.classList.remove('bg-secondary');
      avBadge.classList.add(av >= 0 ? 'bg-success' : 'bg-danger');
    }
    if (!isNaN(c) && !isNaN(a) && caBadge){
      const ca = c - a;
      caBadge.textContent = `C−A: ${fmt2(ca)}`;
      caBadge.classList.remove('bg-secondary');
      caBadge.classList.add(ca >= 0 ? 'bg-success' : 'bg-danger');
    }

    // Validation: enforce V <= A <= C
    if (!isNaN(v) && !isNaN(a) && a < v && aInp) aInp.classList.add('is-invalid');
    if (!isNaN(a) && !isNaN(c) && c < a && cInp) cInp.classList.add('is-invalid');
  };

  const table = pane.querySelector('table');
  if (!table) return;
  table.querySelectorAll('tbody tr').forEach(tr => recalcRow(tr));
  table.addEventListener('input', (e) => {
    const tr = e.target.closest('tr');
    if (tr) recalcRow(tr);
  });
  table.addEventListener('blur', (e) => {
    const t = e.target;
    if (t.matches('.num-input')) { t.value = fmt2(parseNum(t.value)); }
  }, true);
})();
</script>
