<?php
namespace App\Services;

use PDO;

class PricingEngine
{
    private PDO $pdo;
    public function __construct(PDO $pdo) { $this->pdo = $pdo; }

    /**
     * Calculate price breakdown.
     * Steps: vendor base -> group (pax) -> admin markup -> corporate -> promo -> taxes
     */
    public function quote(array $input): array
    {
        $module = $input['module']; // hotel|activity|taxi|evisa
        $refId = (int)$input['ref_id']; // room/tier/etc
        $pax = max(1, (int)($input['pax'] ?? 1));
        $date = $input['date'] ?? null;
        $promo = $input['promo'] ?? null;
        $agentId = $input['agent_id'] ?? null;

        // Load app config (markup, tax)
        $app = require dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'app.php';
        $markupPercent = (float)($app['admin_markup_percent'] ?? 10);
        $taxPercent = (float)($app['tax_percent'] ?? 7);

        $steps = [];
        // Vendor base (calendar override > default)
        $vb = $this->vendorBase($module, $refId, $date);
        if (!empty($vb['blackout'])) {
            return [
                'module' => $module,
                'ref_id' => $refId,
                'pax' => $pax,
                'date' => $date,
                'steps' => [['step' => 'blackout', 'amount' => 0.0, 'applied' => true]],
                'total' => 0.0,
                'blackout' => true,
            ];
        }
        $base = (float)$vb['amount'];
        $steps[] = ['step' => 'vendor_base', 'amount' => $base, 'override' => (bool)($vb['override'] ?? false)];

        // Group pricing
        $grouped = $this->applyGroup($module, $refId, $pax, $base);
        $steps[] = ['step' => 'group', 'amount' => $grouped['amount'], 'is_group' => $grouped['is_group']];

        // Admin markup (simple % for starter)
        $markup = round($grouped['amount'] * ($markupPercent/100), 2);
        $steps[] = ['step' => 'admin_markup', 'amount' => $markup];
        $subtotal = $grouped['amount'] + $markup;

        // Corporate plan (stub: no corporate applied)
        $steps[] = ['step' => 'corporate', 'amount' => 0.00, 'applied' => false];

        // Promo (stub: no promo)
        $steps[] = ['step' => 'promo', 'amount' => 0.00, 'code' => $promo, 'applied' => false];

        // Taxes (stub: 7%)
        $tax = round($subtotal * ($taxPercent/100), 2);
        $steps[] = ['step' => 'tax', 'amount' => $tax];

        $total = round(($subtotal + $tax) * $pax, 2);
        return [
            'module' => $module,
            'ref_id' => $refId,
            'pax' => $pax,
            'date' => $date,
            'steps' => $steps,
            'total' => $total,
        ];
    }

    private function vendorBase(string $module, int $refId, ?string $date): array
    {
        $table = $module . 's';
        // Check calendars: blackout and per-date override
        if ($date) {
            $c = $this->pdo->prepare('SELECT price, blackout FROM calendars WHERE module=:m AND item_id=:id AND date=:d LIMIT 1');
            $c->execute(['m' => $module, 'id' => $refId, 'd' => $date]);
            $cal = $c->fetch();
            if ($cal) {
                if ((int)($cal['blackout'] ?? 0) === 1) {
                    return ['amount' => 0.0, 'blackout' => true];
                }
                if ($cal['price'] !== null) {
                    return ['amount' => (float)$cal['price'], 'override' => true];
                }
            }
        }
        // Fallback to base_price on module table
        $stmt = $this->pdo->prepare("SELECT base_price FROM {$table} WHERE id=:id");
        $stmt->execute(['id' => $refId]);
        $row = $stmt->fetch();
        return ['amount' => ($row ? (float)$row['base_price'] : 0.0), 'override' => false];
    }

    private function applyGroup(string $module, int $refId, int $pax, float $amount): array
    {
        $stmt = $this->pdo->prepare('SELECT pax_threshold, group_price FROM group_pricing WHERE module=:m AND item_id=:id LIMIT 1');
        $stmt->execute(['m' => $module, 'id' => $refId]);
        $gp = $stmt->fetch();
        if ($gp && $pax >= (int)$gp['pax_threshold']) {
            return ['amount' => (float)$gp['group_price'], 'is_group' => true];
        }
        return ['amount' => $amount, 'is_group' => false];
    }
}
