<?php
namespace App\Services;

use PDO;

class ChannelPartnerWalletService
{
    public function __construct(private PDO $pdo) {}

    public function getOrCreateWallet(int $partnerId): int
    {
        $stmt = $this->pdo->prepare('SELECT id FROM channel_partner_wallet WHERE partner_id = :p LIMIT 1');
        $stmt->execute(['p' => $partnerId]);
        $w = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($w) return (int)$w['id'];
        $this->pdo->prepare('INSERT INTO channel_partner_wallet (partner_id, balance) VALUES (:p, 0)')->execute(['p' => $partnerId]);
        return (int)$this->pdo->lastInsertId();
    }

    public function creditApproved(int $partnerId, float $amount, string $method, array $meta = []): bool
    {
        if ($amount <= 0) return false;
        $this->pdo->beginTransaction();
        try {
            $walletId = $this->getOrCreateWallet($partnerId);
            // direct credit to partner wallet uses method 'manual'
            $this->pdo->prepare('INSERT INTO channel_partner_wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "credit", :a, "manual", "approved", :meta)')
                ->execute(['w'=>$walletId,'a'=>$amount,'meta'=>json_encode($meta)]);
            $this->pdo->prepare('UPDATE channel_partner_wallet SET balance = balance + :a WHERE id = :w')
                ->execute(['a'=>$amount,'w'=>$walletId]);
            $this->pdo->commit();
            return true;
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) $this->pdo->rollBack();
            return false;
        }
    }

    public function transferFromUserApproved(int $fromUserId, int $toPartnerId, float $amount, string $method, array $meta = []): bool
    {
        if ($amount <= 0) return false;
        $this->pdo->beginTransaction();
        try {
            // lock user wallet (source)
            $fromWallet = $this->pdo->prepare('SELECT id, balance FROM wallets WHERE user_id = :u FOR UPDATE');
            $fromWallet->execute(['u'=>$fromUserId]);
            $fw = $fromWallet->fetch(PDO::FETCH_ASSOC);
            if (!$fw || (float)$fw['balance'] < $amount) { $this->pdo->rollBack(); return false; }
            $toWalletId = $this->getOrCreateWallet($toPartnerId);
            $metaJson = json_encode($meta);
            // user ledger entries
            // user wallet ledger method limited to 'manual' or 'stripe' -> choose 'manual'
            $this->pdo->prepare('INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "debit", :a, "manual", "approved", :meta)')
                ->execute(['w'=>(int)$fw['id'],'a'=>$amount,'meta'=>$metaJson]);
            $this->pdo->prepare('UPDATE wallets SET balance = balance - :a WHERE id = :w')
                ->execute(['a'=>$amount,'w'=>(int)$fw['id']]);
            // partner ledger entries
            $this->pdo->prepare('INSERT INTO channel_partner_wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "credit", :a, "transfer", "approved", :meta)')
                ->execute(['w'=>$toWalletId,'a'=>$amount,'meta'=>$metaJson]);
            $this->pdo->prepare('UPDATE channel_partner_wallet SET balance = balance + :a WHERE id = :w')
                ->execute(['a'=>$amount,'w'=>$toWalletId]);
            $this->pdo->commit();
            return true;
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) $this->pdo->rollBack();
            return false;
        }
    }

    public function transferToUserApproved(int $fromPartnerId, int $toUserId, float $amount, string $method, array $meta = []): bool
    {
        if ($amount <= 0) return false;
        $this->pdo->beginTransaction();
        try {
            // lock partner wallet (source)
            $src = $this->pdo->prepare('SELECT id, balance FROM channel_partner_wallet WHERE partner_id = :p FOR UPDATE');
            $src->execute(['p'=>$fromPartnerId]);
            $pw = $src->fetch(PDO::FETCH_ASSOC);
            if (!$pw || (float)$pw['balance'] < $amount) { $this->pdo->rollBack(); return false; }
            // ensure user wallet
            $userWalletId = null;
            $stmt = $this->pdo->prepare('SELECT id FROM wallets WHERE user_id = :u LIMIT 1');
            $stmt->execute(['u' => $toUserId]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($row) { $userWalletId = (int)$row['id']; }
            else {
                $this->pdo->prepare('INSERT INTO wallets (user_id, balance) VALUES (:u, 0)')->execute(['u'=>$toUserId]);
                $userWalletId = (int)$this->pdo->lastInsertId();
            }
            $metaJson = json_encode($meta);
            // partner ledger
            $this->pdo->prepare('INSERT INTO channel_partner_wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "debit", :a, "transfer", "approved", :meta)')
                ->execute(['w'=>(int)$pw['id'],'a'=>$amount,'meta'=>$metaJson]);
            $this->pdo->prepare('UPDATE channel_partner_wallet SET balance = balance - :a WHERE id = :w')
                ->execute(['a'=>$amount,'w'=>(int)$pw['id']]);
            // user ledger
            $this->pdo->prepare('INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "credit", :a, "manual", "approved", :meta)')
                ->execute(['w'=>$userWalletId,'a'=>$amount,'meta'=>$metaJson]);
            $this->pdo->prepare('UPDATE wallets SET balance = balance + :a WHERE id = :w')
                ->execute(['a'=>$amount,'w'=>$userWalletId]);
            $this->pdo->commit();
            return true;
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) $this->pdo->rollBack();
            return false;
        }
    }

    // Reverse original flow: User -> Partner (credit partner). Compensating move: debit partner, credit user.
    public function reverseUserToPartnerApproved(int $originalUserId, int $partnerId, float $amount, string $reason, array $extraMeta = []): bool
    {
        if ($amount <= 0) return false;
        $this->pdo->beginTransaction();
        try {
            // lock partner wallet (will be debited)
            $src = $this->pdo->prepare('SELECT id, balance FROM channel_partner_wallet WHERE partner_id = :p FOR UPDATE');
            $src->execute(['p'=>$partnerId]);
            $pw = $src->fetch(PDO::FETCH_ASSOC);
            if (!$pw || (float)$pw['balance'] < $amount) { $this->pdo->rollBack(); return false; }
            // ensure user wallet
            $userWalletId = null;
            $stmt = $this->pdo->prepare('SELECT id FROM wallets WHERE user_id = :u LIMIT 1');
            $stmt->execute(['u' => $originalUserId]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($row) { $userWalletId = (int)$row['id']; }
            else {
                $this->pdo->prepare('INSERT INTO wallets (user_id, balance) VALUES (:u, 0)')->execute(['u'=>$originalUserId]);
                $userWalletId = (int)$this->pdo->lastInsertId();
            }
            $meta = array_merge([
                'flow' => 'reversal',
                'reason' => $reason,
                'reversed_at' => date('Y-m-d H:i:s'),
                'original_user_id' => $originalUserId,
                'partner_id' => $partnerId,
            ], $extraMeta);
            $metaJson = json_encode($meta);
            // partner debit
            $this->pdo->prepare('INSERT INTO channel_partner_wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "debit", :a, "transfer", "approved", :meta)')
                ->execute(['w'=>(int)$pw['id'],'a'=>$amount,'meta'=>$metaJson]);
            $this->pdo->prepare('UPDATE channel_partner_wallet SET balance = balance - :a WHERE id = :w')
                ->execute(['a'=>$amount,'w'=>(int)$pw['id']]);
            // user credit
            $this->pdo->prepare('INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "credit", :a, "manual", "approved", :meta)')
                ->execute(['w'=>$userWalletId,'a'=>$amount,'meta'=>$metaJson]);
            $this->pdo->prepare('UPDATE wallets SET balance = balance + :a WHERE id = :w')
                ->execute(['a'=>$amount,'w'=>$userWalletId]);
            $this->pdo->commit();
            return true;
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) $this->pdo->rollBack();
            return false;
        }
    }

    // Reverse original flow: Partner -> User (credit user). Compensating move: debit user, credit partner.
    public function reversePartnerToUserApproved(int $partnerId, int $originalUserId, float $amount, string $reason, array $extraMeta = []): bool
    {
        if ($amount <= 0) return false;
        $this->pdo->beginTransaction();
        try {
            // lock user wallet (will be debited)
            $src = $this->pdo->prepare('SELECT id, balance FROM wallets WHERE user_id = :u FOR UPDATE');
            $src->execute(['u'=>$originalUserId]);
            $uw = $src->fetch(PDO::FETCH_ASSOC);
            if (!$uw || (float)$uw['balance'] < $amount) { $this->pdo->rollBack(); return false; }
            // ensure partner wallet
            $partnerWalletId = null;
            $stmt = $this->pdo->prepare('SELECT id FROM channel_partner_wallet WHERE partner_id = :p LIMIT 1');
            $stmt->execute(['p' => $partnerId]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($row) { $partnerWalletId = (int)$row['id']; }
            else {
                $this->pdo->prepare('INSERT INTO channel_partner_wallet (partner_id, balance) VALUES (:p, 0)')->execute(['p'=>$partnerId]);
                $partnerWalletId = (int)$this->pdo->lastInsertId();
            }
            $meta = array_merge([
                'flow' => 'reversal',
                'reason' => $reason,
                'reversed_at' => date('Y-m-d H:i:s'),
                'original_user_id' => $originalUserId,
                'partner_id' => $partnerId,
            ], $extraMeta);
            $metaJson = json_encode($meta);
            // user debit
            $this->pdo->prepare('INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "debit", :a, "manual", "approved", :meta)')
                ->execute(['w'=>(int)$uw['id'],'a'=>$amount,'meta'=>$metaJson]);
            $this->pdo->prepare('UPDATE wallets SET balance = balance - :a WHERE id = :w')
                ->execute(['a'=>$amount,'w'=>(int)$uw['id']]);
            // partner credit
            $this->pdo->prepare('INSERT INTO channel_partner_wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "credit", :a, "transfer", "approved", :meta)')
                ->execute(['w'=>$partnerWalletId,'a'=>$amount,'meta'=>$metaJson]);
            $this->pdo->prepare('UPDATE channel_partner_wallet SET balance = balance + :a WHERE id = :w')
                ->execute(['a'=>$amount,'w'=>$partnerWalletId]);
            $this->pdo->commit();
            return true;
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) $this->pdo->rollBack();
            return false;
        }
    }
}
