<?php
namespace App\Middleware;

class AgentGuard
{
    public static function requireLogin(): void
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }
        if (empty($_SESSION['agent']) || empty($_SESSION['agent']['id'])) {
            // Remember where the agent wanted to go, to redirect after successful login
            if (!isset($_SESSION['agent_redirect_to'])) {
                $_SESSION['agent_redirect_to'] = $_SERVER['REQUEST_URI'] ?? '/b2b/agent';
            }
            header('Location: ' . rtrim(\APP_URL, '/') . '/b2b/agent/login');
            exit;
        }

        // Hardened security headers for all authenticated agent pages
        // Safe to send multiple times; PHP header() will set them if not already sent
        // Clickjacking protection
        header('X-Frame-Options: DENY');
        // MIME sniffing protection
        header('X-Content-Type-Options: nosniff');
        // Better referrer privacy
        header('Referrer-Policy: strict-origin-when-cross-origin');
        // Lock down powerful features by default
        header('Permissions-Policy: camera=(), microphone=(), geolocation=()');
        // Conservative Content Security Policy tailored for this portal
        // Allows self, https CDNs, inline styles/scripts for current UI (Tailwind/DaisyUI)
        header("Content-Security-Policy: default-src 'self' https: data: blob:; img-src 'self' https: data: blob:; style-src 'self' 'unsafe-inline' https:; script-src 'self' 'unsafe-inline' https:; frame-ancestors 'none';");
    }

    // CSRF helpers for Agent portal forms
    public static function generateCsrfToken(string $key): string
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }
        if (!isset($_SESSION['csrf']) || !is_array($_SESSION['csrf'])) {
            $_SESSION['csrf'] = [];
        }
        $token = bin2hex(random_bytes(16));
        $_SESSION['csrf'][$key] = $token;
        return $token;
    }

    public static function validateCsrf(string $key, ?string $token): bool
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }
        if (!isset($_SESSION['csrf'][$key])) { return false; }
        $ok = hash_equals((string)$_SESSION['csrf'][$key], (string)$token);
        // One-time token: invalidate after check
        unset($_SESSION['csrf'][$key]);
        return $ok;
    }
}
