<?php
namespace App\Core\WhatsApp;

use App\Core\WhatsApp\Providers\AiSensyProvider;
use App\Core\WhatsApp\Providers\TwilioProvider;
use PDO;

class WhatsAppService
{
    private PDO $pdo;
    private ?bool $logsHasQueueId = null;

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    public function loadSettings(): array
    {
        $st = $this->pdo->query("SELECT * FROM whatsapp_settings WHERE id=1");
        $row = $st?->fetch(PDO::FETCH_ASSOC) ?: [];
        if (!$row) {
            $this->pdo->exec("INSERT IGNORE INTO whatsapp_settings (id) VALUES (1)");
            $st = $this->pdo->query("SELECT * FROM whatsapp_settings WHERE id=1");
            $row = $st?->fetch(PDO::FETCH_ASSOC) ?: [];
        }
        return $row;
    }

    private function providerFrom(array $settings): ProviderInterface|null
    {
        $active = $settings['active_provider'] ?? 'disabled';
        if ($active === 'aisensy') {
            return new AiSensyProvider(
                (string)($settings['aisensy_api_base'] ?? ''),
                (string)($settings['aisensy_api_key'] ?? ''),
                $settings['aisensy_sender'] ?? null
            );
        }
        if ($active === 'twilio') {
            return new TwilioProvider((string)($settings['twilio_sid'] ?? ''), (string)($settings['twilio_token'] ?? ''), (string)($settings['twilio_from'] ?? ''));
        }
        return null;
    }

    public function enqueue(string $eventKey, string $audience, string $module, ?int $bookingId, string $toPhone, string $messageText, string $locale = 'en'): int
    {
        $sql = "INSERT INTO whatsapp_queue (event_key, audience, module, booking_id, to_phone, message_text, locale) VALUES (:e,:a,:m,:b,:t,:msg,:l)";
        $st = $this->pdo->prepare($sql);
        $st->execute([
            ':e' => $eventKey,
            ':a' => $audience,
            ':m' => $module,
            ':b' => $bookingId,
            ':t' => $toPhone,
            ':msg' => $messageText,
            ':l' => $locale,
        ]);
        return (int)$this->pdo->lastInsertId();
    }

    public function sendQueueItem(array $row): array
    {
        $settings = $this->loadSettings();
        $provider = $this->providerFrom($settings);
        if (!$provider) {
            $this->logAttempt((int)$row['id'], $settings['active_provider'] ?? 'disabled', $row['to_phone'], 'failed', null, 'Provider disabled');
            $this->updateQueueStatus((int)$row['id'], 'failed', 'Provider disabled');
            return ['ok'=>false,'message'=>'Provider disabled'];
        }
        // Determine default sender from active provider settings
        $from = $settings['aisensy_sender'] ?? ($settings['twilio_from'] ?? null);
        $msg = new Message($this->normalizePhone($row['to_phone']), (string)$row['message_text'], $from, 'text', []);
        $res = $provider->send($msg);
        $this->logAttempt((int)$row['id'], $provider->name(), $row['to_phone'], $res->ok ? 'sent' : 'failed', $res->messageId, $res->error);
        $this->updateQueueStatus((int)$row['id'], $res->ok ? 'sent' : 'failed', $res->error);
        return ['ok'=>$res->ok,'message'=>$res->error ?? 'Sent'];
    }

    public function sendImmediate(string $toPhone, string $text, ?string $fromOverride = null, array $meta = []): array
    {
        $settings = $this->loadSettings();
        $provider = $this->providerFrom($settings);
        if (!$provider) return ['ok'=>false,'message'=>'Provider disabled'];
        $from = $fromOverride ?: ($settings['aisensy_sender'] ?? ($settings['twilio_from'] ?? null));
        $msg = new Message($this->normalizePhone($toPhone), $text, $from, 'text', $meta);
        $res = $provider->send($msg);
        $this->logAttempt(null, $provider->name(), $toPhone, $res->ok ? 'sent' : 'failed', $res->messageId, $res->error);
        return ['ok'=>$res->ok,'message'=>$res->error ?? 'Sent'];
    }

    private function updateQueueStatus(int $id, string $status, ?string $error = null): void
    {
        $st = $this->pdo->prepare("UPDATE whatsapp_queue SET status=:s, attempts=attempts+1, last_error=:e, sent_at = CASE WHEN :s='sent' THEN NOW() ELSE sent_at END WHERE id=:id");
        $st->execute([':s'=>$status, ':e'=>$error, ':id'=>$id]);
    }

    private function logAttempt(?int $queueId, string $provider, string $to, string $status, ?string $providerId, ?string $error): void
    {
        // Ensure table and required columns exist; then insert with available columns
        $cols = $this->getTableColumns('whatsapp_logs');
        // If table is missing entirely, try to create a minimal version
        if ($cols === null) {
            try {
                $this->pdo->exec("CREATE TABLE IF NOT EXISTS whatsapp_logs (
                    id BIGINT PRIMARY KEY AUTO_INCREMENT,
                    provider VARCHAR(32) NOT NULL,
                    to_phone VARCHAR(32) NOT NULL,
                    status VARCHAR(16) NOT NULL,
                    provider_message_id VARCHAR(100) NULL,
                    error_text TEXT NULL,
                    created_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP
                )");
            } catch (\Throwable $e) {
                // give up silently; avoid blocking sends
                return;
            }
            $cols = $this->getTableColumns('whatsapp_logs') ?? [];
        }
        // Add missing standard columns if possible
        $required = [
            'queue_id' => "ALTER TABLE whatsapp_logs ADD COLUMN queue_id BIGINT NULL AFTER id",
            'provider' => "ALTER TABLE whatsapp_logs ADD COLUMN provider VARCHAR(32) NOT NULL AFTER queue_id",
            'to_phone' => "ALTER TABLE whatsapp_logs ADD COLUMN to_phone VARCHAR(32) NOT NULL AFTER provider",
            'status' => "ALTER TABLE whatsapp_logs ADD COLUMN status VARCHAR(16) NOT NULL AFTER to_phone",
            'provider_message_id' => "ALTER TABLE whatsapp_logs ADD COLUMN provider_message_id VARCHAR(100) NULL AFTER status",
            'error_text' => "ALTER TABLE whatsapp_logs ADD COLUMN error_text TEXT NULL AFTER provider_message_id",
            'created_at' => "ALTER TABLE whatsapp_logs ADD COLUMN created_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP AFTER error_text",
        ];
        foreach ($required as $c => $ddl) {
            if (!in_array($c, $cols, true)) {
                try { $this->pdo->exec($ddl); } catch (\Throwable $_) {}
            }
        }
        $cols = $this->getTableColumns('whatsapp_logs') ?? [];
        // Build dynamic insert using intersection of available columns
        $data = [
            'queue_id' => $queueId,
            'provider' => $provider,
            'to_phone' => $to,
            'status' => $status,
            'provider_message_id' => $providerId,
            'error_text' => $error,
        ];
        $available = array_keys(array_intersect_key($data, array_flip($cols)));
        if (empty($available)) return; // nothing we can insert
        $fields = implode(', ', $available);
        $placeholders = implode(', ', array_map(fn($k)=>":".$k, $available));
        $st = $this->pdo->prepare("INSERT INTO whatsapp_logs ($fields) VALUES ($placeholders)");
        $bind = [];
        foreach ($available as $k) { $bind[":$k"] = $data[$k]; }
        try { $st->execute($bind); } catch (\Throwable $_) { /* suppress */ }
    }

    private function getTableColumns(string $table): ?array
    {
        try {
            $rs = $this->pdo->query("SHOW COLUMNS FROM `".$table."`");
            if (!$rs) return null;
            $cols = [];
            while ($r = $rs->fetch(PDO::FETCH_ASSOC)) { $cols[] = (string)$r['Field']; }
            return $cols;
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function normalizePhone(string $phone): string
    {
        $p = preg_replace('/[^0-9+]/', '', $phone) ?: $phone;
        if (str_starts_with($p, '00')) $p = '+' . substr($p, 2);
        if ($p && $p[0] !== '+') {
            // assume country prefix if missing? leave as-is to avoid wrong numbers
            return $p;
        }
        return $p;
    }
}
