<?php
namespace App\Core\WhatsApp\Providers;

use App\Core\WhatsApp\Message;
use App\Core\WhatsApp\ProviderInterface;
use App\Core\WhatsApp\SendResult;

class TwilioProvider implements ProviderInterface
{
    private string $sid;
    private string $token;
    private string $from;

    public function __construct(string $sid, string $token, string $from)
    {
        $this->sid = $sid;
        $this->token = $token;
        $this->from = $from; // expected format: 'whatsapp:+E164'
    }

    public function name(): string { return 'twilio'; }

    public function send(Message $msg): SendResult
    {
        // Prefer SDK if available, else REST fallback
        if (class_exists('Twilio\\Rest\\Client')) {
            try {
                $client = new \Twilio\Rest\Client($this->sid, $this->token);
                $m = $client->messages->create(
                    'whatsapp:' . ltrim($msg->to, 'whatsapp:'),
                    [
                        'from' => $this->from,
                        'body' => $msg->text,
                    ]
                );
                return new SendResult(true, $m->sid ?? null, null);
            } catch (\Throwable $e) {
                return new SendResult(false, null, $e->getMessage());
            }
        }
        // REST API fallback
        $to = 'whatsapp:' . ltrim($msg->to, 'whatsapp:');
        $url = "https://api.twilio.com/2010-04-01/Accounts/{$this->sid}/Messages.json";
        $payload = http_build_query([
            'From' => $this->from,
            'To' => $to,
            'Body' => $msg->text,
        ]);
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $payload,
            CURLOPT_USERPWD => $this->sid . ':' . $this->token,
            CURLOPT_RETURNTRANSFER => true,
        ]);
        $resp = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE) ?: 0;
        $err = curl_error($ch);
        curl_close($ch);
        if ($resp === false) return new SendResult(false, null, $err ?: 'HTTP error');
        $data = json_decode($resp, true);
        $ok = ($code >= 200 && $code < 300);
        return new SendResult($ok, $data['sid'] ?? null, $ok ? null : ($data['message'] ?? $resp));
    }
}
