<?php
namespace App\Core\WhatsApp\Providers;

use App\Core\WhatsApp\Message;
use App\Core\WhatsApp\ProviderInterface;
use App\Core\WhatsApp\SendResult;

/**
 * AiSensy Provider
 * Implements template/text send using AiSensy campaign API v2
 * Docs sample (as shared): POST {base}/campaign/t1/api/v2 with JSON
 */
class AiSensyProvider implements ProviderInterface
{
    private string $apiBase;
    private string $apiKey;
    private ?string $sender;

    public function __construct(string $apiBase, string $apiKey, ?string $sender = null)
    {
        $this->apiBase = rtrim($apiBase ?: 'https://backend.aisensy.com', '/');
        $this->apiKey = $apiKey;
        $this->sender = $sender;
    }

    public function name(): string { return 'aisensy'; }

    public function send(Message $msg): SendResult
    {
        if ($this->apiKey === '') {
            return new SendResult(false, null, 'AiSensy API key missing');
        }
        $endpoint = $this->apiBase . '/campaign/t1/api/v2';
        $destination = $this->normalizeDigits($msg->to);
        // Build payload per docs
        $payload = [
            'apiKey' => $this->apiKey,
            'campaignName' => (string)($msg->meta['campaignName'] ?? $msg->meta['templateName'] ?? $msg->meta['template'] ?? 'default'),
            'destination' => $destination,
        ];
        if (!empty($msg->meta['userName'])) $payload['userName'] = (string)$msg->meta['userName'];
        if (!empty($msg->meta['source'])) $payload['source'] = (string)$msg->meta['source'];
        if (!empty($msg->meta['media']) && is_array($msg->meta['media'])) $payload['media'] = $msg->meta['media'];
        if (!empty($msg->meta['buttons']) && is_array($msg->meta['buttons'])) $payload['buttons'] = $msg->meta['buttons'];
        if (!empty($msg->meta['carouselCards']) && is_array($msg->meta['carouselCards'])) $payload['carouselCards'] = $msg->meta['carouselCards'];
        if (!empty($msg->meta['location']) && is_array($msg->meta['location'])) $payload['location'] = $msg->meta['location'];
        // Template params: accepts array of strings
        if (!empty($msg->meta['templateParams'])) {
            $params = $msg->meta['templateParams'];
            if (is_string($params)) {
                // allow comma-separated in test modal
                $params = array_values(array_filter(array_map('trim', explode(',', $params)), fn($v)=>$v!==''));
            }
            if (is_array($params)) $payload['templateParams'] = array_map('strval', $params);
        }
        if (!empty($msg->meta['tags']) && is_array($msg->meta['tags'])) $payload['tags'] = $msg->meta['tags'];
        if (!empty($msg->meta['attributes']) && is_array($msg->meta['attributes'])) $payload['attributes'] = $msg->meta['attributes'];
        if (!empty($msg->meta['paramsFallbackValue'])) {
            // allow JSON string or array
            $pf = $msg->meta['paramsFallbackValue'];
            if (is_string($pf)) {
                $dec = json_decode($pf, true);
                if (is_array($dec)) $pf = $dec; else $pf = [];
            }
            if (is_array($pf)) $payload['paramsFallbackValue'] = $pf;
        }

        // For plain text cases without template, AiSensy often uses a template configured as campaignName
        // We'll pass message text as first template param if no templateParams given but text present
        if (empty($payload['templateParams']) && $msg->text !== '') {
            $payload['templateParams'] = [ $msg->text ];
        }

        $ch = curl_init($endpoint);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [ 'Content-Type: application/json' ],
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 25,
        ]);
        $resp = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE) ?: 0;
        $err = curl_error($ch);
        curl_close($ch);
        if ($resp === false) return new SendResult(false, null, $err ?: 'HTTP error');
        $data = json_decode($resp, true);
        $ok = ($code >= 200 && $code < 300);
        // Common AiSensy response keys include: success, msgId/messageId, error
        $ok = $ok && (!isset($data['success']) || $data['success'] === true);
        $messageId = $data['messageId'] ?? ($data['msgId'] ?? null);
        if (!$ok) {
            $error = $data['error'] ?? ($data['message'] ?? ("HTTP $code: " . $resp));
            return new SendResult(false, null, is_string($error) ? $error : json_encode($error));
        }
        return new SendResult(true, is_string($messageId) ? $messageId : null, null);
    }

    private function normalizeDigits(string $phone): string
    {
        // AiSensy sample uses digits-only with country code (no plus)
        $p = preg_replace('/[^0-9]/', '', $phone) ?: $phone;
        if (str_starts_with($p, '00')) $p = substr($p, 2);
        return $p;
    }
}
