<?php
namespace App\Core;

class Security
{
    public static function csrfToken(): string
    {
        // Ensure session is active
        if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }
        // TTL fallback (1 hour) if constant not defined
        $ttl = \defined('CSRF_TTL') ? (int)CSRF_TTL : 3600;
        $now = time();
        $val = $_SESSION['csrf'] ?? null;
        $ts  = (int)($_SESSION['csrf_time'] ?? 0);
        // Regenerate if missing, wrong type, empty, or expired
        if (!is_string($val) || $val === '' || ($ttl > 0 && ($now - $ts) > $ttl)) {
            $_SESSION['csrf'] = bin2hex(random_bytes(16));
            $_SESSION['csrf_time'] = $now;
        }
        return (string)$_SESSION['csrf'];
    }

    public static function verifyCsrf(?string $token): bool
    {
        return is_string($token) && isset($_SESSION['csrf']) && hash_equals($_SESSION['csrf'], $token);
    }

    // Convenience: enforce CSRF and fail fast with 400
    public static function requireCsrf(?string $token = null): void
    {
        if ($token === null) {
            $token = $_POST['csrf'] ?? $_GET['csrf'] ?? $_POST['csrf_token'] ?? $_GET['csrf_token'] ?? ($_SERVER['HTTP_X_CSRF_TOKEN'] ?? null);
        }
        if (!self::verifyCsrf($token)) {
            http_response_code(400);
            // Return JSON for AJAX to avoid HTML in XHR responses
            $accept = (string)($_SERVER['HTTP_ACCEPT'] ?? '');
            $xhr = (string)($_SERVER['HTTP_X_REQUESTED_WITH'] ?? '');
            $isAjax = (stripos($accept, 'application/json') !== false) || (strcasecmp($xhr, 'XMLHttpRequest') === 0);
            if ($isAjax) {
                header('Content-Type: application/json');
                echo json_encode(['error' => 'Invalid CSRF']);
            } else {
                echo 'Invalid CSRF';
            }
            exit;
        }
    }

    // Non-exiting variant for controllers that want to redirect back with flash
    public static function verifyMasterPassword(?string $provided = null): bool
    {
        if ($provided === null) {
            $provided = $_POST['master_password'] ?? $_GET['master_password'] ?? null;
        }
        $expected = '';
        if (function_exists('env')) {
            $expected = (string)env('MASTER_PASSWORD', '');
        }
        if ($expected === '') {
            $expected = getenv('MASTER_PASSWORD') ?: '';
        }
        if (is_string($expected)) {
            $expected = trim($expected);
            $len = strlen($expected);
            if ($len >= 2) {
                $first = $expected[0];
                $last  = $expected[$len - 1];
                if (($first === '"' && $last === '"') || ($first === "'" && $last === "'")) {
                    $expected = substr($expected, 1, -1);
                }
            }
        }
        if (is_string($provided)) {
            $provided = trim($provided);
            $len2 = strlen($provided);
            if ($len2 >= 2) {
                $first2 = $provided[0];
                $last2  = $provided[$len2 - 1];
                if (($first2 === '"' && $last2 === '"') || ($first2 === "'" && $last2 === "'")) {
                    $provided = substr($provided, 1, -1);
                }
            }
        }
        if ($expected === '') return true; // allow in dev if not configured
        return is_string($provided) && hash_equals($expected, $provided);
    }

    public static function sanitize(string $s): string
    {
        return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
    }

    public static function enforceIpAllowlist(): void
    {
        $allow = getenv('ALLOWED_IPS') ?: '';
        if (!$allow) return;
        $list = array_filter(array_map('trim', explode(',', $allow)));
        if ($list && !in_array($_SERVER['REMOTE_ADDR'] ?? '', $list, true)) {
            http_response_code(403);
            die('Access denied');
        }
    }

    // Require master password for sensitive mutations
    public static function requireMasterPassword(?string $provided = null): void
    {
        if ($provided === null) {
            $provided = $_POST['master_password'] ?? $_GET['master_password'] ?? null;
        }
        // Prefer app env() loader (parses .env) if available; fallback to getenv
        $expected = '';
        if (function_exists('env')) {
            $expected = (string)env('MASTER_PASSWORD', '');
        }
        if ($expected === '') {
            $expected = getenv('MASTER_PASSWORD') ?: '';
        }
        // Normalize both values to avoid mismatches due to quotes in .env or inputs
        if (is_string($expected)) {
            $expected = trim($expected);
            // Trim surrounding quotes if present (PHP 7 compatible)
            $len = strlen($expected);
            if ($len >= 2) {
                $first = $expected[0];
                $last  = $expected[$len - 1];
                if (($first === '"' && $last === '"') || ($first === "'" && $last === "'")) {
                    $expected = substr($expected, 1, -1);
                }
            }
        }
        if (is_string($provided)) {
            $provided = trim($provided);
            $len2 = strlen($provided);
            if ($len2 >= 2) {
                $first2 = $provided[0];
                $last2  = $provided[$len2 - 1];
                if (($first2 === '"' && $last2 === '"') || ($first2 === "'" && $last2 === "'")) {
                    $provided = substr($provided, 1, -1);
                }
            }
        }
        // If not configured, allow to prevent lockout in dev
        if ($expected === '') return;
        if (!is_string($provided) || !hash_equals($expected, $provided)) {
            http_response_code(403);
            // Detect AJAX to return JSON instead of plain text
            $accept = (string)($_SERVER['HTTP_ACCEPT'] ?? '');
            $xhr = (string)($_SERVER['HTTP_X_REQUESTED_WITH'] ?? '');
            $isAjax = (stripos($accept, 'application/json') !== false) || (strcasecmp($xhr, 'XMLHttpRequest') === 0);
            if ($isAjax) {
                header('Content-Type: application/json');
                echo json_encode(['error' => 'Invalid master password']);
            } else {
                echo 'Invalid master password';
            }
            exit;
        }
    }
}
