<?php
namespace App\Core\Mail\Providers;

use App\Core\Mail\ProviderInterface;
use App\Core\Mail\Envelope;
use App\Core\Mail\SendResult;
use Aws\SesV2\SesV2Client;
use Aws\Exception\AwsException;

class SesProvider implements ProviderInterface
{
    private array $profile;
    private SesV2Client $client;

    public function __construct(array $profile)
    {
        $this->profile = $profile;
        $this->client = new SesV2Client([
            'version' => 'latest',
            'region' => $profile['region'] ?? 'ap-southeast-1',
            'credentials' => [
                'key' => $profile['access_key'] ?? '',
                'secret' => $profile['secret_key'] ?? '',
            ],
        ]);
    }

    public function name(): string { return 'ses'; }

    public function send(Envelope $env): SendResult
    {
        $fromEmail = $env->fromEmail ?: ($this->profile['from']['address'] ?? 'noreply@example.com');
        $fromName = $env->fromName ?: ($this->profile['from']['name'] ?? '');
        $fromHeader = $fromName ? sprintf('%s <%s>', $fromName, $fromEmail) : $fromEmail;
        $bcc = $env->bcc ?? [];
        // Merge profile BCC list if provided as comma-separated
        if (!empty($this->profile['copy_to'])) {
            foreach (explode(',', (string)$this->profile['copy_to']) as $addr) {
                $addr = trim($addr);
                if ($addr !== '') $bcc[] = $addr;
            }
        }
        try {
            $result = $this->client->sendEmail([
                'FromEmailAddress' => $fromHeader,
                'Destination' => [
                    'ToAddresses' => [ $env->to ],
                    'BccAddresses' => $bcc,
                ],
                'Content' => [
                    'Simple' => [
                        'Subject' => [ 'Data' => $env->subject, 'Charset' => 'UTF-8' ],
                        'Body' => [
                            'Html' => [ 'Data' => $env->html, 'Charset' => 'UTF-8' ],
                            'Text' => [ 'Data' => $env->altText ?: strip_tags($env->html), 'Charset' => 'UTF-8' ],
                        ],
                    ],
                ],
                // ConfigurationSetName optional
            ]);
            $messageId = $result->get('MessageId') ?: null;
            return new SendResult(true, 'Sent', $messageId, 200);
        } catch (AwsException $e) {
            return new SendResult(false, $e->getAwsErrorMessage() ?: $e->getMessage(), null, (int)($e->getStatusCode() ?: 0));
        } catch (\Throwable $e) {
            return new SendResult(false, $e->getMessage(), null, 0);
        }
    }
}
