<?php
namespace App\Core;

class FileCrypto
{
    private static function key(): string
    {
        $b64 = (string)Env::get('KYC_ENC_KEY', '');
        $b64 = trim($b64);
        // Strip surrounding quotes if present
        if ((str_starts_with($b64, '"') && str_ends_with($b64, '"')) || (str_starts_with($b64, "'") && str_ends_with($b64, "'"))) {
            $b64 = substr($b64, 1, -1);
        }
        if ($b64 === '') { throw new \RuntimeException('KYC_ENC_KEY missing'); }
        $key = base64_decode($b64, true);
        if ($key === false || strlen($key) !== 32) { throw new \RuntimeException('KYC_ENC_KEY must be base64 for 32 bytes'); }
        return $key;
    }

    /**
     * Encrypt a file using AES-256-GCM and write sidecar meta JSON.
     * Writes $destEnc (ciphertext) and $destEnc.meta.json (iv, tag, mime, v, ts).
     */
    public static function encryptFile(string $srcPath, string $destEncPath, string $mime, ?array &$meta = null): void
    {
        $key = self::key();
        $iv = random_bytes(12);
        $plaintext = file_get_contents($srcPath);
        if ($plaintext === false) { throw new \RuntimeException('Read failed'); }
        $tag = '';
        $cipher = openssl_encrypt($plaintext, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $iv, $tag);
        if ($cipher === false) { throw new \RuntimeException('Encrypt failed'); }
        $dir = dirname($destEncPath);
        if (!is_dir($dir)) { @mkdir($dir, 0775, true); }
        if (file_put_contents($destEncPath, $cipher) === false) { throw new \RuntimeException('Write failed'); }
        $metaArr = [
            'iv' => base64_encode($iv),
            'tag' => base64_encode($tag),
            'mime' => $mime,
            'v' => 1,
            'ts' => time(),
        ];
        // Write meta as <name>.meta.json (replace .enc suffix when present)
        $metaPath = preg_match('/\.enc$/', $destEncPath) ? preg_replace('/\.enc$/', '.meta.json', $destEncPath) : ($destEncPath . '.meta.json');
        @file_put_contents($metaPath, json_encode($metaArr));
        if ($meta !== null) { $meta = $metaArr; }
    }
}
