<?php
namespace App\Core;

class Audit
{
    private \PDO $pdo;

    public function __construct(\PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    /**
     * Write a security/audit event.
     * @param array $data keys: event_type (required), action, message, details (array), status_code, user_id, agent_id, order_id
     */
    public function event(array $data): void
    {
        try {
            $eventType  = (string)($data['event_type'] ?? 'unknown');
            $action     = isset($data['action']) ? (string)$data['action'] : null;
            $message    = isset($data['message']) ? (string)$data['message'] : null;
            $detailsArr = isset($data['details']) && is_array($data['details']) ? $data['details'] : null;
            $statusCode = isset($data['status_code']) ? (int)$data['status_code'] : null;
            $userId     = isset($data['user_id']) ? (int)$data['user_id'] : null;
            $agentId    = isset($data['agent_id']) ? (int)$data['agent_id'] : null;
            $orderId    = isset($data['order_id']) ? (int)$data['order_id'] : null;

            $route  = isset($_SERVER['REQUEST_URI']) ? (string)$_SERVER['REQUEST_URI'] : null;
            $method = isset($_SERVER['REQUEST_METHOD']) ? (string)$_SERVER['REQUEST_METHOD'] : null;
            $ip     = isset($_SERVER['REMOTE_ADDR']) ? (string)$_SERVER['REMOTE_ADDR'] : null;
            $ua     = isset($_SERVER['HTTP_USER_AGENT']) ? substr((string)$_SERVER['HTTP_USER_AGENT'], 0, 255) : null;

            $sql = 'INSERT INTO security_audit_events (user_id, agent_id, order_id, event_type, action, message, details, route, method, ip, user_agent, status_code) 
                    VALUES (:user_id, :agent_id, :order_id, :event_type, :action, :message, :details, :route, :method, :ip, :user_agent, :status_code)';
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([
                'user_id' => $userId,
                'agent_id' => $agentId,
                'order_id' => $orderId,
                'event_type' => $eventType,
                'action' => $action,
                'message' => $message,
                'details' => $detailsArr ? json_encode($detailsArr) : null,
                'route' => $route,
                'method' => $method,
                'ip' => $ip,
                'user_agent' => $ua,
                'status_code' => $statusCode,
            ]);
        } catch (\Throwable $_) {
            // Never throw from audit path
        }
    }
}
