<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;

class YachtsAdminController extends Controller
{
    private function normalizeUploadPath(string $urlOrPath): string
    {
        $s = trim($urlOrPath);
        if ($s === '') return $s;
        if (strpos($s, '/assets/uploads/') === 0) return $s;
        if (preg_match('#^https?://[^/]+(/.*)$#i', $s, $m)) {
            $s = $m[1];
        }
        $pos = strpos($s, '/assets/uploads/');
        if ($pos !== false) {
            return substr($s, $pos);
        }
        return $s;
    }

    private function saveYachtImages(int $yachtId, string $mediaJson, int $coverIndex): void
    {
        $list = json_decode($mediaJson ?: '[]', true);
        if (!is_array($list)) $list = [];
        try {
            $del = $this->pdo->prepare("DELETE FROM yacht_images WHERE yacht_id = :id");
            $del->execute(['id' => $yachtId]);
        } catch (\Throwable $e) {
            return; // tolerate missing table in some environments
        }
        if (!$list) return;
        try {
            $ins = $this->pdo->prepare("INSERT INTO yacht_images (yacht_id, file_path, is_cover) VALUES (:id, :path, :cover)");
            foreach ($list as $idx => $item) {
                $orig = is_array($item) ? ($item['originalUrl'] ?? ($item['url'] ?? '')) : '';
                if (!$orig) continue;
                $path = $this->normalizeUploadPath((string)$orig);
                $ins->execute([
                    'id' => $yachtId,
                    'path' => $path,
                    'cover' => ($idx === $coverIndex) ? 1 : 0,
                ]);
            }
        } catch (\Throwable $e) { /* swallow */ }
    }

    public function unlock(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $_SESSION['admin_yachts_unlocked'] = true;
        $this->redirect('/admin/yachts');
    }

    public function index(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $unlocked = !empty($_SESSION['admin_yachts_unlocked']);
        $rows = [];
        if ($unlocked) {
            try {
                $q = "SELECT y.id, y.name, y.subtitle, y.city, y.capacity, y.max_guests, y.from_price, y.currency, y.active, y.booking_code, y.visible_agent,
                            (
                              SELECT yi.file_path FROM yacht_images yi
                              WHERE yi.yacht_id = y.id
                              ORDER BY yi.is_cover DESC, yi.id ASC
                              LIMIT 1
                            ) AS cover_path
                      FROM yachts y
                      ORDER BY y.id DESC LIMIT 300";
                $rows = $this->pdo->query($q)->fetchAll();
            } catch (\Throwable $e) { $rows = []; }
        }
        $this->view('admin/yachts_index', compact('csrf', 'unlocked', 'rows'));
    }

    public function create(): void
    {
        Auth::requireRole(['Admin']);
        if (empty($_SESSION['admin_yachts_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock yachts with the update password first.';
            $this->redirect('/admin/yachts');
            return;
        }
        $csrf = Security::csrfToken();
        // Currency options
        $currencyCodes = [];
        try {
            $currencyCodes = $this->pdo->query("SELECT code FROM currencies ORDER BY code")->fetchAll();
        } catch (\Throwable $e) { $currencyCodes = []; }
        $yacht = [
            'id' => null,
            'name' => '',
            'subtitle' => '',
            'city' => '',
            'capacity' => '',
            'max_guests' => '',
            'from_price' => '0.00',
            'booking_code' => '',
            'vendor_cost' => '',
            'agent_cost' => '',
            'customer_cost' => '',
            'visible_agent' => 1,
            'currency' => 'THB',
            'active' => 1,
            'description' => '',
        ];
        $this->view('admin/yachts_form', compact('csrf', 'yacht', 'currencyCodes'));
    }

    public function edit(): void
    {
        Auth::requireRole(['Admin']);
        if (empty($_SESSION['admin_yachts_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock yachts with the update password first.';
            $this->redirect('/admin/yachts');
            return;
        }
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/yachts'); return; }
        $stmt = $this->pdo->prepare("SELECT * FROM yachts WHERE id = :id");
        $stmt->execute(['id' => $id]);
        $yacht = $stmt->fetch();
        if (!$yacht) { $this->redirect('/admin/yachts'); return; }
        $csrf = Security::csrfToken();
        // Currency options
        $currencyCodes = [];
        try {
            $currencyCodes = $this->pdo->query("SELECT code FROM currencies ORDER BY code")->fetchAll();
        } catch (\Throwable $e) { $currencyCodes = []; }
        // Prefill media
        try {
            $ms = $this->pdo->prepare("SELECT file_path, is_cover FROM yacht_images WHERE yacht_id = :id ORDER BY id ASC");
            $ms->execute(['id' => $id]);
            $imgs = $ms->fetchAll();
            if ($imgs && is_array($imgs)) {
                $files = [];
                $coverIdx = 0; $i = 0;
                foreach ($imgs as $row) {
                    $path = (string)($row['file_path'] ?? '');
                    if ($path === '') { $i++; continue; }
                    if (strpos($path, 'http://') !== 0 && strpos($path, 'https://') !== 0 && strpos($path, '/') !== 0) {
                        $path = '/' . ltrim($path, '/');
                    }
                    $files[] = ['originalUrl' => $path, 'thumbUrl' => ''];
                    if ((int)($row['is_cover'] ?? 0) === 1) { $coverIdx = $i; }
                    $i++;
                }
                if ($files) {
                    $yacht['media_json'] = json_encode($files);
                    $yacht['cover_index'] = (string)$coverIdx;
                }
            }
        } catch (\Throwable $e) { /* tolerate */ }
        $this->view('admin/yachts_form', compact('csrf', 'yacht', 'currencyCodes'));
    }

    public function store(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['admin_yachts_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock yachts with the update password first.';
            $this->redirect('/admin/yachts');
            return;
        }
        $name = trim((string)($_POST['name'] ?? ''));
        $subtitle = trim((string)($_POST['subtitle'] ?? ''));
        $city = trim((string)($_POST['city'] ?? ''));
        $capacity = (int)($_POST['capacity'] ?? 0);
        $max_guests = (int)($_POST['max_guests'] ?? 0);
        $from_price = (string)($_POST['from_price'] ?? '');
        $booking_code = trim((string)($_POST['booking_code'] ?? ''));
        $vendor_cost = $_POST['vendor_cost'] !== '' ? (string)$_POST['vendor_cost'] : null;
        $agent_cost = $_POST['agent_cost'] !== '' ? (string)$_POST['agent_cost'] : null;
        $customer_cost = $_POST['customer_cost'] !== '' ? (string)$_POST['customer_cost'] : null;
        $visible_agent = isset($_POST['visible_agent']) && $_POST['visible_agent'] === '1' ? 1 : 0;
        $currency = trim((string)($_POST['currency'] ?? 'THB')) ?: 'THB';
        $active = isset($_POST['active']) && $_POST['active'] === '1' ? 1 : 0;
        $description = trim((string)($_POST['description'] ?? ''));

        $errors = [];
        if ($name === '') $errors[] = 'Name is required';
        if ($capacity < 1) $errors[] = 'Capacity must be at least 1';
        if ($max_guests < 1) $errors[] = 'Max guests must be at least 1';
        if ($errors) {
            $_SESSION['errors'] = $errors;
            $this->redirect('/admin/yachts/create');
            return;
        }
        // If customer_cost provided but from_price empty, use customer_cost as from_price for display
        if ($from_price === '' && $customer_cost !== null && $customer_cost !== '') { $from_price = $customer_cost; }
        $stmt = $this->pdo->prepare("INSERT INTO yachts (name, subtitle, city, capacity, max_guests, from_price, booking_code, vendor_cost, agent_cost, customer_cost, visible_agent, currency, active) VALUES (:name,:subtitle,:city,:capacity,:max_guests,:from_price,:booking_code,:vendor_cost,:agent_cost,:customer_cost,:visible_agent,:currency,:active)");
        $stmt->execute([
            'name' => $name,
            'subtitle' => $subtitle ?: null,
            'city' => $city ?: null,
            'capacity' => $capacity ?: null,
            'max_guests' => $max_guests ?: null,
            'from_price' => ($from_price !== '' ? $from_price : '0'),
            'booking_code' => $booking_code ?: null,
            'vendor_cost' => $vendor_cost,
            'agent_cost' => $agent_cost,
            'customer_cost' => $customer_cost,
            'visible_agent' => $visible_agent,
            'currency' => $currency,
            'active' => $active,
        ]);
        $yachtId = (int)$this->pdo->lastInsertId();
        // Save media
        $mediaJson = (string)($_POST['media_json'] ?? '');
        $coverIndex = (int)($_POST['cover_index'] ?? 0);
        if ($mediaJson !== '') {
            $this->saveYachtImages($yachtId, $mediaJson, $coverIndex);
        }
        $_SESSION['flash'] = 'Yacht created.';
        $this->redirect('/admin/yachts');
    }

    public function update(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['admin_yachts_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock yachts with the update password first.';
            $this->redirect('/admin/yachts');
            return;
        }
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/yachts'); return; }
        $name = trim((string)($_POST['name'] ?? ''));
        $subtitle = trim((string)($_POST['subtitle'] ?? ''));
        $city = trim((string)($_POST['city'] ?? ''));
        $capacity = (int)($_POST['capacity'] ?? 0);
        $max_guests = (int)($_POST['max_guests'] ?? 0);
        $from_price = (string)($_POST['from_price'] ?? '');
        $currency = trim((string)($_POST['currency'] ?? 'THB')) ?: 'THB';
        $active = isset($_POST['active']) && $_POST['active'] === '1' ? 1 : 0;
        $description = trim((string)($_POST['description'] ?? ''));

        $errors = [];
        if ($name === '') $errors[] = 'Name is required';
        if ($capacity < 1) $errors[] = 'Capacity must be at least 1';
        if ($max_guests < 1) $errors[] = 'Max guests must be at least 1';
        if ($errors) {
            $_SESSION['errors'] = $errors;
            $this->redirect('/admin/yachts/edit?id=' . $id);
            return;
        }
        if ($from_price === '' && $customer_cost !== null && $customer_cost !== '') { $from_price = $customer_cost; }
        $stmt = $this->pdo->prepare("UPDATE yachts SET name=:name, subtitle=:subtitle, city=:city, capacity=:capacity, max_guests=:max_guests, from_price=:from_price, booking_code=:booking_code, vendor_cost=:vendor_cost, agent_cost=:agent_cost, customer_cost=:customer_cost, visible_agent=:visible_agent, currency=:currency, active=:active WHERE id=:id");
        $stmt->execute([
            'id' => $id,
            'name' => $name,
            'subtitle' => $subtitle ?: null,
            'city' => $city ?: null,
            'capacity' => $capacity ?: null,
            'max_guests' => $max_guests ?: null,
            'from_price' => ($from_price !== '' ? $from_price : '0'),
            'booking_code' => $booking_code ?: null,
            'vendor_cost' => $vendor_cost,
            'agent_cost' => $agent_cost,
            'customer_cost' => $customer_cost,
            'visible_agent' => $visible_agent,
            'currency' => $currency,
            'active' => $active,
        ]);
        // Save media
        $mediaJson = (string)($_POST['media_json'] ?? '');
        $coverIndex = (int)($_POST['cover_index'] ?? 0);
        if ($mediaJson !== '') {
            $this->saveYachtImages($id, $mediaJson, $coverIndex);
        }
        $_SESSION['flash'] = 'Yacht updated.';
        $this->redirect('/admin/yachts');
    }

    public function enable(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            $s = $this->pdo->prepare("UPDATE yachts SET active=1 WHERE id=:id");
            $s->execute(['id' => $id]);
        }
        $this->redirect('/admin/yachts');
    }

    public function disable(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            $s = $this->pdo->prepare("UPDATE yachts SET active=0 WHERE id=:id");
            $s->execute(['id' => $id]);
        }
        $this->redirect('/admin/yachts');
    }

    public function delete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            try {
                $d = $this->pdo->prepare("DELETE FROM yachts WHERE id = :id");
                $d->execute(['id' => $id]);
            } catch (\Throwable $e) { /* ignore */ }
        }
        $this->redirect('/admin/yachts');
    }
}
