<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Security;
use App\Core\Auth;
use App\Core\WhatsApp\WhatsAppService;
use PDO;

class WhatsAppAdminController extends Controller
{
    public function settings(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $svc = new WhatsAppService($this->pdo);
        $settings = $svc->loadSettings();
        $this->view('admin/whatsapp_settings', compact('csrf','settings'));
    }

    public function save(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $fields = [
            'active_provider','aisensy_api_base','aisensy_api_key','aisensy_sender',
            'twilio_sid','twilio_token','twilio_from','default_locale',
            'enable_taxi_vendor','enable_taxi_agent','enable_taxi_internal',
            'enable_hotel_vendor','enable_hotel_agent','enable_hotel_internal',
            'enable_activity_vendor','enable_activity_agent','enable_activity_internal',
        ];
        $vals = [];
        foreach ($fields as $f) {
            if (str_starts_with($f, 'enable_')) {
                $vals[$f] = isset($_POST[$f]) ? 1 : 0;
            } else {
                $vals[$f] = trim((string)($_POST[$f] ?? ''));
            }
        }
        if (!in_array($vals['active_provider'], ['disabled','aisensy','twilio'], true)) {
            $vals['active_provider'] = 'disabled';
        }
        $sql = "UPDATE whatsapp_settings SET 
            active_provider=:active_provider,
            aisensy_api_base=:aisensy_api_base,
            aisensy_api_key=:aisensy_api_key,
            aisensy_sender=:aisensy_sender,
            twilio_sid=:twilio_sid,
            twilio_token=:twilio_token,
            twilio_from=:twilio_from,
            default_locale=:default_locale,
            enable_taxi_vendor=:enable_taxi_vendor,
            enable_taxi_agent=:enable_taxi_agent,
            enable_taxi_internal=:enable_taxi_internal,
            enable_hotel_vendor=:enable_hotel_vendor,
            enable_hotel_agent=:enable_hotel_agent,
            enable_hotel_internal=:enable_hotel_internal,
            enable_activity_vendor=:enable_activity_vendor,
            enable_activity_agent=:enable_activity_agent,
            enable_activity_internal=:enable_activity_internal
            WHERE id=1";
        $st = $this->pdo->prepare($sql);
        $st->execute($vals);
        $_SESSION['flash'] = 'WhatsApp settings saved.';
        $this->redirect('/admin/settings/whatsapp');
    }

    public function test(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $to = trim((string)($_POST['to'] ?? ''));
        $from = trim((string)($_POST['from'] ?? ''));
        $text = trim((string)($_POST['text'] ?? 'Test message from B2B'));
        if ($to === '') { $_SESSION['errors'] = ['Enter a recipient phone']; $this->redirect('/admin/settings/whatsapp'); return; }
        $svc = new WhatsAppService($this->pdo);
        // Collect optional AiSensy fields from the modal
        $meta = [
            'campaignName' => trim((string)($_POST['campaignName'] ?? '')),
            'templateParams' => trim((string)($_POST['templateParams'] ?? '')),
            'userName' => trim((string)($_POST['userName'] ?? '')),
            'source' => trim((string)($_POST['source'] ?? '')),
        ];
        // Drop empties
        foreach ($meta as $k=>$v) { if ($v === '') unset($meta[$k]); }
        $res = $svc->sendImmediate($to, $text, $from !== '' ? $from : null, $meta);
        if (!empty($res['ok'])) $_SESSION['flash'] = 'Test sent.'; else $_SESSION['errors'] = ['Test failed: ' . (string)($res['message'] ?? 'unknown')];
        $this->redirect('/admin/settings/whatsapp');
    }

    public function logs(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $st = $this->pdo->query("SELECT * FROM whatsapp_logs ORDER BY id DESC LIMIT 200");
        $logs = $st?->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $this->view('admin/whatsapp_logs', compact('csrf','logs'));
    }

    public function resend(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $queueId = (int)($_POST['queue_id'] ?? 0);
        if ($queueId <= 0) { $_SESSION['errors']=['Invalid queue id']; $this->redirect('/admin/logs/whatsapp'); return; }
        $st = $this->pdo->prepare('SELECT * FROM whatsapp_queue WHERE id=:id');
        $st->execute([':id'=>$queueId]);
        $row = $st->fetch(PDO::FETCH_ASSOC);
        if (!$row) { $_SESSION['errors']=['Queue item not found']; $this->redirect('/admin/logs/whatsapp'); return; }
        $svc = new WhatsAppService($this->pdo);
        $res = $svc->sendQueueItem($row);
        $_SESSION[!empty($res['ok']) ? 'flash' : 'errors'] = !empty($res['ok']) ? 'Resent.' : ['Resend failed: ' . (string)($res['message'] ?? 'unknown')];
        $this->redirect('/admin/logs/whatsapp');
    }
}
