<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Security;

class VendorAuthController extends Controller
{
    // GET /vendor/login
    public function login(): void
    {
        // If already logged in as vendor
        if (!empty($_SESSION['user']) && ($_SESSION['user']['role'] ?? '') === 'Vendor') { 
            $this->redirect('/vendor/dashboard'); 
            return; 
        }
        
        $csrf = Security::csrfToken();
        $this->view('vendor/login', ['title' => 'Vendor Login', 'csrf' => $csrf]);
    }

    // POST /vendor/login
    public function loginPost(): void
    {
        Security::requireCsrf();
        $email = trim((string)($_POST['email'] ?? ''));
        $password = (string)($_POST['password'] ?? '');
        
        if ($email === '' || $password === '') {
            $_SESSION['flash'] = 'Email and Password are required.';
            $this->redirect('/vendor/login');
            return;
        }
        
        error_log("Vendor login attempt for: " . $email);
        
        try {
            // Look up user by email and role
            $st = $this->pdo->prepare("SELECT * FROM users WHERE email = ? AND role = 'Vendor' LIMIT 1");
            $st->execute([$email]);
            $user = $st->fetch(\PDO::FETCH_ASSOC);
            
            if (!$user) {
                error_log("No vendor user found with email: " . $email);
                $_SESSION['flash'] = 'Invalid email or password.';
                $this->redirect('/vendor/login');
                return;
            }
            
            // Check status
            if (($user['status'] ?? '') !== 'Active') {
                error_log("Vendor account not active: " . $email);
                $_SESSION['flash'] = 'Your account is not active. Please contact support.';
                $this->redirect('/vendor/login');
                return;
            }
            
            // Verify password
            if (!password_verify($password, $user['password'])) {
                error_log("Invalid password for vendor: " . $email);
                $_SESSION['flash'] = 'Invalid email or password.';
                $this->redirect('/vendor/login');
                return;
            }
            
            // Get vendor details
            $st = $this->pdo->prepare('SELECT id, name FROM vendors WHERE contact_email = ? LIMIT 1');
            $st->execute([$email]);
            $vendor = $st->fetch(\PDO::FETCH_ASSOC);
            
            if (!$vendor) {
                error_log("No vendor record found for email: " . $email);
                $_SESSION['flash'] = 'Vendor record not found. Please contact support.';
                $this->redirect('/vendor/login');
                return;
            }
            
            // Update user's vendor_id if not set
            if (empty($user['vendor_id']) && !empty($vendor['id'])) {
                $this->pdo->prepare('UPDATE users SET vendor_id = ? WHERE id = ?')
                    ->execute([$vendor['id'], $user['id']]);
                $user['vendor_id'] = $vendor['id'];
            }
            
            // Set session data
            $_SESSION['user'] = [
                'id' => (int)$user['id'],
                'name' => $user['name'],
                'email' => $user['email'],
                'role' => 'Vendor',
                'vendor_id' => (int)($user['vendor_id'] ?? $vendor['id']),
                'vendor_name' => $vendor['name']
            ];
            // Also store a dedicated vendor session block for portal controllers
            $_SESSION['vendor'] = [
                'id' => (int)($user['vendor_id'] ?? $vendor['id']),
                'name' => (string)($vendor['name'] ?? ''),
                'email' => (string)$user['email'],
            ];
            
            // Update last login
            $this->pdo->prepare('UPDATE users SET last_login = NOW() WHERE id = ?')
                ->execute([$user['id']]);
            
            // Clear any old flash messages
            unset($_SESSION['flash']);
            
            // Redirect to vendor dashboard
            $this->redirect('/vendor/dashboard');
            
        } catch (\Throwable $e) {
            error_log("Error during vendor login: " . $e->getMessage());
            $_SESSION['flash'] = 'An error occurred. Please try again.';
            $this->redirect('/vendor/login');
        }
        if (!$u) {
            error_log("No user found with email: " . $email);
            $_SESSION['flash'] = 'Invalid email or password.';
            $this->redirect('/vendor/login');
            return;
        }
        
        if ((string)($u['role'] ?? '') !== 'Vendor') {
            error_log("User is not a vendor: " . $email);
            $_SESSION['flash'] = 'Access denied. Vendor account required.';
            $this->redirect('/vendor/login');
            return;
        }
        
        if ((string)($u['status'] ?? '') !== 'Active') {
            error_log("Vendor account not active: " . $email);
            $_SESSION['flash'] = 'Your account is not active. Please contact support.';
            $this->redirect('/vendor/login');
            return;
        }
        // Verify password
        $hash = (string)($u['password_hash'] ?? '');
        $passwordMatch = false;
        
        if ($hash !== '' && password_verify($password, $hash)) {
            $passwordMatch = true;
            error_log("Password verified using password_hash");
        } else {
            // Fallback to plain text password (for backward compatibility)
            $plain = (string)($u['password'] ?? '');
            if ($plain !== '' && hash_equals($plain, $password)) {
                $passwordMatch = true;
                error_log("Password verified using plain text (legacy)");
                
                // Upgrade to hashed password
                try {
                    $newHash = password_hash($password, PASSWORD_DEFAULT);
                    $st = $this->pdo->prepare("UPDATE users SET password_hash = ?, password = '' WHERE id = ?");
                    $st->execute([$newHash, $u['id']]);
                    error_log("Upgraded password to hashed version for user: " . $u['id']);
                } catch (\Exception $e) {
                    error_log("Failed to upgrade password hash: " . $e->getMessage());
                }
            }
        }
        
        if (!$passwordMatch) {
            error_log("Invalid password for user: " . $email);
            $_SESSION['flash'] = 'Invalid email or password.';
            $this->redirect('/vendor/login');
            return;
        }
        // Map vendor_id by matching vendors.contact_email
        $vendorId = 0;
        try {
            $sv = $this->pdo->prepare('SELECT id FROM vendors WHERE contact_email = :e LIMIT 1');
            $sv->execute([':e' => (string)$u['email']]);
            $vendorId = (int)($sv->fetchColumn() ?: 0);
        } catch (\Throwable $e) { $vendorId = 0; }
        // set session
        $_SESSION['vendor'] = [ 'id' => (int)$u['id'], 'vendor_id' => $vendorId, 'name' => (string)$u['name'], 'email' => (string)$u['email'] ];
        $this->redirect('/vendor/jobs');
    }

    // GET /vendor/logout
    public function logout(): void
    {
        unset($_SESSION['vendor']);
        $this->redirect('/vendor/login');
    }
}
