<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;
use App\Core\Mail\Template;
use App\Core\Mail\Mailer;

class ToolsController extends Controller
{
    public function countryCurrency(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $unlocked = isset($_SESSION['tools_cc_unlocked']) && $_SESSION['tools_cc_unlocked'] === true;

        if (!$unlocked) {
            // Render view in locked mode (only unlock form)
            $this->view('admin/tools_country_currency', compact('csrf', 'unlocked'));
            return;
        }

        // Ensure mapping table exists
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS country_currencies (country VARCHAR(100) PRIMARY KEY, currency_code CHAR(3) NULL)");
        // Countries with optional mapped currency
        $countries = $this->pdo->query(
            "SELECT DISTINCT l.country, cc.currency_code
             FROM locations l
             LEFT JOIN country_currencies cc ON cc.country = l.country
             WHERE l.country IS NOT NULL AND l.country<>''
             ORDER BY l.country"
        )->fetchAll();
        $currencies = $this->pdo->query("SELECT code, rate_to_base, updated_at FROM currencies ORDER BY code")->fetchAll();
        $locations = $this->pdo->query("SELECT country, city FROM locations WHERE city IS NOT NULL AND city<>'' ORDER BY country, city")->fetchAll();
        $this->view('admin/tools_country_currency', compact('countries','currencies','locations','csrf','unlocked'));
    }

    public function unlockCountryCurrency(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        // Validate update password (master password)
        Security::requireMasterPassword();
        $_SESSION['tools_cc_unlocked'] = true;
        $this->redirect('/admin/tools/country-currency');
    }

    public function storeCountry(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();

        $country = trim((string)($_POST['country'] ?? ''));
        $currencyCode = strtoupper(trim((string)($_POST['currency_code'] ?? '')));
        $errors = [];
        if ($country === '') {
            $errors[] = 'Country name is required.';
        }
        if ($currencyCode === '' || strlen($currencyCode) !== 3) {
            $errors[] = 'Currency code (3 letters) is required with country.';
        }
        if ($errors) {
            $_SESSION['errors'] = $errors;
            $_SESSION['old'] = ['country' => $country, 'currency_code' => $currencyCode];
            $this->redirect('/admin/tools/country-currency');
        }

        // If not exists, insert a stub row in locations with country only
        $exists = $this->pdo->prepare('SELECT 1 FROM locations WHERE country = ? LIMIT 1');
        $exists->execute([$country]);
        if (!$exists->fetch()) {
            $ins = $this->pdo->prepare('INSERT INTO locations (country) VALUES (?)');
            $ok = $ins->execute([$country]);
            $_SESSION['flash'] = $ok ? 'Country added.' : 'Failed to add country.';
        } else {
            $_SESSION['flash'] = 'Country already exists.';
        }

        // Ensure currency exists with default rate 1.0 (upsert behavior)
        $upd = $this->pdo->prepare('UPDATE currencies SET rate_to_base = COALESCE(rate_to_base, 1), updated_at = NOW() WHERE code = ?');
        $upd->execute([$currencyCode]);
        if ($upd->rowCount() === 0) {
            $insCur = $this->pdo->prepare('INSERT INTO currencies (code, rate_to_base, updated_at) VALUES (?, 1, NOW())');
            $insCur->execute([$currencyCode]);
        }
        // Upsert country->currency mapping
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS country_currencies (country VARCHAR(100) PRIMARY KEY, currency_code CHAR(3) NULL)");
        $mapUpd = $this->pdo->prepare('UPDATE country_currencies SET currency_code = ? WHERE country = ?');
        $mapUpd->execute([$currencyCode, $country]);
        if ($mapUpd->rowCount() === 0) {
            $mapIns = $this->pdo->prepare('INSERT INTO country_currencies (country, currency_code) VALUES (?, ?)');
            $mapIns->execute([$country, $currencyCode]);
        }

        unset($_SESSION['old']);
        $this->redirect('/admin/tools/country-currency');
    }

    public function updateCountry(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();

        $original = trim((string)($_POST['original_country'] ?? ''));
        $new = trim((string)($_POST['country'] ?? ''));
        $currencyCode = strtoupper(trim((string)($_POST['currency_code'] ?? '')));
        $errors = [];
        if ($original === '') {
            $errors[] = 'Original country is required.';
        }
        if ($new === '') {
            $errors[] = 'New country name is required.';
        }
        if ($currencyCode !== '' && strlen($currencyCode) !== 3) {
            $errors[] = 'Currency code must be 3 letters.';
        }
        if ($errors) {
            $_SESSION['errors'] = $errors;
            $_SESSION['old'] = ['country' => $new, 'currency_code' => $currencyCode];
            $this->redirect('/admin/tools/country-currency');
        }

        // Rename country across locations
        $stmt = $this->pdo->prepare('UPDATE locations SET country = ? WHERE country = ?');
        $stmt->execute([$new, $original]);
        // If currency provided, ensure it exists
        if ($currencyCode !== '') {
            $upd = $this->pdo->prepare('UPDATE currencies SET rate_to_base = COALESCE(rate_to_base, 1), updated_at = NOW() WHERE code = ?');
            $upd->execute([$currencyCode]);
            if ($upd->rowCount() === 0) {
                $insCur = $this->pdo->prepare('INSERT INTO currencies (code, rate_to_base, updated_at) VALUES (?, 1, NOW())');
                $insCur->execute([$currencyCode]);
            }
            // update mapping
            $this->pdo->exec("CREATE TABLE IF NOT EXISTS country_currencies (country VARCHAR(100) PRIMARY KEY, currency_code CHAR(3) NULL)");
            // move mapping key if country name changed
            $move = $this->pdo->prepare('UPDATE country_currencies SET country = ? WHERE country = ?');
            $move->execute([$new, $original]);
            $mm = $this->pdo->prepare('UPDATE country_currencies SET currency_code = ? WHERE country = ?');
            $mm->execute([$currencyCode, $new]);
            if ($mm->rowCount() === 0 && $move->rowCount() === 0) {
                $insMap = $this->pdo->prepare('INSERT INTO country_currencies (country, currency_code) VALUES (?, ?)');
                $insMap->execute([$new, $currencyCode]);
            }
        }
        $_SESSION['flash'] = ($stmt->rowCount() > 0)
            ? 'Country updated successfully.'
            : 'No changes made.';
        unset($_SESSION['old']);
        $this->redirect('/admin/tools/country-currency');
    }

    public function deleteCountry(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();

        $country = trim((string)($_POST['country'] ?? ''));
        if ($country === '') {
            $_SESSION['errors'] = ['Country is required for delete.'];
            $this->redirect('/admin/tools/country-currency');
        }
        // Delete country rows (including city rows) and mapping
        $del = $this->pdo->prepare('DELETE FROM locations WHERE country = ?');
        $del->execute([$country]);
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS country_currencies (country VARCHAR(100) PRIMARY KEY, currency_code CHAR(3) NULL)");
        $dm = $this->pdo->prepare('DELETE FROM country_currencies WHERE country = ?');
        $dm->execute([$country]);
        $_SESSION['flash'] = 'Country deleted.';
        $this->redirect('/admin/tools/country-currency');
    }

    public function deleteLocation(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();

        $country = trim((string)($_POST['country'] ?? ''));
        $city = trim((string)($_POST['city'] ?? ''));
        if ($country === '' || $city === '') {
            $_SESSION['errors'] = ['Country and City are required for delete.'];
            $this->redirect('/admin/tools/country-currency');
        }
        $stmt = $this->pdo->prepare('DELETE FROM locations WHERE country = ? AND city = ?');
        $ok = $stmt->execute([$country, $city]);
        $_SESSION['flash'] = $ok && $stmt->rowCount() > 0 ? 'Location deleted.' : 'No matching location found.';
        $this->redirect('/admin/tools/country-currency');
    }

    public function storeCurrency(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();

        $code = strtoupper(trim((string)($_POST['code'] ?? '')));
        $rate = (string)($_POST['rate_to_base'] ?? '');
        $errors = [];
        if ($code === '' || strlen($code) !== 3) {
            $errors[] = 'Currency code must be a 3-letter code (e.g., USD).';
        }
        // Rate optional: if blank, default to 1.0; if provided, must be positive number
        if ($rate === '') {
            $rate = '1';
        } elseif (!is_numeric($rate) || (float)$rate <= 0) {
            $errors[] = 'Rate to base must be a positive number.';
        }
        if ($errors) {
            $_SESSION["errors"] = $errors;
            $_SESSION['old'] = ['code' => $code, 'rate_to_base' => $rate];
            $this->redirect('/admin/tools/country-currency');
        }

        // Upsert-like behavior: try update, if 0 rows, insert
        $upd = $this->pdo->prepare('UPDATE currencies SET rate_to_base = ?, updated_at = NOW() WHERE code = ?');
        $upd->execute([(float)$rate, $code]);
        if ($upd->rowCount() === 0) {
            $ins = $this->pdo->prepare('INSERT INTO currencies (code, rate_to_base, updated_at) VALUES (?, ?, NOW())');
            $ok = $ins->execute([$code, (float)$rate]);
            $_SESSION['flash'] = $ok ? 'Currency added.' : 'Failed to add currency.';
        } else {
            $_SESSION['flash'] = 'Currency updated.';
        }
        unset($_SESSION['old']);
        $this->redirect('/admin/tools/country-currency');
    }

    public function storeLocation(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();

        $country = trim((string)($_POST['country'] ?? ''));
        $city = trim((string)($_POST['city'] ?? ''));
        if ($country === '' || $city === '') {
            $_SESSION['errors'] = ['Country and City are required.'];
            $_SESSION['old'] = ['country' => $country, 'city' => $city];
            $this->redirect('/admin/tools/country-currency');
        }

        // avoid duplicates
        $exists = $this->pdo->prepare('SELECT 1 FROM locations WHERE country = ? AND city = ? LIMIT 1');
        $exists->execute([$country, $city]);
        if ($exists->fetch()) {
            $_SESSION['flash'] = 'Location already exists.';
            $this->redirect('/admin/tools/country-currency');
        }
        $ins = $this->pdo->prepare('INSERT INTO locations (country, city) VALUES (?, ?)');
        $ok = $ins->execute([$country, $city]);
        $_SESSION['flash'] = $ok ? 'Location added.' : 'Failed to add location.';
        unset($_SESSION['old']);
        $this->redirect('/admin/tools/country-currency');
    }

    /**
     * Admin page to preview email templates in an iframe with tabs.
     */
    public function emailTemplates(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $unlocked = isset($_SESSION['tools_email_unlocked']) && $_SESSION['tools_email_unlocked'] === true;
        $this->view('admin/tools_email_templates', compact('csrf','unlocked'));
    }

    /**
     * Unlock the Email Templates tool using master password
     */
    public function emailTemplatesUnlock(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $_SESSION['tools_email_unlocked'] = true;
        $this->redirect('/admin/tools/email-templates');
    }

    /**
     * Raw HTML preview for a given template type. Use ?type=hotel|activity|taxi|visa|login
     */
    public function emailTemplatesPreview(): void
    {
        Auth::requireRole(['Admin']);
        $type = strtolower(trim((string)($_GET['type'] ?? 'hotel')));
        $brandName = trim((string)($_GET['brand_name'] ?? ''));
        $logoUrl = trim((string)($_GET['logo_url'] ?? ''));
        $brandColor = trim((string)($_GET['brand_color'] ?? ''));
        $mode = strtolower(trim((string)($_GET['mode'] ?? 'b2c')));
        if (!in_array($mode, ['b2c','b2b','vendor'], true)) { $mode = 'b2c'; }
        $hideBranding = (string)($_GET['hide_branding'] ?? '0') === '1';
        $signatureHtmlOverride = (string)($_GET['signature_html'] ?? '');
        $footerTextOverride = (string)($_GET['footer_text'] ?? '');

        if ($type === 'login') {
            $base = @file_get_contents(BASE_PATH . '/app/Views/emails/system/base.html') ?: '<html><body>{{signature_html}}</body></html>';
            $block = @file_get_contents(BASE_PATH . '/app/Views/emails/system/login_otp.html') ?: '<h3>{{email_title}}</h3><p>OTP: {{otp}}</p>';
            $data = [
                'brand_name' => 'B2B Travel',
                'email_title' => 'Login OTP',
                'signature_html' => 'Regards,<br>Support Team',
                'footer_text' => 'This is an automated message.',
                'otp' => '123456',
                'otp_ttl_minutes' => '10',
            ];
            // Apply branding overrides before rendering
            if ($brandName !== '') $data['brand_name'] = $brandName;
            if ($logoUrl !== '') $data['logo_url'] = $logoUrl;
            if ($hideBranding) { $data['brand_name'] = ''; $data['logo_url'] = ''; }
            if ($signatureHtmlOverride !== '') $data['signature_html'] = $signatureHtmlOverride;
            if ($footerTextOverride !== '') $data['footer_text'] = $footerTextOverride;

            $html = Template::render($base, $block, $data);
            if ($brandColor !== '') {
                // Update multiple CSS vars to reflect brand color
                foreach (['--brand-color', '--btn', '--badge'] as $var) {
                    $html = preg_replace('/(' . preg_quote($var, '/') . '\s*:\s*)(#[0-9a-fA-F]{3,6}|rgb\([^\)]+\)|[a-zA-Z]+)/', '$1' . $brandColor, $html, 1) ?? $html;
                }
            }
            if ($hideBranding) {
                // Remove logo and brand containers and fix <title>
                $html = preg_replace('/<img[^>]*class=("|\')?[^>]*\blogo\b[^>]*>/i', '', $html) ?? $html;
                $html = preg_replace('/<div[^>]*class=("|\')?[^>]*\bbrand\b[^>]*>.*?<\/div>/is', '', $html) ?? $html;
                $html = preg_replace('/<title>\s*\|\s*/i', '<title>', $html) ?? $html;
                // Remove empty header if it only contains whitespace after removals
                $html = preg_replace('/<div[^>]*class=("|\')?[^>]*\bhead\b[^>]*>\s*<\/div>/is', '', $html) ?? $html;
            }
        } else {
            // vendor modules
            $base = @file_get_contents(BASE_PATH . '/app/Views/emails/vendor/base.html') ?: '<html><body><!-- MODULE_BLOCK --></body></html>';
            $map = [
                'hotel' => 'hotel_block.html',
                'activity' => 'activity_block.html',
                'taxi' => 'taxi_block.html',
                'visa' => 'visa_block.html',
            ];
            $type = array_key_exists($type, $map) ? $type : 'hotel';
            $block = @file_get_contents(BASE_PATH . '/app/Views/emails/vendor/' . $map[$type]) ?: '<p>Missing block</p>';

            $common = [
                'brand_name' => 'B2B Travel',
                'email_title' => ucfirst($type) . ' Booking – Preview',
                'logo_url' => 'https://dummyimage.com/140x40/0d6efd/ffffff&text=B2B',
                'booking_ref' => 'BK-45821',
                'status' => 'Pending Confirmation',
                'created_at' => date('d M Y H:i'),
                'signature_html' => 'Regards,<br>Operations Team',
                'footer_text' => 'This is a system-generated booking preview.',
                'action_url' => 'https://admin.example.com/bookings/BK-45821',
            ];

            // Inject branding overrides into common data BEFORE composing specific $data
            if ($brandName !== '') $common['brand_name'] = $brandName;
            if ($logoUrl !== '') $common['logo_url'] = $logoUrl;
            if ($hideBranding) { $common['brand_name'] = ''; $common['logo_url'] = ''; }
            if ($signatureHtmlOverride !== '') $common['signature_html'] = $signatureHtmlOverride;
            if ($footerTextOverride !== '') $common['footer_text'] = $footerTextOverride;

            // Mode-specific tweaks for common values
            if ($mode === 'vendor') {
                $common['status'] = 'Confirmed';
                $common['footer_text'] = 'Final supplier confirmation. Please proceed as per the details above.';
            } elseif ($mode === 'b2b') {
                $common['footer_text'] = 'For support contact your account manager or reply to this email.';
            }

            switch ($type) {
                case 'activity':
                    $data = $common + [
                        'activity_name' => 'Safari World + Marine Park',
                        'activity_dt' => '22 Aug 2025, 10:00',
                        'duration' => '6h',
                        'location' => 'Bangkok',
                        'participants' => '2 Adults',
                        'notes' => 'Hotel pickup required',
                        'currency' => 'THB',
                        'net_amount' => '2,400',
                        'payment_terms' => 'Payable within 7 days',
                    ];
                    break;
                case 'taxi':
                    $data = $common + [
                        'pickup' => 'BKK Airport',
                        'drop' => 'Sukhumvit 11',
                        'service_dt' => '20 Aug 2025, 19:30',
                        'vehicle_type' => 'SUV',
                        'flight_no' => 'TG123',
                        'pax' => '2',
                        'luggage' => '2',
                        'notes' => 'Meet at Gate 3',
                        'currency' => 'THB',
                        'net_amount' => '900',
                        'payment_terms' => 'Payable within 7 days',
                    ];
                    break;
                case 'visa':
                    $data = $common + [
                        'applicants' => 'John Doe, Jane Doe',
                        'nationality' => 'India',
                        'visa_type' => 'Tourist',
                        'visa_entry' => 'Single',
                        'processing_time' => '3-5 working days',
                        'docs_provided' => 'Passport, Photo',
                        'docs_required' => 'Flight, Hotel, Bank Statement',
                    ];
                    break;
                case 'hotel':
                default:
                    $data = $common + [
                        'hotel_name' => 'Amari Bangkok',
                        'hotel_city' => 'Bangkok',
                        'hotel_address' => '123 Sukhumvit Rd',
                        'checkin' => '20 Aug 2025',
                        'checkout' => '22 Aug 2025',
                        'nights' => '2',
                        'room_type' => 'Deluxe King',
                        'meal_plan' => 'Breakfast',
                        'guest_summary' => '2 Adults',
                        'special_requests' => 'High floor',
                        'currency' => 'THB',
                        'net_amount' => '6,500',
                        'payment_terms' => 'Payable within 7 days',
                    ];
            }

            $html = Template::render($base, $block, $data);
            if ($brandColor !== '') {
                foreach (['--brand-color', '--btn', '--badge'] as $var) {
                    $html = preg_replace('/(' . preg_quote($var, '/') . '\s*:\s*)(#[0-9a-fA-F]{3,6}|rgb\([^\)]+\)|[a-zA-Z]+)/', '$1' . $brandColor, $html, 1) ?? $html;
                }
            }
            // In B2B agent mode, hide the public booking button from blocks
            if ($mode === 'b2b') {
                $html = preg_replace('/<a[^>]*class="[^"]*btn[^"]*"[^>]*>\s*View\s+Booking\s*<\/a>/i', '', $html) ?? $html;
            }
            if ($hideBranding) {
                $html = preg_replace('/<img[^>]*class=("|\')?[^>]*\blogo\b[^>]*>/i', '', $html) ?? $html;
                $html = preg_replace('/<div[^>]*class=("|\')?[^>]*\bbrand\b[^>]*>.*?<\/div>/is', '', $html) ?? $html;
                $html = preg_replace('/<title>\s*\|\s*/i', '<title>', $html) ?? $html;
                $html = preg_replace('/<div[^>]*class=("|\')?[^>]*\bhead\b[^>]*>\s*<\/div>/is', '', $html) ?? $html;
            }
        }

        header('Content-Type: text/html; charset=utf-8');
        echo $html;
    }

    /**
     * Send test email for a given template to a target address.
     * POST: type, to, subject, [profile]
     */
    public function emailTemplatesTestSend(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $type = strtolower(trim((string)($_POST['type'] ?? 'hotel')));
        $to = trim((string)($_POST['to'] ?? ''));
        $subject = trim((string)($_POST['subject'] ?? 'Template Test Email'));
        $profile = trim((string)($_POST['profile'] ?? '')) ?: null;
        $mode = strtolower(trim((string)($_POST['mode'] ?? 'b2c')));
        if (!in_array($mode, ['b2c','b2b','vendor'], true)) { $mode = 'b2c'; }
        $brandName = trim((string)($_POST['brand_name'] ?? ''));
        $logoUrl = trim((string)($_POST['logo_url'] ?? ''));
        $brandColor = trim((string)($_POST['brand_color'] ?? ''));
        $hideBranding = (string)($_POST['hide_branding'] ?? '0') === '1';
        $signatureHtmlOverride = (string)($_POST['signature_html'] ?? '');
        $footerTextOverride = (string)($_POST['footer_text'] ?? '');

        if ($to === '' || !filter_var($to, FILTER_VALIDATE_EMAIL)) {
            $_SESSION['errors'] = ['Please provide a valid test email address.'];
            $this->redirect('/admin/tools/email-templates');
        }

        // Render same as preview
        if ($type === 'login') {
            $base = @file_get_contents(BASE_PATH . '/app/Views/emails/system/base.html') ?: '<html><body>{{signature_html}}</body></html>';
            $block = @file_get_contents(BASE_PATH . '/app/Views/emails/system/login_otp.html') ?: '<h3>{{email_title}}</h3><p>OTP: {{otp}}</p>';
            $data = [
                'brand_name' => 'B2B Travel',
                'email_title' => 'Login OTP',
                'content_html' => '',
                'signature_html' => 'Regards,<br>B2B Travel Team',
                'otp' => '123456',
                'otp_expires' => '10 minutes',
            ];
            if ($brandName !== '') $data['brand_name'] = $brandName;
            if ($logoUrl !== '') $data['logo_url'] = $logoUrl;
            if ($hideBranding) { $data['brand_name'] = ''; $data['logo_url'] = ''; }
            if ($signatureHtmlOverride !== '') $data['signature_html'] = $signatureHtmlOverride;
            if ($footerTextOverride !== '') $data['footer_text'] = $footerTextOverride;
        } else {
            $base = @file_get_contents(BASE_PATH . '/app/Views/emails/vendor/base.html') ?: '<html><body>{{module_block}}</body></html>';
            $map = [
                'hotel' => 'hotel_block.html',
                'activity' => 'activity_block.html',
                'taxi' => 'taxi_block.html',
                'visa' => 'visa_block.html',
            ];
            $file = $map[$type] ?? 'hotel_block.html';
            $block = @file_get_contents(BASE_PATH . '/app/Views/emails/vendor/' . $file) ?: '<p>Booking: {{booking_ref}}</p>';
            $common = [
                'brand_name' => 'B2B Travel',
                'email_title' => ucfirst($type) . ' Booking – Preview',
                'logo_url' => 'https://dummyimage.com/140x40/0d6efd/ffffff&text=B2B',
                'booking_ref' => 'BK-45821',
                'status' => 'Pending Confirmation',
                'created_at' => date('d M Y H:i'),
                'signature_html' => 'Regards,<br>Operations Team',
                'footer_text' => 'This is a system-generated booking preview.',
                'action_url' => APP_URL . '/admin',
            ];
            if ($brandName !== '') $common['brand_name'] = $brandName;
            if ($logoUrl !== '') $common['logo_url'] = $logoUrl;
            if ($signatureHtmlOverride !== '') $common['signature_html'] = $signatureHtmlOverride;
            if ($footerTextOverride !== '') $common['footer_text'] = $footerTextOverride;
            if ($mode === 'vendor') {
                $common['status'] = 'Confirmed';
                $common['footer_text'] = 'Final supplier confirmation. Please proceed as per the details above.';
            } elseif ($mode === 'b2b') {
                $common['footer_text'] = 'For support contact your account manager or reply to this email.';
            }
            if ($hideBranding) { $common['brand_name'] = ''; $common['logo_url'] = ''; }
            switch ($type) {
                case 'activity':
                    $data = $common + [
                        'booking_ref' => 'AC-90231',
                        'status' => 'Pending Confirmation',
                        'generated_at' => date('d M Y H:i'),
                        'activity_name' => 'Safari World Ticket',
                        'date' => '20 Aug 2025',
                        'participants' => '2 Adults, 1 Child',
                        'notes' => 'Pickup at 9:00 AM',
                        'currency' => 'THB',
                        'net_amount' => '2,400',
                        'payment_terms' => 'Payable within 7 days',
                    ];
                    break;
                case 'taxi':
                    $data = $common + [
                        'booking_ref' => 'TX-33211',
                        'status' => 'Pending Confirmation',
                        'generated_at' => date('d M Y H:i'),
                        'pickup' => 'Suvarnabhumi Airport (BKK)',
                        'dropoff' => 'Amari Bangkok, Sukhumvit',
                        'flight_info' => 'TG123 Arr 13:10',
                        'pax' => '2 Adults',
                        'luggage' => '2 Bags',
                        'notes' => 'Meet at Gate 3',
                        'currency' => 'THB',
                        'net_amount' => '800',
                        'payment_terms' => 'Payable within 7 days',
                    ];
                    break;
                case 'visa':
                    $data = $common + [
                        'booking_ref' => 'VI-77881',
                        'status' => 'Pending',
                        'generated_at' => date('d M Y H:i'),
                        'applicant_name' => 'John Doe',
                        'visa_type' => 'Tourist',
                        'processing_time' => '3-5 Business Days',
                        'documents_provided' => 'Passport, Photo',
                        'documents_required' => 'Bank Statement',
                        'currency' => 'THB',
                        'net_amount' => '3,000',
                    ];
                    break;
                default:
                    $data = $common + [
                        'booking_ref' => 'BK-45821',
                        'status' => 'Pending Confirmation',
                        'generated_at' => date('d M Y H:i'),
                        'hotel_name' => 'Amari Bangkok',
                        'hotel_address' => '123 Sukhumvit Rd, Bangkok',
                        'checkin' => '20 Aug 2025',
                        'checkout' => '22 Aug 2025',
                        'nights' => '2',
                        'room_type' => 'Deluxe King',
                        'meal_plan' => 'Breakfast',
                        'guest_summary' => '2 Adults',
                        'special_requests' => 'High floor',
                        'currency' => 'THB',
                        'net_amount' => '6,500',
                        'payment_terms' => 'Payable within 7 days',
                    ];
            }
        }

        $html = Template::render($base, $block, $data);
        if ($brandColor !== '') {
            foreach (['--brand-color', '--btn', '--badge'] as $var) {
                $html = preg_replace('/(' . preg_quote($var, '/') . '\s*:\s*)(#[0-9a-fA-F]{3,6}|rgb\([^\)]+\)|[a-zA-Z]+)/', '$1' . $brandColor, $html, 1) ?? $html;
            }
        }
        if ($mode === 'b2b') {
            $html = preg_replace('/<a[^>]*class="[^"]*btn[^"]*"[^>]*>\s*View\s+Booking\s*<\/a>/i', '', $html) ?? $html;
        }
        if ($hideBranding) {
            $html = preg_replace('/<img[^>]*class=("|\')?[^>]*\blogo\b[^>]*>/i', '', $html) ?? $html;
            $html = preg_replace('/<div[^>]*class=("|\')?[^>]*\bbrand\b[^>]*>.*?<\/div>/is', '', $html) ?? $html;
            $html = preg_replace('/<title>\s*\|\s*/i', '<title>', $html) ?? $html;
            $html = preg_replace('/<div[^>]*class=("|\')?[^>]*\bhead\b[^>]*>\s*<\/div>/is', '', $html) ?? $html;
        }
        $alt = trim(strip_tags($html));

        // Pick profile if not provided explicitly
        if ($profile === null) {
            $profile = in_array($type, ['hotel','activity','taxi','visa'], true) ? $type : null;
        }
        $mailCfg = require BASE_PATH . '/config/mail.php';
        $mailer = new Mailer($mailCfg);
        $result = $mailer->sendHtml($to, $subject, $html, $alt, $profile);

        $_SESSION['flash'] = $result['ok'] ? 'Test email sent to ' . htmlspecialchars($to) : ('Send failed: ' . $result['message']);
        $this->redirect('/admin/tools/email-templates');
    }

    /**
     * Return raw file contents for the selected template as JSON: { path, content }
     * GET params: type=hotel|activity|taxi|visa|login, part=base|block
     */
    public function emailTemplatesFile(): void
    {
        Auth::requireRole(['Admin']);
        header('Content-Type: application/json');
        $type = strtolower(trim((string)($_GET['type'] ?? 'hotel')));
        $part = strtolower(trim((string)($_GET['part'] ?? 'block')));

        $path = $this->resolveTemplatePath($type, $part);
        if (!$path || !file_exists($path)) {
            http_response_code(404);
            echo json_encode(['error' => 'Template file not found.']);
            return;
        }
        $content = file_get_contents($path);
        echo json_encode(['path' => $path, 'content' => $content]);
    }

    /**
     * Save raw file contents for the selected template. POST: type, part, content.
     */
    public function emailTemplatesFileSave(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $type = strtolower(trim((string)($_POST['type'] ?? '')));
        $part = strtolower(trim((string)($_POST['part'] ?? '')));
        $content = (string)($_POST['content'] ?? '');

        $path = $this->resolveTemplatePath($type, $part);
        if (!$path) {
            $_SESSION['errors'] = ['Invalid template selection.'];
            $this->redirect('/admin/tools/email-templates');
        }
        // Small safety check: only allow edits within Views/emails/
        if (strpos(realpath($path) ?: '', realpath(BASE_PATH . '/app/Views/emails')) !== 0) {
            $_SESSION['errors'] = ['Permission denied.'];
            $this->redirect('/admin/tools/email-templates');
        }
        $ok = file_put_contents($path, $content) !== false;
        $_SESSION['flash'] = $ok ? 'Template saved.' : 'Failed to save template.';
        $this->redirect('/admin/tools/email-templates');
    }

    /**
     * Resolve file path for a given template type and part.
     */
    private function resolveTemplatePath(string $type, string $part): ?string
    {
        $type = strtolower($type);
        $part = strtolower($part);
        if ($type === 'login') {
            if ($part === 'base') return BASE_PATH . '/app/Views/emails/system/base.html';
            if ($part === 'block') return BASE_PATH . '/app/Views/emails/system/login_otp.html';
            return null;
        }
        $vendorBase = BASE_PATH . '/app/Views/emails/vendor/base.html';
        $map = [
            'hotel' => 'hotel_block.html',
            'activity' => 'activity_block.html',
            'taxi' => 'taxi_block.html',
            'visa' => 'visa_block.html',
        ];
        if (!isset($map[$type])) $type = 'hotel';
        if ($part === 'base') return $vendorBase;
        if ($part === 'block') return BASE_PATH . '/app/Views/emails/vendor/' . $map[$type];
        return null;
    }

    /**
     * Tools: Payment Gateways settings page (Wallet Balance, Stripe, PayPal)
     */
    public function paymentGateways(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $cfg = $this->loadPaymentGatewayConfig();
        $this->view('admin/tools_payment_gateways', ['csrf' => $csrf, 'cfg' => $cfg]);
    }

    /**
     * Toggle enable/disable for a gateway. POST: gateway, enabled(0|1)
     */
    public function paymentGatewayToggle(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (!Security::verifyMasterPassword()) {
            $_SESSION['errors'] = ['Invalid master password'];
            $this->redirect('/admin/tools/payment-gateways');
        }
        $gateway = strtolower(trim((string)($_POST['gateway'] ?? '')));
        $enabled = (string)($_POST['enabled'] ?? '0') === '1';
        $cfg = $this->loadPaymentGatewayConfig();
        if (!isset($cfg[$gateway])) {
            $_SESSION['errors'] = ['Invalid gateway.'];
            $this->redirect('/admin/tools/payment-gateways');
        }
        $cfg[$gateway]['enabled'] = $enabled;
        $ok = $this->savePaymentGatewayConfig($cfg);
        $_SESSION['flash'] = $ok ? ucfirst($gateway) . ' status updated.' : 'Failed to save.';
        $this->redirect('/admin/tools/payment-gateways');
    }

    /**
     * Save settings for a gateway. POST: gateway, settings[...] per gateway
     */
    public function paymentGatewaySave(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (!Security::verifyMasterPassword()) {
            $_SESSION['errors'] = ['Invalid master password'];
            $this->redirect('/admin/tools/payment-gateways');
        }
        $gateway = strtolower(trim((string)($_POST['gateway'] ?? '')));
        $cfg = $this->loadPaymentGatewayConfig();
        if (!isset($cfg[$gateway])) {
            $_SESSION['errors'] = ['Invalid gateway.'];
            $this->redirect('/admin/tools/payment-gateways');
        }
        // Map per-gateway settings
        if ($gateway === 'wallet') {
            // nothing extra, only enabled handled in toggle
        } elseif ($gateway === 'stripe') {
            $cfg['stripe']['publishable_key'] = trim((string)($_POST['publishable_key'] ?? ''));
            $cfg['stripe']['secret_key'] = trim((string)($_POST['secret_key'] ?? ''));
            $cfg['stripe']['webhook_secret'] = trim((string)($_POST['webhook_secret'] ?? ''));
            $cfg['stripe']['currency'] = strtoupper(trim((string)($_POST['currency'] ?? 'THB')));
        } elseif ($gateway === 'paypal') {
            $cfg['paypal']['client_id'] = trim((string)($_POST['client_id'] ?? ''));
            $cfg['paypal']['client_secret'] = trim((string)($_POST['client_secret'] ?? ''));
            $cfg['paypal']['mode'] = in_array($_POST['mode'] ?? 'sandbox', ['sandbox','live'], true) ? $_POST['mode'] : 'sandbox';
            $cfg['paypal']['currency'] = strtoupper(trim((string)($_POST['currency'] ?? 'USD')));
        }
        $ok = $this->savePaymentGatewayConfig($cfg);
        $_SESSION['flash'] = $ok ? ucfirst($gateway) . ' settings saved.' : 'Failed to save.';
        $this->redirect('/admin/tools/payment-gateways');
    }

    private function loadPaymentGatewayConfig(): array
    {
        $path = BASE_PATH . '/config/payment_gateways.php';
        if (!file_exists($path)) {
            return [
                'wallet' => ['enabled' => true],
                'stripe' => [
                    'enabled' => false,
                    'publishable_key' => '',
                    'secret_key' => '',
                    'webhook_secret' => '',
                    'currency' => 'THB',
                ],
                'paypal' => [
                    'enabled' => false,
                    'client_id' => '',
                    'client_secret' => '',
                    'mode' => 'sandbox',
                    'currency' => 'USD',
                ],
            ];
        }
        return require $path;
    }

    private function savePaymentGatewayConfig(array $cfg): bool
    {
        $path = BASE_PATH . '/config/payment_gateways.php';
        $export = "<?php\nreturn " . var_export($cfg, true) . ";\n";
        return (bool)file_put_contents($path, $export);
    }

    /**
     * Admin-only DB patch to ensure hotels.active exists (and stars too, defensively).
     * GET: /admin/tools/db/patch-hotels-active
     */
    public function dbPatchHotelsActive(): void
    {
        Auth::requireRole(['Admin']);
        header('Content-Type: text/html; charset=utf-8');
        $logs = [];

        // Helper to check column existence
        $hasColumn = function(string $table, string $column) {
            $stmt = $this->pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
            $stmt->execute([$column]);
            return (bool)$stmt->fetch();
        };

        // Ensure hotels table exists
        try {
            $this->pdo->query('SELECT 1 FROM hotels LIMIT 1');
        } catch (\Throwable $e) {
            echo '<h3>Hotels table missing</h3><p>Please import database/schema.sql first.</p>';
            return;
        }

        // Ensure stars column (in case old DBs are missing it)
        try {
            if (!$hasColumn('hotels', 'stars')) {
                $this->pdo->exec("ALTER TABLE hotels ADD COLUMN stars TINYINT UNSIGNED NOT NULL DEFAULT 0 AFTER base_price");
                $logs[] = 'Added hotels.stars column.';
            } else {
                $logs[] = 'hotels.stars already exists.';
            }
        } catch (\Throwable $e) {
            $logs[] = 'Failed to add stars: ' . htmlspecialchars($e->getMessage());
        }

        // Ensure active column
        try {
            if (!$hasColumn('hotels', 'active')) {
                $this->pdo->exec("ALTER TABLE hotels ADD COLUMN active TINYINT(1) NOT NULL DEFAULT 1 AFTER base_price");
                $logs[] = 'Added hotels.active column.';
            } else {
                $logs[] = 'hotels.active already exists.';
            }
        } catch (\Throwable $e) {
            $logs[] = 'Failed to add active: ' . htmlspecialchars($e->getMessage());
        }

        // Render minimal result
        echo '<!doctype html><meta charset="utf-8"><title>DB Patch – Hotels Active</title>';
        echo '<div style="font-family:system-ui,Segoe UI,Arial;padding:20px;max-width:800px">';
        echo '<h2>DB Patch – Hotels Columns</h2>';
        echo '<ul>';
        foreach ($logs as $line) {
            echo '<li>' . htmlspecialchars($line) . '</li>';
        }
        echo '</ul>';
        echo '<p><a href="/admin/hotels">Return to Admin Hotels</a></p>';
        echo '</div>';
    }

    /**
     * Admin-only DB patch to ensure per-audience flags exist on hotels table.
     * GET: /admin/tools/db/patch-hotels-audience-flags
     */
    public function dbPatchHotelsAudienceFlags(): void
    {
        Auth::requireRole(['Admin']);
        header('Content-Type: text/html; charset=utf-8');
        $logs = [];

        // Helper to check column existence
        $hasColumn = function(string $table, string $column) {
            $stmt = $this->pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
            $stmt->execute([$column]);
            return (bool)$stmt->fetch();
        };

        // Ensure hotels table exists
        try {
            $this->pdo->query('SELECT 1 FROM hotels LIMIT 1');
        } catch (\Throwable $e) {
            echo '<h3>Hotels table missing</h3><p>Please import database/schema.sql first.</p>';
            return;
        }

        // visible_customer
        try {
            if (!$hasColumn('hotels', 'visible_customer')) {
                $this->pdo->exec("ALTER TABLE hotels ADD COLUMN visible_customer TINYINT(1) NOT NULL DEFAULT 1 AFTER stars");
                $logs[] = 'Added hotels.visible_customer column.';
            } else {
                $logs[] = 'hotels.visible_customer already exists.';
            }
        } catch (\Throwable $e) {
            $logs[] = 'Failed to add visible_customer: ' . htmlspecialchars($e->getMessage());
        }

        // visible_agent
        try {
            if (!$hasColumn('hotels', 'visible_agent')) {
                $this->pdo->exec("ALTER TABLE hotels ADD COLUMN visible_agent TINYINT(1) NOT NULL DEFAULT 1 AFTER visible_customer");
                $logs[] = 'Added hotels.visible_agent column.';
            } else {
                $logs[] = 'hotels.visible_agent already exists.';
            }
        } catch (\Throwable $e) {
            $logs[] = 'Failed to add visible_agent: ' . htmlspecialchars($e->getMessage());
        }

        // featured_customer
        try {
            if (!$hasColumn('hotels', 'featured_customer')) {
                $this->pdo->exec("ALTER TABLE hotels ADD COLUMN featured_customer TINYINT(1) NOT NULL DEFAULT 0 AFTER visible_agent");
                $logs[] = 'Added hotels.featured_customer column.';
            } else {
                $logs[] = 'hotels.featured_customer already exists.';
            }
        } catch (\Throwable $e) {
            $logs[] = 'Failed to add featured_customer: ' . htmlspecialchars($e->getMessage());
        }

        // featured_agent
        try {
            if (!$hasColumn('hotels', 'featured_agent')) {
                $this->pdo->exec("ALTER TABLE hotels ADD COLUMN featured_agent TINYINT(1) NOT NULL DEFAULT 0 AFTER featured_customer");
                $logs[] = 'Added hotels.featured_agent column.';
            } else {
                $logs[] = 'hotels.featured_agent already exists.';
            }
        } catch (\Throwable $e) {
            $logs[] = 'Failed to add featured_agent: ' . htmlspecialchars($e->getMessage());
        }

        echo '<!doctype html><meta charset="utf-8"><title>DB Patch – Hotels Audience Flags</title>';
        echo '<div style="font-family:system-ui,Segoe UI,Arial;padding:20px;max-width:800px">';
        echo '<h2>DB Patch – Hotels Audience Flags</h2>';
        echo '<ul>';
        foreach ($logs as $line) {
            echo '<li>' . htmlspecialchars($line) . '</li>';
        }
        echo '</ul>';
        echo '<p><a href="/admin/hotels">Return to Admin Hotels</a></p>';
        echo '</div>';
    }
}
