<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;

class TaxiItinerariesAdminController extends Controller
{
    public function unlock(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $_SESSION['admin_taxi_itins_unlocked'] = true;
        $this->redirect('/admin/taxi/itineraries');
    }

    public function index(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $unlocked = !empty($_SESSION['admin_taxi_itins_unlocked']);
        $rows = [];
        if ($unlocked) {
            try {
                $q = "SELECT ti.id, ti.name, ti.active, t.name AS taxi
                      FROM taxi_itineraries ti
                      LEFT JOIN taxis t ON t.id = ti.taxi_id
                      ORDER BY ti.id DESC LIMIT 200";
                $rows = $this->pdo->query($q)->fetchAll() ?: [];
            } catch (\Throwable $_) { $rows = []; }
        }
        $this->view('admin/taxi_itineraries_index', compact('csrf','unlocked','rows'));
    }

    public function create(): void
    {
        Auth::requireRole(['Admin']);
        if (empty($_SESSION['admin_taxi_itins_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock itineraries with the update password first.';
            $this->redirect('/admin/taxi/itineraries');
            return;
        }
        $csrf = Security::csrfToken();
        // dropdown taxis
        $taxis = [];
        try { $taxis = $this->pdo->query('SELECT id, name FROM taxis ORDER BY id DESC LIMIT 500')->fetchAll() ?: []; } catch (\Throwable $_) {}
        $prefillTaxiId = (int)($_GET['taxi_id'] ?? 0);
        $itin = ['id'=>null,'taxi_id'=>($prefillTaxiId ?: ''),'name'=>'','description'=>'','active'=>1];
        $stops = [];
        $existing = [];
        if ($prefillTaxiId > 0) {
            try {
                $se = $this->pdo->prepare('SELECT id, name, active FROM taxi_itineraries WHERE taxi_id = :tid ORDER BY id DESC');
                $se->execute([':tid'=>$prefillTaxiId]);
                $existing = $se->fetchAll(\PDO::FETCH_ASSOC) ?: [];
            } catch (\Throwable $_) { $existing = []; }
        }
        $this->view('admin/taxi_itineraries_form', compact('csrf','taxis','itin','stops','existing'));
    }

    public function edit(): void
    {
        Auth::requireRole(['Admin']);
        if (empty($_SESSION['admin_taxi_itins_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock itineraries with the update password first.';
            $this->redirect('/admin/taxi/itineraries');
            return;
        }
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/taxi/itineraries'); return; }
        $csrf = Security::csrfToken();
        $itin = null; $stops = [];
        try {
            $s = $this->pdo->prepare('SELECT * FROM taxi_itineraries WHERE id = :id');
            $s->execute([':id'=>$id]);
            $itin = $s->fetch(\PDO::FETCH_ASSOC) ?: null;
        } catch (\Throwable $_) { $itin = null; }
        if (!$itin) { $this->redirect('/admin/taxi/itineraries'); return; }
        try {
            $ss = $this->pdo->prepare('SELECT id, seq, label, address, wait_minutes, extra_price, notes FROM taxi_itinerary_stops WHERE itinerary_id = :id ORDER BY seq ASC, id ASC');
            $ss->execute([':id'=>$id]);
            $stops = $ss->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $stops = []; }
        $taxis = [];
        try { $taxis = $this->pdo->query('SELECT id, name FROM taxis ORDER BY id DESC LIMIT 500')->fetchAll() ?: []; } catch (\Throwable $_) {}
        $existing = [];
        $taxiIdSel = (int)($itin['taxi_id'] ?? 0);
        if ($taxiIdSel > 0) {
            try {
                $se = $this->pdo->prepare('SELECT id, name, active FROM taxi_itineraries WHERE taxi_id = :tid ORDER BY id DESC');
                $se->execute([':tid'=>$taxiIdSel]);
                $existing = $se->fetchAll(\PDO::FETCH_ASSOC) ?: [];
            } catch (\Throwable $_) { $existing = []; }
        }
        $this->view('admin/taxi_itineraries_form', compact('csrf','taxis','itin','stops','existing'));
    }

    public function store(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['admin_taxi_itins_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock itineraries with the update password first.';
            $this->redirect('/admin/taxi/itineraries');
            return;
        }
        $taxi_id = (int)($_POST['taxi_id'] ?? 0) ?: null;
        $name = trim((string)($_POST['name'] ?? ''));
        $description = trim((string)($_POST['description'] ?? ''));
        $active = isset($_POST['active']) && $_POST['active'] === '1' ? 1 : 0;
        if ($name === '') { $_SESSION['errors']=['Name is required']; $this->redirect('/admin/taxi/itineraries/create'); return; }
        try {
            $ins = $this->pdo->prepare('INSERT INTO taxi_itineraries (vendor_id, taxi_id, name, description, active) VALUES (NULL, :taxi_id, :name, :description, :active)');
            $ins->execute([':taxi_id'=>$taxi_id, ':name'=>$name, ':description'=>($description!==''?$description:null), ':active'=>$active]);
            $iid = (int)$this->pdo->lastInsertId();
            $this->persistStops($iid, $_POST['stops'] ?? []);
            $_SESSION['flash'] = 'Itinerary created.';
        } catch (\Throwable $e) {
            $_SESSION['errors'] = ['Failed to save itinerary.'];
        }
        $this->redirect('/admin/taxi/itineraries');
    }

    public function update(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['admin_taxi_itins_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock itineraries with the update password first.';
            $this->redirect('/admin/taxi/itineraries');
            return;
        }
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/taxi/itineraries'); return; }
        $taxi_id = (int)($_POST['taxi_id'] ?? 0) ?: null;
        $name = trim((string)($_POST['name'] ?? ''));
        $description = trim((string)($_POST['description'] ?? ''));
        $active = isset($_POST['active']) && $_POST['active'] === '1' ? 1 : 0;
        if ($name === '') { $_SESSION['errors']=['Name is required']; $this->redirect('/admin/taxi/itineraries/edit?id='.$id); return; }
        try {
            $up = $this->pdo->prepare('UPDATE taxi_itineraries SET taxi_id=:taxi_id, name=:name, description=:description, active=:active WHERE id=:id');
            $up->execute([':id'=>$id, ':taxi_id'=>$taxi_id, ':name'=>$name, ':description'=>($description!==''?$description:null), ':active'=>$active]);
            $this->persistStops($id, $_POST['stops'] ?? []);
            $_SESSION['flash'] = 'Itinerary updated.';
        } catch (\Throwable $e) {
            $_SESSION['errors'] = ['Failed to update itinerary.'];
        }
        $this->redirect('/admin/taxi/itineraries');
    }

    public function delete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            try {
                $st = $this->pdo->prepare('DELETE FROM taxi_itinerary_stops WHERE itinerary_id=:id');
                $st->execute([':id'=>$id]);
                $this->pdo->prepare('DELETE FROM taxi_itineraries WHERE id=:id')->execute([':id'=>$id]);
                $_SESSION['flash'] = 'Itinerary deleted.';
            } catch (\Throwable $e) { $_SESSION['errors']=['Failed to delete itinerary.']; }
        }
        // Redirect back where user came from (create/edit) for better UX
        $ref = (string)($_SERVER['HTTP_REFERER'] ?? '');
        if ($ref !== '' && strpos($ref, '/admin/taxi/itineraries') !== false) {
            header('Location: ' . $ref);
            exit;
        }
        $this->redirect('/admin/taxi/itineraries');
    }

    private function persistStops(int $itineraryId, $stopsPost): void
    {
        if (!is_array($stopsPost)) { return; }
        try { $this->pdo->prepare('DELETE FROM taxi_itinerary_stops WHERE itinerary_id=:id')->execute([':id'=>$itineraryId]); } catch (\Throwable $_) {}
        $ins = $this->pdo->prepare('INSERT INTO taxi_itinerary_stops (itinerary_id, seq, label, address, wait_minutes, extra_price, notes) VALUES (:iid,:seq,:label,:addr,:wait,:price,:notes)');
        $seq = 1;
        foreach ($stopsPost as $s) {
            if (!is_array($s)) continue;
            $label = trim((string)($s['label'] ?? ''));
            if ($label === '') continue;
            $addr = trim((string)($s['address'] ?? ''));
            $wait = (int)($s['wait_minutes'] ?? 0);
            $price = isset($s['extra_price']) && $s['extra_price'] !== '' ? (string)$s['extra_price'] : null;
            $notes = trim((string)($s['notes'] ?? ''));
            try {
                $ins->execute([':iid'=>$itineraryId, ':seq'=>$seq++, ':label'=>$label, ':addr'=>($addr!==''?$addr:null), ':wait'=>($wait>0?$wait:null), ':price'=>$price, ':notes'=>($notes!==''?$notes:null)]);
            } catch (\Throwable $_) { /* ignore a bad row but continue */ }
        }
    }
}
