<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;

class TaxiAdminController extends Controller
{
    /**
     * Normalize an uploaded media URL to a relative file path under /assets/uploads.
     * Falls back to returning the input if it already looks like a relative path.
     */
    private function normalizeUploadPath(string $urlOrPath): string
    {
        $s = trim($urlOrPath);
        if ($s === '') return $s;
        // If already relative (starts with /assets/uploads), keep as-is
        if (strpos($s, '/assets/uploads/') === 0) return $s;
        // Strip scheme+host if absolute URL
        if (preg_match('#^https?://[^/]+(/.*)$#i', $s, $m)) {
            $s = $m[1];
        }
        // Ensure we only keep uploads subtree
        $pos = strpos($s, '/assets/uploads/');
        if ($pos !== false) {
            return substr($s, $pos);
        }
        // Unknown format, return as-is
        return $s;
    }

    /**
     * Persist taxi images from media_json and cover index.
     * On update, it replaces existing rows for this taxi.
     */
    private function saveTaxiImages(int $taxiId, string $mediaJson, int $coverIndex): void
    {
        // Decode JSON: expected [{originalUrl, thumbUrl, name, mime}, ...]
        $list = json_decode($mediaJson ?: '[]', true);
        if (!is_array($list)) $list = [];
        try {
            // Clear existing images and reinsert for simplicity
            $del = $this->pdo->prepare("DELETE FROM taxi_images WHERE taxi_id = :tid");
            $del->execute(['tid' => $taxiId]);
        } catch (\Throwable $e) {
            // Table might not exist in some environments; tolerate silently
            return;
        }

        if (!$list) return;
        try {
            $ins = $this->pdo->prepare("INSERT INTO taxi_images (taxi_id, file_path, is_cover) VALUES (:tid, :path, :cover)");
            foreach ($list as $idx => $item) {
                $orig = is_array($item) ? ($item['originalUrl'] ?? ($item['url'] ?? '')) : '';
                if (!$orig) continue;
                $path = $this->normalizeUploadPath((string)$orig);
                $ins->execute([
                    'tid' => $taxiId,
                    'path' => $path,
                    'cover' => ($idx === $coverIndex) ? 1 : 0,
                ]);
            }
        } catch (\Throwable $e) {
            // swallow; do not break main flow
        }
    }
    public function unlock(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $_SESSION['admin_taxis_unlocked'] = true;
        $this->redirect('/admin/taxis');
    }

    public function index(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $unlocked = !empty($_SESSION['admin_taxis_unlocked']);
        $rows = [];
        if ($unlocked) {
            // Include cover image path if available (prefer is_cover=1, then earliest)
            try {
                $q = "SELECT t.id, t.name AS title, t.name, t.route, t.base_price, t.active, v.name AS vendor,
                             (
                               SELECT ti.file_path FROM taxi_images ti
                               WHERE ti.taxi_id = t.id
                               ORDER BY ti.is_cover DESC, ti.id ASC
                               LIMIT 1
                             ) AS cover_path,
                             (
                               SELECT tp.agent_price FROM taxi_pricing tp
                               WHERE tp.taxi_id = t.id
                               ORDER BY tp.id DESC
                               LIMIT 1
                             ) AS b2b_price,
                             (
                               SELECT COALESCE(tp.customer_price, tp.base_fare) FROM taxi_pricing tp
                               WHERE tp.taxi_id = t.id
                               ORDER BY tp.id DESC
                               LIMIT 1
                             ) AS b2c_price
                      FROM taxis t JOIN vendors v ON v.id = t.vendor_id
                      ORDER BY t.id DESC LIMIT 200";
                $rows = $this->pdo->query($q)->fetchAll();
            } catch (\Throwable $e) {
                // Fallback if customer_price/base_fare column not present
                try {
                    $q = "SELECT t.id, t.name AS title, t.name, t.route, t.base_price, t.active, v.name AS vendor,
                                 (
                                   SELECT ti.file_path FROM taxi_images ti
                                   WHERE ti.taxi_id = t.id
                                   ORDER BY ti.is_cover DESC, ti.id ASC
                                   LIMIT 1
                                 ) AS cover_path,
                                 (
                                   SELECT tp.agent_price FROM taxi_pricing tp
                                   WHERE tp.taxi_id = t.id
                                   ORDER BY tp.id DESC
                                   LIMIT 1
                                 ) AS b2b_price,
                                 (
                                   SELECT tp.base_fare FROM taxi_pricing tp
                                   WHERE tp.taxi_id = t.id
                                   ORDER BY tp.id DESC
                                   LIMIT 1
                                 ) AS b2c_price
                          FROM taxis t JOIN vendors v ON v.id = t.vendor_id
                          ORDER BY t.id DESC LIMIT 200";
                    $rows = $this->pdo->query($q)->fetchAll();
                } catch (\Throwable $e2) {
                    // Last fallback without pricing join
                    $q = "SELECT t.id, t.name AS title, t.name, t.route, t.base_price, t.active, v.name AS vendor,
                                 (
                                   SELECT ti.file_path FROM taxi_images ti
                                   WHERE ti.taxi_id = t.id
                                   ORDER BY ti.is_cover DESC, ti.id ASC
                                   LIMIT 1
                                 ) AS cover_path
                          FROM taxis t JOIN vendors v ON v.id = t.vendor_id
                          ORDER BY t.id DESC LIMIT 200";
                    $rows = $this->pdo->query($q)->fetchAll();
                }
            }
        }
        $this->view('admin/taxis_index', compact('csrf', 'unlocked', 'rows'));
    }

    public function create(): void
    {
        Auth::requireRole(['Admin']);
        if (empty($_SESSION['admin_taxis_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock taxis with the update password first.';
            $this->redirect('/admin/taxis');
            return;
        }
        $csrf = Security::csrfToken();
        
        // Country filter (GET) and countries list
        $selectedCountry = trim((string)($_GET['country'] ?? ''));
        // Load countries primarily from locations; fallback to vendors if needed
        try {
            $countries = $this->pdo->query("SELECT DISTINCT country FROM locations WHERE country IS NOT NULL AND country<>'' ORDER BY country")->fetchAll();
        } catch (\Throwable $e) { $countries = []; }
        if (!$countries || count($countries) === 0) {
            try {
                $countries = $this->pdo->query("SELECT DISTINCT country FROM vendors WHERE country IS NOT NULL AND country<>'' ORDER BY country")->fetchAll();
            } catch (\Throwable $e) { $countries = []; }
        }

        // Vendors: only those with active taxi packages (fallbacks included); optionally filter by country
        $vendors = [];
        try {
            if ($selectedCountry !== '') {
                // Taxi vendors in selected country: active taxi package OR vendors.module='taxi'
                $stmt = $this->pdo->prepare(
                    "SELECT DISTINCT v.id, v.name
                     FROM vendors v
                     LEFT JOIN vendor_packages p ON p.vendor_id = v.id AND p.module='taxi' AND p.active=1
                     WHERE v.country = ? AND (p.id IS NOT NULL OR v.module='taxi')
                     ORDER BY v.name ASC"
                );
                $stmt->execute([$selectedCountry]);
                $vendors = $stmt->fetchAll();
            } else {
                $vendors = $this->pdo->query(
                    "SELECT DISTINCT v.id, v.name
                     FROM vendors v
                     JOIN vendor_packages p ON p.vendor_id = v.id
                     WHERE p.module='taxi' AND p.active=1
                     ORDER BY v.name ASC"
                )->fetchAll();
            }
        } catch (\Throwable $e) {
            // Fallback 1: vendors.module = 'taxi' if column exists
            try {
                if ($selectedCountry !== '') {
                    $stmt = $this->pdo->prepare("SELECT id, name FROM vendors WHERE module='taxi' AND country = ? ORDER BY name ASC");
                    $stmt->execute([$selectedCountry]);
                    $vendors = $stmt->fetchAll();
                } else {
                    $vendors = $this->pdo->query("SELECT id, name FROM vendors WHERE module='taxi' ORDER BY name ASC")->fetchAll();
                }
            } catch (\Throwable $e2) {
                // Fallback 2: all vendors (no filtering)
                try {
                    if ($selectedCountry !== '') {
                        $stmt = $this->pdo->prepare("SELECT id, name FROM vendors WHERE country = ? ORDER BY name ASC");
                        $stmt->execute([$selectedCountry]);
                        $vendors = $stmt->fetchAll();
                    } else {
                        $vendors = $this->pdo->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetchAll();
                    }
                } catch (\Throwable $e3) { $vendors = []; }
            }
        }
        // If query succeeded but returned empty, use broader fallbacks
        if (!$vendors) {
            try {
                $vendors = $this->pdo->query(
                    "SELECT DISTINCT v.id, v.name
                     FROM vendors v
                     JOIN vendor_packages p ON p.vendor_id = v.id
                     WHERE p.module='taxi' AND p.active=1
                     ORDER BY v.name ASC"
                )->fetchAll();
            } catch (\Throwable $e) {
                try {
                    $vendors = $this->pdo->query("SELECT id, name FROM vendors WHERE module='taxi' ORDER BY name ASC")->fetchAll();
                } catch (\Throwable $e2) {
                    try { $vendors = $this->pdo->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetchAll(); }
                    catch (\Throwable $e3) { $vendors = []; }
                }
            }
        }
        // Currency mapping: prefer country-specific mapping, fallback to global currencies
        $currencyCodes = [];
        $mappedCurrency = null;
        if ($selectedCountry !== '') {
            try {
                $s = $this->pdo->prepare("SELECT currency_code FROM country_currencies WHERE country = ?");
                $s->execute([$selectedCountry]);
                $mappedCurrency = strtoupper(trim((string)($s->fetchColumn() ?: '')));
                if ($mappedCurrency === '') { $mappedCurrency = null; }
            } catch (\Throwable $e) { $mappedCurrency = null; }
        }
        if ($mappedCurrency) {
            // Pass as a list with a single mapped code
            $currencyCodes = [['code' => $mappedCurrency]];
        } else {
            try {
                $currencyCodes = $this->pdo->query("SELECT code FROM currencies ORDER BY code")->fetchAll();
            } catch (\Throwable $e) { $currencyCodes = []; }
        }
        $taxi = [
            'id' => null,
            'vendor_id' => '',
            'title' => '',
            'name' => '',
            'vehicle_type' => '',
            'capacity' => '',
            'luggage_capacity' => '',
            'route' => '',
            'base_price' => '0.00',
            'active' => 1,
            'description' => '',
            // Pricing defaults for UI
            'currency' => $mappedCurrency ?: 'THB',
            'vendor_cost' => '',
            'agent_price' => '',
            'customer_price' => '',
            'pricing_model' => 'fixed',
            'per_km_price' => '',
            'per_hour_price' => '',
            'wait_per_hour' => '',
            'tax_percent' => '',
            'fees_amount' => '',
        ];
        $this->view('admin/taxis_form', compact('csrf', 'vendors', 'taxi', 'countries', 'selectedCountry', 'currencyCodes'));
    }

    public function edit(): void
    {
        Auth::requireRole(['Admin']);
        if (empty($_SESSION['admin_taxis_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock taxis with the update password first.';
            $this->redirect('/admin/taxis');
            return;
        }
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/taxis'); return; }
        $stmt = $this->pdo->prepare("SELECT * FROM taxis WHERE id = :id");
        $stmt->execute(['id' => $id]);
        $taxi = $stmt->fetch();
        if (!$taxi) { $this->redirect('/admin/taxis'); return; }
        $csrf = Security::csrfToken();
        // Load current vendor (for defaults)
        $currentVendor = null;
        try {
            $vs = $this->pdo->prepare("SELECT id, name, country FROM vendors WHERE id = ?");
            $vs->execute([(int)$taxi['vendor_id']]);
            $currentVendor = $vs->fetch();
        } catch (\Throwable $e) { $currentVendor = null; }
        // Country filter (GET) and countries list
        $selectedCountry = trim((string)($_GET['country'] ?? ''));
        if ($selectedCountry === '' && $currentVendor && !empty($currentVendor['country'])) {
            $selectedCountry = (string)$currentVendor['country'];
        }
        try {
            $countries = $this->pdo->query("SELECT DISTINCT country FROM locations WHERE country IS NOT NULL AND country<>'' ORDER BY country")->fetchAll();
        } catch (\Throwable $e) { $countries = []; }
        if (!$countries || count($countries) === 0) {
            try {
                $countries = $this->pdo->query("SELECT DISTINCT country FROM vendors WHERE country IS NOT NULL AND country<>'' ORDER BY country")->fetchAll();
            } catch (\Throwable $e) { $countries = []; }
        }

        // Vendors: only those with active taxi packages; optionally filter by country
        $vendors = [];
        try {
            if ($selectedCountry !== '') {
                // Taxi vendors in selected country: active taxi package OR vendors.module='taxi'
                $stmt = $this->pdo->prepare(
                    "SELECT DISTINCT v.id, v.name
                     FROM vendors v
                     LEFT JOIN vendor_packages p ON p.vendor_id = v.id AND p.module='taxi' AND p.active=1
                     WHERE v.country = ? AND (p.id IS NOT NULL OR v.module='taxi')
                     ORDER BY v.name ASC"
                );
                $stmt->execute([$selectedCountry]);
                $vendors = $stmt->fetchAll();
            } else {
                $vendors = $this->pdo->query(
                    "SELECT DISTINCT v.id, v.name
                     FROM vendors v
                     JOIN vendor_packages p ON p.vendor_id = v.id
                     WHERE p.module='taxi' AND p.active=1
                     ORDER BY v.name ASC"
                )->fetchAll();
            }
        } catch (\Throwable $e) {
            try {
                if ($selectedCountry !== '') {
                    $stmt = $this->pdo->prepare("SELECT id, name FROM vendors WHERE module='taxi' AND country = ? ORDER BY name ASC");
                    $stmt->execute([$selectedCountry]);
                    $vendors = $stmt->fetchAll();
                } else {
                    $vendors = $this->pdo->query("SELECT id, name FROM vendors WHERE module='taxi' ORDER BY name ASC")->fetchAll();
                }
            } catch (\Throwable $e2) {
                try {
                    if ($selectedCountry !== '') {
                        $stmt = $this->pdo->prepare("SELECT id, name FROM vendors WHERE country = ? ORDER BY name ASC");
                        $stmt->execute([$selectedCountry]);
                        $vendors = $stmt->fetchAll();
                    } else {
                        $vendors = $this->pdo->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetchAll();
                    }
                } catch (\Throwable $e3) { $vendors = []; }
            }
        }
        // If query succeeded but returned empty, use broader fallbacks
        if (!$vendors) {
            try {
                $vendors = $this->pdo->query(
                    "SELECT DISTINCT v.id, v.name
                     FROM vendors v
                     JOIN vendor_packages p ON p.vendor_id = v.id
                     WHERE p.module='taxi' AND p.active=1
                     ORDER BY v.name ASC"
                )->fetchAll();
            } catch (\Throwable $e) {
                try {
                    $vendors = $this->pdo->query("SELECT id, name FROM vendors WHERE module='taxi' ORDER BY name ASC")->fetchAll();
                } catch (\Throwable $e2) {
                    try { $vendors = $this->pdo->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetchAll(); }
                    catch (\Throwable $e3) { $vendors = []; }
                }
            }
        }
        // Currency mapping: prefer country-specific mapping, fallback to global currencies
        $currencyCodes = [];
        $mappedCurrency = null;
        if ($selectedCountry !== '') {
            try {
                $s = $this->pdo->prepare("SELECT currency_code FROM country_currencies WHERE country = ?");
                $s->execute([$selectedCountry]);
                $mappedCurrency = strtoupper(trim((string)($s->fetchColumn() ?: '')));
                if ($mappedCurrency === '') { $mappedCurrency = null; }
            } catch (\Throwable $e) { $mappedCurrency = null; }
        }
        if ($mappedCurrency) {
            $currencyCodes = [['code' => $mappedCurrency]];
        } else {
            try {
                $currencyCodes = $this->pdo->query("SELECT code FROM currencies ORDER BY code")->fetchAll();
            } catch (\Throwable $e) { $currencyCodes = []; }
        }
        // Load pricing row (if exists) for this taxi and map to form fields
        try {
            $ps = $this->pdo->prepare("SELECT * FROM taxi_pricing WHERE taxi_id = :tid ORDER BY id DESC LIMIT 1");
            $ps->execute(['tid' => $id]);
            $pr = $ps->fetch();
            if ($pr) {
                $taxi['currency'] = $pr['currency'] ?? 'THB';
                $taxi['pricing_model'] = $pr['model'] ?? 'fixed';
                $taxi['per_km_price'] = $pr['per_km'] ?? '';
                $taxi['per_hour_price'] = $pr['per_hour'] ?? '';
                $taxi['wait_per_hour'] = $pr['wait_per_hour'] ?? '';
                $taxi['tax_percent'] = $pr['tax_pct'] ?? '';
                $taxi['fees_amount'] = $pr['fee_pct'] ?? '';
                if (array_key_exists('vendor_cost', $pr)) { $taxi['vendor_cost'] = $pr['vendor_cost']; }
                if (array_key_exists('agent_price', $pr)) { $taxi['agent_price'] = $pr['agent_price']; }
                if (array_key_exists('customer_price', $pr)) { $taxi['customer_price'] = $pr['customer_price']; }
                // Backfill legacy base_fare -> customer_price
                if (!isset($taxi['customer_price']) && isset($pr['base_fare'])) {
                    $taxi['customer_price'] = $pr['base_fare'];
                }
                // Also backfill vendor_cost and agent_price from base_fare if missing (schema lacks these cols)
                if (!isset($taxi['vendor_cost']) && isset($pr['base_fare'])) {
                    $taxi['vendor_cost'] = $pr['base_fare'];
                }
                if (!isset($taxi['agent_price']) && isset($pr['base_fare'])) {
                    $taxi['agent_price'] = $pr['base_fare'];
                }
                // Heuristic for distance toggle
                $distOn = 0;
                if (!empty($taxi['pricing_model']) && $taxi['pricing_model'] === 'distance') { $distOn = 1; }
                if (!$distOn) {
                    $pkm = (string)($taxi['per_km_price'] ?? '');
                    if ($pkm !== '' && (float)$pkm > 0) { $distOn = 1; }
                }
                $taxi['distance_pricing_enabled'] = $distOn;
            }
        } catch (\Throwable $e) { /* ignore if table/cols missing */ }

        // Load latest route (if exists) and map for form
        try {
            $rs = $this->pdo->prepare("SELECT * FROM taxi_routes WHERE taxi_id = :tid ORDER BY id DESC LIMIT 1");
            $rs->execute(['tid' => $id]);
            $rt = $rs->fetch();
            if ($rt) {
                $taxi['pickup_label'] = $rt['pickup_name'] ?? '';
                $taxi['pickup_lat'] = $rt['pickup_lat'] ?? '';
                $taxi['pickup_lng'] = $rt['pickup_lng'] ?? '';
                $taxi['drop_label'] = $rt['drop_name'] ?? '';
                $taxi['drop_lat'] = $rt['drop_lat'] ?? '';
                $taxi['drop_lng'] = $rt['drop_lng'] ?? '';
            } else {
                // Fallback: derive from taxis.route label (e.g., "BKK -> City" or "BKK → City")
                $routeLabel = (string)($taxi['route'] ?? '');
                if ($routeLabel !== '') {
                    $parts = preg_split('/\s*(?:->|→|-)\s*/u', $routeLabel);
                    if (is_array($parts) && count($parts) >= 2) {
                        if (empty($taxi['pickup_label'])) $taxi['pickup_label'] = trim((string)$parts[0]);
                        if (empty($taxi['drop_label'])) $taxi['drop_label'] = trim((string)$parts[1]);
                    }
                }
            }
        } catch (\Throwable $e) { /* tolerate missing table */ }

        // Load media (existing images) for prefill in uploader
        try {
            $ms = $this->pdo->prepare("SELECT file_path, is_cover FROM taxi_images WHERE taxi_id = :tid ORDER BY id ASC");
            $ms->execute(['tid' => $id]);
            $imgs = $ms->fetchAll();
            if ($imgs && is_array($imgs)) {
                $files = [];
                $coverIdx = 0; $i = 0;
                foreach ($imgs as $row) {
                    $path = (string)($row['file_path'] ?? '');
                    if ($path === '') { $i++; continue; }
                    // Normalize to absolute web path if stored as relative
                    if (strpos($path, 'http://') !== 0 && strpos($path, 'https://') !== 0 && strpos($path, '/') !== 0) {
                        $path = '/' . ltrim($path, '/');
                    }
                    // Assume stored path is a web path already or relative under /; pass through
                    $files[] = ['originalUrl' => $path, 'thumbUrl' => ''];
                    if ((int)($row['is_cover'] ?? 0) === 1) { $coverIdx = $i; }
                    $i++;
                }
                if ($files) {
                    $taxi['media_json'] = json_encode($files);
                    $taxi['cover_index'] = (string)$coverIdx;
                }
            }
        } catch (\Throwable $e) { /* tolerate missing table */ }

        // Load availability and map to simple fields expected by form
        try {
            $as = $this->pdo->prepare("SELECT * FROM taxi_availability WHERE taxi_id = :tid ORDER BY id DESC LIMIT 1");
            $as->execute(['tid' => $id]);
            $av = $as->fetch();
            if ($av) {
                // Map days (robust across shapes)
                $daysArr = [];
                $firstStart = '';
                $firstEnd = '';
                $dow = $av['dow'] ?? null;
                // Try JSON decode if string
                $dow = is_string($dow) ? json_decode($dow, true) : (is_array($dow) ? $dow : null);
                if (is_array($dow)) {
                    $labelMap = [
                        'Mon' => ['Mon','mon','monday'],
                        'Tue' => ['Tue','tue','tuesday'],
                        'Wed' => ['Wed','wed','wednesday'],
                        'Thu' => ['Thu','thu','thursday'],
                        'Fri' => ['Fri','fri','friday'],
                        'Sat' => ['Sat','sat','saturday'],
                        'Sun' => ['Sun','sun','sunday'],
                    ];
                    // Case A: array of strings e.g., ["Mon","Tue"]
                    $isStrings = array_values($dow) === $dow; // sequential indices
                    if ($isStrings) {
                        foreach ($labelMap as $label => $alts) {
                            foreach ($dow as $d) {
                                if (in_array($d, $alts, true)) { $daysArr[] = $label; break; }
                            }
                        }
                    } else {
                        // Case B: object of day->(bool|object)
                        foreach ($labelMap as $label => $alts) {
                            foreach ($alts as $k) {
                                if (array_key_exists($k, $dow)) {
                                    $v = $dow[$k];
                                    if (is_array($v)) {
                                        $on = (int)($v['on'] ?? 0) === 1 || ($v['on'] ?? false) === true;
                                        if ($on) { $daysArr[] = $label; }
                                        if ($firstStart === '' && !empty($v['start'])) { $firstStart = (string)$v['start']; }
                                        if ($firstEnd === '' && !empty($v['end'])) { $firstEnd = (string)$v['end']; }
                                    } else {
                                        // boolean-like
                                        if ($v) { $daysArr[] = $label; }
                                    }
                                    break;
                                }
                            }
                        }
                    }
                }

                // Support optional explicit columns if present
                if ($firstStart === '' && !empty($av['start_time'])) { $firstStart = (string)$av['start_time']; }
                if ($firstEnd === '' && !empty($av['end_time'])) { $firstEnd = (string)$av['end_time']; }

                if ($daysArr) { $taxi['available_days'] = $daysArr; }
                if ($firstStart !== '') { $taxi['start_time'] = $firstStart; }
                if ($firstEnd !== '') { $taxi['end_time'] = $firstEnd; }

                // Blackout dates: accept JSON array or CSV string
                $bd = $av['blackout_dates'] ?? null;
                $asArray = null;
                if (is_string($bd)) {
                    $decoded = json_decode($bd, true);
                    if (is_array($decoded)) { $asArray = $decoded; }
                    else { $asArray = array_map('trim', array_filter(explode(',', $bd))); }
                } elseif (is_array($bd)) {
                    $asArray = $bd;
                }
                if (is_array($asArray)) {
                    $asArray = array_filter(array_map('strval', $asArray));
                    if ($asArray) { $taxi['blackout_dates'] = implode(', ', $asArray); }
                }
            }
        } catch (\Throwable $e) { /* tolerate missing table */ }

        $this->view('admin/taxis_form', compact('csrf', 'vendors', 'taxi', 'countries', 'selectedCountry', 'currencyCodes'));
    }

    public function store(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['admin_taxis_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock taxis with the update password first.';
            $this->redirect('/admin/taxis');
            return;
        }
        // Basic validation
        $vendor_id = (int)($_POST['vendor_id'] ?? 0);
        $name = trim((string)($_POST['name'] ?? ''));
        $title = trim((string)($_POST['title'] ?? ''));
        $vehicle_type = trim((string)($_POST['vehicle_type'] ?? ''));
        $capacity = (int)($_POST['capacity'] ?? 0);
        $luggage_capacity = (int)($_POST['luggage_capacity'] ?? 0);
        $route = trim((string)($_POST['route'] ?? ''));
        $trip_type = (string)($_POST['trip_type'] ?? 'both');
        if (!in_array($trip_type, ['single','round','both'], true)) { $trip_type = 'both'; }
        $base_price = (string)($_POST['base_price'] ?? '');
        $active = isset($_POST['active']) && $_POST['active'] === '1' ? 1 : 0;
        $description = trim((string)($_POST['description'] ?? ''));

        $errors = [];
        if ($vendor_id <= 0) $errors[] = 'Vendor is required';
        if ($name === '' && $title === '') $errors[] = 'Name or Title is required';
        if ($capacity < 1) $errors[] = 'Capacity must be at least 1';
        if ($route === '') $errors[] = 'Route is required';

        if ($errors) {
            $_SESSION['errors'] = $errors;
            $this->redirect('/admin/taxis/create');
            return;
        }

        // Pricing fields (read first so we can compute effective base price)
        $currency = trim((string)($_POST['currency'] ?? 'THB')) ?: 'THB';
        $pricing_model = trim((string)($_POST['pricing_model'] ?? 'fixed')) ?: 'fixed';
        $vendor_cost = (string)($_POST['vendor_cost'] ?? '');
        $agent_price = (string)($_POST['agent_price'] ?? '');
        $customer_price = (string)($_POST['customer_price'] ?? '');
        $per_km = (string)($_POST['per_km_price'] ?? '');
        $per_hour = (string)($_POST['per_hour_price'] ?? '');
        $wait_per_hour = (string)($_POST['wait_per_hour'] ?? '');
        $tax_pct = (string)($_POST['tax_percent'] ?? '');
        $fee_pct = (string)($_POST['fees_amount'] ?? '');

        // Compute effective base price: prefer customer_price, else provided base_price, else 0
        $effectiveBasePrice = $customer_price !== '' ? $customer_price : ($base_price !== '' ? $base_price : '0');

        $stmt = $this->pdo->prepare("INSERT INTO taxis (vendor_id, name, route, trip_type, base_price, title, vehicle_type, capacity, luggage_capacity, description, active)
            VALUES (:vendor_id, :name, :route, :trip_type, :base_price, :title, :vehicle_type, :capacity, :luggage_capacity, :description, :active)");
        $stmt->execute([
            'vendor_id' => $vendor_id,
            'name' => $name ?: $title,
            'route' => $route,
            'trip_type' => $trip_type,
            'base_price' => $effectiveBasePrice,
            'title' => $title ?: $name,
            'vehicle_type' => $vehicle_type ?: null,
            'capacity' => $capacity ?: null,
            'luggage_capacity' => $luggage_capacity ?: null,
            'description' => $description ?: null,
            'active' => $active,
        ]);
        $taxiId = (int)$this->pdo->lastInsertId();

        // Generate a unique human-friendly series number and persist
        try {
            $baseSeries = 'TX-' . date('ymd') . '-' . str_pad((string)$taxiId, 4, '0', STR_PAD_LEFT);
            $seriesNo = $baseSeries;
            $attempts = 0;
            do {
                $ok = true;
                try {
                    $u = $this->pdo->prepare("UPDATE taxis SET series_no = :s WHERE id = :id");
                    $u->execute(['s' => $seriesNo, 'id' => $taxiId]);
                } catch (\PDOException $ex) {
                    // On duplicate, append a short suffix and retry a few times
                    $ok = false;
                    $attempts++;
                    $seriesNo = $baseSeries . '-' . substr((string)uniqid('', true), -3);
                }
            } while (!$ok && $attempts < 3);
        } catch (\Throwable $e) { /* ignore series_no failures */ }

        // Insert pricing row; tolerate missing columns
        try {
            $sql = "INSERT INTO taxi_pricing (taxi_id, model, currency, base_fare, per_km, per_hour, wait_per_hour, tax_pct, fee_pct, vendor_cost, agent_price, customer_price)
                    VALUES (:taxi_id, :model, :currency, :base_fare, :per_km, :per_hour, :wait_per_hour, :tax_pct, :fee_pct, :vendor_cost, :agent_price, :customer_price)";
            $ps = $this->pdo->prepare($sql);
            $ps->execute([
                'taxi_id' => $taxiId,
                'model' => $pricing_model,
                'currency' => $currency,
                'base_fare' => $effectiveBasePrice,
                'per_km' => $per_km !== '' ? $per_km : null,
                'per_hour' => $per_hour !== '' ? $per_hour : null,
                'wait_per_hour' => $wait_per_hour !== '' ? $wait_per_hour : null,
                'tax_pct' => $tax_pct !== '' ? $tax_pct : null,
                'fee_pct' => $fee_pct !== '' ? $fee_pct : null,
                'vendor_cost' => $vendor_cost !== '' ? $vendor_cost : null,
                'agent_price' => $agent_price !== '' ? $agent_price : null,
                'customer_price' => $customer_price !== '' ? $customer_price : null,
            ]);
        } catch (\Throwable $e) {
            // Retry without optional columns if schema lacks them
            try {
                $sql2 = "INSERT INTO taxi_pricing (taxi_id, model, currency, base_fare, per_km, per_hour, wait_per_hour, tax_pct, fee_pct)
                         VALUES (:taxi_id, :model, :currency, :base_fare, :per_km, :per_hour, :wait_per_hour, :tax_pct, :fee_pct)";
                $ps2 = $this->pdo->prepare($sql2);
                $ps2->execute([
                    'taxi_id' => $taxiId,
                    'model' => $pricing_model,
                    'currency' => $currency,
                    'base_fare' => $customer_price !== '' ? $customer_price : $base_price,
                    'per_km' => $per_km !== '' ? $per_km : null,
                    'per_hour' => $per_hour !== '' ? $per_hour : null,
                    'wait_per_hour' => $wait_per_hour !== '' ? $wait_per_hour : null,
                    'tax_pct' => $tax_pct !== '' ? $tax_pct : null,
                    'fee_pct' => $fee_pct !== '' ? $fee_pct : null,
                ]);
            } catch (\Throwable $e2) { /* swallow */ }
        }
        // Persist media if provided
        $mediaJson = (string)($_POST['media_json'] ?? '');
        $coverIndex = (int)($_POST['cover_index'] ?? 0);
        if ($mediaJson !== '') {
            $this->saveTaxiImages($taxiId, $mediaJson, $coverIndex);
        }

        $_SESSION['flash'] = 'Taxi created.';
        $this->redirect('/admin/taxis');
    }

    public function update(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['admin_taxis_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock taxis with the update password first.';
            $this->redirect('/admin/taxis');
            return;
        }
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/taxis'); return; }

        $vendor_id = (int)($_POST['vendor_id'] ?? 0);
        $name = trim((string)($_POST['name'] ?? ''));
        $title = trim((string)($_POST['title'] ?? ''));
        $vehicle_type = trim((string)($_POST['vehicle_type'] ?? ''));
        $capacity = (int)($_POST['capacity'] ?? 0);
        $luggage_capacity = (int)($_POST['luggage_capacity'] ?? 0);
        $route = trim((string)($_POST['route'] ?? ''));
        // Trip type (ensure defined for UPDATE below)
        $trip_type = (string)($_POST['trip_type'] ?? 'both');
        if (!in_array($trip_type, ['single','round','both'], true)) { $trip_type = 'both'; }
        $base_price = (string)($_POST['base_price'] ?? '');
        $active = isset($_POST['active']) && $_POST['active'] === '1' ? 1 : 0;
        $description = trim((string)($_POST['description'] ?? ''));

        $errors = [];
        if ($vendor_id <= 0) $errors[] = 'Vendor is required';
        if ($name === '' && $title === '') $errors[] = 'Name or Title is required';
        if ($capacity < 1) $errors[] = 'Capacity must be at least 1';
        if ($route === '') $errors[] = 'Route is required';
        if ($errors) {
            $_SESSION['errors'] = $errors;
            $this->redirect('/admin/taxis/edit?id=' . $id);
            return;
        }

        // Pricing fields
        $currency = trim((string)($_POST['currency'] ?? 'THB')) ?: 'THB';
        $pricing_model = trim((string)($_POST['pricing_model'] ?? 'fixed')) ?: 'fixed';
        $vendor_cost = (string)($_POST['vendor_cost'] ?? '');
        $agent_price = (string)($_POST['agent_price'] ?? '');
        $customer_price = (string)($_POST['customer_price'] ?? '');
        $per_km = (string)($_POST['per_km_price'] ?? '');
        $per_hour = (string)($_POST['per_hour_price'] ?? '');
        $wait_per_hour = (string)($_POST['wait_per_hour'] ?? '');
        $tax_pct = (string)($_POST['tax_percent'] ?? '');
        $fee_pct = (string)($_POST['fees_amount'] ?? '');

        // Compute effective base price: prefer customer_price, else provided base_price, else 0
        $effectiveBasePrice = $customer_price !== '' ? $customer_price : ($base_price !== '' ? $base_price : '0');

        $stmt = $this->pdo->prepare("UPDATE taxis SET vendor_id=:vendor_id, name=:name, route=:route, trip_type=:trip_type, base_price=:base_price, title=:title, vehicle_type=:vehicle_type, capacity=:capacity, luggage_capacity=:luggage_capacity, description=:description, active=:active
            WHERE id=:id");
        $stmt->execute([
            'id' => $id,
            'vendor_id' => $vendor_id,
            'name' => $name ?: $title,
            'route' => $route,
            'trip_type' => $trip_type,
            'base_price' => $effectiveBasePrice,
            'title' => $title ?: $name,
            'vehicle_type' => $vehicle_type ?: null,
            'capacity' => $capacity ?: null,
            'luggage_capacity' => $luggage_capacity ?: null,
            'description' => $description ?: null,
            'active' => $active,
        ]);

        // Upsert pricing: try UPDATE; if 0 rows, INSERT. Be tolerant of missing optional columns.
        try {
            $up = $this->pdo->prepare("UPDATE taxi_pricing SET model=:model, currency=:currency, base_fare=:base_fare,
                     per_km=:per_km, per_hour=:per_hour, wait_per_hour=:wait_per_hour, tax_pct=:tax_pct, fee_pct=:fee_pct,
                     vendor_cost=:vendor_cost, agent_price=:agent_price, customer_price=:customer_price
                     WHERE taxi_id=:taxi_id");
            $up->execute([
                'taxi_id' => $id,
                'model' => $pricing_model,
                'currency' => $currency,
                'base_fare' => $effectiveBasePrice,
                'per_km' => $per_km !== '' ? $per_km : null,
                'per_hour' => $per_hour !== '' ? $per_hour : null,
                'wait_per_hour' => $wait_per_hour !== '' ? $wait_per_hour : null,
                'tax_pct' => $tax_pct !== '' ? $tax_pct : null,
                'fee_pct' => $fee_pct !== '' ? $fee_pct : null,
                'vendor_cost' => $vendor_cost !== '' ? $vendor_cost : null,
                'agent_price' => $agent_price !== '' ? $agent_price : null,
                'customer_price' => $customer_price !== '' ? $customer_price : null,
            ]);
            if ($up->rowCount() === 0) {
                $ins = $this->pdo->prepare("INSERT INTO taxi_pricing (taxi_id, model, currency, base_fare, per_km, per_hour, wait_per_hour, tax_pct, fee_pct, vendor_cost, agent_price, customer_price)
                    VALUES (:taxi_id, :model, :currency, :base_fare, :per_km, :per_hour, :wait_per_hour, :tax_pct, :fee_pct, :vendor_cost, :agent_price, :customer_price)");
                $ins->execute([
                    'taxi_id' => $id,
                    'model' => $pricing_model,
                    'currency' => $currency,
                    'base_fare' => $effectiveBasePrice,
                    'per_km' => $per_km !== '' ? $per_km : null,
                    'per_hour' => $per_hour !== '' ? $per_hour : null,
                    'wait_per_hour' => $wait_per_hour !== '' ? $wait_per_hour : null,
                    'tax_pct' => $tax_pct !== '' ? $tax_pct : null,
                    'fee_pct' => $fee_pct !== '' ? $fee_pct : null,
                    'vendor_cost' => $vendor_cost !== '' ? $vendor_cost : null,
                    'agent_price' => $agent_price !== '' ? $agent_price : null,
                    'customer_price' => $customer_price !== '' ? $customer_price : null,
                ]);
            }
        } catch (\Throwable $e) {
            try {
                $up2 = $this->pdo->prepare("UPDATE taxi_pricing SET model=:model, currency=:currency, base_fare=:base_fare,
                         per_km=:per_km, per_hour=:per_hour, wait_per_hour=:wait_per_hour, tax_pct=:tax_pct, fee_pct=:fee_pct
                         WHERE taxi_id=:taxi_id");
                $up2->execute([
                    'taxi_id' => $id,
                    'model' => $pricing_model,
                    'currency' => $currency,
                    'base_fare' => $effectiveBasePrice,
                    'per_km' => $per_km !== '' ? $per_km : null,
                    'per_hour' => $per_hour !== '' ? $per_hour : null,
                    'wait_per_hour' => $wait_per_hour !== '' ? $wait_per_hour : null,
                    'tax_pct' => $tax_pct !== '' ? $tax_pct : null,
                    'fee_pct' => $fee_pct !== '' ? $fee_pct : null,
                ]);
                if ($up2->rowCount() === 0) {
                    $ins2 = $this->pdo->prepare("INSERT INTO taxi_pricing (taxi_id, model, currency, base_fare, per_km, per_hour, wait_per_hour, tax_pct, fee_pct)
                        VALUES (:taxi_id, :model, :currency, :base_fare, :per_km, :per_hour, :wait_per_hour, :tax_pct, :fee_pct)");
                    $ins2->execute([
                        'taxi_id' => $id,
                        'model' => $pricing_model,
                        'currency' => $currency,
                        'base_fare' => $effectiveBasePrice,
                        'per_km' => $per_km !== '' ? $per_km : null,
                        'per_hour' => $per_hour !== '' ? $per_hour : null,
                        'wait_per_hour' => $wait_per_hour !== '' ? $wait_per_hour : null,
                        'tax_pct' => $tax_pct !== '' ? $tax_pct : null,
                        'fee_pct' => $fee_pct !== '' ? $fee_pct : null,
                    ]);
                }
            } catch (\Throwable $e2) { /* swallow */ }
        }

        // Upsert Routes (pickup/drop labels and coordinates)
        try {
            $pickup_label = trim((string)($_POST['pickup_label'] ?? ''));
            $pickup_lat = trim((string)($_POST['pickup_lat'] ?? ''));
            $pickup_lng = trim((string)($_POST['pickup_lng'] ?? ''));
            $drop_label = trim((string)($_POST['drop_label'] ?? ''));
            $drop_lat = trim((string)($_POST['drop_lat'] ?? ''));
            $drop_lng = trim((string)($_POST['drop_lng'] ?? ''));

            // Consider a route update if any field provided
            $hasRouteInput = ($pickup_label !== '' || $drop_label !== '' || $pickup_lat !== '' || $pickup_lng !== '' || $drop_lat !== '' || $drop_lng !== '');
            if ($hasRouteInput) {
                // Simplest approach: clear old rows and insert one new row
                try {
                    $del = $this->pdo->prepare("DELETE FROM taxi_routes WHERE taxi_id = :tid");
                    $del->execute(['tid' => $id]);
                } catch (\Throwable $e) { /* ignore */ }
                try {
                    $ins = $this->pdo->prepare("INSERT INTO taxi_routes (taxi_id, pickup_name, pickup_lat, pickup_lng, drop_name, drop_lat, drop_lng) VALUES (:tid, :pname, :plat, :plng, :dname, :dlat, :dlng)");
                    $ins->execute([
                        'tid' => $id,
                        'pname' => $pickup_label !== '' ? $pickup_label : null,
                        'plat' => $pickup_lat !== '' ? $pickup_lat : null,
                        'plng' => $pickup_lng !== '' ? $pickup_lng : null,
                        'dname' => $drop_label !== '' ? $drop_label : null,
                        'dlat' => $drop_lat !== '' ? $drop_lat : null,
                        'dlng' => $drop_lng !== '' ? $drop_lng : null,
                    ]);
                } catch (\Throwable $e) { /* tolerate missing table */ }
            }
        } catch (\Throwable $e) { /* ignore */ }

        // Upsert Availability (days, start/end, blackout dates)
        try {
            $days = isset($_POST['available_days']) && is_array($_POST['available_days']) ? $_POST['available_days'] : [];
            $start_time = trim((string)($_POST['start_time'] ?? ''));
            $end_time = trim((string)($_POST['end_time'] ?? ''));
            $blackout = trim((string)($_POST['blackout_dates'] ?? ''));
            // Build normalized DOW object
            $labels = ['Mon','Tue','Wed','Thu','Fri','Sat','Sun'];
            $mapKeys = [
                'Mon' => 'mon','Tue' => 'tue','Wed' => 'wed','Thu' => 'thu','Fri' => 'fri','Sat' => 'sat','Sun' => 'sun'
            ];
            $dow = [];
            foreach ($labels as $L) {
                $k = $mapKeys[$L];
                $on = in_array($L, $days, true);
                $dow[$k] = ['on' => $on ? 1 : 0, 'start' => $start_time ?: null, 'end' => $end_time ?: null];
            }
            $dowJson = json_encode($dow);
            // Normalize blackout dates to array JSON
            $blackoutArr = [];
            if ($blackout !== '') {
                // If looks like JSON array, keep; else split by comma
                $decoded = json_decode($blackout, true);
                if (is_array($decoded)) { $blackoutArr = $decoded; }
                else { $blackoutArr = array_values(array_filter(array_map('trim', explode(',', $blackout)))); }
            }
            $bdJson = json_encode($blackoutArr);

            // Clear and reinsert
            try {
                $delA = $this->pdo->prepare("DELETE FROM taxi_availability WHERE taxi_id = :tid");
                $delA->execute(['tid' => $id]);
            } catch (\Throwable $e) { /* ignore */ }
            // Try with optional start/end columns first
            try {
                $insA = $this->pdo->prepare("INSERT INTO taxi_availability (taxi_id, dow, blackout_dates, start_time, end_time) VALUES (:tid, :dow, :bd, :st, :et)");
                $insA->execute(['tid' => $id, 'dow' => $dowJson, 'bd' => $bdJson, 'st' => $start_time !== '' ? $start_time : null, 'et' => $end_time !== '' ? $end_time : null]);
            } catch (\Throwable $e) {
                // Fallback without start/end columns
                try {
                    $insA2 = $this->pdo->prepare("INSERT INTO taxi_availability (taxi_id, dow, blackout_dates) VALUES (:tid, :dow, :bd)");
                    $insA2->execute(['tid' => $id, 'dow' => $dowJson, 'bd' => $bdJson]);
                } catch (\Throwable $e2) { /* swallow */ }
            }
        } catch (\Throwable $e) { /* ignore */ }

        // Persist media if provided (replace existing)
        $mediaJson = (string)($_POST['media_json'] ?? '');
        $coverIndex = (int)($_POST['cover_index'] ?? 0);
        if ($mediaJson !== '') {
            $this->saveTaxiImages($id, $mediaJson, $coverIndex);
        } else {
            // If explicitly empty, also clear existing images to reflect removal
            try {
                $del = $this->pdo->prepare("DELETE FROM taxi_images WHERE taxi_id = :tid");
                $del->execute(['tid' => $id]);
            } catch (\Throwable $e) { /* ignore */ }
        }

        $this->redirect('/admin/taxis');
    }

    public function enable(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['admin_taxis_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock taxis with the update password first.';
            $this->redirect('/admin/taxis');
            return;
        }
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            $stmt = $this->pdo->prepare("UPDATE taxis SET active = 1 WHERE id = :id");
            $stmt->execute(['id' => $id]);
            $_SESSION['flash'] = 'Taxi enabled.';
        }
        $this->redirect('/admin/taxis');
    }

    public function disable(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['admin_taxis_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock taxis with the update password first.';
            $this->redirect('/admin/taxis');
            return;
        }
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            $stmt = $this->pdo->prepare("UPDATE taxis SET active = 0 WHERE id = :id");
            $stmt->execute(['id' => $id]);
            $_SESSION['flash'] = 'Taxi disabled.';
        }
        $this->redirect('/admin/taxis');
    }

    public function show(): void
    {
        Auth::requireRole(['Admin']);
        if (empty($_SESSION['admin_taxis_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock taxis with the update password first.';
            $this->redirect('/admin/taxis');
            return;
        }
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/taxis'); return; }
        $csrf = Security::csrfToken();
        // Load taxi
        $stmt = $this->pdo->prepare("SELECT t.*, v.name AS vendor_name FROM taxis t JOIN vendors v ON v.id=t.vendor_id WHERE t.id = :id");
        $stmt->execute(['id' => $id]);
        $taxi = $stmt->fetch();
        if (!$taxi) { $this->redirect('/admin/taxis'); return; }
        // Pricing (latest)
        $pricing = null;
        try {
            $ps = $this->pdo->prepare("SELECT * FROM taxi_pricing WHERE taxi_id = :tid ORDER BY id DESC LIMIT 1");
            $ps->execute(['tid' => $id]);
            $pricing = $ps->fetch();
        } catch (\Throwable $e) { $pricing = null; }
        // Images
        $images = [];
        try {
            $is = $this->pdo->prepare("SELECT file_path, is_cover FROM taxi_images WHERE taxi_id = :tid ORDER BY is_cover DESC, id ASC");
            $is->execute(['tid' => $id]);
            $images = $is->fetchAll();
        } catch (\Throwable $e) { $images = []; }
        $this->view('admin/taxis_view', compact('csrf', 'taxi', 'pricing', 'images'));
    }
}
