<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Middleware\AgentGuard;
use App\Core\Auth;

class SupportController extends Controller
{
    // GET /b2b/agent/support
    public function agentSupport(): void
    {
        AgentGuard::requireLogin();
        $agent = $_SESSION['agent'] ?? null;
        $uid = (int)($agent['id'] ?? 0);
        $tickets = [];
        $status = trim((string)($_GET['status'] ?? ''));
        $keyword = trim((string)($_GET['q'] ?? ''));
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 10; // default 10 per page
        $offset = ($page - 1) * $perPage;
        try {
            // Build WHERE
            $where = ['user_id = :uid'];
            $params = [':uid'=>$uid];
            if ($status !== '') { $where[] = 'status = :st'; $params[':st'] = $status; }
            if ($keyword !== '') { $where[] = '(subject LIKE :kw OR message LIKE :kw)'; $params[':kw'] = '%'.$keyword.'%'; }
            $whereSql = ' WHERE ' . implode(' AND ', $where);

            // Count total
            $count = $this->pdo->prepare("SELECT COUNT(*) AS c FROM support_tickets".$whereSql);
            $count->execute($params);
            $total = (int)($count->fetch(\PDO::FETCH_ASSOC)['c'] ?? 0);

            // Fetch page
            $sql = "SELECT * FROM support_tickets".$whereSql." ORDER BY COALESCE(updated_at, created_at) DESC LIMIT {$perPage} OFFSET {$offset}";
            $st = $this->pdo->prepare($sql);
            $st->execute($params);
            $tickets = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $tickets = []; $total = 0; }
        $csrf = AgentGuard::generateCsrfToken('support_create');
        $this->view('agent/support', [
            'title' => 'Support Center',
            'tickets' => $tickets,
            'csrf' => $csrf,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
            'statusFilter' => $status,
            'keyword' => $keyword,
        ]);
    }

    // GET /b2b/agent/support/view?id=123
    public function agentView(): void
    {
        AgentGuard::requireLogin();
        $agent = $_SESSION['agent'] ?? null;
        $agentId = (int)($agent['id'] ?? 0);
        $ticketId = (int)($_GET['id'] ?? 0);
        
        if ($ticketId <= 0) {
            $this->redirect('/b2b/agent/support?err=invalid_ticket');
            return;
        }
        
        try {
            // Get the ticket
            $ticketStmt = $this->pdo->prepare('SELECT t.*, u.email AS user_email 
                FROM support_tickets t 
                LEFT JOIN users u ON u.id = t.user_id 
                WHERE t.id = :id AND t.user_id = :agent_id');
                
            $ticketStmt->execute([':id' => $ticketId, ':agent_id' => $agentId]);
            $ticket = $ticketStmt->fetch(\PDO::FETCH_ASSOC);
            
            if (!$ticket) {
                $this->redirect('/b2b/agent/support?err=not_found');
                return;
            }
            
            // Get all replies
            $repliesStmt = $this->pdo->prepare('SELECT * FROM support_replies WHERE ticket_id = :ticket_id ORDER BY created_at ASC');
            $repliesStmt->execute([':ticket_id' => $ticketId]);
            $replies = $repliesStmt->fetchAll(\PDO::FETCH_ASSOC) ?: [];
            
            // Mark as read or update last viewed if needed
            // (Add this if you have a read/unread system)
            
        } catch (\Throwable $e) {
            error_log("Error in agentView: " . $e->getMessage());
            $this->redirect('/b2b/agent/support?err=load_failed');
            return;
        }
        
        $csrf = AgentGuard::generateCsrfToken('support_reply_' . $ticketId);
        $this->view('agent/support_view', [
            'title' => 'Ticket #' . $ticketId,
            'ticket' => $ticket,
            'replies' => $replies,
            'csrf' => $csrf
        ]);
    }

    // POST /b2b/agent/support (create ticket)
    public function agentSupportStore(): void
    {
        AgentGuard::requireLogin();
        $agent = $_SESSION['agent'] ?? null;
        $uid = (int)($agent['id'] ?? 0);
        $csrf = (string)($_POST['csrf_token'] ?? '');
        if (!AgentGuard::validateCsrf('support_create', $csrf)) { $this->redirect('/b2b/agent/support?err=csrf'); return; }

        $subject = trim((string)($_POST['subject'] ?? ''));
        $category = trim((string)($_POST['category'] ?? 'general'));
        $priority = trim((string)($_POST['priority'] ?? 'normal'));
        $message = trim((string)($_POST['message'] ?? ''));
        if ($subject === '' || $message === '') { $this->redirect('/b2b/agent/support?err=required'); return; }
        try {
            $st = $this->pdo->prepare("INSERT INTO support_tickets (user_id, subject, category, priority, message, status, created_at) VALUES (:uid,:sub,:cat,:pri,:msg,'open',NOW())");
            $st->execute([':uid'=>$uid, ':sub'=>$subject, ':cat'=>$category, ':pri'=>$priority, ':msg'=>$message]);
        } catch (\Throwable $_) { /* ignore */ }
        $this->redirect('/b2b/agent/support?ok=1');
    }

    // ===== Agent side =====

    // GET /b2b/agent/support
    public function agentSupportView(): void
    {
        AgentGuard::requireLogin();
        $status = trim((string)($_GET['status'] ?? ''));
        $where = [];$params=[];
        if ($status !== '') { $where[] = 't.status = :st'; $params[':st'] = $status; }
        $sql = 'SELECT t.*, u.email AS user_email FROM support_tickets t LEFT JOIN users u ON u.id = t.user_id';
        if ($where) { $sql .= ' WHERE ' . implode(' AND ', $where); }
        $sql .= " ORDER BY FIELD(t.status,'open','in_progress','resolved','closed'), COALESCE(t.updated_at, t.created_at) DESC LIMIT 200";
        $tickets = [];
        try { $st = $this->pdo->prepare($sql); $st->execute($params); $tickets = $st->fetchAll(\PDO::FETCH_ASSOC) ?: []; } catch (\Throwable $_) { $tickets = []; }
        $csrf = \App\Core\Security::csrfToken();
        $this->view('agent/support_view', ['title'=>'Support Tickets','tickets'=>$tickets,'csrf'=>$csrf]);
    }

    // ===== Admin side =====

    // GET /admin/support
    public function adminIndex(): void
    {
        Auth::requireRole(['Admin']);
        $status = trim((string)($_GET['status'] ?? ''));
        $where = [];$params=[];
        if ($status !== '') { $where[] = 't.status = :st'; $params[':st'] = $status; }
        $sql = 'SELECT t.*, u.email AS user_email FROM support_tickets t LEFT JOIN users u ON u.id = t.user_id';
        if ($where) { $sql .= ' WHERE ' . implode(' AND ', $where); }
        $sql .= " ORDER BY FIELD(t.status,'open','in_progress','resolved','closed'), COALESCE(t.updated_at, t.created_at) DESC LIMIT 200";
        $tickets = [];
        try { $st = $this->pdo->prepare($sql); $st->execute($params); $tickets = $st->fetchAll(\PDO::FETCH_ASSOC) ?: []; } catch (\Throwable $_) { $tickets = []; }
        $csrf = \App\Core\Security::csrfToken();
        $this->view('admin/support_index', ['title'=>'Support Tickets','tickets'=>$tickets,'csrf'=>$csrf]);
    }

    // GET /admin/support/view?id=
    public function adminView(): void
    {
        Auth::requireRole(['Admin']);
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/support'); return; }
        $ticket = null; $replies = [];
        try {
            $st = $this->pdo->prepare('SELECT t.*, u.email AS user_email FROM support_tickets t LEFT JOIN users u ON u.id=t.user_id WHERE t.id=:id');
            $st->execute([':id'=>$id]);
            $ticket = $st->fetch(\PDO::FETCH_ASSOC);
        } catch (\Throwable $_) { $ticket = null; }
        try {
            $sr = $this->pdo->prepare('SELECT * FROM support_replies WHERE ticket_id = :id ORDER BY id ASC');
            $sr->execute([':id'=>$id]);
            $replies = $sr->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $replies = []; }
        if (!$ticket) { $this->redirect('/admin/support'); return; }
        $csrf = \App\Core\Security::csrfToken();
        $this->view('admin/support_view', ['title'=>'Ticket #'.$id,'ticket'=>$ticket,'replies'=>$replies,'csrf'=>$csrf]);
    }

    // POST /admin/support/reply
    public function adminReply(): void
    {
        Auth::requireRole(['Admin']);
        \App\Core\Security::requireCsrf();
        $id = (int)($_POST['ticket_id'] ?? 0);
        $message = trim((string)($_POST['message'] ?? ''));
        $isNote = isset($_POST['is_note']) && $_POST['is_note'] === '1';
        
        if ($id <= 0 || $message === '') { 
            $this->redirect('/admin/support?err=invalid_input'); 
            return; 
        }
        
        try {
            // Start transaction
            $this->pdo->beginTransaction();
            
            // Store the reply
            $role = $isNote ? 'note' : 'admin';
            $ins = $this->pdo->prepare('INSERT INTO support_replies (ticket_id, user_id, role, message, created_at) 
                VALUES (:tid, :uid, :role, :msg, NOW())');
                
            $ins->execute([
                ':tid' => $id, 
                ':uid' => $_SESSION['user_id'] ?? null,
                ':role' => $role,
                ':msg' => $message
            ]);
            $replyId = (int)$this->pdo->lastInsertId();
            
            // Update ticket timestamps and status
            $statusUpdate = $isNote ? '' : ", status = IF(status='open','in_progress',status)";
            $this->pdo->prepare("UPDATE support_tickets 
                SET updated_at = NOW(), 
                    last_replied_at = IF(:is_note, last_replied_at, NOW())
                    {$statusUpdate}
                WHERE id = :id")
                ->execute([
                    ':id' => $id,
                    ':is_note' => $isNote ? 1 : 0
                ]);
                
            $this->pdo->commit();
            
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) {
                $this->pdo->rollBack();
            }
            error_log("Error in adminReply: " . $e->getMessage());
            $accept = (string)($_SERVER['HTTP_ACCEPT'] ?? '');
            $xhr = (string)($_SERVER['HTTP_X_REQUESTED_WITH'] ?? '');
            $isAjax = (stripos($accept, 'application/json') !== false) || (strcasecmp($xhr, 'XMLHttpRequest') === 0);
            if ($isAjax) {
                header('Content-Type: application/json');
                http_response_code(500);
                echo json_encode(['error' => 'reply_failed']);
                return;
            } else {
                $this->redirect('/admin/support/view?id='.$id.'&err=reply_failed');
                return;
            }
        }
        $accept = (string)($_SERVER['HTTP_ACCEPT'] ?? '');
        $xhr = (string)($_SERVER['HTTP_X_REQUESTED_WITH'] ?? '');
        $isAjax = (stripos($accept, 'application/json') !== false) || (strcasecmp($xhr, 'XMLHttpRequest') === 0);
        if ($isAjax) {
            header('Content-Type: application/json');
            echo json_encode([
                'ok' => true,
                'reply' => [
                    'id' => $replyId,
                    'ticket_id' => $id,
                    'role' => $role,
                    'message' => $message,
                    'created_at' => date('Y-m-d H:i:s')
                ]
            ]);
            return;
        }
        $this->redirect('/admin/support/view?id='.$id.'#reply-latest');
    }
    
    // POST /b2b/agent/support/status - Update ticket status (agent)
    public function agentStatus(): void
    {
        AgentGuard::requireLogin();
        $agent = $_SESSION['agent'] ?? null;
        $agentId = (int)($agent['id'] ?? 0);
        
        $ticketId = (int)($_POST['ticket_id'] ?? 0);
        $status = trim((string)($_POST['status'] ?? ''));
        
        if ($ticketId <= 0 || !in_array($status, ['open', 'in_progress', 'resolved', 'closed'])) {
            $this->redirect('/b2b/agent/support?err=invalid_input');
            return;
        }
        
        try {
            // Verify the agent has access to this ticket
            $ticketStmt = $this->pdo->prepare('SELECT id FROM support_tickets WHERE id = :id AND user_id = :user_id');
            $ticketStmt->execute([':id' => $ticketId, ':user_id' => $agentId]);
            $ticket = $ticketStmt->fetch(\PDO::FETCH_ASSOC);
            
            if (!$ticket) {
                $this->redirect('/b2b/agent/support?err=not_found');
                return;
            }
            
            // Update ticket status
            $this->pdo->prepare('UPDATE support_tickets SET status = :status, updated_at = NOW() WHERE id = :id')
                ->execute([':status' => $status, ':id' => $ticketId]);
                
            // Add a system note about the status change
            $statusText = ucfirst(str_replace('_', ' ', $status));
            $this->pdo->prepare('INSERT INTO support_replies (ticket_id, role, message, created_at) VALUES (:tid, "system", :msg, NOW())')
                ->execute([
                    ':tid' => $ticketId,
                    ':msg' => "Status changed to: {$statusText}"
                ]);
                
        } catch (\Throwable $e) {
            error_log("Error in agentStatus: " . $e->getMessage());
            $this->redirect('/b2b/agent/support/view?id='.$ticketId.'&err=update_failed');
            return;
        }
        
        $this->redirect('/b2b/agent/support/view?id='.$ticketId.'#reply-latest');
    }

    // GET /b2b/agent/support/replies?ticket_id=&since_id=
    public function agentRepliesJson(): void
    {
        header('Content-Type: application/json');
        AgentGuard::requireLogin();
        $agent = $_SESSION['agent'] ?? null;
        $agentId = (int)($agent['id'] ?? 0);
        $ticketId = (int)($_GET['ticket_id'] ?? 0);
        $sinceId = (int)($_GET['since_id'] ?? 0);
        if ($ticketId <= 0) { echo json_encode(['ok'=>true,'replies'=>[]]); return; }
        try {
            // Ensure ownership
            $chk = $this->pdo->prepare('SELECT id FROM support_tickets WHERE id=:id AND user_id=:uid');
            $chk->execute([':id'=>$ticketId, ':uid'=>$agentId]);
            if (!$chk->fetch(\PDO::FETCH_ASSOC)) { http_response_code(403); echo json_encode(['error'=>'forbidden']); return; }
            // Fetch new replies
            $sql = 'SELECT id, ticket_id, role, message, created_at FROM support_replies WHERE ticket_id = :tid';
            $params = [':tid'=>$ticketId];
            if ($sinceId > 0) { $sql .= ' AND id > :sid'; $params[':sid'] = $sinceId; }
            $sql .= ' ORDER BY id ASC LIMIT 100';
            $st = $this->pdo->prepare($sql);
            $st->execute($params);
            $rows = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
            echo json_encode(['ok'=>true,'replies'=>$rows]);
        } catch (\Throwable $e) {
            http_response_code(500);
            echo json_encode(['error'=>'server_error']);
        }
    }
    
    // POST /b2b/agent/support/reply - Agent reply to ticket
    public function agentReply(): void
    {
        AgentGuard::requireLogin();
        $agent = $_SESSION['agent'] ?? null;
        $agentId = (int)($agent['id'] ?? 0);
        
        $ticketId = (int)($_POST['ticket_id'] ?? 0);
        $message = trim((string)($_POST['message'] ?? ''));
        
        if ($ticketId <= 0 || $message === '') {
            $this->redirect('/b2b/agent/support?err=invalid_input');
            return;
        }
        
        try {
            // Verify the agent has access to this ticket
            $ticketStmt = $this->pdo->prepare('SELECT id FROM support_tickets WHERE id = :id AND user_id = :user_id');
            $ticketStmt->execute([':id' => $ticketId, ':user_id' => $agentId]);
            $ticket = $ticketStmt->fetch(\PDO::FETCH_ASSOC);
            
            if (!$ticket) {
                $this->redirect('/b2b/agent/support?err=not_found');
                return;
            }
            
            // Start transaction
            $this->pdo->beginTransaction();
            
            // Store the agent's reply
            $ins = $this->pdo->prepare('INSERT INTO support_replies (ticket_id, agent_id, role, message, created_at) 
                VALUES (:tid, :aid, "agent", :msg, NOW())');
                
            $ins->execute([
                ':tid' => $ticketId, 
                ':aid' => $agentId,
                ':msg' => $message
            ]);
            $replyId = (int)$this->pdo->lastInsertId();
            
            // Update ticket timestamps and status
            $this->pdo->prepare("UPDATE support_tickets 
                SET updated_at = NOW(), 
                    last_replied_at = NOW(),
                    status = IF(status='open','in_progress',status)
                WHERE id = :id")
                ->execute([':id' => $ticketId]);
                
            $this->pdo->commit();
            
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) {
                $this->pdo->rollBack();
            }
            error_log("Error in agentReply: " . $e->getMessage());
            $accept = (string)($_SERVER['HTTP_ACCEPT'] ?? '');
            $xhr = (string)($_SERVER['HTTP_X_REQUESTED_WITH'] ?? '');
            $isAjax = (stripos($accept, 'application/json') !== false) || (strcasecmp($xhr, 'XMLHttpRequest') === 0);
            if ($isAjax) {
                header('Content-Type: application/json');
                http_response_code(500);
                echo json_encode(['error' => 'reply_failed']);
                return;
            } else {
                $this->redirect('/b2b/agent/support/view?id='.$ticketId.'&err=reply_failed');
                return;
            }
        }
        // AJAX JSON
        $accept = (string)($_SERVER['HTTP_ACCEPT'] ?? '');
        $xhr = (string)($_SERVER['HTTP_X_REQUESTED_WITH'] ?? '');
        $isAjax = (stripos($accept, 'application/json') !== false) || (strcasecmp($xhr, 'XMLHttpRequest') === 0);
        if ($isAjax) {
            header('Content-Type: application/json');
            echo json_encode([
                'ok' => true,
                'reply' => [
                    'id' => $replyId,
                    'ticket_id' => $ticketId,
                    'role' => 'agent',
                    'message' => $message,
                    'created_at' => date('Y-m-d H:i:s')
                ]
            ]);
            return;
        }
        $this->redirect('/b2b/agent/support/view?id='.$ticketId.'#reply-latest');
    }

    // POST /admin/support/status
    public function adminStatus(): void
    {
        Auth::requireRole(['Admin']);
        \App\Core\Security::requireCsrf();
        $id = (int)($_POST['ticket_id'] ?? 0);
        $status = (string)($_POST['status'] ?? 'open');
        if ($id <= 0) { $this->redirect('/admin/support'); return; }
        try { $this->pdo->prepare('UPDATE support_tickets SET status=:s, updated_at=NOW() WHERE id=:id')->execute([':s'=>$status, ':id'=>$id]); } catch (\Throwable $_) {}
        $this->redirect('/admin/support/view?id='.$id);
    }

    // GET /admin/support/replies?ticket_id=&since_id=
    public function adminRepliesJson(): void
    {
        header('Content-Type: application/json');
        Auth::requireRole(['Admin']);
        $ticketId = (int)($_GET['ticket_id'] ?? 0);
        $sinceId = (int)($_GET['since_id'] ?? 0);
        if ($ticketId <= 0) { echo json_encode(['ok'=>true,'replies'=>[]]); return; }
        try {
            $sql = 'SELECT id, ticket_id, role, message, created_at FROM support_replies WHERE ticket_id = :tid';
            $params = [':tid'=>$ticketId];
            if ($sinceId > 0) { $sql .= ' AND id > :sid'; $params[':sid'] = $sinceId; }
            $sql .= ' ORDER BY id ASC LIMIT 100';
            $st = $this->pdo->prepare($sql);
            $st->execute($params);
            $rows = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
            echo json_encode(['ok'=>true,'replies'=>$rows]);
        } catch (\Throwable $e) {
            http_response_code(500);
            echo json_encode(['error'=>'server_error']);
        }
    }
}
