<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use Stripe\StripeClient;

class PaymentsController extends Controller
{
    // POST /b2b/api/checkout/pay-gateway
    // Body JSON: { order_id: number, gateway: 'stripe' }
    public function payGateway(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');

        $raw = file_get_contents('php://input') ?: '';
        $body = json_decode($raw, true) ?: [];
        $orderId = (int)($body['order_id'] ?? 0);
        $gateway = strtolower(trim((string)($body['gateway'] ?? 'stripe')));
        if ($orderId <= 0 || $gateway !== 'stripe') {
            http_response_code(422);
            echo json_encode(['error' => 'order_id required and gateway must be stripe']);
            return;
        }
        // Load gateway config
        $cfgPath = dirname(__DIR__, 2) . '/config/payment_gateways.php';
        $cfg = file_exists($cfgPath) ? (require $cfgPath) : [];
        if (empty($cfg['stripe']['enabled']) || empty($cfg['stripe']['secret_key'])) {
            http_response_code(503);
            echo json_encode(['error' => 'Stripe is disabled or not configured']);
            return;
        }
        $stripeSecret = (string)$cfg['stripe']['secret_key'];
        $currency = strtoupper((string)($cfg['stripe']['currency'] ?? 'THB'));
        $appCfg = require dirname(__DIR__, 2) . '/config/app.php';
        $appUrl = rtrim((string)($appCfg['url'] ?? 'http://localhost:8000'), '/');

        // Ensure order belongs to user and is payable
        $user = $_SESSION['user'] ?? null;
        if (!$user) { http_response_code(401); echo json_encode(['error' => 'Unauthorized']); return; }
        $st = $this->pdo->prepare('SELECT id, user_id, total_amount, currency, payment_status FROM orders WHERE id = :id AND user_id = :uid LIMIT 1');
        $st->execute(['id' => $orderId, 'uid' => (int)$user['id']]);
        $order = $st->fetch();
        if (!$order) { http_response_code(404); echo json_encode(['error' => 'Order not found']); return; }
        if (in_array($order['payment_status'] ?? 'pending', ['paid','failed','canceled'], true)) {
            http_response_code(409); echo json_encode(['error' => 'Order not payable']); return;
        }

        $amount = (float)($order['total_amount'] ?? 0);
        $ordCurrency = strtoupper((string)($order['currency'] ?? $currency));
        if ($amount <= 0) { http_response_code(422); echo json_encode(['error' => 'Invalid order amount']); return; }

        // Convert to minor units for Stripe depending on zero-decimal currency
        $zeroDecimal = in_array($ordCurrency, ['JPY','KRW'], true);
        $unitAmount = $zeroDecimal ? (int)round($amount) : (int)round($amount * 100);

        // Create a pending payment row with idempotency key
        $idem = 'pay_' . bin2hex(random_bytes(12));
        $ins = $this->pdo->prepare('INSERT INTO payments (order_id, user_id, gateway, amount, currency, status, idempotency_key) VALUES (:oid, :uid, :gw, :amt, :cur, "pending", :idem)');
        $ins->execute([
            'oid' => $orderId,
            'uid' => (int)$user['id'],
            'gw'  => 'stripe',
            'amt' => $amount,
            'cur' => $ordCurrency,
            'idem'=> $idem,
        ]);
        $paymentId = (int)$this->pdo->lastInsertId();

        // Prepare success/cancel URLs (signed with APP_KEY like CheckoutController)
        $appKey = function_exists('env') ? (string)env('APP_KEY', '') : '';
        if ($appKey === '') { $appKey = getenv('APP_KEY') ?: 'devkey'; }
        $sig = hash_hmac('sha256', $orderId . ':' . (int)$user['id'], $appKey);
        $successUrl = $appUrl . '/b2b/checkout/confirm?id=' . $orderId . '&sig=' . $sig;
        $cancelUrl  = $appUrl . '/b2b/checkout/pay?id=' . $orderId . '&sig=' . $sig;

        // Create Stripe Checkout Session via official SDK
        try {
            $stripe = new StripeClient($stripeSecret);
            $session = $stripe->checkout->sessions->create([
                'mode' => 'payment',
                'success_url' => $successUrl . '&src=stripe',
                'cancel_url' => $cancelUrl . '&src=stripe',
                'line_items' => [[
                    'quantity' => 1,
                    'price_data' => [
                        'currency' => $ordCurrency,
                        'product_data' => [ 'name' => 'Order #' . $orderId ],
                        'unit_amount' => $unitAmount,
                    ],
                ]],
                'client_reference_id' => (string)$orderId,
                'metadata' => [
                    'order_id' => (string)$orderId,
                    'payment_id' => (string)$paymentId,
                ],
            ], [
                'idempotency_key' => $idem,
            ]);

            if (!$session || empty($session->id) || empty($session->url)) {
                throw new \RuntimeException('Invalid session from Stripe');
            }

            $raw = json_encode($session);
            $upd = $this->pdo->prepare('UPDATE payments SET provider_ref = :ref, session_url = :url, raw_session = :raw WHERE id = :id');
            $upd->execute(['ref' => $session->id, 'url' => $session->url, 'raw' => $raw, 'id' => $paymentId]);

            echo json_encode(['redirect_url' => $session->url, 'payment_id' => $paymentId]);
        } catch (\Throwable $e) {
            $upd = $this->pdo->prepare('UPDATE payments SET status="failed", raw_session = :raw WHERE id = :id');
            $upd->execute(['raw' => json_encode(['error' => $e->getMessage()]), 'id' => $paymentId]);
            http_response_code(502);
            echo json_encode(['error' => 'Stripe session failed']);
            return;
        }
    }

    // POST /b2b/api/taxi/pay-gateway
    // Body JSON: { booking_id?: number, intent_id?: number, gateway: 'stripe' }
    public function taxiPayGateway(): void
    {
        header('Content-Type: application/json');
        // Agent session required
        $agent = $_SESSION['agent'] ?? null;
        if (!$agent || empty($agent['id'])) { http_response_code(401); echo json_encode(['error'=>'Unauthorized']); return; }

        $raw = file_get_contents('php://input') ?: '';
        $body = json_decode($raw, true) ?: [];
        $bookingId = (int)($body['booking_id'] ?? 0);
        $intentId  = (int)($body['intent_id'] ?? 0);
        $gateway = strtolower(trim((string)($body['gateway'] ?? 'stripe')));
        if (($bookingId <= 0 && $intentId <= 0) || $gateway !== 'stripe') { http_response_code(422); echo json_encode(['error'=>'booking_id or intent_id required and gateway must be stripe']); return; }

        // Load gateway config
        $cfgPath = dirname(__DIR__, 2) . '/config/payment_gateways.php';
        $cfg = file_exists($cfgPath) ? (require $cfgPath) : [];
        if (empty($cfg['stripe']['enabled']) || empty($cfg['stripe']['secret_key'])) {
            http_response_code(503);
            echo json_encode(['error' => 'Stripe is disabled or not configured']);
            return;
        }
        $stripeSecret = (string)$cfg['stripe']['secret_key'];
        $appCfg = require dirname(__DIR__, 2) . '/config/app.php';
        $appUrl = rtrim((string)($appCfg['url'] ?? 'http://localhost:8000'), '/');

        // Determine amount/currency from booking or intent
        $amount = 0.0; $bkCurrency = 'THB';
        $meta = [];
        if ($bookingId > 0) {
            $bst = $this->pdo->prepare('SELECT id, agent_id, amount_total, currency FROM taxi_bookings WHERE id=:id LIMIT 1');
            $bst->execute([':id'=>$bookingId]);
            $bk = $bst->fetch();
            if (!$bk || (int)$bk['agent_id'] !== (int)$agent['id']) { http_response_code(404); echo json_encode(['error'=>'Booking not found']); return; }
            $amount = (float)($bk['amount_total'] ?? 0);
            $bkCurrency = strtoupper((string)($bk['currency'] ?? 'THB'));
            $meta['taxi_booking_id'] = (string)$bookingId;
        } else {
            $ist = $this->pdo->prepare('SELECT id, agent_id, amount, currency FROM taxi_payment_intents WHERE id=:id AND status=\'pending\' LIMIT 1');
            $ist->execute([':id'=>$intentId]);
            $in = $ist->fetch();
            if (!$in || (int)$in['agent_id'] !== (int)$agent['id']) { http_response_code(404); echo json_encode(['error'=>'Intent not found']); return; }
            $amount = (float)($in['amount'] ?? 0);
            $bkCurrency = strtoupper((string)($in['currency'] ?? 'THB'));
            $meta['taxi_intent_id'] = (string)$intentId;
        }
        if ($amount <= 0) { http_response_code(422); echo json_encode(['error'=>'Invalid booking amount']); return; }

        $zeroDecimal = in_array($bkCurrency, ['JPY','KRW'], true);
        $unitAmount = $zeroDecimal ? (int)round($amount) : (int)round($amount * 100);

        $idem = ($bookingId>0? 'txb_' : 'txi_') . bin2hex(random_bytes(12));
        try {
            $stripe = new StripeClient($stripeSecret);
            $successUrl = $bookingId>0
                ? $appUrl . '/b2b/agent/taxi/voucher?id=' . $bookingId . '&src=stripe'
                : $appUrl . '/b2b/agent/taxi/payment?intent_id=' . $intentId . '&src=stripe';
            $cancelUrl  = $bookingId>0
                ? $appUrl . '/b2b/agent/taxi/payment?id=' . $bookingId . '&src=stripe'
                : $appUrl . '/b2b/agent/taxi/payment?intent_id=' . $intentId . '&src=stripe';

            $session = $stripe->checkout->sessions->create([
                'mode' => 'payment',
                'success_url' => $successUrl,
                'cancel_url'  => $cancelUrl,
                'line_items' => [[
                    'quantity' => 1,
                    'price_data' => [
                        'currency' => $bkCurrency,
                        'product_data' => [ 'name' => 'Taxi Booking' ],
                        'unit_amount' => $unitAmount,
                    ],
                ]],
                'client_reference_id' => (string)($bookingId > 0 ? $bookingId : $intentId),
                'metadata' => $meta,
            ], [ 'idempotency_key' => $idem ]);

            if (!$session || empty($session->id) || empty($session->url)) { throw new \RuntimeException('Invalid session from Stripe'); }

            // Store session reference
            if ($bookingId > 0) {
                try {
                    $insPay = $this->pdo->prepare('INSERT INTO taxi_payments (booking_id, method, gateway_name, amount, currency, status, idempotency_key, provider_ref, session_url, raw_session, created_at, updated_at) VALUES (:bid, "gateway", "stripe", :amt, :cur, "pending", :idem, :ref, :url, :raw, NOW(), NOW())');
                    $insPay->execute([':bid'=>$bookingId, ':amt'=>$amount, ':cur'=>$bkCurrency, ':idem'=>$idem, ':ref'=>$session->id, ':url'=>$session->url, ':raw'=>json_encode($session)]);
                } catch (\Throwable $e) { /* ignore */ }
            } else {
                try {
                    $this->pdo->prepare('UPDATE taxi_payment_intents SET gateway_name="stripe", gateway_session_id=:sid, updated_at=NOW() WHERE id=:id')->execute([':sid'=>$session->id, ':id'=>$intentId]);
                } catch (\Throwable $e) { /* ignore */ }
            }

            echo json_encode(['redirect_url' => $session->url]);
        } catch (\Throwable $e) {
            http_response_code(502);
            echo json_encode(['error'=>'Stripe session failed']);
            return;
        }
    }

    // POST /b2b/api/checkout/yacht-pay-gateway
    // Body JSON: { booking_id: number, gateway: 'stripe' }
    public function yachtPayGateway(): void
    {
        header('Content-Type: application/json');
        // Agent session required
        $agent = $_SESSION['agent'] ?? null;
        if (!$agent || empty($agent['id'])) { http_response_code(401); echo json_encode(['error'=>'Unauthorized']); return; }

        $raw = file_get_contents('php://input') ?: '';
        $body = json_decode($raw, true) ?: [];
        $bookingId = (int)($body['booking_id'] ?? 0);
        $intentId  = (int)($body['intent_id'] ?? 0);
        $gateway = strtolower(trim((string)($body['gateway'] ?? 'stripe')));
        if (($bookingId <= 0 && $intentId <= 0) || $gateway !== 'stripe') { http_response_code(422); echo json_encode(['error'=>'booking_id or intent_id required and gateway must be stripe']); return; }

        // Load gateway config
        $cfgPath = dirname(__DIR__, 2) . '/config/payment_gateways.php';
        $cfg = file_exists($cfgPath) ? (require $cfgPath) : [];
        if (empty($cfg['stripe']['enabled']) || empty($cfg['stripe']['secret_key'])) {
            http_response_code(503);
            echo json_encode(['error' => 'Stripe is disabled or not configured']);
            return;
        }
        $stripeSecret = (string)$cfg['stripe']['secret_key'];
        $currency = strtoupper((string)($cfg['stripe']['currency'] ?? 'THB'));
        $appCfg = require dirname(__DIR__, 2) . '/config/app.php';
        $appUrl = rtrim((string)($appCfg['url'] ?? 'http://localhost:8000'), '/');

        // Determine amount/currency from booking or intent
        $amount = 0.0; $bkCurrency = 'THB';
        $meta = [];
        if ($bookingId > 0) {
            $bst = $this->pdo->prepare('SELECT id, agent_id, amount_total, currency FROM yacht_bookings WHERE id=:id LIMIT 1');
            $bst->execute([':id'=>$bookingId]);
            $bk = $bst->fetch();
            if (!$bk || (int)$bk['agent_id'] !== (int)$agent['id']) { http_response_code(404); echo json_encode(['error'=>'Booking not found']); return; }
            $amount = (float)($bk['amount_total'] ?? 0);
            $bkCurrency = strtoupper((string)($bk['currency'] ?? 'THB'));
            $meta['yacht_booking_id'] = (string)$bookingId;
        } else {
            $ist = $this->pdo->prepare('SELECT id, agent_id, amount, currency FROM yacht_payment_intents WHERE id=:id AND status=\'pending\' LIMIT 1');
            $ist->execute([':id'=>$intentId]);
            $in = $ist->fetch();
            if (!$in || (int)$in['agent_id'] !== (int)$agent['id']) { http_response_code(404); echo json_encode(['error'=>'Intent not found']); return; }
            $amount = (float)($in['amount'] ?? 0);
            $bkCurrency = strtoupper((string)($in['currency'] ?? 'THB'));
            $meta['yacht_intent_id'] = (string)$intentId;
        }
        if ($amount <= 0) { http_response_code(422); echo json_encode(['error'=>'Invalid booking amount']); return; }

        // Convert to minor units for Stripe depending on zero-decimal currency
        $zeroDecimal = in_array($bkCurrency, ['JPY','KRW'], true);
        $unitAmount = $zeroDecimal ? (int)round($amount) : (int)round($amount * 100);

        // Idempotency for session creation
        $idem = ($bookingId>0? 'ypb_' : 'ypi_') . bin2hex(random_bytes(12));
        $ins = $this->pdo->prepare('INSERT INTO yacht_payments (booking_id, method, gateway_name, amount, currency, status, idempotency_key) VALUES (:bid, :m, :g, :amt, :cur, "pending", :idem)');
        $ins->execute([
            'bid' => $bookingId,
            'm'   => 'gateway',
            'g'   => 'stripe',
            'amt' => $amount,
            'cur' => $bkCurrency,
            'idem'=> $idem,
        ]);
        $yachtPaymentId = (int)$this->pdo->lastInsertId();

        // Prepare success/cancel URLs (signed with APP_KEY like CheckoutController)
        $appKey = function_exists('env') ? (string)env('APP_KEY', '') : '';
        if ($appKey === '') { $appKey = getenv('APP_KEY') ?: 'devkey'; }
        $sig = hash_hmac('sha256', $bookingId . ':' . (int)$agent['id'], $appKey);
        if ($bookingId > 0) {
            $successUrl = $appUrl . '/b2b/agent/yacht/voucher?id=' . $bookingId . '&src=stripe';
            $cancelUrl  = $appUrl . '/b2b/agent/yacht/payment?id=' . $bookingId . '&src=stripe';
        } else {
            $successUrl = $appUrl . '/b2b/agent/yacht/payment?intent_id=' . $intentId . '&src=stripe';
            $cancelUrl  = $appUrl . '/b2b/agent/yacht/payment?intent_id=' . $intentId . '&src=stripe';
        }

        // Create Stripe Checkout Session via official SDK
        try {
            $stripe = new StripeClient($stripeSecret);
            $session = $stripe->checkout->sessions->create([
                'mode' => 'payment',
                'success_url' => $successUrl . '&src=stripe',
                'cancel_url' => $cancelUrl . '&src=stripe',
                'line_items' => [[
                    'quantity' => 1,
                    'price_data' => [
                        'currency' => $bkCurrency,
                        'product_data' => [ 'name' => 'Booking #' . $bookingId ],
                        'unit_amount' => $unitAmount,
                    ],
                ]],
                'client_reference_id' => (string)($bookingId > 0 ? $bookingId : $intentId),
                'metadata' => $meta,
            ], [
                'idempotency_key' => $idem,
            ]);

            if (!$session || empty($session->id) || empty($session->url)) { throw new \RuntimeException('Invalid session from Stripe'); }
            $raw = json_encode($session);
            if ($bookingId > 0) {
                // Optional: record on booking payments table if pre-existing booking flow
                try {
                    $insPay = $this->pdo->prepare('INSERT INTO yacht_payments (booking_id, method, gateway_name, amount, currency, status, idempotency_key, provider_ref, session_url, raw_session, created_at, updated_at) VALUES (:bid, :m, :g, :amt, :cur, "pending", :idem, :ref, :url, :raw, NOW(), NOW())');
                    $insPay->execute([':bid'=>$bookingId, ':m'=>'gateway', ':g'=>'stripe', ':amt'=>$amount, ':cur'=>$bkCurrency, ':idem'=>$idem, ':ref'=>$session->id, ':url'=>$session->url, ':raw'=>$raw]);
                } catch (\Throwable $e) { /* ignore */ }
            } else {
                // Store session id on intent for traceability
                try {
                    $this->pdo->prepare('UPDATE yacht_payment_intents SET gateway_name="stripe", gateway_session_id=:sid, updated_at=NOW() WHERE id=:id')->execute([':sid'=>$session->id, ':id'=>$intentId]);
                } catch (\Throwable $e) { /* ignore */ }
            }
            echo json_encode(['redirect_url' => $session->url, 'payment_id' => $yachtPaymentId ?? 0]);
        } catch (\Throwable $e) {
            if (!empty($yachtPaymentId)) {
                $upd = $this->pdo->prepare('UPDATE yacht_payments SET status="failed", raw_session=:raw WHERE id=:id');
                $upd->execute([':raw'=>json_encode(['error'=>$e->getMessage()]), ':id'=>$yachtPaymentId]);
            }
            http_response_code(502);
            echo json_encode(['error'=>'Stripe session failed']);
            return;
        }
    }

    // POST /webhook/stripe
    public function stripeWebhook(): void
    {
        // Stripe sends application/json
        $payload = file_get_contents('php://input') ?: '';
        $sigHeader = $_SERVER['HTTP_STRIPE_SIGNATURE'] ?? '';
        $cfgPath = dirname(__DIR__, 2) . '/config/payment_gateways.php';
        $cfg = file_exists($cfgPath) ? (require $cfgPath) : [];
        $secret = (string)($cfg['stripe']['webhook_secret'] ?? '');
        if ($secret === '') { http_response_code(500); echo 'whsec not set'; return; }
        
        // Verify signature using Stripe SDK
        try {
            $event = \Stripe\Webhook::constructEvent($payload, $sigHeader, $secret);
        } catch (\UnexpectedValueException $e) {
            http_response_code(400); echo 'invalid payload'; return;
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            http_response_code(400); echo 'invalid signature'; return;
        }

        $event = is_array($event) ? $event : json_decode(json_encode($event), true);
        $eventId = (string)($event['id'] ?? '');
        $type = (string)($event['type'] ?? '');

        // Idempotency: ignore if already processed
        if ($eventId !== '') {
            $exists = $this->pdo->prepare('SELECT id FROM payments WHERE event_id = :eid LIMIT 1');
            $exists->execute(['eid' => $eventId]);
            if ($exists->fetch()) { http_response_code(200); echo 'ok'; return; }
        }

        if ($type === 'checkout.session.completed') {
            $session = $event['data']['object'] ?? [];
            $sessionId = (string)($session['id'] ?? '');
            $orderId = isset($session['metadata']['order_id']) ? (int)$session['metadata']['order_id'] : null;
            $paymentId = isset($session['metadata']['payment_id']) ? (int)$session['metadata']['payment_id'] : null;
            $yachtBookingId = isset($session['metadata']['yacht_booking_id']) ? (int)$session['metadata']['yacht_booking_id'] : null;
            $yachtIntentId = isset($session['metadata']['yacht_intent_id']) ? (int)$session['metadata']['yacht_intent_id'] : null;
            $taxiBookingId = isset($session['metadata']['taxi_booking_id']) ? (int)$session['metadata']['taxi_booking_id'] : null;
            $taxiIntentId = isset($session['metadata']['taxi_intent_id']) ? (int)$session['metadata']['taxi_intent_id'] : null;

            if ($sessionId === '') { http_response_code(200); echo 'ok'; return; }

            // Checkout order flow
            if ($orderId && $paymentId) {
                $upd = $this->pdo->prepare('UPDATE payments SET status = "succeeded", provider_ref = :ref, event_id = :eid, raw_event = :raw WHERE id = :pid AND gateway = "stripe"');
                $upd->execute(['ref' => $sessionId, 'eid' => $eventId, 'raw' => $payload, 'pid' => $paymentId]);
                $uo = $this->pdo->prepare('UPDATE orders SET payment_status = "paid", payment_gateway = "stripe" WHERE id = :oid');
                $uo->execute(['oid' => $orderId]);
                http_response_code(200); echo 'ok'; return;
            }

            // Taxi booking flow - existing booking path
            if ($taxiBookingId) {
                try {
                    $ub = $this->pdo->prepare('UPDATE taxi_bookings SET status="confirmed", payment_status="paid", payment_method="gateway", gateway_name="stripe", payment_txn_id=:tx, paid_at=NOW() WHERE id=:bid');
                    $ub->execute(['tx'=>$sessionId, 'bid'=>$taxiBookingId]);
                    $p = $this->pdo->prepare('INSERT INTO taxi_payments (booking_id, method, gateway_name, amount, currency, status, provider_ref, event_id, raw_event, created_at, updated_at) VALUES (:bid, "gateway", "stripe", 0, "THB", "captured", :ref, :eid, :raw, NOW(), NOW())');
                    $p->execute([':bid'=>$taxiBookingId, ':ref'=>$sessionId, ':eid'=>$eventId, ':raw'=>$payload]);

                    // Assign vendor based on taxi.vendor_id so booking shows in vendor portal
                    try {
                        $stTaxi = $this->pdo->prepare('SELECT t.vendor_id FROM taxi_bookings b JOIN taxis t ON t.id=b.taxi_id WHERE b.id=:id');
                        $stTaxi->execute([':id'=>$taxiBookingId]);
                        $vid = (int)($stTaxi->fetchColumn() ?: 0);
                        if ($vid > 0) {
                            $this->pdo->prepare('UPDATE taxi_bookings SET vendor_id = :vid, vendor_assigned_at = IFNULL(vendor_assigned_at, NOW()) WHERE id = :id')->execute([':vid'=>$vid, ':id'=>$taxiBookingId]);
                        }
                    } catch (\Throwable $e) { /* ignore vendor assignment */ }
                } catch (\Throwable $e) { /* ignore */ }
                http_response_code(200); echo 'ok'; return;
            }

            // Taxi intent flow - create booking now
            if ($taxiIntentId) {
                $si = $this->pdo->prepare('SELECT * FROM taxi_payment_intents WHERE id=:id LIMIT 1');
                $si->execute([':id'=>$taxiIntentId]);
                $in = $si->fetch(\PDO::FETCH_ASSOC) ?: null;
                if ($in && (string)($in['status'] ?? '') === 'pending') {
                    $agentId = (int)($in['agent_id'] ?? 0);
                    $taxiId = (int)($in['taxi_id'] ?? 0);
                    $tripDate = (string)($in['trip_date'] ?? '');
                    $pickupTime = (string)($in['pickup_time'] ?? '');
                    $flightNo = (string)($in['flight_no'] ?? '');
                    $notes = (string)($in['notes'] ?? '');
                    $pax = (int)($in['pax'] ?? 1);
                    $from = (string)($in['from_text'] ?? '');
                    $to = (string)($in['to_text'] ?? '');
                    $amount = (float)($in['amount'] ?? 0);
                    $currency = (string)($in['currency'] ?? 'THB');
                    // Compute vendor_cost using taxi_pricing.vendor_cost (fallback vendor_price, then base_fare) + extras from itinerary_json
                    $vendorCost = 0.0; $perKmRate = 0.0; $vendorBase = 0.0;
                    try {
                        $pp = $this->pdo->prepare('SELECT vendor_cost, vendor_price, base_fare, per_km FROM taxi_pricing WHERE taxi_id=:id ORDER BY id DESC LIMIT 1');
                        $pp->execute([':id'=>$taxiId]);
                        $pr = $pp->fetch(\PDO::FETCH_ASSOC) ?: [];
                        $vendorBase = (float)(($pr['vendor_cost'] ?? null) !== null ? $pr['vendor_cost'] : (($pr['vendor_price'] ?? null) !== null ? $pr['vendor_price'] : ($pr['base_fare'] ?? 0)));
                        $perKmRate = (float)($pr['per_km'] ?? 0);
                    } catch (\Throwable $e) { /* ignore */ }
                    $vendorCost = $vendorBase;
                    $itineraryJson = (string)($in['itinerary_json'] ?? '');
                    if ($itineraryJson !== '') {
                        $stops = json_decode($itineraryJson, true);
                        if (is_array($stops)) {
                            $includeKm = 7;
                            foreach ($stops as $sp) {
                                $addr = trim((string)($sp['address'] ?? ''));
                                if ($addr === '') { break; }
                                $vendorCost += (float)($sp['extra_price'] ?? 0);
                                if ($perKmRate > 0) {
                                    $dkm = isset($sp['distance_km']) ? (float)$sp['distance_km'] : 0.0;
                                    if ($dkm > $includeKm) { $vendorCost += ($dkm - $includeKm) * $perKmRate; }
                                }
                            }
                        }
                    }
                    try {
                        $insB = $this->pdo->prepare('INSERT INTO taxi_bookings (agent_id, taxi_id, trip_date, pickup_time, flight_no, notes, pax, from_text, to_text, amount_total, vendor_cost, vendor_currency, currency, status, payment_status, payment_method, gateway_name, payment_txn_id, paid_at, channel, created_by, updated_by, created_at, updated_at) VALUES (:agent_id,:taxi_id,:trip_date,:pickup_time,:flight_no,:notes,:pax,:from,:to,:amount,:vendor_cost,\'THB\',:currency,\'confirmed\',\'paid\',\'gateway\',\'stripe\',:tx,NOW(),\'portal\',:uid,:uid,NOW(),NOW())');
                        $insB->execute([':agent_id'=>$agentId, ':taxi_id'=>$taxiId, ':trip_date'=>$tripDate, ':pickup_time'=>$pickupTime, ':flight_no'=>$flightNo, ':notes'=>$notes, ':pax'=>$pax, ':from'=>$from, ':to'=>$to, ':amount'=>$amount, ':vendor_cost'=>$vendorCost, ':currency'=>$currency, ':tx'=>$sessionId, ':uid'=>$agentId]);
                        $bid = (int)$this->pdo->lastInsertId();
                        if ($bid > 0) {
                            $code = 'TAX-' . date('Ymd') . '-' . $bid;
                            $this->pdo->prepare('UPDATE taxi_bookings SET booking_code=:c WHERE id=:id')->execute([':c'=>$code, ':id'=>$bid]);
                            // Assign vendor based on taxi.vendor_id
                            try {
                                $stTaxi = $this->pdo->prepare('SELECT vendor_id FROM taxis WHERE id = :id');
                                $stTaxi->execute([':id'=>$taxiId]);
                                $vid = (int)($stTaxi->fetchColumn() ?: 0);
                                if ($vid > 0) {
                                    $this->pdo->prepare('UPDATE taxi_bookings SET vendor_id = :vid, vendor_assigned_at = NOW() WHERE id = :id')->execute([':vid'=>$vid, ':id'=>$bid]);
                                }
                            } catch (\Throwable $e) { /* ignore vendor assignment */ }
                            try {
                                $pp = $this->pdo->prepare('INSERT INTO taxi_payments (booking_id, method, gateway_name, amount, currency, status, provider_ref, event_id, raw_event, created_at, updated_at) VALUES (:bid, "gateway", "stripe", :amt, :cur, "captured", :ref, :eid, :raw, NOW(), NOW())');
                                $pp->execute([':bid'=>$bid, ':amt'=>$amount, ':cur'=>$currency, ':ref'=>$sessionId, ':eid'=>$eventId, ':raw'=>$payload]);
                            } catch (\Throwable $e) {}
                            $this->pdo->prepare('UPDATE taxi_payment_intents SET status=\'succeeded\', booking_id=:bid, updated_at=NOW() WHERE id=:id')->execute([':bid'=>$bid, ':id'=>$taxiIntentId]);
                        }
                    } catch (\Throwable $e) { /* ignore */ }
                }
                http_response_code(200); echo 'ok'; return;
            }

            // Yacht booking flow - existing booking path (if metadata contains booking)
            if ($yachtBookingId) {
                // Mark booking paid/confirmed; insert payment row
                try {
                    $ub = $this->pdo->prepare('UPDATE yacht_bookings SET status="confirmed", payment_status="paid", payment_method="gateway", gateway_name="stripe", payment_txn_id=:tx, paid_at=NOW() WHERE id=:bid');
                    $ub->execute(['tx'=>$sessionId, 'bid'=>$yachtBookingId]);
                    $p = $this->pdo->prepare('INSERT INTO yacht_payments (booking_id, method, gateway_name, amount, currency, status, provider_ref, event_id, raw_event, created_at, updated_at) VALUES (:bid, "gateway", "stripe", 0, "THB", "captured", :ref, :eid, :raw, NOW(), NOW())');
                    $p->execute([':bid'=>$yachtBookingId, ':ref'=>$sessionId, ':eid'=>$eventId, ':raw'=>$payload]);
                } catch (\Throwable $e) { /* ignore */ }
                http_response_code(200); echo 'ok'; return;
            }

            // Yacht intent flow - create booking now
            if ($yachtIntentId) {
                // Load intent
                $ist = $this->pdo->prepare('SELECT * FROM yacht_payment_intents WHERE id=:id LIMIT 1');
                $ist->execute([':id'=>$yachtIntentId]);
                $in = $ist->fetch(\PDO::FETCH_ASSOC) ?: null;
                if ($in && (string)($in['status'] ?? '') === 'pending') {
                    $agentId = (int)($in['agent_id'] ?? 0);
                    $yachtId = (int)($in['yacht_id'] ?? 0);
                    $tripDate = (string)($in['trip_date'] ?? '');
                    $hours = (int)($in['hours'] ?? 0);
                    $guests = (int)($in['guests'] ?? 0);
                    $amount = (float)($in['amount'] ?? 0);
                    $currency = (string)($in['currency'] ?? 'THB');
                    // Create booking paid
                    try {
                        $insB = $this->pdo->prepare('INSERT INTO yacht_bookings (agent_id, yacht_id, trip_date, hours, guests, amount_total, currency, status, payment_status, payment_method, gateway_name, payment_txn_id, paid_at, channel, created_by, updated_by, created_at, updated_at) VALUES (:agent_id,:yacht_id,:trip_date,:hours,:guests,:amount,:currency,\'confirmed\',\'paid\',\'gateway\',\'stripe\',:tx,NOW(),\'portal\',:uid,:uid,NOW(),NOW())');
                        $insB->execute([':agent_id'=>$agentId, ':yacht_id'=>$yachtId, ':trip_date'=>$tripDate, ':hours'=>$hours, ':guests'=>$guests, ':amount'=>$amount, ':currency'=>$currency, ':tx'=>$sessionId, ':uid'=>$agentId]);
                        $bid = (int)$this->pdo->lastInsertId();
                        if ($bid > 0) {
                            $code = 'YCH-' . date('Ymd') . '-' . $bid;
                            $this->pdo->prepare('UPDATE yacht_bookings SET booking_code=:c WHERE id=:id')->execute([':c'=>$code, ':id'=>$bid]);
                            // Payment row
                            try {
                                $pp = $this->pdo->prepare('INSERT INTO yacht_payments (booking_id, method, gateway_name, amount, currency, status, provider_ref, event_id, raw_event, created_at, updated_at) VALUES (:bid, "gateway", "stripe", :amt, :cur, "captured", :ref, :eid, :raw, NOW(), NOW())');
                                $pp->execute([':bid'=>$bid, ':amt'=>$amount, ':cur'=>$currency, ':ref'=>$sessionId, ':eid'=>$eventId, ':raw'=>$payload]);
                            } catch (\Throwable $e) {}
                            // Link intent
                            $this->pdo->prepare('UPDATE yacht_payment_intents SET status=\'succeeded\', booking_id=:bid, updated_at=NOW() WHERE id=:id')->execute([':bid'=>$bid, ':id'=>$yachtIntentId]);
                        }
                    } catch (\Throwable $e) { /* ignore */ }
                }
                http_response_code(200); echo 'ok'; return;
            }

            http_response_code(200);
            echo 'ok';
            return;
        }

        // Other events: record raw event minimally
        if ($eventId !== '') {
            $ins = $this->pdo->prepare('INSERT INTO payments (order_id, user_id, gateway, amount, currency, status, provider_ref, event_id, raw_event) VALUES (0,0, "stripe", 0, "THB", "pending", NULL, :eid, :raw)');
            try { $ins->execute(['eid' => $eventId, 'raw' => $payload]); } catch (\Throwable $e) { /* ignore duplicates */ }
        }
        http_response_code(200);
        echo 'ok';
    }

    // Signature verification handled by Stripe SDK
}
