<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;

class PackagesController extends Controller
{
    // ===== Admin Screens =====
    public function index(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();

        $vendorId = (int)($_GET['vendor_id'] ?? 0);
        $selectedCountry = trim((string)($_GET['country'] ?? ''));
        $selectedCity = trim((string)($_GET['city'] ?? ''));
        // Vendors for selector (optionally filtered by country/city) – only Activity vendors
        try {
            if ($selectedCountry !== '' && $selectedCity !== '') {
                $stmtVendors = $this->pdo->prepare("SELECT id, name FROM vendors WHERE module='activity' AND country=? AND city=? ORDER BY name");
                $stmtVendors->execute([$selectedCountry, $selectedCity]);
                $vendors = $stmtVendors->fetchAll();
            } elseif ($selectedCountry !== '') {
                $stmtVendors = $this->pdo->prepare("SELECT id, name FROM vendors WHERE module='activity' AND country=? ORDER BY name");
                $stmtVendors->execute([$selectedCountry]);
                $vendors = $stmtVendors->fetchAll();
            } else {
                $vendors = $this->pdo->query("SELECT id, name FROM vendors WHERE module='activity' ORDER BY name")->fetchAll();
            }
        } catch (\Throwable $e) { $vendors = []; }

        $packages = [];
        $variantsByPkg = [];
        $pricesByVariant = [];
        $timesByPkg = [];
        if ($vendorId > 0) {
            try {
                $stmt = $this->pdo->prepare("SELECT * FROM vendor_packages WHERE vendor_id=? ORDER BY id DESC");
                $stmt->execute([$vendorId]);
                $packages = $stmt->fetchAll();

                if ($packages) {
                    $pkgIds = array_column($packages, 'id');
                    // Variants
                    $in = implode(',', array_fill(0, count($pkgIds), '?'));
                    $stmtV = $this->pdo->prepare("SELECT * FROM vendor_package_variants WHERE package_id IN ($in) ORDER BY id DESC");
                    $stmtV->execute($pkgIds);
                    $variants = $stmtV->fetchAll();
                    foreach ($variants as $v) { $variantsByPkg[$v['package_id']][] = $v; }

                    // Prices
                    if (!empty($variants)) {
                        $variantIds = array_column($variants, 'id');
                        $in2 = implode(',', array_fill(0, count($variantIds), '?'));
                        $stmtP = $this->pdo->prepare("SELECT * FROM vendor_package_prices WHERE variant_id IN ($in2) ORDER BY price_type, min_quantity");
                        $stmtP->execute($variantIds);
                        $prices = $stmtP->fetchAll();
                        foreach ($prices as $p) { $pricesByVariant[$p['variant_id']][] = $p; }
                    }

                    // Showtimes overrides (package-level)
                    $stmtT = $this->pdo->prepare("SELECT * FROM vendor_package_showtimes WHERE package_id IN ($in) AND (variant_id IS NULL OR variant_id=0) AND active=1 ORDER BY time");
                    $stmtT->execute($pkgIds);
                    $times = $stmtT->fetchAll();
                    foreach ($times as $t) { $timesByPkg[$t['package_id']][] = $t['time']; }
                }
            } catch (\Throwable $e) { /* ignore */ }
        }

        $this->view('admin/packages_index', compact('csrf', 'vendors', 'vendorId', 'packages', 'variantsByPkg', 'pricesByVariant', 'timesByPkg', 'selectedCountry', 'selectedCity'));
    }

    // Create/Update package
    public function store(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        // Ensure schema has thumbnail column
        $this->ensurePackageSchema();

        $id = (int)($_POST['id'] ?? 0);
        $vendorId = (int)($_POST['vendor_id'] ?? 0);
        $name = trim((string)($_POST['name'] ?? ''));
        $module = trim((string)($_POST['module'] ?? 'activity'));
        $requiresTime = isset($_POST['requires_show_time']) ? 1 : 0;
        $agePolicy = in_array(($_POST['age_policy'] ?? 'adult_only'), ['adult_only','family'], true) ? $_POST['age_policy'] : 'adult_only';
        $addrOverride = trim((string)($_POST['address_override'] ?? ''));
        $active = isset($_POST['active']) ? 1 : 0;

        if ($vendorId <= 0 || $name === '') {
            $_SESSION['errors'] = ['Vendor and name are required.'];
            $this->redirect('/admin/vendors/package_manage?vendor_id=' . $vendorId);
        }

        // Handle thumbnail upload if provided
        $uploadedThumbPath = null;
        if (isset($_FILES['thumbnail']) && is_array($_FILES['thumbnail']) && (int)$_FILES['thumbnail']['error'] === UPLOAD_ERR_OK) {
            try {
                $maxBytes = 2 * 1024 * 1024; // 2MB
                if ((int)$_FILES['thumbnail']['size'] > $maxBytes) { throw new \RuntimeException('Thumbnail too large.'); }
                $tmp = $_FILES['thumbnail']['tmp_name'];
                $finfo = function_exists('finfo_open') ? finfo_open(FILEINFO_MIME_TYPE) : null;
                $mime = $finfo ? finfo_file($finfo, $tmp) : mime_content_type($tmp);
                if ($finfo) { finfo_close($finfo); }
                $allowed = ['image/jpeg'=>'jpg','image/png'=>'png','image/webp'=>'webp'];
                $ext = null;
                if (isset($allowed[$mime])) { $ext = $allowed[$mime]; }
                else {
                    // fallback to extension from name
                    $namePart = strtolower(pathinfo((string)($_FILES['thumbnail']['name'] ?? ''), PATHINFO_EXTENSION));
                    if (in_array($namePart, ['jpg','jpeg','png','webp'], true)) { $ext = ($namePart==='jpeg'?'jpg':$namePart); }
                }
                if (!$ext) { throw new \RuntimeException('Unsupported image type.'); }
                $root = dirname(__DIR__, 2);
                $uploadDir = $root . '/public/assets/uploads/packages';
                if (!is_dir($uploadDir)) { @mkdir($uploadDir, 0777, true); }
                $safeBase = preg_replace('/[^a-z0-9_-]+/i', '-', $name);
                $filename = 'pkg_' . $vendorId . '_' . time() . '_' . substr(sha1($safeBase . random_int(1, PHP_INT_MAX)), 0, 8) . '.' . $ext;
                $dest = $uploadDir . '/' . $filename;
                if (!@move_uploaded_file($tmp, $dest)) { throw new \RuntimeException('Failed to save thumbnail.'); }
                // Store as web path
                $uploadedThumbPath = '/assets/uploads/packages/' . $filename;
            } catch (\Throwable $e) {
                // ignore upload errors for now, do not block package save
            }
        }

        if ($id > 0) {
            $stmt = $this->pdo->prepare("UPDATE vendor_packages SET vendor_id=?, name=?, module=?, requires_show_time=?, age_policy=?, address_override=?, active=?, updated_at=NOW() WHERE id=?");
            $ok = $stmt->execute([$vendorId, $name, $module, (int)$requiresTime, $agePolicy, $addrOverride !== '' ? $addrOverride : null, (int)$active, $id]);
            // Save thumbnail if new one uploaded
            if ($ok && $uploadedThumbPath) {
                $stmtU = $this->pdo->prepare('UPDATE vendor_packages SET thumbnail_path=? WHERE id=?');
                $stmtU->execute([$uploadedThumbPath, $id]);
            }
            // Save show times if provided
            if ($ok && $requiresTime && isset($_POST['show_times'])) {
                $times = array_filter(array_map('trim', explode(',', (string)$_POST['show_times'])));
                // filter HH:MM 24h
                $valid = [];
                foreach ($times as $t) { if (preg_match('/^(?:[01]\d|2[0-3]):[0-5]\d$/', $t)) { $valid[] = $t; } }
                if (!empty($valid)) {
                    // Remove existing package-level showtimes and insert new
                    $del = $this->pdo->prepare('DELETE FROM vendor_package_showtimes WHERE package_id=? AND (variant_id IS NULL OR variant_id=0)');
                    $del->execute([$id]);
                    $ins = $this->pdo->prepare('INSERT INTO vendor_package_showtimes (package_id, variant_id, time, active) VALUES (?, NULL, ?, 1)');
                    foreach ($valid as $tv) { $ins->execute([$id, $tv]); }
                }
            }
            $_SESSION['flash'] = $ok ? 'Package updated.' : 'Failed to update package.';
        } else {
            $stmt = $this->pdo->prepare("INSERT INTO vendor_packages (vendor_id, name, module, requires_show_time, age_policy, address_override, active) VALUES (?,?,?,?,?,?,?)");
            $ok = $stmt->execute([$vendorId, $name, $module, (int)$requiresTime, $agePolicy, $addrOverride !== '' ? $addrOverride : null, (int)$active]);
            if ($ok) {
                $newId = (int)$this->pdo->lastInsertId();
                if ($uploadedThumbPath) {
                    $stmtU = $this->pdo->prepare('UPDATE vendor_packages SET thumbnail_path=? WHERE id=?');
                    $stmtU->execute([$uploadedThumbPath, $newId]);
                }
                if ($requiresTime && isset($_POST['show_times'])) {
                    $times = array_filter(array_map('trim', explode(',', (string)$_POST['show_times'])));
                    $valid = [];
                    foreach ($times as $t) { if (preg_match('/^(?:[01]\d|2[0-3]):[0-5]\d$/', $t)) { $valid[] = $t; } }
                    if (!empty($valid)) {
                        $ins = $this->pdo->prepare('INSERT INTO vendor_package_showtimes (package_id, variant_id, time, active) VALUES (?, NULL, ?, 1)');
                        foreach ($valid as $tv) { $ins->execute([$newId, $tv]); }
                    }
                }
            }
            $_SESSION['flash'] = $ok ? 'Package created.' : 'Failed to create package.';
        }
        // Redirect to the single package manage page when possible; otherwise back to vendor-filtered list
        $redirectId = $id > 0 ? $id : ($newId ?? 0);
        if ($redirectId > 0) {
            $this->redirect('/admin/vendors/package_manage?id=' . $redirectId);
        }
        $this->redirect('/admin/vendors/package_manage?vendor_id=' . $vendorId);
    }

    private function ensurePackageSchema(): void
    {
        try {
            // Check for column existence
            $stmt = $this->pdo->query("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='vendor_packages' AND COLUMN_NAME='thumbnail_path'");
            $exists = (bool)$stmt->fetchColumn();
            if (!$exists) {
                $this->pdo->exec("ALTER TABLE vendor_packages ADD COLUMN thumbnail_path VARCHAR(255) NULL AFTER address_override");
            }
        } catch (\Throwable $e) {
            // As a fallback, attempt altering without pre-check
            try { $this->pdo->exec("ALTER TABLE vendor_packages ADD COLUMN thumbnail_path VARCHAR(255) NULL"); } catch (\Throwable $e2) { /* ignore */ }
        }
    }

    public function delete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        $vendorId = (int)($_POST['vendor_id'] ?? 0);
        if ($id <= 0) { $_SESSION['errors'] = ['Invalid id']; $this->redirect('/admin/vendors/package_manage?vendor_id=' . $vendorId); }
        $stmt = $this->pdo->prepare('DELETE FROM vendor_packages WHERE id=?');
        $ok = $stmt->execute([$id]);
        $_SESSION['flash'] = $ok ? 'Package deleted.' : 'Failed to delete package.';
        $this->redirect('/admin/vendors/package_manage?vendor_id=' . $vendorId);
    }

    // Toggle package active status
    public function togglePackageActive(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        $vendorId = (int)($_POST['vendor_id'] ?? 0);
        $active = isset($_POST['active']) ? 1 : 0;
        if ($id <= 0) { $_SESSION['errors'] = ['Invalid id']; $this->redirect('/admin/vendor-packages?vendor_id=' . $vendorId); }
        $stmt = $this->pdo->prepare('UPDATE vendor_packages SET active=?, updated_at=NOW() WHERE id=?');
        $ok = $stmt->execute([(int)$active, $id]);
        $_SESSION['flash'] = $ok ? ('Package ' . ($active? 'activated':'deactivated') . '.') : 'Failed to update package status.';
        $this->redirect('/admin/vendors/package_manage?id=' . $id);
    }

    // Toggle variant active status
    public function toggleVariantActive(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        $vendorId = (int)($_POST['vendor_id'] ?? 0);
        $active = isset($_POST['active']) ? 1 : 0;
        if ($id <= 0) { $_SESSION['errors'] = ['Invalid id']; $this->redirect('/admin/vendor-packages?vendor_id=' . $vendorId); }
        // Find package id for redirect
        $packageId = 0;
        try {
            $stmtPkg = $this->pdo->prepare('SELECT package_id FROM vendor_package_variants WHERE id=?');
            $stmtPkg->execute([$id]);
            $packageId = (int)($stmtPkg->fetchColumn() ?: 0);
        } catch (\Throwable $e) { /* ignore */ }
        $stmt = $this->pdo->prepare('UPDATE vendor_package_variants SET active=?, updated_at=NOW() WHERE id=?');
        $ok = $stmt->execute([(int)$active, $id]);
        $_SESSION['flash'] = $ok ? ('Variant ' . ($active? 'activated':'deactivated') . '.') : 'Failed to update variant status.';
        if ($packageId > 0) { $this->redirect('/admin/vendors/package_manage?id=' . $packageId); }
        $this->redirect('/admin/vendors/package_manage?vendor_id=' . $vendorId);
    }

    // Create/Update variant
    public function variantStore(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        $packageId = (int)($_POST['package_id'] ?? 0);
        $vendorId = (int)($_POST['vendor_id'] ?? 0);
        $name = trim((string)($_POST['name'] ?? ''));
        $notes = trim((string)($_POST['notes'] ?? ''));
        $active = isset($_POST['active']) ? 1 : 0;
        if ($packageId <= 0 || $name === '') { $_SESSION['errors'] = ['Variant name required.']; $this->redirect('/admin/vendor-packages?vendor_id=' . $vendorId); }
        if ($id > 0) {
            $stmt = $this->pdo->prepare('UPDATE vendor_package_variants SET name=?, notes=?, active=?, updated_at=NOW() WHERE id=? AND package_id=?');
            $ok = $stmt->execute([$name, $notes !== '' ? $notes : null, (int)$active, $id, $packageId]);
            $_SESSION['flash'] = $ok ? 'Variant updated.' : 'Failed to update variant.';
        } else {
            $stmt = $this->pdo->prepare('INSERT INTO vendor_package_variants (package_id, name, notes, active) VALUES (?,?,?,?)');
            $ok = $stmt->execute([$packageId, $name, $notes !== '' ? $notes : null, (int)$active]);
            $_SESSION['flash'] = $ok ? 'Variant created.' : 'Failed to create variant.';
        }
        $this->redirect('/admin/vendors/package_manage?id=' . $packageId);
    }

    public function variantDelete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        $vendorId = (int)($_POST['vendor_id'] ?? 0);
        if ($id <= 0) { $_SESSION['errors'] = ['Invalid id']; $this->redirect('/admin/vendor-packages?vendor_id=' . $vendorId); }
        // find package for redirect
        $packageId = 0;
        try {
            $stmtPkg = $this->pdo->prepare('SELECT package_id FROM vendor_package_variants WHERE id=?');
            $stmtPkg->execute([$id]);
            $packageId = (int)($stmtPkg->fetchColumn() ?: 0);
        } catch (\Throwable $e) { /* ignore */ }
        $stmt = $this->pdo->prepare('DELETE FROM vendor_package_variants WHERE id=?');
        $ok = $stmt->execute([$id]);
        $_SESSION['flash'] = $ok ? 'Variant deleted.' : 'Failed to delete variant.';
        if ($packageId > 0) { $this->redirect('/admin/vendors/package_manage?id=' . $packageId); }
        $this->redirect('/admin/vendor-packages?vendor_id=' . $vendorId);
    }

    // Create/Update price
    public function priceStore(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        // Ensure schema has customer pricing columns
        $this->ensurePriceSchema();
        $id = (int)($_POST['id'] ?? 0);
        $packageId = (int)($_POST['package_id'] ?? 0);
        $variantId = (int)($_POST['variant_id'] ?? 0);
        $vendorId = (int)($_POST['vendor_id'] ?? 0);
        $priceType = in_array(($_POST['price_type'] ?? 'base'), ['base','group_tier'], true) ? $_POST['price_type'] : 'base';
        $paxType = in_array(($_POST['pax_type'] ?? 'adult'), ['adult','child','flat'], true) ? $_POST['pax_type'] : 'adult';
        $minQty = isset($_POST['min_quantity']) && $_POST['min_quantity'] !== '' ? (int)$_POST['min_quantity'] : null;
        $vendorCost = (float)($_POST['vendor_cost'] ?? 0);
        $agentPrice = (float)($_POST['agent_price'] ?? 0);
        $customerPrice = isset($_POST['customer_price']) && $_POST['customer_price'] !== '' ? (float)$_POST['customer_price'] : null;
        $customerPromoPrice = isset($_POST['customer_promo_price']) && $_POST['customer_promo_price'] !== '' ? (float)$_POST['customer_promo_price'] : null;
        $promoStartAt = trim((string)($_POST['promo_start_at'] ?? ''));
        $promoEndAt = trim((string)($_POST['promo_end_at'] ?? ''));
        $promoStartAt = $promoStartAt !== '' ? $promoStartAt : null;
        $promoEndAt = $promoEndAt !== '' ? $promoEndAt : null;
        $currency = strtoupper(trim((string)($_POST['currency'] ?? 'THB')));
        $active = isset($_POST['active']) ? 1 : 0;

        // Pickup/Drop fields (support combined UI select: pickup_combo)
        $pickupCombo = (string)($_POST['pickup_combo'] ?? '');
        if ($pickupCombo === 'pickup_only') {
            $pickupType = 'included';
            $pickupScope = 'pickup_only';
        } elseif ($pickupCombo === 'pickup_and_drop') {
            $pickupType = 'included';
            $pickupScope = 'pickup_and_drop';
        } elseif ($pickupCombo === 'none') {
            $pickupType = 'none';
            $pickupScope = 'pickup_and_drop'; // default, ignored when none
        } else {
            // fallback to legacy fields if combo not provided
            $pickupType = in_array(($_POST['pickup_type'] ?? 'none'), ['none','included','optional'], true) ? (string)$_POST['pickup_type'] : 'none';
            $pickupScope = in_array(($_POST['pickup_scope'] ?? 'pickup_and_drop'), ['pickup_only','pickup_and_drop'], true) ? (string)$_POST['pickup_scope'] : 'pickup_and_drop';
        }
        $pickupRadiusKm = isset($_POST['pickup_radius_km']) && $_POST['pickup_radius_km'] !== '' ? (float)$_POST['pickup_radius_km'] : null;
        if ($pickupType !== 'none') {
            // enforce radius when pickup is offered
            if ($pickupRadiusKm === null) { $pickupRadiusKm = 5.0; }
            if ($pickupRadiusKm < 0 || $pickupRadiusKm > 5) {
                $_SESSION['errors'] = ['Pickup radius must be between 0 and 5 km.'];
                $this->redirect('/admin/vendors/package_manage?id=' . $packageId);
            }
        } else {
            $pickupRadiusKm = null; // not applicable when no pickup
        }
        $pickupFee = null;
        if ($pickupType === 'optional') {
            // allow 0.00 as valid fee
            $pickupFee = isset($_POST['pickup_fee']) && $_POST['pickup_fee'] !== '' ? (float)$_POST['pickup_fee'] : null;
            if ($pickupFee === null || $pickupFee < 0) {
                $_SESSION['errors'] = ['Pickup fee is required and must be >= 0 when pickup is optional.'];
                $this->redirect('/admin/vendors/package_manage?id=' . $packageId);
            }
        } else {
            // No optional mode in UI now; ensure fee is not stored
            $pickupFee = null;
        }
        $pickupNotes = trim((string)($_POST['pickup_notes'] ?? ''));
        $pickupNotes = $pickupNotes !== '' ? $pickupNotes : null;
        if ($packageId <= 0 || $variantId <= 0) { $_SESSION['errors'] = ['Price needs package and variant.']; $this->redirect('/admin/vendors/package_manage?vendor_id=' . $vendorId); }

        // Enforce age policy: if package is adult_only, disallow child pax prices
        try {
            $stmtPkg = $this->pdo->prepare('SELECT age_policy FROM vendor_packages WHERE id=?');
            $stmtPkg->execute([$packageId]);
            $agePolicy = (string)($stmtPkg->fetchColumn() ?: 'adult_only');
            if ($agePolicy === 'adult_only' && $paxType === 'child') {
                $_SESSION['errors'] = ['This package is Adult Only. Child pax pricing is not allowed.'];
                $this->redirect('/admin/vendors/package_manage?id=' . $packageId);
            }
        } catch (\Throwable $e) { /* ignore on read error */ }

        if ($id > 0) {
            $stmt = $this->pdo->prepare('UPDATE vendor_package_prices SET price_type=?, pax_type=?, min_quantity=?, vendor_cost=?, agent_price=?, customer_price=?, customer_promo_price=?, promo_start_at=?, promo_end_at=?, currency=?, pickup_type=?, pickup_scope=?, pickup_radius_km=?, pickup_fee=?, pickup_notes=?, active=?, updated_at=NOW() WHERE id=? AND package_id=? AND variant_id=?');
            $ok = $stmt->execute([$priceType, $paxType, $minQty, $vendorCost, $agentPrice, $customerPrice, $customerPromoPrice, $promoStartAt, $promoEndAt, $currency, $pickupType, $pickupScope, $pickupRadiusKm, $pickupFee, $pickupNotes, (int)$active, $id, $packageId, $variantId]);
            $_SESSION['flash'] = $ok ? 'Price updated.' : 'Failed to update price.';
        } else {
            $stmt = $this->pdo->prepare('INSERT INTO vendor_package_prices (package_id, variant_id, price_type, pax_type, min_quantity, vendor_cost, agent_price, customer_price, customer_promo_price, promo_start_at, promo_end_at, currency, pickup_type, pickup_scope, pickup_radius_km, pickup_fee, pickup_notes, active) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)');
            $ok = $stmt->execute([$packageId, $variantId, $priceType, $paxType, $minQty, $vendorCost, $agentPrice, $customerPrice, $customerPromoPrice, $promoStartAt, $promoEndAt, $currency, $pickupType, $pickupScope, $pickupRadiusKm, $pickupFee, $pickupNotes, (int)$active]);
            $_SESSION['flash'] = $ok ? 'Price added.' : 'Failed to add price.';
        }
        $this->redirect('/admin/vendors/package_manage?id=' . $packageId);
    }

    private function ensurePriceSchema(): void
    {
        try {
            // customer_price
            $stmt = $this->pdo->query("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='vendor_package_prices' AND COLUMN_NAME='customer_price'");
            $exists = (bool)$stmt->fetchColumn();
            if (!$exists) { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN customer_price DECIMAL(10,2) NULL AFTER agent_price"); }
        } catch (\Throwable $e) {
            try { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN customer_price DECIMAL(10,2) NULL"); } catch (\Throwable $e2) { /* ignore */ }
        }
        try {
            // customer_promo_price
            $stmt = $this->pdo->query("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='vendor_package_prices' AND COLUMN_NAME='customer_promo_price'");
            $exists = (bool)$stmt->fetchColumn();
            if (!$exists) { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN customer_promo_price DECIMAL(10,2) NULL AFTER customer_price"); }
        } catch (\Throwable $e) {
            try { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN customer_promo_price DECIMAL(10,2) NULL"); } catch (\Throwable $e2) { /* ignore */ }
        }
        try {
            // promo_start_at
            $stmt = $this->pdo->query("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='vendor_package_prices' AND COLUMN_NAME='promo_start_at'");
            $exists = (bool)$stmt->fetchColumn();
            if (!$exists) { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN promo_start_at DATETIME NULL AFTER customer_promo_price"); }
        } catch (\Throwable $e) {
            try { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN promo_start_at DATETIME NULL"); } catch (\Throwable $e2) { /* ignore */ }
        }
        try {
            // promo_end_at
            $stmt = $this->pdo->query("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='vendor_package_prices' AND COLUMN_NAME='promo_end_at'");
            $exists = (bool)$stmt->fetchColumn();
            if (!$exists) { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN promo_end_at DATETIME NULL AFTER promo_start_at"); }
        } catch (\Throwable $e) {
            try { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN promo_end_at DATETIME NULL"); } catch (\Throwable $e2) { /* ignore */ }
        }
        // Ensure pickup columns
        try {
            $stmt = $this->pdo->query("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='vendor_package_prices' AND COLUMN_NAME='pickup_type'");
            $exists = (bool)$stmt->fetchColumn();
            if (!$exists) { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN pickup_type ENUM('none','included','optional') NOT NULL DEFAULT 'none' AFTER currency"); }
        } catch (\Throwable $e) {
            try { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN pickup_type ENUM('none','included','optional') NOT NULL DEFAULT 'none'"); } catch (\Throwable $e2) { /* ignore */ }
        }
        try {
            $stmt = $this->pdo->query("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='vendor_package_prices' AND COLUMN_NAME='pickup_scope'");
            $exists = (bool)$stmt->fetchColumn();
            if (!$exists) { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN pickup_scope ENUM('pickup_only','pickup_and_drop') NOT NULL DEFAULT 'pickup_and_drop' AFTER pickup_type"); }
        } catch (\Throwable $e) {
            try { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN pickup_scope ENUM('pickup_only','pickup_and_drop') NOT NULL DEFAULT 'pickup_and_drop'"); } catch (\Throwable $e2) { /* ignore */ }
        }
        try {
            $stmt = $this->pdo->query("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='vendor_package_prices' AND COLUMN_NAME='pickup_radius_km'");
            $exists = (bool)$stmt->fetchColumn();
            if (!$exists) { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN pickup_radius_km DECIMAL(4,1) NULL AFTER pickup_scope"); }
        } catch (\Throwable $e) {
            try { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN pickup_radius_km DECIMAL(4,1) NULL"); } catch (\Throwable $e2) { /* ignore */ }
        }
        try {
            $stmt = $this->pdo->query("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='vendor_package_prices' AND COLUMN_NAME='pickup_fee'");
            $exists = (bool)$stmt->fetchColumn();
            if (!$exists) { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN pickup_fee DECIMAL(10,2) NULL AFTER pickup_radius_km"); }
        } catch (\Throwable $e) {
            try { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN pickup_fee DECIMAL(10,2) NULL"); } catch (\Throwable $e2) { /* ignore */ }
        }
        try {
            $stmt = $this->pdo->query("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='vendor_package_prices' AND COLUMN_NAME='pickup_notes'");
            $exists = (bool)$stmt->fetchColumn();
            if (!$exists) { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN pickup_notes VARCHAR(255) NULL AFTER pickup_fee"); }
        } catch (\Throwable $e) {
            try { $this->pdo->exec("ALTER TABLE vendor_package_prices ADD COLUMN pickup_notes VARCHAR(255) NULL"); } catch (\Throwable $e2) { /* ignore */ }
        }
    }

    public function priceDelete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        $vendorId = (int)($_POST['vendor_id'] ?? 0);
        if ($id <= 0) { $_SESSION['errors'] = ['Invalid id']; $this->redirect('/admin/vendor-packages?vendor_id=' . $vendorId); }
        // find package for redirect
        $packageId = 0;
        try {
            $stmtPkg = $this->pdo->prepare('SELECT package_id FROM vendor_package_prices WHERE id=?');
            $stmtPkg->execute([$id]);
            $packageId = (int)($stmtPkg->fetchColumn() ?: 0);
        } catch (\Throwable $e) { /* ignore */ }
        $stmt = $this->pdo->prepare('DELETE FROM vendor_package_prices WHERE id=?');
        $ok = $stmt->execute([$id]);
        $_SESSION['flash'] = $ok ? 'Price deleted.' : 'Failed to delete price.';
        if ($packageId > 0) { $this->redirect('/admin/vendors/package_manage?id=' . $packageId); }
        $this->redirect('/admin/vendor-packages?vendor_id=' . $vendorId);
    }

    // ===== APIs for Agent/Booking UIs =====
    private function json($data, int $code = 200): void
    {
        http_response_code($code);
        header('Content-Type: application/json');
        echo json_encode($data);
    }

    public function apiVendorPackages(): void
    {
        Auth::requireRole(['Admin','Agent','Employee','Partner']);
        $vendorId = (int)($_GET['vendor_id'] ?? 0);
        if ($vendorId <= 0) { $this->json(['error' => 'vendor_id required'], 400); return; }
        $stmt = $this->pdo->prepare('SELECT id, name, module, requires_show_time, age_policy, address_override, active FROM vendor_packages WHERE vendor_id=? AND active=1 ORDER BY name');
        $stmt->execute([$vendorId]);
        $rows = $stmt->fetchAll();
        $this->json($rows); return;
    }

    public function apiPackageVariants(): void
    {
        Auth::requireRole(['Admin','Agent','Employee','Partner']);
        $packageId = (int)($_GET['package_id'] ?? 0);
        if ($packageId <= 0) { $this->json(['error' => 'package_id required'], 400); return; }
        $stmt = $this->pdo->prepare('SELECT id, name, notes, active FROM vendor_package_variants WHERE package_id=? AND active=1 ORDER BY id');
        $stmt->execute([$packageId]);
        $rows = $stmt->fetchAll();
        $this->json($rows); return;
    }

    public function apiVariantTimes(): void
    {
        Auth::requireRole(['Admin','Agent','Employee','Partner']);
        $variantId = (int)($_GET['variant_id'] ?? 0);
        $packageId = (int)($_GET['package_id'] ?? 0);
        if ($variantId <= 0 && $packageId <= 0) { $this->json(['error' => 'variant_id or package_id required'], 400); return; }

        // Prefer explicit overrides
        $times = [];
        if ($variantId > 0) {
            $stmt = $this->pdo->prepare('SELECT time FROM vendor_package_showtimes WHERE variant_id=? AND active=1 ORDER BY time');
            $stmt->execute([$variantId]);
            $times = array_column($stmt->fetchAll(), 'time');
            if (!empty($times)) { $this->json($times); return; }
            // fallback to package
            $stmt = $this->pdo->prepare('SELECT package_id FROM vendor_package_variants WHERE id=?');
            $stmt->execute([$variantId]);
            $packageId = (int)($stmt->fetchColumn() ?: 0);
        }

        if ($packageId > 0) {
            // package-level overrides
            $stmt = $this->pdo->prepare('SELECT time FROM vendor_package_showtimes WHERE package_id=? AND (variant_id IS NULL OR variant_id=0) AND active=1 ORDER BY time');
            $stmt->execute([$packageId]);
            $times = array_column($stmt->fetchAll(), 'time');
            if (!empty($times)) { $this->json($times); return; }
            // fallback to vendor.show_times
            $stmt2 = $this->pdo->prepare('SELECT v.show_times FROM vendor_packages p JOIN vendors v ON v.id=p.vendor_id WHERE p.id=?');
            $stmt2->execute([$packageId]);
            $json = $stmt2->fetchColumn();
            $arr = $json ? json_decode($json, true) : [];
            $this->json(is_array($arr) ? $arr : []); return;
        }

        $this->json([]); return;
    }

    public function apiVariantPrice(): void
    {
        Auth::requireRole(['Admin','Agent','Employee','Partner']);
        Security::requireCsrf();
        $variantId = (int)($_POST['variant_id'] ?? 0);
        $adultQty = max(0, (int)($_POST['adult_qty'] ?? 0));
        $childQty = max(0, (int)($_POST['child_qty'] ?? 0));
        if ($variantId <= 0) { $this->json(['error' => 'variant_id required'], 400); return; }

        // Load package (for age policy)
        $stmt = $this->pdo->prepare('SELECT p.id as package_id, p.age_policy FROM vendor_package_variants v JOIN vendor_packages p ON p.id=v.package_id WHERE v.id=?');
        $stmt->execute([$variantId]);
        $row = $stmt->fetch();
        if (!$row) { $this->json(['error' => 'variant not found'], 404); return; }
        $packageId = (int)$row['package_id'];
        $agePolicy = $row['age_policy'];

        // Fetch prices for variant (include vehicle capacity info)
        $stmt = $this->pdo->prepare("SELECT price_type, pax_type, min_quantity, vendor_cost, agent_price, vehicle_type, max_pax FROM vendor_package_prices WHERE variant_id=? AND active=1");
        $stmt->execute([$variantId]);
        $prices = $stmt->fetchAll();

        // Separate base and tiers
        $base = ['adult'=>null,'child'=>null,'flat'=>null];
        $tiers = [];
        foreach ($prices as $p) {
            if ($p['price_type'] === 'base') {
                $base[$p['pax_type']] = $p;
            } else {
                $tiers[] = $p;
            }
        }
        // Pick best tier by total pax (adult_only: adult only; family: adult+child)
        $totalPax = ($agePolicy === 'family') ? ($adultQty + $childQty) : $adultQty;
        $appliedTier = null;
        // 1) Prefer tiers with min_quantity <= totalPax and enough capacity via max_pax
        foreach ($tiers as $t) {
            $minOk = ($t['min_quantity'] !== null) ? ((int)$t['min_quantity'] <= $totalPax) : false;
            $capOk = (!isset($t['max_pax']) || $t['max_pax'] === null) ? true : ((int)$t['max_pax'] >= $totalPax);
            if ($minOk && $capOk) {
                if ($appliedTier === null || (int)$t['min_quantity'] > (int)$appliedTier['min_quantity']) {
                    $appliedTier = $t;
                }
            }
        }
        // 2) If none found, relax: pick any tier that can accommodate by capacity, with smallest max_pax >= totalPax
        if ($appliedTier === null) {
            $candidate = null;
            foreach ($tiers as $t) {
                $capOk = (!isset($t['max_pax']) || $t['max_pax'] === null) ? true : ((int)$t['max_pax'] >= $totalPax);
                if ($capOk) {
                    if ($candidate === null) { $candidate = $t; continue; }
                    $curMax = $candidate['max_pax'] ?? PHP_INT_MAX;
                    $newMax = $t['max_pax'] ?? PHP_INT_MAX;
                    if ($newMax < $curMax) { $candidate = $t; }
                }
            }
            if ($candidate !== null) { $appliedTier = $candidate; }
        }

        $groupApplied = $appliedTier !== null;
        $priceRowAdult = $groupApplied ? $appliedTier : $base['adult'];
        $priceRowChild = $groupApplied ? $appliedTier : $base['child'];

        $vendorCost = 0.0; $agentPrice = 0.0;
        if ($agePolicy === 'adult_only') {
            $rowUse = $groupApplied ? $appliedTier : $base['adult'];
            if ($rowUse) {
                if ($rowUse['pax_type'] === 'flat') {
                    // flat interpreted as per-person flat in adult_only mode
                    $vendorCost += (float)$rowUse['vendor_cost'] * $adultQty;
                    $agentPrice += (float)$rowUse['agent_price'] * $adultQty;
                } else { // adult
                    $vendorCost += (float)$rowUse['vendor_cost'] * $adultQty;
                    $agentPrice += (float)$rowUse['agent_price'] * $adultQty;
                }
            }
        } else { // family
            if ($groupApplied) {
                if ($appliedTier['pax_type'] === 'flat') {
                    // Flat price per person regardless of adult/child counts
                    $vendorCost += (float)$appliedTier['vendor_cost'] * $totalPax;
                    $agentPrice += (float)$appliedTier['agent_price'] * $totalPax;
                } elseif ($appliedTier['pax_type'] === 'adult') {
                    $vendorCost += (float)$appliedTier['vendor_cost'] * $adultQty;
                    $agentPrice += (float)$appliedTier['agent_price'] * $adultQty;
                    if ($base['child']) { $vendorCost += (float)$base['child']['vendor_cost'] * $childQty; $agentPrice += (float)$base['child']['agent_price'] * $childQty; }
                } elseif ($appliedTier['pax_type'] === 'child') {
                    if ($base['adult']) { $vendorCost += (float)$base['adult']['vendor_cost'] * $adultQty; $agentPrice += (float)$base['adult']['agent_price'] * $adultQty; }
                    $vendorCost += (float)$appliedTier['vendor_cost'] * $childQty;
                    $agentPrice += (float)$appliedTier['agent_price'] * $childQty;
                }
            } else {
                // No group tier applied: use base adult/child
                if ($base['adult']) { $vendorCost += (float)$base['adult']['vendor_cost'] * $adultQty; $agentPrice += (float)$base['adult']['agent_price'] * $adultQty; }
                if ($base['child']) { $vendorCost += (float)$base['child']['vendor_cost'] * $childQty; $agentPrice += (float)$base['child']['agent_price'] * $childQty; }
            }
        }

        $this->json([
            'group_applied' => $groupApplied,
            'vendor_cost_total' => round($vendorCost, 2),
            'agent_price_total' => round($agentPrice, 2),
            'vehicle_type' => $groupApplied ? ($appliedTier['vehicle_type'] ?? null) : null,
            'applied_min_quantity' => $groupApplied ? (int)($appliedTier['min_quantity'] ?? 0) : null,
            'applied_max_pax' => $groupApplied ? ($appliedTier['max_pax'] ?? null) : null,
        ]);
        return;
    }

    // ===== Dedicated Manage Page for a Single Package =====
    public function manage(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $packageId = (int)($_GET['id'] ?? 0);
        // If no specific package selected, show a list filtered by vendor (optional)
        if ($packageId <= 0) {
            $vendorId = (int)($_GET['vendor_id'] ?? 0);
            $selectedCountry = trim((string)($_GET['country'] ?? ''));
            $selectedCity = trim((string)($_GET['city'] ?? ''));

            // Countries (prefer locations table; fallback to vendors table)
            try {
                $countries = $this->pdo->query("SELECT DISTINCT country FROM locations WHERE country IS NOT NULL AND country<>'' ORDER BY country")->fetchAll();
            } catch (\Throwable $e) { $countries = []; }
            if (!$countries || count($countries) === 0) {
                try {
                    $countries = $this->pdo->query("SELECT DISTINCT country FROM vendors WHERE country IS NOT NULL AND country<>'' ORDER BY country")->fetchAll();
                } catch (\Throwable $e) { /* ignore */ }
            }

            // Cities based on selected country (prefer locations; fallback to vendors)
            try {
                if ($selectedCountry !== '') {
                    $stmtC = $this->pdo->prepare("SELECT DISTINCT city FROM locations WHERE country=? AND city IS NOT NULL AND city<>'' ORDER BY city");
                    $stmtC->execute([$selectedCountry]);
                    $cities = $stmtC->fetchAll();
                } else {
                    $cities = $this->pdo->query("SELECT DISTINCT city FROM locations WHERE city IS NOT NULL AND city<>'' ORDER BY city")->fetchAll();
                }
            } catch (\Throwable $e) { $cities = []; }
            if (!$cities || count($cities) === 0) {
                try {
                    if ($selectedCountry !== '') {
                        $stmtCv = $this->pdo->prepare("SELECT DISTINCT city FROM vendors WHERE country=? AND city IS NOT NULL AND city<>'' ORDER BY city");
                        $stmtCv->execute([$selectedCountry]);
                        $cities = $stmtCv->fetchAll();
                    } else {
                        $cities = $this->pdo->query("SELECT DISTINCT city FROM vendors WHERE city IS NOT NULL AND city<>'' ORDER BY city")->fetchAll();
                    }
                } catch (\Throwable $e) { /* ignore */ }
            }

            // Vendors for filter dropdown (optionally filtered by country/city)
            try {
                if ($selectedCountry !== '' && $selectedCity !== '') {
                    $stmtV = $this->pdo->prepare("SELECT id, name FROM vendors WHERE module='activity' AND country=? AND city=? ORDER BY name");
                    $stmtV->execute([$selectedCountry, $selectedCity]);
                    $vendors = $stmtV->fetchAll();
                } elseif ($selectedCountry !== '') {
                    $stmtV = $this->pdo->prepare("SELECT id, name FROM vendors WHERE module='activity' AND country=? ORDER BY name");
                    $stmtV->execute([$selectedCountry]);
                    $vendors = $stmtV->fetchAll();
                } else {
                    $vendors = $this->pdo->query("SELECT id, name FROM vendors WHERE module='activity' ORDER BY name")->fetchAll();
                }
            } catch (\Throwable $e) { $vendors = []; }

            // Packages list
            $packages = [];
            try {
                if ($vendorId > 0) {
                    $stmt = $this->pdo->prepare('SELECT p.id, p.vendor_id, p.name, p.module, p.requires_show_time, p.age_policy, p.active, p.thumbnail_path, v.name AS vendor_name, v.city AS city, v.country AS country FROM vendor_packages p JOIN vendors v ON v.id=p.vendor_id WHERE p.vendor_id=? ORDER BY p.name');
                    $stmt->execute([$vendorId]);
                    $packages = $stmt->fetchAll();
                } elseif ($selectedCountry !== '' && $selectedCity !== '') {
                    $stmt = $this->pdo->prepare('SELECT p.id, p.vendor_id, p.name, p.module, p.requires_show_time, p.age_policy, p.active, p.thumbnail_path, v.name AS vendor_name, v.city AS city, v.country AS country FROM vendor_packages p JOIN vendors v ON v.id=p.vendor_id WHERE v.country=? AND v.city=? ORDER BY v.name, p.name');
                    $stmt->execute([$selectedCountry, $selectedCity]);
                    $packages = $stmt->fetchAll();
                } elseif ($selectedCountry !== '') {
                    $stmt = $this->pdo->prepare('SELECT p.id, p.vendor_id, p.name, p.module, p.requires_show_time, p.age_policy, p.active, p.thumbnail_path, v.name AS vendor_name, v.city AS city, v.country AS country FROM vendor_packages p JOIN vendors v ON v.id=p.vendor_id WHERE v.country=? ORDER BY v.name, p.name');
                    $stmt->execute([$selectedCountry]);
                    $packages = $stmt->fetchAll();
                } else {
                    // Show all with vendor name when no vendor filter is selected
                    $stmt = $this->pdo->query('SELECT p.id, p.vendor_id, p.name, p.module, p.requires_show_time, p.age_policy, p.active, p.thumbnail_path, v.name AS vendor_name, v.city AS city, v.country AS country FROM vendor_packages p JOIN vendors v ON v.id=p.vendor_id ORDER BY v.name, p.name');
                    $packages = $stmt->fetchAll();
                }
            } catch (\Throwable $e) { /* ignore */ }

            $this->view('vendors/activity/package_list', compact('csrf', 'vendors', 'vendorId', 'packages', 'countries', 'cities', 'selectedCountry', 'selectedCity'));
            return;
        }

        // Load package
        $stmt = $this->pdo->prepare('SELECT * FROM vendor_packages WHERE id=?');
        $stmt->execute([$packageId]);
        $package = $stmt->fetch();
        if (!$package) { http_response_code(404); echo 'Package not found'; return; }

        $vendorId = (int)$package['vendor_id'];

        // Load variants
        $stmtV = $this->pdo->prepare('SELECT * FROM vendor_package_variants WHERE package_id=? ORDER BY id DESC');
        $stmtV->execute([$packageId]);
        $variants = $stmtV->fetchAll();

        // Prices grouped by variant
        $pricesByVariant = [];
        if (!empty($variants)) {
            $variantIds = array_column($variants, 'id');
            $in = implode(',', array_fill(0, count($variantIds), '?'));
            $stmtP = $this->pdo->prepare("SELECT * FROM vendor_package_prices WHERE variant_id IN ($in) ORDER BY price_type, min_quantity");
            $stmtP->execute($variantIds);
            $prices = $stmtP->fetchAll();
            foreach ($prices as $p) { $pricesByVariant[$p['variant_id']][] = $p; }
        }

        // Package-level showtimes
        $times = [];
        $stmtT = $this->pdo->prepare('SELECT time FROM vendor_package_showtimes WHERE package_id=? AND (variant_id IS NULL OR variant_id=0) AND active=1 ORDER BY time');
        $stmtT->execute([$packageId]);
        $times = array_column($stmtT->fetchAll(), 'time');

        // Transport options (fixed price) for this package
        $transportOptions = [];
        $transportEnabled = (int)($package['transport_enabled'] ?? 0) === 1;
        if ($packageId > 0) {
            try {
                // ensure table exists (non-destructive)
                $this->ensureTransportSchema();
                $st = $this->pdo->prepare('SELECT id, vehicle_type, max_pax, vendor_cost, agent_price, customer_price, price, is_active, display_order FROM package_transport_options WHERE package_id = :pid ORDER BY is_active DESC, display_order ASC, id ASC');
                $st->execute([':pid' => $packageId]);
                $transportOptions = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
            } catch (\Throwable $_) { $transportOptions = []; }
        }

        $this->view('vendors/activity/package_manage', compact('csrf','package','variants','pricesByVariant','times','vendorId','transportOptions','transportEnabled'));
    }

    private function ensureTransportSchema(): void
    {
        // Add vendor_packages.transport_enabled if missing
        try {
            $stmt = $this->pdo->query("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='vendor_packages' AND COLUMN_NAME='transport_enabled'");
            $exists = (bool)$stmt->fetchColumn();
            if (!$exists) {
                $this->pdo->exec("ALTER TABLE vendor_packages ADD COLUMN transport_enabled TINYINT(1) NOT NULL DEFAULT 0 AFTER requires_show_time");
            }
        } catch (\Throwable $e) { /* ignore */ }
        // Create package_transport_options table if not exists
        try {
            $this->pdo->exec("CREATE TABLE IF NOT EXISTS package_transport_options (\n  id INT UNSIGNED NOT NULL AUTO_INCREMENT,\n  package_id INT UNSIGNED NOT NULL,\n  vehicle_type VARCHAR(50) NOT NULL,\n  price DECIMAL(10,2) NOT NULL,\n  is_active TINYINT(1) NOT NULL DEFAULT 1,\n  display_order INT NOT NULL DEFAULT 0,\n  created_at DATETIME NOT NULL DEFAULT NOW(),\n  updated_at DATETIME NOT NULL DEFAULT NOW() ON UPDATE NOW(),\n  PRIMARY KEY (id),\n  KEY idx_pkg_active (package_id, is_active, display_order)\n) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        } catch (\Throwable $e) { /* ignore */ }
    }

    // Toggle transport_enabled for a package
    public function transportToggle(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $packageId = (int)($_POST['package_id'] ?? 0);
        $enabled = isset($_POST['transport_enabled']) ? 1 : 0;
        if ($packageId <= 0) { $_SESSION['errors'] = ['Invalid package id']; $this->redirect('/admin/vendor-packages'); }
        $this->ensureTransportSchema();
        try {
            $st = $this->pdo->prepare('UPDATE vendor_packages SET transport_enabled = :en, updated_at = NOW() WHERE id = :id');
            $ok = $st->execute([':en' => (int)$enabled, ':id' => $packageId]);
            $_SESSION['flash'] = $ok ? 'Transport setting updated.' : 'Failed to update transport setting.';
        } catch (\Throwable $e) { $_SESSION['errors'] = ['Failed to update transport setting.']; }
        $this->redirect('/admin/vendors/package_manage?id=' . $packageId);
    }

    // Create/Update a fixed-price transport option
    public function transportOptionStore(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $this->ensureTransportSchema();
        $id = (int)($_POST['id'] ?? 0);
        $packageId = (int)($_POST['package_id'] ?? 0);
        $vehicleType = trim((string)($_POST['vehicle_type'] ?? ''));
        $maxPax = max(1, (int)($_POST['max_pax'] ?? 4));
        // Multi-price fields
        $vendorCost = isset($_POST['vendor_cost']) && $_POST['vendor_cost'] !== '' ? (float)$_POST['vendor_cost'] : 0.0;
        $agentPrice = isset($_POST['agent_price']) && $_POST['agent_price'] !== '' ? (float)$_POST['agent_price'] : 0.0;
        $customerPrice = isset($_POST['customer_price']) && $_POST['customer_price'] !== '' ? (float)$_POST['customer_price'] : null;
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        $order = isset($_POST['display_order']) && $_POST['display_order'] !== '' ? (int)$_POST['display_order'] : 0;
        if ($packageId <= 0 || $vehicleType === '') {
            $_SESSION['errors'] = ['Vehicle type is required.'];
            $this->redirect('/admin/vendor-packages');
        }
        // Legacy single price mirrors agent_price
        $price = $agentPrice;
        try {
            if ($id > 0) {
                $st = $this->pdo->prepare('UPDATE package_transport_options SET vehicle_type=:vt, max_pax=:mp, vendor_cost=:vc, agent_price=:ap, customer_price=:cp, price=:pr, is_active=:ac, display_order=:ord, updated_at=NOW() WHERE id=:id AND package_id=:pid');
                $ok = $st->execute([':vt'=>$vehicleType, ':mp'=>$maxPax, ':vc'=>$vendorCost, ':ap'=>$agentPrice, ':cp'=>$customerPrice, ':pr'=>$price, ':ac'=>$isActive, ':ord'=>$order, ':id'=>$id, ':pid'=>$packageId]);
                $_SESSION['flash'] = $ok ? 'Transport option updated.' : 'Failed to update transport option.';
            } else {
                $st = $this->pdo->prepare('INSERT INTO package_transport_options (package_id, vehicle_type, max_pax, vendor_cost, agent_price, customer_price, price, is_active, display_order, created_at, updated_at) VALUES (:pid,:vt,:mp,:vc,:ap,:cp,:pr,:ac,:ord,NOW(),NOW())');
                $ok = $st->execute([':pid'=>$packageId, ':vt'=>$vehicleType, ':mp'=>$maxPax, ':vc'=>$vendorCost, ':ap'=>$agentPrice, ':cp'=>$customerPrice, ':pr'=>$price, ':ac'=>$isActive, ':ord'=>$order]);
                $_SESSION['flash'] = $ok ? 'Transport option added.' : 'Failed to add transport option.';
            }
        } catch (\Throwable $e) { $_SESSION['errors'] = ['Database error while saving option.']; }
        $this->redirect('/admin/vendors/package_manage?id=' . $packageId);
    }

    public function transportOptionToggle(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        $packageId = (int)($_POST['package_id'] ?? 0);
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        if ($id <= 0 || $packageId <= 0) { $_SESSION['errors'] = ['Invalid id']; $this->redirect('/admin/vendors/package_manage?id=' . $packageId); }
        try {
            $st = $this->pdo->prepare('UPDATE package_transport_options SET is_active=:ac, updated_at=NOW() WHERE id=:id AND package_id=:pid');
            $ok = $st->execute([':ac'=>$isActive, ':id'=>$id, ':pid'=>$packageId]);
            $_SESSION['flash'] = $ok ? ('Transport option ' . ($isActive? 'activated':'deactivated') . '.') : 'Failed to update option.';
        } catch (\Throwable $e) { $_SESSION['errors'] = ['Failed to update option.']; }
        $this->redirect('/admin/vendors/package_manage?id=' . $packageId);
    }

    public function transportOptionDelete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        $packageId = (int)($_POST['package_id'] ?? 0);
        if ($id <= 0 || $packageId <= 0) { $_SESSION['errors'] = ['Invalid id']; $this->redirect('/admin/vendors/package_manage?id=' . $packageId); }
        try {
            $st = $this->pdo->prepare('DELETE FROM package_transport_options WHERE id=:id AND package_id=:pid');
            $ok = $st->execute([':id'=>$id, ':pid'=>$packageId]);
            $_SESSION['flash'] = $ok ? 'Transport option deleted.' : 'Failed to delete transport option.';
        } catch (\Throwable $e) { $_SESSION['errors'] = ['Failed to delete transport option.']; }
        $this->redirect('/admin/vendors/package_manage?id=' . $packageId);
    }
}
