<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;
use App\Core\Upload;

class MediaController extends Controller
{
    // GET /admin/media
    public function index(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $this->view('admin/media_uploader', compact('csrf'));
    }

    // POST /admin/media/upload
    // Expects: files[] or file, CSRF via header X-CSRF-Token or form field csrf
    // Returns JSON with original and thumbnail URLs
    public function upload(): void
    {
        header('Content-Type: application/json');
        Auth::requireRole(['Admin']);
        Security::requireCsrf();

        $uploadsDir = dirname(__DIR__, 2) . '/public/assets/uploads';
        $thumbsDir  = $uploadsDir . '/thumbs';
        if (!is_dir($thumbsDir)) @mkdir($thumbsDir, 0775, true);

        $files = [];
        if (!empty($_FILES['files'])) {
            // Multiple
            $count = is_array($_FILES['files']['name']) ? count($_FILES['files']['name']) : 0;
            for ($i = 0; $i < $count; $i++) {
                $files[] = [
                    'name' => $_FILES['files']['name'][$i] ?? '',
                    'type' => $_FILES['files']['type'][$i] ?? '',
                    'tmp_name' => $_FILES['files']['tmp_name'][$i] ?? '',
                    'error' => $_FILES['files']['error'][$i] ?? UPLOAD_ERR_NO_FILE,
                    'size' => $_FILES['files']['size'][$i] ?? 0,
                ];
            }
        } elseif (!empty($_FILES['file'])) {
            // Single
            $files[] = $_FILES['file'];
        } else {
            http_response_code(400);
            echo json_encode(['error' => 'No files']);
            return;
        }

        $out = [];
        foreach ($files as $f) {
            $res = Upload::image($f, $uploadsDir);
            if (!$res['ok']) {
                http_response_code(400);
                echo json_encode(['error' => $res['error']]);
                return;
            }

            // Create thumbnail 400x400 center-crop
            $thumbName = 'th_' . $res['name'];
            $thumbPath = $thumbsDir . DIRECTORY_SEPARATOR . $thumbName;
            $this->makeThumb($res['path'], $thumbPath, 400, 400, $res['mime']);

            $baseUrl = rtrim((string)($_SERVER['REQUEST_SCHEME'] ?? 'http'), ':/') . '://' . ($_SERVER['HTTP_HOST'] ?? 'localhost');
            $origUrl = $baseUrl . '/assets/uploads/' . $res['name'];
            $thumbUrl = file_exists($thumbPath) ? ($baseUrl . '/assets/uploads/thumbs/' . $thumbName) : '';

            $out[] = [
                'originalUrl' => $origUrl,
                'thumbUrl' => $thumbUrl,
                'name' => $res['name'],
                'mime' => $res['mime'],
            ];
        }

        echo json_encode(['ok' => true, 'files' => $out]);
    }

    private function makeThumb(string $srcPath, string $dstPath, int $tw, int $th, string $mime): void
    {
        [$w, $h] = getimagesize($srcPath) ?: [0,0];
        if ($w <= 0 || $h <= 0) return;
        $src = null;
        $mime = trim($mime);
        if ($mime === 'image/jpeg') {
            if (!function_exists('imagecreatefromjpeg')) return; // GD not installed
            $src = imagecreatefromjpeg($srcPath);
        } elseif ($mime === 'image/png') {
            if (!function_exists('imagecreatefrompng')) return;
            $src = imagecreatefrompng($srcPath);
        } elseif ($mime === 'image/webp') {
            if (!function_exists('imagecreatefromwebp')) return;
            $src = imagecreatefromwebp($srcPath);
        }
        if (!$src) return;
        $srcRatio = $w / $h; $dstRatio = $tw / $th;
        // Compute crop rect for center-crop
        if ($srcRatio > $dstRatio) {
            $newW = (int)($h * $dstRatio);
            $newH = $h;
            $sx = (int)(($w - $newW) / 2);
            $sy = 0;
            $cw = $newW; $ch = $newH;
        } else {
            $newW = $w; $newH = (int)($w / $dstRatio);
            $sx = 0; $sy = (int)(($h - $newH) / 2);
            $cw = $newW; $ch = $newH;
        }
        $dst = imagecreatetruecolor($tw, $th);
        imagecopyresampled($dst, $src, 0, 0, $sx, $sy, $tw, $th, $cw, $ch);
        // Save
        if ($mime === 'image/jpeg') {
            if (function_exists('imagejpeg')) imagejpeg($dst, $dstPath, 85);
        } elseif ($mime === 'image/png') {
            if (function_exists('imagepng')) imagepng($dst, $dstPath, 6);
        } elseif ($mime === 'image/webp') {
            if (function_exists('imagewebp')) imagewebp($dst, $dstPath, 85);
        }
        imagedestroy($src); imagedestroy($dst);
    }
}
