<?php
namespace App\Controllers;

use App\Core\Auth;
use App\Core\Security;
use App\Core\Controller;
use App\Core\Mail\MailService;
use App\Core\Mail\Envelope;

class MailQueueController extends Controller
{
    public function processBatch(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();

        $limit = max(1, (int)($_POST['limit'] ?? 20));

        $svc = new MailService($this->pdo);

        // Pick queued items
        $this->pdo->beginTransaction();
        $st = $this->pdo->prepare("SELECT * FROM mail_queue WHERE status='queued' ORDER BY id ASC LIMIT :lim FOR UPDATE");
        $st->bindValue(':lim', $limit, \PDO::PARAM_INT);
        $st->execute();
        $jobs = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        // Lock them as processing
        if ($jobs) {
            $ids = implode(',', array_map('intval', array_column($jobs, 'id')));
            $this->pdo->exec("UPDATE mail_queue SET status='processing', updated_at=NOW() WHERE id IN ($ids)");
        }
        $this->pdo->commit();

        $processed = 0; $sent = 0; $failed = 0;
        foreach ($jobs as $job) {
            $processed++;
            try {
                $env = new Envelope($job['to_email'], $job['subject'], $job['html'], '');
                $res = $svc->send($env);
                // Log
                $log = $this->pdo->prepare("INSERT INTO mail_logs (queue_id, booking_id, module, audience, provider, to_email, subject, status, message_id, error_text, created_at) VALUES (:qid,:bid,:mod,:aud,:prov,:to,:sub,:status,:msgid,:err,NOW())");
                $prov = $svc->activeProvider()->name();
                $log->execute([
                    ':qid'=>$job['id'], ':bid'=>$job['booking_id'], ':mod'=>$job['module'], ':aud'=>$job['audience'], ':prov'=>$prov,
                    ':to'=>$job['to_email'], ':sub'=>$job['subject'], ':status'=>$res->ok ? 'sent':'failed', ':msgid'=>$res->messageId, ':err'=>$res->ok ? null : ($res->message ?? null)
                ]);
                if ($res->ok) {
                    $upd = $this->pdo->prepare("UPDATE mail_queue SET status='sent', attempt_count=attempt_count+1, updated_at=NOW(), last_error=NULL WHERE id=:id");
                    $upd->execute([':id'=>$job['id']]);
                    $sent++;
                } else {
                    $upd = $this->pdo->prepare("UPDATE mail_queue SET status='failed', attempt_count=attempt_count+1, updated_at=NOW(), last_error=:err WHERE id=:id");
                    $upd->execute([':id'=>$job['id'], ':err'=>substr((string)$res->message,0,480)]);
                    $failed++;
                }
            } catch (\Throwable $e) {
                $upd = $this->pdo->prepare("UPDATE mail_queue SET status='failed', attempt_count=attempt_count+1, updated_at=NOW(), last_error=:err WHERE id=:id");
                $upd->execute([':id'=>$job['id'], ':err'=>substr($e->getMessage(),0,480)]);
                $failed++;
            }
        }

        $_SESSION['flash'] = sprintf('Mail queue processed: %d jobs, %d sent, %d failed.', $processed, $sent, $failed);
        $this->redirect('/admin/logs/emails');
    }
}
