<?php
namespace App\Controllers;

use App\Core\Auth;
use App\Core\Security;
use App\Core\Controller;
use App\Core\Mail\MailService;
use App\Core\Mail\Envelope;

class MailAdminController extends Controller
{
    private function requireAdmin(): void { Auth::requireRole(['Admin']); }

    public function settings(): void
    {
        $this->requireAdmin();
        $csrf = Security::csrfToken();
        $st = $this->pdo->query("SELECT * FROM mail_settings WHERE id=1");
        $settings = $st? $st->fetch(\PDO::FETCH_ASSOC) : null;
        if (!$settings) { $settings = ['active_provider'=>'smtp','from_name'=>'B2B Travel','from_email'=>'noreply@example.com']; }
        $this->view('admin/mail_settings', compact('csrf','settings'));
    }

    public function saveSettings(): void
    {
        $this->requireAdmin();
        Security::requireCsrf();
        Security::requireMasterPassword();
        $p = $_POST;
        $fields = [
            'active_provider','from_name','from_email','reply_to_email','bcc_list','ses_region','ses_smtp_username','ses_smtp_password',
            'sendgrid_api_key','smtp_host','smtp_port','smtp_username','smtp_password','smtp_encryption',
            'enable_taxi_vendor','enable_taxi_agent','enable_taxi_internal',
            'enable_hotel_vendor','enable_hotel_agent','enable_hotel_internal',
            'enable_activity_vendor','enable_activity_agent','enable_activity_internal',
        ];
        $data = [];
        foreach ($fields as $f) { $data[$f] = $p[$f] ?? null; }
        // Normalize checkboxes
        foreach (['enable_taxi_vendor','enable_taxi_agent','enable_taxi_internal','enable_hotel_vendor','enable_hotel_agent','enable_hotel_internal','enable_activity_vendor','enable_activity_agent','enable_activity_internal'] as $f) {
            $data[$f] = isset($p[$f]) ? 1 : 0;
        }
        // Basic validation
        $data['active_provider'] = in_array($data['active_provider'], ['ses','sendgrid','smtp'], true) ? $data['active_provider'] : 'smtp';
        $sql = "UPDATE mail_settings SET
            active_provider=:active_provider,
            from_name=:from_name,
            from_email=:from_email,
            reply_to_email=:reply_to_email,
            bcc_list=:bcc_list,
            ses_region=:ses_region,
            ses_smtp_username=:ses_smtp_username,
            ses_smtp_password=:ses_smtp_password,
            sendgrid_api_key=:sendgrid_api_key,
            smtp_host=:smtp_host,
            smtp_port=:smtp_port,
            smtp_username=:smtp_username,
            smtp_password=:smtp_password,
            smtp_encryption=:smtp_encryption,
            enable_taxi_vendor=:enable_taxi_vendor,
            enable_taxi_agent=:enable_taxi_agent,
            enable_taxi_internal=:enable_taxi_internal,
            enable_hotel_vendor=:enable_hotel_vendor,
            enable_hotel_agent=:enable_hotel_agent,
            enable_hotel_internal=:enable_hotel_internal,
            enable_activity_vendor=:enable_activity_vendor,
            enable_activity_agent=:enable_activity_agent,
            enable_activity_internal=:enable_activity_internal,
            updated_at=NOW()
            WHERE id=1";
        $st = $this->pdo->prepare($sql);
        $st->execute($data);
        $_SESSION['flash'] = 'Mail settings saved.';
        $this->redirect('/admin/settings/email');
    }

    public function testSend(): void
    {
        $this->requireAdmin();
        Security::requireCsrf();
        $to = trim((string)($_POST['to'] ?? ''));
        if ($to === '') { $_SESSION['errors']=['Please enter a test recipient']; $this->redirect('/admin/settings/email'); }
        $svc = new MailService($this->pdo);
        $env = new Envelope($to, 'Test Email', '<p>This is a test email from B2B Travel.</p>');
        $res = $svc->send($env);
        if ($res->ok) { $_SESSION['flash'] = 'Test email sent via ' . $svc->activeProvider()->name(); }
        else { $_SESSION['errors'] = ['Test failed: ' . $res->message]; }
        $this->redirect('/admin/settings/email');
    }

    public function logs(): void
    {
        $this->requireAdmin();
        $csrf = Security::csrfToken();
        $q = $this->pdo->query("SELECT * FROM mail_logs ORDER BY id DESC LIMIT 200");
        $logs = $q? $q->fetchAll(\PDO::FETCH_ASSOC) : [];
        $this->view('admin/mail_logs', compact('csrf','logs'));
    }

    public function resend(): void
    {
        $this->requireAdmin();
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['log_id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/logs/emails'); }
        $st = $this->pdo->prepare("SELECT * FROM mail_logs WHERE id=:id");
        $st->execute([':id'=>$id]);
        $log = $st->fetch(\PDO::FETCH_ASSOC) ?: null;
        if (!$log) { $this->redirect('/admin/logs/emails'); }
        // Requeue
        $ins = $this->pdo->prepare("INSERT INTO mail_queue (event_key,audience,module,booking_id,to_email,subject,html,locale,status,attempt_count,created_at)
            VALUES (:ev,:aud,:mod,:bid,:to,:sub,:html,:loc,'queued',0,NOW())");
        $ins->execute([
            ':ev' => 'manual.resend',
            ':aud' => $log['audience'],
            ':mod' => $log['module'],
            ':bid' => $log['booking_id'],
            ':to' => $log['to_email'],
            ':sub' => $log['subject'],
            ':html' => '(Re-send) ' . ($log['subject'] ?? '') . '<br><br>(Original send attempt ' . $id . ')',
            ':loc' => 'en',
        ]);
        $_SESSION['flash'] = 'Queued for resend.';
        $this->redirect('/admin/logs/emails');
    }

    public function webhookSes(): void
    {
        // Accept JSON from AWS SNS (direct or via subscription). Minimal bounce/complaint handling.
        $raw = file_get_contents('php://input') ?: '';
        $data = json_decode($raw, true);
        if (!$data) { http_response_code(204); return; }
        // If it's an SNS envelope, the actual notification is in Message
        if (isset($data['Type']) && isset($data['Message'])) {
            $msg = json_decode($data['Message'], true);
        } else { $msg = $data; }
        $type = strtolower((string)($msg['notificationType'] ?? ''));
        if ($type === 'bounce') {
            $recipients = (array)($msg['bounce']['bouncedRecipients'] ?? []);
            foreach ($recipients as $r) {
                $email = strtolower(trim((string)($r['emailAddress'] ?? '')));
                if ($email !== '') {
                    $st = $this->pdo->prepare("INSERT INTO mail_suppressions (email,type,provider,meta) VALUES (:e,'bounce','ses',:m)
                        ON DUPLICATE KEY UPDATE provider='ses', meta=:m");
                    $st->execute([':e'=>$email, ':m'=>json_encode($msg)]);
                }
            }
        } elseif ($type === 'complaint') {
            $recipients = (array)($msg['complaint']['complainedRecipients'] ?? []);
            foreach ($recipients as $r) {
                $email = strtolower(trim((string)($r['emailAddress'] ?? '')));
                if ($email !== '') {
                    $st = $this->pdo->prepare("INSERT INTO mail_suppressions (email,type,provider,meta) VALUES (:e,'complaint','ses',:m)
                        ON DUPLICATE KEY UPDATE provider='ses', meta=:m");
                    $st->execute([':e'=>$email, ':m'=>json_encode($msg)]);
                }
            }
        }
        http_response_code(204);
    }

    public function webhookSendGrid(): void
    {
        // SendGrid posts array of events
        $raw = file_get_contents('php://input') ?: '';
        $events = json_decode($raw, true);
        if (!is_array($events)) { http_response_code(204); return; }
        foreach ($events as $ev) {
            $event = strtolower((string)($ev['event'] ?? ''));
            $email = strtolower(trim((string)($ev['email'] ?? '')));
            if ($email === '') continue;
            if (in_array($event, ['bounce','dropped'], true)) {
                $st = $this->pdo->prepare("INSERT INTO mail_suppressions (email,type,provider,meta) VALUES (:e,'bounce','sendgrid',:m)
                    ON DUPLICATE KEY UPDATE provider='sendgrid', meta=:m");
                $st->execute([':e'=>$email, ':m'=>json_encode($ev)]);
            } elseif ($event === 'spamreport') {
                $st = $this->pdo->prepare("INSERT INTO mail_suppressions (email,type,provider,meta) VALUES (:e,'complaint','sendgrid',:m)
                    ON DUPLICATE KEY UPDATE provider='sendgrid', meta=:m");
                $st->execute([':e'=>$email, ':m'=>json_encode($ev)]);
            } elseif ($event === 'unsubscribe') {
                $st = $this->pdo->prepare("INSERT INTO mail_suppressions (email,type,provider,meta) VALUES (:e,'unsubscribe','sendgrid',:m)
                    ON DUPLICATE KEY UPDATE provider='sendgrid', meta=:m");
                $st->execute([':e'=>$email, ':m'=>json_encode($ev)]);
            }
        }
        http_response_code(204);
    }
}
