<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;
use App\Core\Audit;
use App\Services\WalletService;

class B2BApiController extends Controller
{
    // GET /b2b/api/hotels/search?city=&checkin=&checkout=&adults=&rooms=
    public function hotelsSearch(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');

        $city = trim((string)($_GET['city'] ?? ''));
        $checkin = (string)($_GET['checkin'] ?? '');
        $checkout = (string)($_GET['checkout'] ?? '');
        $adults = (int)($_GET['adults'] ?? 2);
        $roomsQty = (int)($_GET['rooms'] ?? 1);
        // New optional filters
        $minPrice = isset($_GET['min_price']) ? (float)$_GET['min_price'] : null;
        $maxPrice = isset($_GET['max_price']) ? (float)$_GET['max_price'] : null;
        $starsParam = trim((string)($_GET['stars'] ?? '')); // e.g. "3,4,5"
        $amenitiesParam = trim((string)($_GET['amenities'] ?? '')); // codes e.g. "wifi,pool"
        $freeCancel = isset($_GET['free_cancellation']) ? (int)$_GET['free_cancellation'] : null; // 0/1
        $payAtProperty = isset($_GET['pay_at_property']) ? (int)$_GET['pay_at_property'] : null; // 0/1
        $minReview = isset($_GET['min_review']) ? (float)$_GET['min_review'] : null; // 0-10
        $sort = trim((string)($_GET['sort'] ?? 'featured'));
        $centerLat = isset($_GET['center_lat']) ? (float)$_GET['center_lat'] : null;
        $centerLng = isset($_GET['center_lng']) ? (float)$_GET['center_lng'] : null;

        // Basic validation
        if ($city === '' || $checkin === '' || $checkout === '') {
            http_response_code(400);
            echo json_encode(['error' => 'Missing required params: city, checkin, checkout']);
            return;
        }

        // Nights calculation
        $nights = 1;
        try {
            $d1 = new \DateTime($checkin);
            $d2 = new \DateTime($checkout);
            $n = (int)$d1->diff($d2)->days;
            $nights = max(1, $n);
        } catch (\Throwable $e) {
            $nights = 1;
        }

        // Build dynamic WHERE for base filters (visibility + city/name tokens)
        $where = ['h.active=1', 'h.visible_agent=1', 'h.country = :ctry'];
        $params = ['cin' => $d1->format('Y-m-d'), 'cout' => $d2->format('Y-m-d'), 'ctry' => 'Thailand'];
        // Tokenize the incoming city/hotel query so inputs like "fx pattaya" work
        $tokens = array_values(array_filter(array_map('trim', preg_split('/\s+/', $city)), function($s){ return $s !== ''; }));
        if (!empty($tokens)) {
            $tokenConds = [];
            foreach ($tokens as $i => $tok) {
                $ph = 't' . $i;
                $tokenConds[] = "(h.city LIKE :$ph OR h.name LIKE :$ph)";
                $params[$ph] = '%' . $tok . '%';
            }
            // Require all tokens to match somewhere in city or name
            $where[] = '(' . implode(' AND ', $tokenConds) . ')';
        } else {
            // Fallback to simple city like
            $where[] = 'h.city LIKE :city';
            $params['city'] = "%$city%";
        }

        // Stars filter
        if ($starsParam !== '') {
            $starsArr = array_values(array_filter(array_map('intval', explode(',', $starsParam)), function($x){ return $x>0; }));
            if (!empty($starsArr)) {
                // create placeholders
                $in = [];
                foreach ($starsArr as $i => $st) { $ph = ":st$i"; $in[] = $ph; $params[substr($ph,1)] = (int)$st; }
                $where[] = 'h.stars IN (' . implode(',', $in) . ')';
            }
        }

        // Flags
        if ($payAtProperty !== null) { $where[] = 'h.pay_at_property = :pap'; $params['pap'] = (int)$payAtProperty; }
        if ($minReview !== null)   { $where[] = '(h.review_score_avg IS NULL OR h.review_score_avg >= :minrev)'; $params['minrev'] = $minReview; }

        // Amenities: require all provided codes
        $amenityJoin = '';
        if ($amenitiesParam !== '') {
            $codes = array_values(array_filter(array_map('trim', explode(',', $amenitiesParam))));
            if (!empty($codes)) {
                // Subselect hotels having all requested amenity codes
                $in = [];
                foreach ($codes as $i => $c) { $ph = ":ac$i"; $in[] = $ph; $params[substr($ph,1)] = $c; }
                $where[] = 'h.id IN (
                    SELECT ha.hotel_id FROM hotel_amenities ha
                    JOIN amenities a ON a.id = ha.amenity_id
                    WHERE a.code IN (' . implode(',', $in) . ')
                    GROUP BY ha.hotel_id
                    HAVING COUNT(DISTINCT a.code) = ' . count($codes) . '
                )';
            }
        }

        // Free cancellation via hotel_policies
        if ($freeCancel !== null) {
            $where[] = 'EXISTS (SELECT 1 FROM hotel_policies hp WHERE hp.hotel_id = h.id AND hp.free_cancellation = :fc)';
            $params['fc'] = (int)$freeCancel;
        }

        // Optional distance select
        $distanceSelect = '';
        if ($centerLat !== null && $centerLng !== null) {
            // Haversine formula (approx) -> distance in KM
            $distanceSelect = ", (6371 * acos( cos(radians(:clat)) * cos(radians(h.lat)) * cos(radians(h.lng) - radians(:clng)) + sin(radians(:clat)) * sin(radians(h.lat)) )) AS distance_km";
            $params['clat'] = $centerLat; $params['clng'] = $centerLng;
        }

        // Compose SQL with computed agent_from and optional distance
        $sql = "SELECT 
                    h.id, h.name, h.city, h.country, h.base_price, h.stars, h.review_score_avg, h.pay_at_property, h.featured_agent, h.featured_agent_label, h.lat, h.lng$distanceSelect,
                    (
                        SELECT hi.file_path FROM hotel_images hi
                        WHERE hi.hotel_id = h.id
                        ORDER BY COALESCE(hi.sort_order, 9999), hi.id DESC
                        LIMIT 1
                    ) AS thumb,
                    (
                        SELECT MIN(COALESCE(rp.agent_price, rp.price))
                        FROM hotel_rooms r
                        JOIN room_prices rp ON rp.room_id = r.id
                        WHERE r.hotel_id = h.id AND rp.date >= :cin AND rp.date < :cout
                    ) AS agent_from,
                    (
                        SELECT MIN(COALESCE(rp.vendor_cost, rp.price))
                        FROM hotel_rooms r
                        JOIN room_prices rp ON rp.room_id = r.id
                        WHERE r.hotel_id = h.id AND rp.date >= :cin AND rp.date < :cout
                    ) AS vendor_from,
                    (
                        SELECT MIN(COALESCE(rp.customer_price, rp.price))
                        FROM hotel_rooms r
                        JOIN room_prices rp ON rp.room_id = r.id
                        WHERE r.hotel_id = h.id AND rp.date >= :cin AND rp.date < :cout
                    ) AS customer_from
                FROM hotels h
                WHERE " . implode(' AND ', $where) . "
                ORDER BY 
                  h.featured_agent DESC,
                  CASE h.featured_agent_label
                    WHEN 'top_choice' THEN 4
                    WHEN 'recommended' THEN 3
                    WHEN 'trusted_partner' THEN 2
                    WHEN 'best_value' THEN 1
                    ELSE 0
                  END DESC,
                  h.stars DESC, h.id DESC
                LIMIT 200";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $hotels = $stmt->fetchAll();

        // Compute an estimated price for the stay using base_price as fallback
        $results = [];
        foreach ($hotels as $h) {
            $baseNight = (float)($h['base_price'] ?? 0);
            $agentFrom = isset($h['agent_from']) ? (float)$h['agent_from'] : null;
            $vendorFrom = isset($h['vendor_from']) ? (float)$h['vendor_from'] : null;
            $customerFrom = isset($h['customer_from']) ? (float)$h['customer_from'] : null;
            $agentBadge = ($agentFrom !== null && $agentFrom > 0) ? $agentFrom : $baseNight;
            $vendorBadge = ($vendorFrom !== null && $vendorFrom > 0) ? $vendorFrom : $baseNight;
            $customerBadge = ($customerFrom !== null && $customerFrom > 0) ? $customerFrom : $baseNight;
            $estimateNight = $agentBadge;
            $totalEstimate = $estimateNight * $nights * max(1, $roomsQty);
            $row = [
                'id' => (int)$h['id'],
                'name' => $h['name'],
                'city' => $h['city'],
                'country' => $h['country'],
                'stars' => (int)$h['stars'],
                'nightly_from' => round($baseNight, 2),
                'nightly_from_vendor' => round($vendorBadge, 2),
                'nightly_from_agent' => round($agentBadge, 2),
                'nightly_from_customer' => round($customerBadge, 2),
                'total_estimate' => round($totalEstimate, 2),
                'currency' => 'THB',
                'thumb' => $h['thumb'] ?? null,
                'review_score_avg' => isset($h['review_score_avg']) ? (float)$h['review_score_avg'] : null,
                'pay_at_property' => isset($h['pay_at_property']) ? (bool)$h['pay_at_property'] : false,
                'featured_agent' => isset($h['featured_agent']) ? (bool)$h['featured_agent'] : false,
                'featured_label' => $h['featured_agent_label'] ?? null,
                'distance_km' => isset($h['distance_km']) ? (float)$h['distance_km'] : null,
            ];
            // Apply price range filter after computing estimateNight if provided
            if ($minPrice !== null && $estimateNight < $minPrice) { continue; }
            if ($maxPrice !== null && $estimateNight > $maxPrice) { continue; }
            $results[] = $row;
        }

        // Client-side sort override
        if ($sort === 'price_asc') {
            usort($results, function($a,$b){ return ($a['nightly_from_agent'] ?? $a['nightly_from'] ?? 0) <=> ($b['nightly_from_agent'] ?? $b['nightly_from'] ?? 0); });
        } elseif ($sort === 'price_desc') {
            usort($results, function($a,$b){ return ($b['nightly_from_agent'] ?? $b['nightly_from'] ?? 0) <=> ($a['nightly_from_agent'] ?? $a['nightly_from'] ?? 0); });
        } elseif ($sort === 'stars_desc') {
            usort($results, function($a,$b){ return ($b['stars'] ?? 0) <=> ($a['stars'] ?? 0); });
        } elseif ($sort === 'distance_asc') {
            usort($results, function($a,$b){ return ( ($a['distance_km'] ?? 99999) <=> ($b['distance_km'] ?? 99999) ); });
        }

        echo json_encode([
            'meta' => [
                'city' => $city,
                'checkin' => $checkin,
                'checkout' => $checkout,
                'nights' => $nights,
                'adults' => $adults,
                'rooms' => $roomsQty,
                'filters' => [
                    'min_price' => $minPrice,
                    'max_price' => $maxPrice,
                    'stars' => $starsParam,
                    'amenities' => $amenitiesParam,
                    'free_cancellation' => $freeCancel,
                    'pay_at_property' => $payAtProperty,
                    'min_review' => $minReview,
                    'sort' => $sort,
                    'center_lat' => $centerLat,
                    'center_lng' => $centerLng,
                ]
            ],
            'data' => $results,
        ]);
    }

    // GET /b2b/api/security/pin-key
    // Provides a temporary RSA public key for encrypting TX PIN on the client.
    // The matching private key is stored in session and rotated on each request.
    public function txPinKey(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        // Generate ephemeral 2048-bit RSA keypair
        $conf = [ 'private_key_bits' => 2048, 'private_key_type' => OPENSSL_KEYTYPE_RSA ];
        $res = openssl_pkey_new($conf);
        if (!$res) { http_response_code(500); echo json_encode(['error' => 'Key generation failed']); return; }
        $privPem = '';
        openssl_pkey_export($res, $privPem);
        $pub = openssl_pkey_get_details($res);
        $pubPem = $pub['key'] ?? '';
        if (!$privPem || !$pubPem) { http_response_code(500); echo json_encode(['error' => 'Key export failed']); return; }
        // Store private key in session (short-lived)
        $_SESSION['txpin_privkey'] = [ 'pem' => $privPem, 'ts' => time() ];
        echo json_encode([ 'alg' => 'RSA-OAEP', 'hash' => 'SHA-1', 'public_key_pem' => $pubPem ]);
    }

    // GET /b2b/api/hotels/details?id=&checkin=&checkout=
    public function hotelDetails(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');

        $id = (int)($_GET['id'] ?? 0);
        $checkin = (string)($_GET['checkin'] ?? '');
        $checkout = (string)($_GET['checkout'] ?? '');
        if ($id <= 0) {
            http_response_code(400);
            echo json_encode(['error' => 'Missing id']);
            return;
        }

        $stmt = $this->pdo->prepare('SELECT * FROM hotels WHERE id=:id AND active=1 AND visible_agent=1');
        $stmt->execute(['id' => $id]);
        $hotel = $stmt->fetch();
        if (!$hotel) {
            http_response_code(404);
            echo json_encode(['error' => 'Hotel not found']);
            return;
        }

        // Rooms
        $rooms = $this->pdo->prepare('SELECT id, name, occupancy_adults as capacity FROM hotel_rooms WHERE hotel_id=:hid AND active=1');
        $rooms->execute(['hid' => $id]);
        $roomRows = $rooms->fetchAll();

        // Compute vendor/agent/customer nightly-from with fallbacks
        $nightlyBase = (float)($hotel['base_price'] ?? 0.0);
        $fromD = null; $toD = null;
        if ($checkin !== '' && $checkout !== '') {
            try { $fromD = new \DateTime($checkin); $toD = new \DateTime($checkout); } catch (\Throwable $e) { $fromD = null; $toD = null; }
        }
        if ($fromD && $toD && $fromD < $toD) {
            // Prepare statement to fetch top meal plan for a room within date range
            $mealPlanStmt = $this->pdo->prepare(
                "SELECT rp2.meal_plan
                 FROM room_prices rp
                 JOIN rate_plans rp2 ON rp2.id = rp.rate_plan_id
                 WHERE rp.room_id = :rid AND rp.date >= :cin AND rp.date < :cout
                 GROUP BY rp2.meal_plan
                 ORDER BY FIELD(rp2.meal_plan,'FB','HB','BB','RO') ASC
                 LIMIT 1"
            );

            foreach ($roomRows as &$r) {
                $rid = (int)$r['id'];
                $minVendor = null; $minAgent = null; $minCustomer = null;
                try {
                    $sql = "SELECT 
                                MIN(COALESCE(rp.vendor_cost, rp.price)) AS v,
                                MIN(COALESCE(rp.agent_price, rp.price))  AS a,
                                MIN(COALESCE(rp.customer_price, rp.price)) AS c
                            FROM room_prices rp 
                            WHERE rp.room_id = :rid AND rp.date >= :cin AND rp.date < :cout";
                    $st = $this->pdo->prepare($sql);
                    $st->execute(['rid' => $rid, 'cin' => $fromD->format('Y-m-d'), 'cout' => $toD->format('Y-m-d')]);
                    $row = $st->fetch(\PDO::FETCH_ASSOC);
                    if ($row) {
                        if ($row['v'] !== null) $minVendor = (float)$row['v'];
                        if ($row['a'] !== null) $minAgent = (float)$row['a'];
                        if ($row['c'] !== null) $minCustomer = (float)$row['c'];
                    }
                } catch (\Throwable $e) { /* ignore */ }
                $r['nightly_from_vendor'] = $minVendor !== null ? $minVendor : $nightlyBase;
                $r['nightly_from_agent'] = $minAgent !== null ? $minAgent : $nightlyBase;
                $r['nightly_from_customer'] = $minCustomer !== null ? $minCustomer : $nightlyBase;
                $r['nightly_from_base'] = $nightlyBase;

                // Attach best-available meal plan for this room over the requested dates (FB > HB > BB > RO)
                try {
                    $mealPlanStmt->execute(['rid' => $rid, 'cin' => $fromD->format('Y-m-d'), 'cout' => $toD->format('Y-m-d')]);
                    $mp = $mealPlanStmt->fetchColumn();
                    if ($mp !== false && $mp !== null && $mp !== '') {
                        $r['meal_plan'] = $mp; // one of RO, BB, HB, FB
                    }
                } catch (\Throwable $e) { /* ignore meal plan */ }
            }
            unset($r);
        } else {
            foreach ($roomRows as &$r) {
                $r['nightly_from_vendor'] = $nightlyBase;
                $r['nightly_from_agent'] = $nightlyBase;
                $r['nightly_from_customer'] = $nightlyBase;
                $r['nightly_from_base'] = $nightlyBase;
                // No date range provided; omit meal_plan as we cannot infer it reliably
            }
            unset($r);
        }

        // Gallery (if any)
        $gal = $this->pdo->prepare('SELECT file_path, caption FROM hotel_images WHERE hotel_id=:hid ORDER BY sort_order ASC, id ASC LIMIT 12');
        $gal->execute(['hid' => $id]);
        $gallery = $gal->fetchAll();

        // Policies
        $pol = $this->pdo->prepare('SELECT checkin_time, checkout_time, child_policy, pet_policy, terms_html FROM hotel_policies WHERE hotel_id=:hid');
        $pol->execute(['hid' => $id]);
        $policies = $pol->fetch() ?: null;

        echo json_encode([
            'hotel' => [
                'id' => (int)$hotel['id'],
                'name' => $hotel['name'],
                'city' => $hotel['city'],
                'country' => $hotel['country'],
                'stars' => (int)($hotel['stars'] ?? 0),
            ],
            'rooms' => $roomRows,
            'gallery' => $gallery,
            'policies' => $policies,
            'meta' => [
                'checkin' => $checkin,
                'checkout' => $checkout,
                'currency' => 'THB',
            ],
        ]);
    }

    // POST /b2b/api/hotels/quote
    // Body JSON: { hotel_id, checkin, checkout, adults?, rooms, selections: [{room_id, rate_plan_id?, qty_rooms}] }
    public function hotelsQuote(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');

        // Parse JSON body
        $raw = file_get_contents('php://input') ?: '';
        $body = json_decode($raw, true) ?: [];

        $user = $_SESSION['user'] ?? null;
        if (!$user) { http_response_code(401); echo json_encode(['error' => 'Unauthorized']); return; }

        $hotelId = (int)($body['hotel_id'] ?? 0);
        $checkin = trim((string)($body['checkin'] ?? ''));
        $checkout = trim((string)($body['checkout'] ?? ''));
        $adults = (int)($body['adults'] ?? 2);
        $roomsQty = max(1, (int)($body['rooms'] ?? 1));
        $selections = is_array($body['selections'] ?? null) ? $body['selections'] : [];

        if ($hotelId <= 0 || $checkin === '' || $checkout === '' || empty($selections)) {
            http_response_code(422);
            echo json_encode(['error' => 'hotel_id, checkin, checkout, selections required']);
            return;
        }

        // Date parse
        try { $d1 = new \DateTime($checkin); $d2 = new \DateTime($checkout); } catch (\Throwable $e) { $d1 = null; $d2 = null; }
        if (!$d1 || !$d2 || $d2 <= $d1) { http_response_code(422); echo json_encode(['error' => 'Invalid date range']); return; }
        $nights = (int)$d1->diff($d2)->days;

        // Validate hotel visibility
        $stH = $this->pdo->prepare('SELECT id, name, base_price FROM hotels WHERE id=:id AND active=1 AND visible_agent=1');
        $stH->execute(['id' => $hotelId]);
        $hotel = $stH->fetch();
        if (!$hotel) { http_response_code(404); echo json_encode(['error' => 'Hotel not found']); return; }

        // Helper: compute nightly agent price sum for a room_id
        $computeRoom = function(int $roomId) use ($d1, $d2, $nights, $hotel) {
            $sum = 0.0; $perNight = []; $missing = 0;
            try {
                $sql = "SELECT rp.date, COALESCE(rp.agent_price, rp.price) AS price FROM room_prices rp WHERE rp.room_id = :rid AND rp.date >= :cin AND rp.date < :cout ORDER BY rp.date ASC";
                $s = $this->pdo->prepare($sql);
                $s->execute(['rid' => $roomId, 'cin' => $d1->format('Y-m-d'), 'cout' => $d2->format('Y-m-d')]);
                $rows = $s->fetchAll(\PDO::FETCH_ASSOC) ?: [];
                foreach ($rows as $r) { $perNight[$r['date']] = (float)($r['price'] ?? 0); }
            } catch (\Throwable $e) { /* ignore */ }
            if (count($perNight) === 0) {
                // fallback base
                $unit = (float)($hotel['base_price'] ?? 0);
                $sum = $unit * $nights;
                return [$sum, $unit, $nights, $missing, 'base'];
            }
            for ($i=0; $i<$nights; $i++) {
                $ds = (clone $d1)->modify("+{$i} day")->format('Y-m-d');
                if (!array_key_exists($ds, $perNight)) { $missing++; }
                else { $sum += (float)$perNight[$ds]; }
            }
            if ($missing > 0) { return [null, null, $nights, $missing, 'incomplete']; }
            $avg = $nights > 0 ? ($sum / $nights) : $sum;
            return [$sum, $avg, $nights, 0, 'room_prices'];
        };

        // Start TX
        $this->pdo->beginTransaction();
        try {
            // Generate codes
            $quoteCode = 'QH-' . substr(bin2hex(random_bytes(8)), 0, 12);
            $expiresAt = (new \DateTimeImmutable('+20 minutes'))->format('Y-m-d H:i:s');

            // Insert quote
            $subtotal = 0.0; $taxes = 0.0; $total = 0.0;
            $stmtQ = $this->pdo->prepare("INSERT INTO hotel_quotes (quote_code, agent_user_id, hotel_id, checkin, checkout, adults, rooms, currency, subtotal, taxes, total_amount, pay_at_property, status, expires_at, meta) VALUES (:qc, :uid, :hid, :cin, :cout, :ad, :rm, 'THB', 0, 0, 0, 0, 'draft', :exp, NULL)");
            $stmtQ->execute([
                'qc' => $quoteCode,
                'uid' => (int)$user['id'],
                'hid' => $hotelId,
                'cin' => $d1->format('Y-m-d'),
                'cout' => $d2->format('Y-m-d'),
                'ad' => $adults,
                'rm' => $roomsQty,
                'exp' => $expiresAt,
            ]);
            $quoteId = (int)$this->pdo->lastInsertId();

            // Insert items
            $insItem = $this->pdo->prepare("INSERT INTO hotel_quote_items (quote_id, room_id, rate_plan_id, offer_id, nightly_agent_price, nights, qty_rooms, total, policy_snapshot) VALUES (:qid, :rid, :rp, :off, :nap, :n, :qty, :tot, NULL)");
            foreach ($selections as $sel) {
                $rid = (int)($sel['room_id'] ?? 0);
                $rpid = isset($sel['rate_plan_id']) && $sel['rate_plan_id'] !== '' ? (int)$sel['rate_plan_id'] : null;
                $qty = max(1, (int)($sel['qty_rooms'] ?? 1));
                if ($rid <= 0) { throw new \Exception('Invalid room_id in selections'); }
                // ensure room belongs to hotel and active
                $chk = $this->pdo->prepare('SELECT id FROM hotel_rooms WHERE id=:rid AND hotel_id=:hid AND active=1');
                $chk->execute(['rid' => $rid, 'hid' => $hotelId]);
                if (!$chk->fetch()) { throw new \Exception('Room not available'); }

                [$sum, $nightlyAvg, $n, $missing, $basis] = $computeRoom($rid);
                if ($sum === null) { throw new \Exception('Price incomplete for selected dates'); }
                $line = $sum * $qty;
                $subtotal += $line;
                $offer = sprintf('OF-%s', substr(sha1($hotelId.'-'.$rid.'-'.($rpid ?? '0').'-'.$d1->format('Ymd').'-'.$d2->format('Ymd').'-'.$adults.'-'.$roomsQty), 0, 10));
                $insItem->execute([
                    'qid' => $quoteId,
                    'rid' => $rid,
                    'rp'  => $rpid,
                    'off' => $offer,
                    'nap' => round((float)$nightlyAvg, 2),
                    'n'   => $n,
                    'qty' => $qty,
                    'tot' => round((float)$line, 2),
                ]);
            }

            $taxes = 0.0; // add tax logic later if needed
            $total = $subtotal + $taxes;
            $upd = $this->pdo->prepare('UPDATE hotel_quotes SET subtotal=:s, taxes=:t, total_amount=:ta WHERE id=:id');
            $upd->execute(['s' => round($subtotal,2), 't' => round($taxes,2), 'ta' => round($total,2), 'id' => $quoteId]);

            $this->pdo->commit();
            echo json_encode([
                'quote_code' => $quoteCode,
                'expires_at' => $expiresAt,
                'currency' => 'THB',
                'totals' => ['subtotal' => round($subtotal,2), 'taxes' => round($taxes,2), 'total' => round($total,2)],
            ]);
        } catch (\Throwable $e) {
            $this->pdo->rollBack();
            http_response_code(400);
            echo json_encode(['error' => 'Quote failed', 'message' => $e->getMessage()]);
        }
    }

    // POST /b2b/api/hotels/book
    // Body JSON: { quote_code, contact: {name,email,mobile}, payment: {method: 'wallet'|'none', pay_at_property: 0|1} }
    public function hotelsBook(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        $raw = file_get_contents('php://input') ?: '';
        $body = json_decode($raw, true) ?: [];
        $user = $_SESSION['user'] ?? null;
        if (!$user) { http_response_code(401); echo json_encode(['error' => 'Unauthorized']); return; }

        $quoteCode = trim((string)($body['quote_code'] ?? ''));
        $payment = $body['payment'] ?? [];
        $payAtProperty = (int)($payment['pay_at_property'] ?? 0) === 1 ? 1 : 0;
        $method = (string)($payment['method'] ?? 'none');

        if ($quoteCode === '') { http_response_code(422); echo json_encode(['error' => 'quote_code required']); return; }

        // Load quote
        $q = $this->pdo->prepare('SELECT * FROM hotel_quotes WHERE quote_code = :qc AND agent_user_id = :uid');
        $q->execute(['qc' => $quoteCode, 'uid' => (int)$user['id']]);
        $quote = $q->fetch();
        if (!$quote) { http_response_code(404); echo json_encode(['error' => 'Quote not found']); return; }
        if ($quote['status'] !== 'draft') { http_response_code(409); echo json_encode(['error' => 'Quote already used or expired']); return; }
        $now = new \DateTimeImmutable('now');
        if ($now > new \DateTimeImmutable($quote['expires_at'])) { http_response_code(409); echo json_encode(['error' => 'Quote expired']); return; }

        // Fetch items
        $it = $this->pdo->prepare('SELECT * FROM hotel_quote_items WHERE quote_id = :qid');
        $it->execute(['qid' => (int)$quote['id']]);
        $items = $it->fetchAll();
        if (!$items || count($items) === 0) { http_response_code(400); echo json_encode(['error' => 'Quote has no items']); return; }

        // Create booking
        $this->pdo->beginTransaction();
        try {
            $bookingCode = 'HB-' . (new \DateTime())->format('Ymd') . '-' . strtoupper(substr(bin2hex(random_bytes(4)), 0, 6));
            $paymentStatus = $payAtProperty === 1 ? 'none' : ($method === 'wallet' ? 'pending' : 'pending');
            $status = 'confirmed'; // or 'pending' based on downstream confirmation rules

            $insB = $this->pdo->prepare("INSERT INTO hotel_bookings (booking_code, agent_user_id, quote_id, hotel_id, checkin, checkout, adults, rooms, currency, subtotal, taxes, total_amount, payment_status, status, pay_at_property, policy_snapshot) VALUES (:bc, :uid, :qid, :hid, :cin, :cout, :ad, :rm, 'THB', :sub, :tax, :tot, :pay, :st, :pap, NULL)");
            $insB->execute([
                'bc' => $bookingCode,
                'uid' => (int)$user['id'],
                'qid' => (int)$quote['id'],
                'hid' => (int)$quote['hotel_id'],
                'cin' => $quote['checkin'],
                'cout' => $quote['checkout'],
                'ad' => (int)$quote['adults'],
                'rm' => (int)$quote['rooms'],
                'sub' => (float)$quote['subtotal'],
                'tax' => (float)$quote['taxes'],
                'tot' => (float)$quote['total_amount'],
                'pay' => $paymentStatus,
                'st'  => $status,
                'pap' => $payAtProperty,
            ]);
            $bookingId = (int)$this->pdo->lastInsertId();

            $insBR = $this->pdo->prepare('INSERT INTO hotel_booking_rooms (booking_id, room_id, rate_plan_id, price_per_night, nights, qty_rooms, total) VALUES (:bid, :rid, :rp, :ppn, :n, :qty, :tot)');
            foreach ($items as $itRow) {
                $insBR->execute([
                    'bid' => $bookingId,
                    'rid' => (int)$itRow['room_id'],
                    'rp'  => $itRow['rate_plan_id'] !== null ? (int)$itRow['rate_plan_id'] : null,
                    'ppn' => (float)$itRow['nightly_agent_price'],
                    'n'   => (int)$itRow['nights'],
                    'qty' => (int)$itRow['qty_rooms'],
                    'tot' => (float)$itRow['total'],
                ]);
            }

            // Mark quote as booked
            $uq = $this->pdo->prepare('UPDATE hotel_quotes SET status = "booked" WHERE id = :id');
            $uq->execute(['id' => (int)$quote['id']]);

            $this->pdo->commit();
            echo json_encode(['booking_code' => $bookingCode, 'status' => $status, 'payment_status' => $paymentStatus, 'total_amount' => (float)$quote['total_amount']]);
        } catch (\Throwable $e) {
            $this->pdo->rollBack();
            http_response_code(400);
            echo json_encode(['error' => 'Booking failed', 'message' => $e->getMessage()]);
        }
    }

    // GET /b2b/api/hotels/booking?code=
    public function hotelBookingGet(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        $code = trim((string)($_GET['code'] ?? ''));
        $user = $_SESSION['user'] ?? null;
        if (!$user) { http_response_code(401); echo json_encode(['error' => 'Unauthorized']); return; }
        if ($code === '') { http_response_code(422); echo json_encode(['error' => 'code required']); return; }

        $b = $this->pdo->prepare('SELECT * FROM hotel_bookings WHERE booking_code = :bc AND agent_user_id = :uid');
        $b->execute(['bc' => $code, 'uid' => (int)$user['id']]);
        $bk = $b->fetch();
        if (!$bk) { http_response_code(404); echo json_encode(['error' => 'Not found']); return; }

        $r = $this->pdo->prepare('SELECT room_id, rate_plan_id, price_per_night, nights, qty_rooms, total FROM hotel_booking_rooms WHERE booking_id = :bid');
        $r->execute(['bid' => (int)$bk['id']]);
        $rooms = $r->fetchAll();

        echo json_encode([
            'booking' => [
                'booking_code' => $bk['booking_code'],
                'hotel_id' => (int)$bk['hotel_id'],
                'checkin' => $bk['checkin'],
                'checkout' => $bk['checkout'],
                'adults' => (int)$bk['adults'],
                'rooms' => (int)$bk['rooms'],
                'currency' => $bk['currency'],
                'subtotal' => (float)$bk['subtotal'],
                'taxes' => (float)$bk['taxes'],
                'total_amount' => (float)$bk['total_amount'],
                'payment_status' => $bk['payment_status'],
                'status' => $bk['status'],
                'pay_at_property' => (bool)$bk['pay_at_property'],
                'created_at' => $bk['created_at'],
            ],
            'items' => $rooms,
        ]);
    }

    // GET /b2b/api/hotels/quote?hotel_id=&room_id=&checkin=&checkout=&rooms=
    public function hotelQuote(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');

        $hotelId = (int)($_GET['hotel_id'] ?? 0);
        $roomId = isset($_GET['room_id']) && $_GET['room_id'] !== '' ? (int)$_GET['room_id'] : null;
        $checkin = trim((string)($_GET['checkin'] ?? ''));
        $checkout = trim((string)($_GET['checkout'] ?? ''));
        $roomsQty = max(1, (int)($_GET['rooms'] ?? 1));

        if ($hotelId <= 0 || $checkin === '' || $checkout === '') {
            http_response_code(422);
            echo json_encode(['error' => 'hotel_id, checkin, checkout required']);
            return;
        }

        // Parse dates robustly
        $parseDate = function(string $s){
            $s = trim($s);
            if ($s === '') return null;
            $fmts = ['d/m/Y','d-m-Y','Y-m-d','m/d/Y'];
            foreach ($fmts as $f) {
                $dt = \DateTime::createFromFormat($f, $s);
                if ($dt instanceof \DateTime) return $dt;
            }
            $ts = strtotime($s);
            if ($ts !== false) { return (new \DateTime())->setTimestamp($ts); }
            return null;
        };
        $d1 = $parseDate($checkin);
        $d2 = $parseDate($checkout);
        if (!$d1 || !$d2) { http_response_code(422); echo json_encode(['error' => 'Invalid date format']); return; }
        $nights = (int)$d1->diff($d2)->days;
        if ($nights <= 0) { http_response_code(422); echo json_encode(['error' => 'checkout must be after checkin']); return; }

        // Fetch hotel for base price and visibility
        $stmt = $this->pdo->prepare('SELECT id, name, base_price FROM hotels WHERE id=:id AND active=1 AND visible_agent=1 LIMIT 1');
        $stmt->execute(['id' => $hotelId]);
        $hotel = $stmt->fetch();
        if (!$hotel) { http_response_code(404); echo json_encode(['error' => 'Hotel not found']); return; }

        // Collect nightly prices when room specified
        $perNight = [];
        if ($roomId) {
            try {
                $sql = "SELECT rp.date, COALESCE(rp.agent_price, rp.price) AS price
                        FROM room_prices rp
                        WHERE rp.room_id = :rid AND rp.date >= :cin AND rp.date < :cout
                        ORDER BY rp.date ASC";
                $s = $this->pdo->prepare($sql);
                $s->execute(['rid' => $roomId, 'cin' => $d1->format('Y-m-d'), 'cout' => $d2->format('Y-m-d')]);
                $rows = $s->fetchAll(\PDO::FETCH_ASSOC) ?: [];
                foreach ($rows as $r) { $perNight[$r['date']] = (float)($r['price'] ?? 0); }
            } catch (\Throwable $e) { /* ignore */ }
            if (count($perNight) === 0) {
                try {
                    $sql2 = "SELECT date, price FROM hotel_room_prices WHERE room_id = :rid AND date >= :cin AND date < :cout ORDER BY date ASC";
                    $s2 = $this->pdo->prepare($sql2);
                    $s2->execute(['rid' => $roomId, 'cin' => $d1->format('Y-m-d'), 'cout' => $d2->format('Y-m-d')]);
                    $rows2 = $s2->fetchAll(\PDO::FETCH_ASSOC) ?: [];
                    foreach ($rows2 as $r) { $perNight[$r['date']] = (float)($r['price'] ?? 0); }
                } catch (\Throwable $e) { /* ignore */ }
            }
        }

        // Build timeline and sum
        $dates = [];
        $sum = 0.0;
        $missing = 0;
        for ($i = 0; $i < $nights; $i++) {
            $d = (clone $d1)->modify("+{$i} day");
            $ds = $d->format('Y-m-d');
            $price = null;
            if (!empty($perNight)) {
                if (array_key_exists($ds, $perNight)) {
                    $price = (float)$perNight[$ds];
                    $sum += $price;
                } else {
                    $missing++;
                }
            }
            $dates[] = ['date' => $ds, 'price' => $price];
        }

        $pricingBasis = 'room_prices_per_night';
        if (empty($perNight)) {
            $unitNight = (float)($hotel['base_price'] ?? 0);
            if ($unitNight < 0) $unitNight = 0;
            $sum = $unitNight * $nights;
            // also fill per-night for clarity
            $dates = [];
            for ($i = 0; $i < $nights; $i++) {
                $d = (clone $d1)->modify("+{$i} day");
                $dates[] = ['date' => $d->format('Y-m-d'), 'price' => $unitNight];
            }
            $pricingBasis = 'base_price_per_night_per_room';
        }

        $lineTotal = $sum * max(1, $roomsQty);

        echo json_encode([
            'hotel_id' => (int)$hotelId,
            'room_id' => $roomId,
            'checkin' => $d1->format('Y-m-d'),
            'checkout' => $d2->format('Y-m-d'),
            'nights' => $nights,
            'rooms' => $roomsQty,
            'dates' => $dates,
            'sum_nightly' => round($sum, 2),
            'total' => round($lineTotal, 2),
            'currency' => 'THB',
            'pricing_basis' => $pricingBasis,
            'missing_nights' => (int)$missing,
            'available' => ($pricingBasis === 'base_price_per_night_per_room') ? true : ($missing === 0),
        ]);
    }

    // GET /b2b/api/wallet
    public function wallet(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        $user = $_SESSION['user'] ?? null;
        if (!$user) {
            http_response_code(401);
            echo json_encode(['error' => 'Unauthorized']);
            return;
        }
        $stmt = $this->pdo->prepare('SELECT w.balance, u.tx_pin_locked_until, u.tx_pin_attempts FROM wallets w JOIN users u ON u.id = w.user_id WHERE w.user_id = :uid');
        $stmt->execute(['uid' => (int)$user['id']]);
        $row = $stmt->fetch();
        $balance = $row ? (float)$row['balance'] : 0.0;
        $lockedUntil = $row['tx_pin_locked_until'] ?? null;
        $locked = false; $lockedMessage = null;
        if ($lockedUntil) {
            $now = new \DateTimeImmutable('now');
            $lu = new \DateTimeImmutable($lockedUntil);
            if ($lu > $now) {
                $locked = true;
                $diffH = (int)$lu->diff($now)->h + ($lu->diff($now)->days * 24);
                $diffM = (int)$lu->diff($now)->i;
                $eta = ($diffH > 0 ? $diffH . 'h ' : '') . $diffM . 'm';
                $lockedMessage = 'Wallet is temporarily disabled due to multiple invalid PIN attempts. Try again in ' . $eta . '.';
            }
        }
        echo json_encode([
            'balance' => round($balance, 2),
            'currency' => 'THB',
            'locked' => $locked,
            'locked_until' => $lockedUntil,
            'locked_message' => $locked ? $lockedMessage : null,
            'attempts' => (int)($row['tx_pin_attempts'] ?? 0),
        ]);
    }

    // GET /b2b/api/activities/search?city=&country=&q=&date=&pax=&type=
    public function activitiesSearch(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        $city = trim((string)($_GET['city'] ?? ''));
        $country = trim((string)($_GET['country'] ?? ''));
        $q = trim((string)($_GET['q'] ?? ''));
        $date = (string)($_GET['date'] ?? '');
        $pax = (int)($_GET['pax'] ?? 2);
        $type = trim((string)($_GET['type'] ?? ''));

        // Source activities from active vendor packages for activity vendors; compute minimal agent base price
        $where = ["p.active=1", "v.module='activity'"];
        $params = [];
        if ($city !== '') { $where[] = 'v.city LIKE :city'; $params['city'] = "%$city%"; }
        if ($q !== '') { $where[] = '(p.name LIKE :q OR v.name LIKE :q)'; $params['q'] = "%$q%"; }

        $sql = "SELECT 
                    p.id AS id,
                    p.name AS name,
                    v.city AS city,
                    p.thumbnail_path AS thumbnail,
                    MIN(CASE WHEN pr.price_type='base' AND pr.pax_type IN ('adult','flat') THEN pr.agent_price ELSE NULL END) AS from_price
                FROM vendor_packages p
                JOIN vendors v ON v.id = p.vendor_id
                LEFT JOIN vendor_package_variants vv ON vv.package_id = p.id AND vv.active=1
                LEFT JOIN vendor_package_prices pr ON pr.variant_id = vv.id AND pr.active=1
                WHERE " . implode(' AND ', $where) . "
                " . (
                    $type === 'family'
                    ? " AND EXISTS (SELECT 1 FROM vendor_package_prices pr2 WHERE pr2.package_id = p.id AND pr2.active=1 AND pr2.pax_type='child')"
                    : ($type === 'adult_only'
                        ? " AND NOT EXISTS (SELECT 1 FROM vendor_package_prices pr3 WHERE pr3.package_id = p.id AND pr3.active=1 AND pr3.pax_type='child')"
                        : ""
                    )
                ) . "
                GROUP BY p.id, p.name, v.city, p.thumbnail_path
                ORDER BY from_price ASC, p.id DESC
                LIMIT 100";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $rows = $stmt->fetchAll();

        // Normalize from_price and ensure numeric values
        foreach ($rows as &$r) {
            $r['from_price'] = isset($r['from_price']) ? (float)$r['from_price'] : 0.0;
        }

        echo json_encode(['meta' => ['city' => $city, 'country' => $country, 'q' => $q, 'date' => $date, 'pax' => $pax, 'type' => $type], 'data' => $rows]);
    }

    // GET /b2b/api/activities/details?id=
    public function activitiesDetails(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { http_response_code(400); echo json_encode(['error' => 'Missing id']); return; }

        // Fetch package basic info
        $pkgSql = "SELECT p.id, p.name, p.requires_show_time, p.age_policy, p.address_override, p.active, p.thumbnail_path, v.id AS vendor_id, v.name AS vendor_name, v.city
                   FROM vendor_packages p
                   JOIN vendors v ON v.id = p.vendor_id
                   WHERE p.id = :id";
        $stmt = $this->pdo->prepare($pkgSql);
        $stmt->execute(['id' => $id]);
        $package = $stmt->fetch();
        if (!$package) { http_response_code(404); echo json_encode(['error' => 'Not found']); return; }

        // Fetch active variants
        $varSql = "SELECT vv.id, vv.name, vv.notes, vv.active
                   FROM vendor_package_variants vv
                   WHERE vv.package_id = :id AND vv.active=1
                   ORDER BY vv.id ASC";
        $stmt = $this->pdo->prepare($varSql);
        $stmt->execute(['id' => $id]);
        $variants = $stmt->fetchAll();

        // Fetch active prices per variant
        $priceSql = "SELECT pr.id, pr.variant_id, pr.price_type, pr.pax_type, pr.min_quantity,
                            pr.agent_price, pr.currency, pr.pickup_type, pr.pickup_scope,
                            pr.pickup_radius_km, pr.pickup_fee, pr.pickup_notes
                     FROM vendor_package_prices pr
                     WHERE pr.package_id = :id AND pr.active=1
                     ORDER BY pr.variant_id, pr.price_type, pr.pax_type";
        $stmt = $this->pdo->prepare($priceSql);
        $stmt->execute(['id' => $id]);
        $prices = $stmt->fetchAll();

        // Fetch showtimes (both package-level and variant-level)
        $timeSql = "SELECT id, variant_id, time
                    FROM vendor_package_showtimes
                    WHERE package_id = :id AND active=1
                    ORDER BY variant_id IS NOT NULL DESC, time ASC";
        $stmt = $this->pdo->prepare($timeSql);
        $stmt->execute(['id' => $id]);
        $times = $stmt->fetchAll();

        // Compute per-variant from_price
        $variantFrom = [];
        foreach ($prices as $pr) {
            if ($pr['price_type'] === 'base' && in_array($pr['pax_type'], ['adult','flat'], true)) {
                $vid = (int)$pr['variant_id'];
                if (!isset($variantFrom[$vid]) || (float)$pr['agent_price'] < $variantFrom[$vid]) {
                    $variantFrom[$vid] = (float)$pr['agent_price'];
                }
            }
        }
        foreach ($variants as &$v) {
            $vid = (int)$v['id'];
            $v['from_price'] = isset($variantFrom[$vid]) ? (float)$variantFrom[$vid] : null;
            // attach showtimes for this variant (or empty)
            $v['showtimes'] = array_values(array_map(function($t){ return $t['time']; }, array_filter($times, function($t) use ($vid){ return (int)($t['variant_id'] ?? 0) === $vid; })));
        }

        echo json_encode([
            'package' => $package,
            'variants' => $variants,
            'prices' => $prices,
            'showtimes' => $times
        ]);
    }

    // GET /b2b/api/activities/cities?q=
    public function activitiesCities(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        $q = trim((string)($_GET['q'] ?? ''));
        $country = trim((string)($_GET['country'] ?? ''));
        $rows = [];
        
        // If a country is specified, return ALL cities from locations for that country
        if ($country !== '') {
            try {
                $params = ['country' => $country];
                $sql = 'SELECT l.city AS city, COUNT(a.id) AS cnt
                        FROM locations l
                        LEFT JOIN activities a ON a.city = l.city
                        WHERE l.country = :country AND l.city IS NOT NULL AND l.city <> \'' . "'";
                if ($q !== '') { $sql .= ' AND l.city LIKE :q'; $params['q'] = "%$q%"; }
                $sql .= ' GROUP BY l.city ORDER BY cnt DESC, l.city ASC LIMIT 500';
                $stmt = $this->pdo->prepare($sql);
                $stmt->execute($params);
                $rows = $stmt->fetchAll();
            } catch (\Throwable $e) { /* ignore, fallback below */ }
        }

        if ($country !== '' && $rows && count($rows) > 0) {
            $data = array_map(function($r){ return ['city' => $r['city'], 'count' => (int)($r['cnt'] ?? 0)]; }, $rows ?: []);
            echo json_encode(['data' => $data]);
            return;
        }
        // 1) Prefer vendors with module='activity'
        try {
            $params = [];
            $sql = "SELECT v.city AS city, COUNT(*) AS cnt FROM vendors v WHERE v.module='activity' AND v.city IS NOT NULL AND v.city<>''";
            if ($q !== '') { $sql .= ' AND v.city LIKE :q'; $params['q'] = "%$q%"; }
            $sql .= ' GROUP BY v.city ORDER BY cnt DESC, v.city ASC LIMIT 100';
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            $rows = $stmt->fetchAll();
        } catch (\Throwable $e) { /* ignore */ }

        // 2) Fallback to activities table if vendors yielded nothing
        if (!$rows || count($rows) === 0) {
            try {
                $params = [];
                $sql = 'SELECT city, COUNT(*) AS cnt FROM activities WHERE city IS NOT NULL AND city<>\'' . "'";
                if ($q !== '') { $sql .= ' AND city LIKE :q'; $params['q'] = "%$q%"; }
                $sql .= ' GROUP BY city ORDER BY cnt DESC, city ASC LIMIT 100';
                $stmt = $this->pdo->prepare($sql);
                $stmt->execute($params);
                $rows = $stmt->fetchAll();
            } catch (\Throwable $e) { /* ignore */ }
        }

        // 3) Fallback to locations master (counts via LEFT JOIN activities)
        if (!$rows || count($rows) === 0) {
            try {
                $params = [];
                $sql = 'SELECT l.city AS city, COUNT(a.id) AS cnt FROM locations l LEFT JOIN activities a ON a.city = l.city WHERE l.city IS NOT NULL AND l.city<>\'' . "'";
                if ($q !== '') { $sql .= ' AND l.city LIKE :q'; $params['q'] = "%$q%"; }
                $sql .= ' GROUP BY l.city ORDER BY cnt DESC, l.city ASC LIMIT 100';
                $stmt = $this->pdo->prepare($sql);
                $stmt->execute($params);
                $rows = $stmt->fetchAll();
            } catch (\Throwable $e) { /* ignore */ }
        }

        $data = array_map(function($r){ return ['city' => $r['city'], 'count' => (int)($r['cnt'] ?? 0)]; }, $rows ?: []);
        echo json_encode(['data' => $data]);
    }

    // GET /b2b/api/activities/suggest?q=&country=Thailand
    // Mixed suggestions of cities and activities for typeahead
    public function activitiesSuggest(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        $q = trim((string)($_GET['q'] ?? ''));
        $country = trim((string)($_GET['country'] ?? 'Thailand'));

        // Guard: empty query -> return top cities in country
        $like = '%' . $q . '%';

        // 1) Cities in the given country with activity vendors
        $cities = [];
        try {
            $params = ['country' => $country];
            $sql = "SELECT v.city AS city, COUNT(p.id) AS cnt\n                    FROM vendors v\n                    JOIN vendor_packages p ON p.vendor_id = v.id AND p.active=1\n                    WHERE v.module='activity' AND v.country = :country AND v.city IS NOT NULL AND v.city<>''";
            if ($q !== '') { $sql .= ' AND v.city LIKE :q'; $params['q'] = $like; }
            $sql .= ' GROUP BY v.city ORDER BY cnt DESC, v.city ASC LIMIT 10';
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            $cities = $stmt->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $e) { $cities = []; }

        // 2) Activities (packages) within the country matching name or vendor name
        $acts = [];
        try {
            $params = ['country' => $country];
            $sql = "SELECT p.id, p.name, v.city\n                    FROM vendor_packages p\n                    JOIN vendors v ON v.id = p.vendor_id\n                    WHERE p.active=1 AND v.module='activity' AND v.country = :country";
            if ($q !== '') { $sql .= ' AND (p.name LIKE :q OR v.name LIKE :q)'; $params['q'] = $like; }
            $sql .= ' ORDER BY p.id DESC LIMIT 15';
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            $acts = $stmt->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $e) { $acts = []; }

        // Normalize to a unified list
        $data = [];
        foreach ($cities as $c) {
            $city = (string)$c['city'];
            if ($city === '') continue;
            $data[] = [
                'type' => 'city',
                'id' => null,
                'name' => $city,
                'city' => $city,
            ];
        }
        foreach ($acts as $a) {
            $data[] = [
                'type' => 'activity',
                'id' => (int)$a['id'],
                'name' => $a['name'],
                'city' => $a['city'] ?? null,
            ];
        }

        // If query provided, lightly prioritize those starting with q
        if ($q !== '') {
            $qLower = mb_strtolower($q);
            usort($data, function($x, $y) use ($qLower) {
                $xa = (int)(mb_stripos($x['name'], $qLower) === 0);
                $ya = (int)(mb_stripos($y['name'], $qLower) === 0);
                return $ya <=> $xa; // starters first
            });
        }

        // Cap results
        if (count($data) > 15) { $data = array_slice($data, 0, 15); }

        echo json_encode(['data' => $data, 'meta' => ['q' => $q, 'country' => $country]]);
    }

    // GET /b2b/api/taxi/quote?from=&to=&at=
    public function taxiQuote(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        $from = trim((string)($_GET['from'] ?? ''));
        $to = trim((string)($_GET['to'] ?? ''));
        $at = (string)($_GET['at'] ?? '');
        if ($from === '' || $to === '') {
            http_response_code(400);
            echo json_encode(['error' => 'Missing from/to']);
            return;
        }
        $routeLike = "%$from%$to%"; // naive
        $stmt = $this->pdo->prepare('SELECT id, name, route, base_price FROM taxis WHERE active=1 AND route LIKE :r ORDER BY id DESC LIMIT 20');
        $stmt->execute(['r' => "%$from%"]);
        $rows = $stmt->fetchAll();
        $quotes = [];
        foreach ($rows as $t) {
            $quotes[] = [
                'id' => (int)$t['id'],
                'name' => $t['name'],
                'route' => $t['route'],
                'price' => (float)$t['base_price'],
                'currency' => 'THB',
                'at' => $at,
            ];
        }
        echo json_encode(['data' => $quotes]);
    }

    // GET /b2b/api/taxi/search?q=&from=&to=&vehicle_type=&min_capacity=
    public function taxiSearch(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');

        $q = trim((string)($_GET['q'] ?? ''));
        $from = trim((string)($_GET['from'] ?? ''));
        $to = trim((string)($_GET['to'] ?? ''));
        $vehicleType = trim((string)($_GET['vehicle_type'] ?? ''));
        $minCapacity = (int)($_GET['min_capacity'] ?? 0);

        $where = ['t.active = 1'];
        $params = [];
        if ($q !== '') { $where[] = '(t.name LIKE :q OR t.route LIKE :q)'; $params['q'] = "%$q%"; }
        if ($from !== '') { $where[] = 't.route LIKE :rf'; $params['rf'] = "%$from%"; }
        if ($to !== '') { $where[] = 't.route LIKE :rt'; $params['rt'] = "%$to%"; }
        if ($vehicleType !== '') { $where[] = 't.vehicle_type = :vt'; $params['vt'] = $vehicleType; }
        if ($minCapacity > 0) { $where[] = '(t.capacity IS NULL OR t.capacity >= :cap)'; $params['cap'] = $minCapacity; }

        // Prefer a cover image if marked; otherwise any image
        $sql = "SELECT 
                    t.id,
                    t.name,
                    t.route,
                    t.base_price AS from_price,
                    t.vehicle_type,
                    t.capacity,
                    (
                        SELECT ti.file_path FROM taxi_images ti 
                        WHERE ti.taxi_id = t.id 
                        ORDER BY ti.is_cover DESC, ti.id ASC
                        LIMIT 1
                    ) AS thumbnail
                FROM taxis t
                WHERE " . implode(' AND ', $where) . "
                ORDER BY t.id DESC
                LIMIT 100";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $rows = $stmt->fetchAll();
        foreach ($rows as &$r) {
            $r['from_price'] = (float)($r['from_price'] ?? 0);
            $r['capacity'] = isset($r['capacity']) ? (int)$r['capacity'] : null;
            // Normalize thumbnail to an absolute-path URL if it's a relative file path
            if (!empty($r['thumbnail'])) {
                $thumb = (string)$r['thumbnail'];
                if (stripos($thumb, 'http://') !== 0 && stripos($thumb, 'https://') !== 0) {
                    if ($thumb[0] !== '/') { $thumb = '/' . $thumb; }
                    $r['thumbnail'] = $thumb;
                }
            }
        }

        echo json_encode([
            'meta' => [
                'q' => $q,
                'from' => $from,
                'to' => $to,
                'vehicle_type' => $vehicleType,
                'min_capacity' => $minCapacity,
            ],
            'data' => $rows,
        ]);
    }

    // GET /b2b/api/yachts/search?q=&city=&min_capacity=&max_guests=&booking_code=
    // Yachts search for B2B agents with booking code support and agent pricing
    public function yachtsSearch(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');

        $q = trim((string)($_GET['q'] ?? ''));
        $city = trim((string)($_GET['city'] ?? ''));
        $minCapacity = (int)($_GET['min_capacity'] ?? 0);
        $maxGuests = (int)($_GET['max_guests'] ?? 0);
        $bookingCode = trim((string)($_GET['booking_code'] ?? ''));

        // If yachts tables exist, return basic list; otherwise return empty data gracefully
        $data = [];
        try {
            $tables = $this->pdo->query("SHOW TABLES LIKE 'yachts'")->fetchAll();
            if (!empty($tables)) {
                $where = ['y.active = 1', 'y.visible_agent = 1'];
                $params = [];
                if ($q !== '') { $where[] = '(y.name LIKE :q OR y.subtitle LIKE :q)'; $params['q'] = "%$q%"; }
                if ($city !== '') { $where[] = 'y.city = :city'; $params['city'] = $city; }
                if ($minCapacity > 0) { $where[] = '(y.capacity IS NULL OR y.capacity >= :cap)'; $params['cap'] = $minCapacity; }
                if ($maxGuests > 0) { $where[] = '(y.max_guests IS NULL OR y.max_guests >= :mg)'; $params['mg'] = $maxGuests; }
                if ($bookingCode !== '') { $where[] = 'y.booking_code = :bc'; $params['bc'] = $bookingCode; }

                $sql = "SELECT y.id, y.name, y.city, y.capacity, y.max_guests, y.from_price, y.agent_cost, y.customer_cost, y.currency, y.booking_code,
                        (
                          SELECT yi.file_path FROM yacht_images yi
                          WHERE yi.yacht_id = y.id
                          ORDER BY yi.is_cover DESC, yi.id ASC
                          LIMIT 1
                        ) AS thumbnail
                      FROM yachts y
                      WHERE " . implode(' AND ', $where) . "
                      ORDER BY y.id DESC
                      LIMIT 100";
                $stmt = $this->pdo->prepare($sql);
                $stmt->execute($params);
                $data = $stmt->fetchAll();
                foreach ($data as &$r) {
                    // Compute agent-visible price: prefer agent_cost, then from_price, then customer_cost
                    $from = isset($r['from_price']) ? (float)$r['from_price'] : 0.0;
                    $agentCost = isset($r['agent_cost']) ? (float)$r['agent_cost'] : null;
                    $customerCost = isset($r['customer_cost']) ? (float)$r['customer_cost'] : null;
                    $agentPrice = $agentCost !== null ? $agentCost : ($from > 0 ? $from : ($customerCost ?? 0.0));
                    $r['from_price'] = $from;
                    $r['agent_price'] = $agentPrice;
                    $r['agent_cost'] = $agentCost; // explicit for agents
                    $r['customer_cost'] = $customerCost;
                    $r['currency'] = $r['currency'] ?? 'THB';
                    $r['capacity'] = isset($r['capacity']) ? (int)$r['capacity'] : null;
                    $r['max_guests'] = isset($r['max_guests']) ? (int)($r['max_guests']) : null;
                    // Normalize thumbnail to an absolute-path URL if it's a relative file path
                    if (!empty($r['thumbnail'])) {
                        $thumb = (string)$r['thumbnail'];
                        if (stripos($thumb, 'http://') !== 0 && stripos($thumb, 'https://') !== 0) {
                            if ($thumb[0] !== '/') { $thumb = '/' . $thumb; }
                            $r['thumbnail'] = $thumb;
                        }
                    }
                }
            }
        } catch (\Throwable $e) {
            // fallback to empty data
        }

        echo json_encode([
            'meta' => [
                'q' => $q,
                'city' => $city,
                'min_capacity' => $minCapacity,
                'max_guests' => $maxGuests,
                'booking_code' => $bookingCode,
            ],
            'data' => $data,
        ]);
    }

    // POST /b2b/api/activities/book
    // JSON: { package_id, variant_id, price_id, qty, show_time, customer_name, customer_email }
    public function activitiesBook(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        $user = $_SESSION['user'] ?? null;
        if (!$user) { http_response_code(401); echo json_encode(['error' => 'Unauthorized']); return; }

        $payload = json_decode(file_get_contents('php://input'), true) ?: [];
        $package_id = (int)($payload['package_id'] ?? 0);
        $variant_id = (int)($payload['variant_id'] ?? 0);
        $price_id = (int)($payload['price_id'] ?? 0);
        $qty = max(1, (int)($payload['qty'] ?? 1));
        $show_time = trim((string)($payload['show_time'] ?? ''));
        $customer_name = trim((string)($payload['customer_name'] ?? ''));
        $customer_email = trim((string)($payload['customer_email'] ?? ''));

        if ($package_id <= 0 || $variant_id <= 0 || $price_id <= 0) {
            http_response_code(422);
            echo json_encode(['error' => 'package_id, variant_id, price_id required']);
            return;
        }

        // Verify price belongs to the variant and package and is active; fetch pricing
        $sql = "SELECT pr.id, pr.variant_id, pr.package_id, pr.price_type, pr.pax_type, pr.min_quantity, pr.agent_price, pr.vendor_cost, pr.currency,
                       p.requires_show_time
                FROM vendor_package_prices pr
                JOIN vendor_packages p ON p.id = pr.package_id
                WHERE pr.id = :pid AND pr.variant_id = :vid AND pr.package_id = :pkg AND pr.active=1
                LIMIT 1";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute(['pid' => $price_id, 'vid' => $variant_id, 'pkg' => $package_id]);
        $row = $stmt->fetch();
        if (!$row) { http_response_code(404); echo json_encode(['error' => 'Price not found']); return; }

        // If requires show time, simple validation that a time is provided
        if ((int)($row['requires_show_time'] ?? 0) === 1) {
            if ($show_time === '') { http_response_code(422); echo json_encode(['error' => 'show_time required']); return; }
        }

        $agentPrice = (float)$row['agent_price'];
        $paxType = (string)$row['pax_type'];
        $minQty = isset($row['min_quantity']) ? (int)$row['min_quantity'] : null;
        // Qty semantics:
        // - flat: price applies per group of min_quantity; require qty >= min_quantity; total = ceil(qty / min_quantity) * agent_price
        // - adult/child: multiply by qty
        $totalQty = $qty;
        if ($paxType === 'flat') {
            $mq = max(1, (int)($minQty ?? 1));
            if ($qty < $mq) {
                http_response_code(422);
                echo json_encode(['error' => 'Flat price requires minimum quantity of ' . $mq]);
                return;
            }
            $groups = (int)ceil($qty / $mq);
            $total = $groups * $agentPrice;
        } else {
            $total = $agentPrice * $totalQty;
        }

        // Derive pax breakdown
        $adults = null; $children = null; $infants = null;
        if ($paxType === 'adult') { $adults = $totalQty; }
        elseif ($paxType === 'child') { $children = $totalQty; }

        // Create generic booking record with details
        $ins = $this->pdo->prepare('INSERT INTO bookings (user_id, module, item_id, variant_id, price_id, pax, pax_type, adults, children, infants, show_time, price, status, agent_price, details_json) VALUES (:uid, :module, :item_id, :variant_id, :price_id, :pax, :pax_type, :adults, :children, :infants, :show_time, :price, :status, :agent_price, :details_json)');
        $ins->execute([
            'uid' => (int)$user['id'],
            'module' => 'activity',
            'item_id' => $package_id,
            'variant_id' => $variant_id,
            'price_id' => $price_id,
            'pax' => $totalQty,
            'pax_type' => $paxType,
            'adults' => $adults,
            'children' => $children,
            'infants' => $infants,
            'show_time' => ($show_time !== '' ? $show_time : null),
            'price' => $total,
            'status' => 'pending',
            'agent_price' => $agentPrice,
            'details_json' => json_encode([
                'package_id' => $package_id,
                'variant_id' => $variant_id,
                'price_id' => $price_id,
                'qty' => $qty,
                'pax_type' => $paxType,
                'min_quantity' => $minQty,
                'show_time' => $show_time,
            ]),
        ]);
        $bookingId = (int)$this->pdo->lastInsertId();

        echo json_encode([
            'status' => 'ok',
            'booking_id' => $bookingId,
            'total' => $total,
            'currency' => $row['currency'] ?? 'THB',
            'meta' => [
                'package_id' => $package_id,
                'variant_id' => $variant_id,
                'price_id' => $price_id,
                'pax_type' => $paxType,
                'qty' => $qty,
                'show_time' => $show_time,
            ],
        ]);
    }

    // POST /b2b/api/activities/book-multi
    // JSON: { items: [{ package_id, variant_id, price_id, qty, show_time }...] }
    public function activitiesBookMulti(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        $user = $_SESSION['user'] ?? null;
        if (!$user) { http_response_code(401); echo json_encode(['error' => 'Unauthorized']); return; }

        $payload = json_decode(file_get_contents('php://input'), true) ?: [];
        $items = $payload['items'] ?? [];
        if (!is_array($items) || count($items) === 0) { http_response_code(422); echo json_encode(['error' => 'No items']); return; }

        $results = [];
        $totalAll = 0.0;
        $this->pdo->beginTransaction();
        try {
            foreach ($items as $it) {
                $package_id = (int)($it['package_id'] ?? 0);
                $variant_id = (int)($it['variant_id'] ?? 0);
                $price_id = (int)($it['price_id'] ?? 0);
                $qty = max(1, (int)($it['qty'] ?? 1));
                $show_time = trim((string)($it['show_time'] ?? ''));
                if ($package_id <= 0 || $variant_id <= 0 || $price_id <= 0) { throw new \Exception('Invalid item'); }

                $sql = "SELECT pr.id, pr.variant_id, pr.package_id, pr.price_type, pr.pax_type, pr.min_quantity, pr.agent_price, pr.vendor_cost, pr.currency, p.requires_show_time
                        FROM vendor_package_prices pr
                        JOIN vendor_packages p ON p.id = pr.package_id
                        WHERE pr.id = :pid AND pr.variant_id = :vid AND pr.package_id = :pkg AND pr.active=1
                        LIMIT 1";
                $stmt = $this->pdo->prepare($sql);
                $stmt->execute(['pid' => $price_id, 'vid' => $variant_id, 'pkg' => $package_id]);
                $row = $stmt->fetch();
                if (!$row) { throw new \Exception('Price not found'); }

                if ((int)($row['requires_show_time'] ?? 0) === 1) {
                    if ($show_time === '') { throw new \Exception('show_time required'); }
                }

                $agentPrice = (float)$row['agent_price'];
                $paxType = (string)$row['pax_type'];
                $minQty = isset($row['min_quantity']) ? (int)$row['min_quantity'] : null;
                $totalQty = $qty;
                if ($paxType === 'flat') {
                    $mq = max(1, (int)($minQty ?? 1));
                    if ($qty < $mq) { throw new \Exception('Flat requires min ' . $mq); }
                    $groups = (int)ceil($qty / $mq);
                    $total = $groups * $agentPrice;
                } else {
                    $total = $agentPrice * $totalQty;
                }

                // Derive pax breakdown
                $adults = null; $children = null; $infants = null;
                if ($paxType === 'adult') { $adults = $totalQty; }
                elseif ($paxType === 'child') { $children = $totalQty; }

                $ins = $this->pdo->prepare('INSERT INTO bookings (user_id, module, item_id, variant_id, price_id, pax, pax_type, adults, children, infants, show_time, price, status, agent_price, details_json) VALUES (:uid, :module, :item_id, :variant_id, :price_id, :pax, :pax_type, :adults, :children, :infants, :show_time, :price, :status, :agent_price, :details_json)');
                $ins->execute([
                    'uid' => (int)$user['id'],
                    'module' => 'activity',
                    'item_id' => $package_id,
                    'variant_id' => $variant_id,
                    'price_id' => $price_id,
                    'pax' => $totalQty,
                    'pax_type' => $paxType,
                    'adults' => $adults,
                    'children' => $children,
                    'infants' => $infants,
                    'show_time' => ($show_time !== '' ? $show_time : null),
                    'price' => $total,
                    'status' => 'pending',
                    'agent_price' => $agentPrice,
                    'details_json' => json_encode([
                        'package_id' => $package_id,
                        'variant_id' => $variant_id,
                        'price_id' => $price_id,
                        'qty' => $qty,
                        'pax_type' => $paxType,
                        'min_quantity' => $minQty,
                        'show_time' => $show_time,
                    ]),
                ]);
                $bookingId = (int)$this->pdo->lastInsertId();
                $totalAll += $total;
                $results[] = [
                    'booking_id' => $bookingId,
                    'package_id' => $package_id,
                    'variant_id' => $variant_id,
                    'price_id' => $price_id,
                    'qty' => $qty,
                    'pax_type' => $paxType,
                    'total' => $total,
                    'currency' => $row['currency'] ?? 'THB',
                ];
            }
            $this->pdo->commit();
        } catch (\Throwable $e) {
            $this->pdo->rollBack();
            http_response_code(422);
            echo json_encode(['error' => $e->getMessage()]);
            return;
        }

        echo json_encode([
            'status' => 'ok',
            'count' => count($results),
            'total' => $totalAll,
            'currency' => $results[0]['currency'] ?? 'THB',
            'items' => $results,
        ]);
    }

    // POST /b2b/api/checkout/create
    // JSON: { items: [{ package_id, variant_id, price_id, qty, show_time }...] }
    public function checkoutCreate(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        // CSRF protection for state-changing request
        Security::requireCsrf();
        $user = $_SESSION['user'] ?? null;
        if (!$user) { http_response_code(401); echo json_encode(['error' => 'Unauthorized']); return; }

        $payload = json_decode(file_get_contents('php://input'), true) ?: [];
        $items = $payload['items'] ?? [];
        // Customer contact (optional at creation; will be completed on checkout page)
        $customer_name = trim((string)($payload['customer_name'] ?? ''));
        $customer_mobile = trim((string)($payload['customer_mobile'] ?? ''));
        $customer_email = trim((string)($payload['customer_email'] ?? ''));
        $customer_whatsapp = trim((string)($payload['customer_whatsapp'] ?? ''));
        if (!is_array($items) || count($items) === 0) { http_response_code(422); echo json_encode(['error' => 'No items']); return; }
        // Validate contact only if provided
        if ($customer_email !== '' && !filter_var($customer_email, FILTER_VALIDATE_EMAIL)) { http_response_code(422); echo json_encode(['error' => 'Invalid customer_email']); return; }

        $orderItems = [];
        $totalAll = 0.0;
        $currency = 'THB';
        $this->pdo->beginTransaction();
        try {
            // Create order shell
            $insOrder = $this->pdo->prepare('INSERT INTO orders (user_id, status, total_amount, currency, payment_method, customer_name, customer_mobile, customer_email, customer_whatsapp) VALUES (:uid, "pending", 0, :cur, "none", :cn, :cm, :ce, :cw)');
            $insOrder->execute([
                'uid' => (int)$user['id'],
                'cur' => $currency,
                'cn' => ($customer_name !== '' ? $customer_name : null),
                'cm' => ($customer_mobile !== '' ? $customer_mobile : null),
                'ce' => ($customer_email !== '' ? $customer_email : null),
                'cw' => ($customer_whatsapp !== '' ? $customer_whatsapp : null),
            ]);
            $orderId = (int)$this->pdo->lastInsertId();

            foreach ($items as $it) {
                $package_id = (int)($it['package_id'] ?? 0);
                $variant_id = (int)($it['variant_id'] ?? 0);
                $price_id   = (int)($it['price_id'] ?? 0);
                $qty        = max(1, (int)($it['qty'] ?? 1));
                $show_time  = trim((string)($it['show_time'] ?? ''));
                if ($package_id <= 0 || $variant_id <= 0 || $price_id <= 0) { throw new \Exception('Invalid item'); }

                $sql = "SELECT pr.id, pr.variant_id, pr.package_id, pr.price_type, pr.pax_type, pr.min_quantity, pr.agent_price, pr.vendor_cost, pr.currency, p.requires_show_time
                        FROM vendor_package_prices pr
                        JOIN vendor_packages p ON p.id = pr.package_id
                        WHERE pr.id = :pid AND pr.variant_id = :vid AND pr.package_id = :pkg AND pr.active=1
                        LIMIT 1";
                $stmt = $this->pdo->prepare($sql);
                $stmt->execute(['pid' => $price_id, 'vid' => $variant_id, 'pkg' => $package_id]);
                $row = $stmt->fetch();
                if (!$row) { throw new \Exception('Price not found'); }
                if ((int)($row['requires_show_time'] ?? 0) === 1 && $show_time === '') { throw new \Exception('show_time required'); }

                $agentPrice = (float)$row['agent_price'];
                $paxType = (string)$row['pax_type'];
                $minQty = isset($row['min_quantity']) ? (int)$row['min_quantity'] : null;
                $totalQty = $qty;
                if ($paxType === 'flat') {
                    $mq = max(1, (int)($minQty ?? 1));
                    if ($qty < $mq) { throw new \Exception('Flat requires min ' . $mq); }
                    $groups = (int)ceil($qty / $mq);
                    $lineTotal = $groups * $agentPrice;
                } else {
                    $lineTotal = $agentPrice * $totalQty;
                }

                $currency = $row['currency'] ?? $currency;
                $totalAll += $lineTotal;
                $orderItems[] = [
                    'module' => 'activity',
                    'item_id' => $package_id,
                    'variant_id' => $variant_id,
                    'price_id' => $price_id,
                    'qty' => $totalQty,
                    'unit_price' => $agentPrice,
                    'line_total' => $lineTotal,
                    'currency' => $currency,
                ];
            }

            // Persist items
            $insItem = $this->pdo->prepare('INSERT INTO order_items (order_id, module, item_id, variant_id, price_id, qty, unit_price, line_total, currency) VALUES (:oid, :m, :iid, :vid, :pid, :q, :up, :lt, :cur)');
            foreach ($orderItems as $oi) {
                $insItem->execute([
                    'oid' => $orderId,
                    'm' => $oi['module'],
                    'iid' => $oi['item_id'],
                    'vid' => $oi['variant_id'],
                    'pid' => $oi['price_id'],
                    'q' => $oi['qty'],
                    'up' => $oi['unit_price'],
                    'lt' => $oi['line_total'],
                    'cur' => $oi['currency'],
                ]);
            }
            $this->pdo->prepare('UPDATE orders SET total_amount = :t, currency = :c WHERE id = :id')->execute(['t'=>$totalAll,'c'=>$currency,'id'=>$orderId]);
            $this->pdo->commit();

            // Prepare signed checkout URL (HMAC id:user_id)
            $appKey = function_exists('env') ? (string)env('APP_KEY', '') : '';
            if ($appKey === '') { $appKey = getenv('APP_KEY') ?: 'devkey'; }
            $sig = hash_hmac('sha256', $orderId . ':' . (int)$user['id'], $appKey);
            $checkoutUrl = '/b2b/checkout?id=' . urlencode((string)$orderId) . '&sig=' . urlencode($sig);

            echo json_encode([
                'status' => 'ok',
                'order_id' => $orderId,
                'total' => $totalAll,
                'currency' => $currency,
                'items' => $orderItems,
                'payment_options' => ['wallet' => true, 'stripe' => true],
                'sig' => $sig,
                'checkout_url' => $checkoutUrl,
                'contact' => [
                    'customer_name' => $customer_name,
                    'customer_mobile' => $customer_mobile,
                    'customer_email' => ($customer_email !== '' ? $customer_email : null),
                    'customer_whatsapp' => ($customer_whatsapp !== '' ? $customer_whatsapp : null),
                ],
            ]);
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) $this->pdo->rollBack();
            http_response_code(422);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // POST /b2b/api/checkout/pay-wallet
    // JSON: { order_id }
    public function checkoutPayWallet(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        // CSRF protection for state-changing request
        Security::requireCsrf();
        $user = $_SESSION['user'] ?? null;
        if (!$user) { http_response_code(401); echo json_encode(['error' => 'Unauthorized']); return; }

        $audit = new Audit($this->pdo);

        $payload = json_decode(file_get_contents('php://input'), true) ?: [];
        $orderId = (int)($payload['order_id'] ?? 0);
        $txPassword = isset($payload['tx_password']) ? (string)$payload['tx_password'] : null;
        $txPasswordEnc = isset($payload['tx_password_enc']) ? (string)$payload['tx_password_enc'] : null;
        $txPasswordB64 = isset($payload['tx_password_b64']) ? (string)$payload['tx_password_b64'] : null;
        $txPasswordUrl = isset($payload['tx_password_url']) ? (string)$payload['tx_password_url'] : null;
        // Resolve PIN from provided formats: RSA-OAEP enc -> Base64 -> URL-encoded -> plaintext
        if (($txPassword === null || $txPassword === '') && $txPasswordEnc && isset($_SESSION['txpin_privkey']['pem'])) {
            try {
                $privPem = (string)$_SESSION['txpin_privkey']['pem'];
                $cipher = base64_decode($txPasswordEnc, true);
                if ($cipher !== false) {
                    $plain = '';
                    $ok = openssl_private_decrypt($cipher, $plain, $privPem, OPENSSL_PKCS1_OAEP_PADDING);
                    if ($ok) { $txPassword = $plain; }
                }
            } catch (\Throwable $_) { /* ignore */ }
        }
        if (($txPassword === null || $txPassword === '') && $txPasswordB64) {
            $dec = base64_decode($txPasswordB64, true);
            if ($dec !== false) { $txPassword = $dec; }
        }
        if (($txPassword === null || $txPassword === '') && $txPasswordUrl) {
            $txPassword = urldecode($txPasswordUrl);
        }
        if ($orderId <= 0) {
            http_response_code(422);
            echo json_encode(['error' => 'order_id required']);
            $audit->event([
                'event_type' => 'wallet_pay_validation',
                'action' => 'missing_order_id',
                'message' => 'Order ID missing in wallet payment request',
                'status_code' => 422,
                'user_id' => (int)$user['id'],
            ]);
            return;
        }

        // Load order
        $o = $this->pdo->prepare('SELECT id, user_id, status, total_amount, currency FROM orders WHERE id=:id AND user_id=:uid LIMIT 1');
        $o->execute(['id'=>$orderId,'uid'=>(int)$user['id']]);
        $order = $o->fetch();
        if (!$order) {
            http_response_code(404);
            echo json_encode(['error' => 'Order not found']);
            $audit->event([
                'event_type' => 'wallet_pay_validation',
                'action' => 'order_not_found',
                'message' => 'Order not found or does not belong to agent',
                'status_code' => 404,
                'user_id' => (int)$user['id'],
                'order_id' => $orderId,
            ]);
            return;
        }
        if ($order['status'] !== 'pending') {
            http_response_code(422);
            echo json_encode(['error' => 'Order not pending']);
            $audit->event([
                'event_type' => 'wallet_pay_validation',
                'action' => 'order_not_pending',
                'message' => 'Attempted to pay non-pending order',
                'status_code' => 422,
                'user_id' => (int)$user['id'],
                'order_id' => $orderId,
                'details' => ['status' => $order['status']],
            ]);
            return;
        }

        // Enforce transaction password for agents if configured/required, with lockout policy
        try {
            $uStmt = $this->pdo->prepare('SELECT transaction_password_hash, require_tx_password, tx_pin_attempts, tx_pin_locked_until FROM users WHERE id = :id LIMIT 1');
            $uStmt->execute(['id' => (int)$user['id']]);
            $uRow = $uStmt->fetch();
            $txHash = (string)($uRow['transaction_password_hash'] ?? '');
            $requireTx = (int)($uRow['require_tx_password'] ?? 0) === 1;
            $attempts = (int)($uRow['tx_pin_attempts'] ?? 0);
            $lockedUntil = $uRow['tx_pin_locked_until'] ?? null;

            // If locked, block immediately
            if ($lockedUntil) {
                $now = new \DateTimeImmutable('now');
                $lu = new \DateTimeImmutable($lockedUntil);
                if ($lu > $now) {
                    http_response_code(422);
                    echo json_encode(['error' => 'Wallet is temporarily disabled due to multiple invalid PIN attempts. Please try again after the lock period.']);
                    $audit->event([
                        'event_type' => 'wallet_pay_blocked',
                        'action' => 'wallet_locked',
                        'message' => 'Wallet locked due to previous invalid PIN attempts',
                        'status_code' => 422,
                        'user_id' => (int)$user['id'],
                        'order_id' => $orderId,
                        'details' => ['locked_until' => $lockedUntil],
                    ]);
                    return;
                }
            }

            if ($requireTx || $txHash !== '') {
                if ($txPassword === null || $txPassword === '') {
                    http_response_code(422);
                    echo json_encode(['error' => 'Please enter your 6-digit transaction PIN to proceed.']);
                    $audit->event([
                        'event_type' => 'wallet_pay_validation',
                        'action' => 'missing_pin',
                        'message' => 'PIN missing in wallet payment request',
                        'status_code' => 422,
                        'user_id' => (int)$user['id'],
                        'order_id' => $orderId,
                    ]);
                    return;
                }
                if (!preg_match('/^\d{4,6}$/', $txPassword)) {
                    http_response_code(422);
                    echo json_encode(['error' => 'PIN format is invalid. Enter a 4–6 digit PIN.']);
                    $audit->event([
                        'event_type' => 'wallet_pay_validation',
                        'action' => 'invalid_pin_format',
                        'message' => 'PIN format invalid',
                        'status_code' => 422,
                        'user_id' => (int)$user['id'],
                        'order_id' => $orderId,
                    ]);
                    return;
                }
                if ($txHash === '' || !password_verify($txPassword, $txHash)) {
                    // Increment attempts and lock after 10 invalid tries
                    $attempts++;
                    if ($attempts >= 10) {
                        $lockStmt = $this->pdo->prepare('UPDATE users SET tx_pin_attempts = :a, tx_pin_locked_until = DATE_ADD(NOW(), INTERVAL 24 HOUR) WHERE id = :id');
                        $lockStmt->execute(['a' => $attempts, 'id' => (int)$user['id']]);
                        http_response_code(422);
                        echo json_encode(['error' => 'Too many invalid attempts. Your wallet has been temporarily disabled for 24 hours.']);
                        $audit->event([
                            'event_type' => 'wallet_pay_pin',
                            'action' => 'pin_lockout',
                            'message' => 'Too many invalid PIN attempts; wallet locked',
                            'status_code' => 422,
                            'user_id' => (int)$user['id'],
                            'order_id' => $orderId,
                            'details' => ['attempts' => $attempts],
                        ]);
                        return;
                    } else {
                        $updStmt = $this->pdo->prepare('UPDATE users SET tx_pin_attempts = :a WHERE id = :id');
                        $updStmt->execute(['a' => $attempts, 'id' => (int)$user['id']]);
                        $remaining = max(0, 10 - $attempts);
                        http_response_code(422);
                        echo json_encode(['error' => 'Invalid PIN. Attempts remaining: ' . $remaining]);
                        $audit->event([
                            'event_type' => 'wallet_pay_pin',
                            'action' => 'pin_invalid',
                            'message' => 'Invalid PIN submitted',
                            'status_code' => 422,
                            'user_id' => (int)$user['id'],
                            'order_id' => $orderId,
                            'details' => ['attempts' => $attempts, 'remaining' => $remaining],
                        ]);
                        return;
                    }
                }
            }
        } catch (\Throwable $e) {
            http_response_code(422);
            echo json_encode(['error' => 'We couldn\'t verify your PIN right now. Please try again.']);
            $audit->event([
                'event_type' => 'wallet_pay_error',
                'action' => 'pin_check_exception',
                'message' => 'Exception during PIN verification',
                'status_code' => 422,
                'user_id' => (int)$user['id'],
                'order_id' => $orderId,
                'details' => ['error' => substr($e->getMessage(), 0, 200)],
            ]);
            return;
        }

        // Wallet charge
        $walletSvc = new WalletService($this->pdo);

        $this->pdo->beginTransaction();
        try {
            // Check balance with FOR UPDATE
            $wid = $walletSvc->getOrCreateWallet((int)$user['id']);
            $row = $this->pdo->query('SELECT balance FROM wallets WHERE id='.(int)$wid.' FOR UPDATE')->fetch();
            $amount = (float)$order['total_amount'];
            if ((float)($row['balance'] ?? 0) < $amount) {
                $audit->event([
                    'event_type' => 'wallet_pay_validation',
                    'action' => 'insufficient_balance',
                    'message' => 'Insufficient wallet balance at debit step',
                    'status_code' => 422,
                    'user_id' => (int)$user['id'],
                    'order_id' => $orderId,
                    'details' => ['balance' => (float)($row['balance'] ?? 0), 'required' => $amount],
                ]);
                throw new \Exception('Insufficient wallet balance');
            }

            // Ledger entries and balance update (approved debit)
            $this->pdo->prepare('INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "debit", :a, "manual", "approved", :meta)')
                ->execute(['w'=>$wid,'a'=>$amount,'meta'=>json_encode(['flow'=>'checkout','order_id'=>$orderId])]);
            $this->pdo->prepare('UPDATE wallets SET balance = balance - :a WHERE id=:w')->execute(['a'=>$amount,'w'=>$wid]);

            // Mark order paid
            $this->pdo->prepare('UPDATE orders SET status="paid", payment_method="wallet", updated_at=NOW() WHERE id=:id')->execute(['id'=>$orderId]);
            $this->pdo->prepare('INSERT INTO payments (order_id, method, amount, currency, status, meta) VALUES (:oid, "wallet", :a, :c, "succeeded", :m)')
                ->execute(['oid'=>$orderId,'a'=>$amount,'c'=>$order['currency'] ?? 'THB','m'=>json_encode(['flow'=>'checkout'])]);

            // Create bookings for order items
            $q = $this->pdo->prepare('SELECT id, module, item_id, variant_id, price_id, qty, unit_price, line_total, currency FROM order_items WHERE order_id = :oid');
            $q->execute(['oid'=>$orderId]);
            $items = $q->fetchAll();
            $bookings = [];
            foreach ($items as $it) {
                if ($it['module'] !== 'activity') { continue; }
                $this->pdo->prepare('INSERT INTO bookings (user_id, order_id, module, item_id, pax, price, status, payment_status, agent_price) VALUES (:uid, :oid, :m, :iid, :pax, :price, "pending", "paid", :ap)')
                    ->execute([
                        'uid' => (int)$user['id'],
                        'oid' => $orderId,
                        'm'   => 'activity',
                        'iid' => (int)$it['item_id'],
                        'pax' => (int)$it['qty'],
                        'price' => (float)$it['line_total'],
                        'ap'   => (float)$it['unit_price'],
                    ]);
                $bid = (int)$this->pdo->lastInsertId();
                // Link back to order_items
                $this->pdo->prepare('UPDATE order_items SET booking_id = :bid WHERE id = :id')->execute(['bid'=>$bid,'id'=>(int)$it['id']]);
                $bookings[] = [ 'booking_id' => $bid, 'total' => (float)$it['line_total'], 'currency' => $it['currency'] ?? 'THB' ];
            }

            // Reset attempts on success
            try {
                $this->pdo->prepare('UPDATE users SET tx_pin_attempts = 0, tx_pin_locked_until = NULL WHERE id = :id')->execute(['id' => (int)$user['id']]);
            } catch (\Throwable $_) { /* ignore */ }
            $this->pdo->commit();
            $audit->event([
                'event_type' => 'wallet_pay_success',
                'action' => 'wallet_debit_and_order_paid',
                'message' => 'Wallet debit succeeded and order marked paid',
                'status_code' => 200,
                'user_id' => (int)$user['id'],
                'order_id' => $orderId,
                'details' => ['amount' => $amount, 'currency' => $order['currency'] ?? 'THB', 'bookings_count' => count($bookings)],
            ]);
            echo json_encode(['status'=>'ok','order_id'=>$orderId,'bookings'=>$bookings,'currency'=>$order['currency'] ?? 'THB','total'=>$amount]);
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) $this->pdo->rollBack();
            http_response_code(422);
            echo json_encode(['error'=>$e->getMessage()]);
            $audit->event([
                'event_type' => 'wallet_pay_error',
                'action' => 'wallet_charge_exception',
                'message' => 'Exception during wallet charge or order finalization',
                'status_code' => 422,
                'user_id' => (int)$user['id'],
                'order_id' => $orderId,
                'details' => ['error' => substr($e->getMessage(), 0, 200)],
            ]);
        }
    }

    // GET /b2b/api/checkout/order?id=
    public function checkoutOrder(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        $user = $_SESSION['user'] ?? null;
        if (!$user) { http_response_code(401); echo json_encode(['error' => 'Unauthorized']); return; }

        $orderId = (int)($_GET['id'] ?? 0);
        if ($orderId <= 0) { http_response_code(400); echo json_encode(['error' => 'id required']); return; }

        // Ensure order belongs to user; also fetch customer contact fields
        $o = $this->pdo->prepare('SELECT id, user_id, status, total_amount, currency, payment_method, created_at, updated_at, customer_name, customer_mobile, customer_email, customer_whatsapp FROM orders WHERE id=:id AND user_id=:uid LIMIT 1');
        $o->execute(['id'=>$orderId,'uid'=>(int)$user['id']]);
        $order = $o->fetch();
        if (!$order) { http_response_code(404); echo json_encode(['error' => 'Order not found']); return; }

        // Load items with friendly names when module is activity; include booking_id for enrichment
        $q = $this->pdo->prepare('SELECT 
                                          oi.id, oi.module, oi.item_id, oi.variant_id, oi.price_id, oi.qty, oi.unit_price, oi.line_total, oi.currency, oi.booking_id,
                                          CASE WHEN oi.module = "activity" THEN vp.name ELSE NULL END AS name,
                                          CASE WHEN oi.module = "activity" THEN vv.name ELSE NULL END AS variant_name,
                                          CASE WHEN oi.module = "activity" THEN pr.pax_type ELSE NULL END AS pax_type
                                  FROM order_items oi
                                  LEFT JOIN vendor_packages vp ON (oi.module = "activity" AND vp.id = oi.item_id)
                                  LEFT JOIN vendor_package_variants vv ON (oi.module = "activity" AND vv.id = oi.variant_id)
                                  LEFT JOIN vendor_package_prices pr ON (oi.module = "activity" AND pr.id = oi.price_id)
                                  WHERE oi.order_id = :oid
                                  ORDER BY oi.id ASC');
        $q->execute(['oid'=>$orderId]);
        $items = $q->fetchAll();

        // Enrich hotel items with hotel name, dates, room and guest details
        $respItems = [];
        foreach ($items ?: [] as $it) {
            $module = (string)($it['module'] ?? '');
            $name = $it['name'] ?? null;
            $variantName = $it['variant_name'] ?? null;
            $details = null;
            if ($module === 'hotel') {
                $hotelId = (int)($it['item_id'] ?? 0);
                $bookingId = isset($it['booking_id']) ? (int)$it['booking_id'] : 0;
                $hotelName = null;
                $roomName = null;
                $hotelDetails = [
                    'hotel_id' => $hotelId,
                    'hotel_name' => null,
                    'check_in_date' => null,
                    'check_out_date' => null,
                    'nights' => null,
                    'rooms' => null,
                    'room' => null,
                    'guests' => null,
                    'guest_names' => [],
                    // Optional mappings for multi-room selections
                    'room_names_by_index' => null,
                    'guest_groups' => [],
                ];
                try {
                    // Hotel name
                    if ($hotelId > 0) {
                        $qh = $this->pdo->prepare('SELECT name FROM hotels WHERE id = :id LIMIT 1');
                        $qh->execute(['id' => $hotelId]);
                        $hotelName = $qh->fetchColumn();
                        if ($hotelName !== false && $hotelName !== null) { $hotelDetails['hotel_name'] = (string)$hotelName; }
                    }
                } catch (\Throwable $e) { /* ignore */ }
                // Booking-level info from generic bookings table (preferred for pre-hotel_book stage)
                if ($bookingId > 0) {
                    try {
                        $qb = $this->pdo->prepare('SELECT user_id, details_json, adults, children FROM bookings WHERE id = :id LIMIT 1');
                        $qb->execute(['id' => $bookingId]);
                        $bk = $qb->fetch(\PDO::FETCH_ASSOC) ?: null;
                        if ($bk && (int)$bk['user_id'] === (int)$user['id']) {
                            $det = null;
                            try { $det = json_decode((string)($bk['details_json'] ?? 'null'), true); } catch (\Throwable $e) { $det = null; }
                            if (is_array($det)) {
                                $hotelDetails['check_in_date'] = $det['check_in_date'] ?? null;
                                $hotelDetails['check_out_date'] = $det['check_out_date'] ?? null;
                                $hotelDetails['nights'] = isset($det['nights']) ? (int)$det['nights'] : null;
                                $hotelDetails['rooms'] = isset($det['rooms']) ? (int)$det['rooms'] : null;
                                // Room info
                                $rid = isset($det['room_id']) ? (int)$det['room_id'] : 0;
                                if ($rid > 0) {
                                    try {
                                        $qr = $this->pdo->prepare('SELECT id, name FROM hotel_rooms WHERE id = :id LIMIT 1');
                                        $qr->execute(['id' => $rid]);
                                        $rrow = $qr->fetch(\PDO::FETCH_ASSOC) ?: null;
                                        if ($rrow) { $roomName = (string)($rrow['name'] ?? ''); $hotelDetails['room'] = ['id' => (int)$rrow['id'], 'name' => $roomName]; }
                                    } catch (\Throwable $e) { /* ignore */ }
                                }
                                // Attach optional room name mapping by 1-based index
                                if (isset($det['room_names_by_index']) && is_array($det['room_names_by_index'])) {
                                    // Normalize keys to int
                                    $map = [];
                                    foreach ($det['room_names_by_index'] as $k => $v) {
                                        $ki = (int)$k; if ($ki <= 0) { continue; }
                                        $nm = trim((string)$v);
                                        if ($nm !== '') { $map[$ki] = $nm; }
                                    }
                                    if (!empty($map)) { $hotelDetails['room_names_by_index'] = $map; }
                                }

                                // Guests summary
                                if (isset($det['guests']) && is_array($det['guests'])) {
                                    $hotelDetails['guests'] = [
                                        'adults' => isset($det['guests']['adults']) ? (int)$det['guests']['adults'] : (isset($bk['adults']) ? (int)$bk['adults'] : null),
                                        'children' => isset($det['guests']['children']) ? (int)$det['guests']['children'] : (isset($bk['children']) ? (int)$bk['children'] : null),
                                    ];
                                } else {
                                    $hotelDetails['guests'] = [
                                        'adults' => isset($bk['adults']) ? (int)$bk['adults'] : null,
                                        'children' => isset($bk['children']) ? (int)$bk['children'] : null,
                                    ];
                                }
                            }
                        }
                    } catch (\Throwable $e) { /* ignore */ }
                    // Guest names
                    try {
                        $qg = $this->pdo->prepare('SELECT room_index, guest_index, type, full_name FROM booking_guests WHERE booking_id = :bid ORDER BY room_index ASC, guest_index ASC');
                        $qg->execute(['bid' => $bookingId]);
                        $grows = $qg->fetchAll(\PDO::FETCH_ASSOC) ?: [];
                        if (!empty($grows)) {
                            $grouped = [];
                            foreach ($grows as $g) {
                                $ri = (int)($g['room_index'] ?? 0);
                                if (!isset($grouped[$ri])) { $grouped[$ri] = []; }
                                $label = trim((string)($g['full_name'] ?? ''));
                                if ($label === '') { $label = trim((string)($g['type'] ?? '')); }
                                if ($label !== '') { $grouped[$ri][] = $label; }
                            }
                            // Convert to sorted array of {room_index, guests: [...]} and also guest_groups with room_name
                            ksort($grouped);
                            $guestNames = [];
                            $guestGroups = [];
                            foreach ($grouped as $ri => $arr) {
                                $riInt = (int)$ri;
                                $names = array_values($arr);
                                $guestNames[] = ['room_index' => $riInt, 'guests' => $names];
                                $roomName = null;
                                if (isset($hotelDetails['room_names_by_index']) && is_array($hotelDetails['room_names_by_index'])) {
                                    $roomName = $hotelDetails['room_names_by_index'][$riInt] ?? null;
                                }
                                $guestGroups[] = ['room_index' => $riInt, 'room_name' => $roomName, 'guests' => $names];
                            }
                            $hotelDetails['guest_names'] = $guestNames;
                            $hotelDetails['guest_groups'] = $guestGroups;
                        }
                    } catch (\Throwable $e) { /* ignore */ }
                    // Fallback: build guest_names from details_json.guests_by_room if booking_guests is empty or unavailable
                    if (empty($hotelDetails['guest_names'])) {
                        try {
                            if (isset($det) && is_array($det) && isset($det['guests_by_room']) && is_array($det['guests_by_room'])) {
                                $guestNames = [];
                                $guestGroups = [];
                                foreach ($det['guests_by_room'] as $ri => $arr) {
                                    $riInt = (int)$ri; if ($riInt <= 0) { $riInt = count($guestNames) + 1; }
                                    $names = [];
                                    if (is_array($arr)) {
                                        foreach ($arr as $g) {
                                            $nm = trim((string)($g['name'] ?? ''));
                                            if ($nm !== '') { $names[] = $nm; }
                                        }
                                    }
                                    if (!empty($names)) {
                                        $guestNames[] = ['room_index' => $riInt, 'guests' => $names];
                                        $roomName = null;
                                        if (isset($hotelDetails['room_names_by_index']) && is_array($hotelDetails['room_names_by_index'])) {
                                            $roomName = $hotelDetails['room_names_by_index'][$riInt] ?? null;
                                        }
                                        $guestGroups[] = ['room_index' => $riInt, 'room_name' => $roomName, 'guests' => $names];
                                    }
                                }
                                if (!empty($guestNames)) { $hotelDetails['guest_names'] = $guestNames; $hotelDetails['guest_groups'] = $guestGroups; }
                            }
                        } catch (\Throwable $e) { /* ignore */ }
                    }
                }

                // Set presentation name/variant when available
                if ($hotelName && !$name) { $name = (string)$hotelName; }
                if ($roomName && !$variantName) { $variantName = (string)$roomName; }
                $details = ['hotel' => $hotelDetails];
            }

            $respItems[] = [
                'id' => (int)$it['id'],
                'module' => $module,
                'item_id' => (int)$it['item_id'],
                'name' => $name,
                'variant_name' => $variantName,
                'pax_type' => $it['pax_type'] ?? null,
                'variant_id' => (int)$it['variant_id'],
                'price_id' => (int)$it['price_id'],
                'qty' => (int)$it['qty'],
                'unit_price' => (float)$it['unit_price'],
                'line_total' => (float)$it['line_total'],
                'currency' => $it['currency'] ?? 'THB',
                'details' => $details,
            ];
        }

        echo json_encode([
            'status' => 'ok',
            'order' => [
                'id' => (int)$order['id'],
                'status' => $order['status'],
                'total' => (float)$order['total_amount'],
                'currency' => $order['currency'] ?? 'THB',
                'payment_method' => $order['payment_method'] ?? 'none',
                'created_at' => $order['created_at'] ?? null,
                'updated_at' => $order['updated_at'] ?? null,
                'customer_name' => $order['customer_name'] ?? null,
                'customer_mobile' => $order['customer_mobile'] ?? null,
                'customer_email' => $order['customer_email'] ?? null,
                'customer_whatsapp' => $order['customer_whatsapp'] ?? null,
            ],
            'items' => $respItems,
            'payment_options' => ['wallet' => true, 'stripe' => true],
        ]);

    }

    public function checkoutOrderContact(): void
    {
        Auth::checkRole(['B2B Agent']);
        header('Content-Type: application/json');
        // CSRF (use shared helper which reads header/POST/GET and compares against $_SESSION['csrf'])
        Security::requireCsrf();

        $user = $_SESSION['user'] ?? null;
        if (!$user) { http_response_code(401); echo json_encode(['error' => 'Unauthorized']); return; }

        $input = json_decode(file_get_contents('php://input'), true) ?: [];
        $orderId = (int)($input['order_id'] ?? 0);
        if ($orderId <= 0) { http_response_code(400); echo json_encode(['error' => 'order_id required']); return; }

        // Ownership
        $o = $this->pdo->prepare('SELECT id, user_id, status FROM orders WHERE id=:id AND user_id=:uid LIMIT 1');
        $o->execute(['id'=>$orderId,'uid'=>(int)$user['id']]);
        $order = $o->fetch();
        if (!$order) { http_response_code(404); echo json_encode(['error' => 'Order not found']); return; }
        if ($order['status'] !== 'pending') { http_response_code(400); echo json_encode(['error' => 'Cannot update contact for non-pending order']); return; }

        // Validate
        $customer_name = trim((string)($input['customer_name'] ?? ''));
        $customer_mobile = trim((string)($input['customer_mobile'] ?? ''));
        $customer_email = trim((string)($input['customer_email'] ?? ''));
        $customer_whatsapp = trim((string)($input['customer_whatsapp'] ?? ''));

        if ($customer_name === '' || mb_strlen($customer_name) < 2) {
            http_response_code(422); echo json_encode(['error' => 'Customer name required']); return;
        }
        if ($customer_mobile === '' || mb_strlen($customer_mobile) < 6) {
            http_response_code(422); echo json_encode(['error' => 'Valid mobile number required']); return;
        }
        if ($customer_email !== '' && !filter_var($customer_email, FILTER_VALIDATE_EMAIL)) {
            http_response_code(422); echo json_encode(['error' => 'Invalid email']); return;
        }

        $u = $this->pdo->prepare('UPDATE orders SET customer_name=:n, customer_mobile=:m, customer_email=:e, customer_whatsapp=:w, updated_at=NOW() WHERE id=:id');
        $u->execute([
            'n'=>$customer_name,
            'm'=>$customer_mobile,
            'e'=>$customer_email !== '' ? $customer_email : null,
            'w'=>$customer_whatsapp !== '' ? $customer_whatsapp : null,
            'id'=>$orderId,
        ]);

        echo json_encode(['status'=>'ok']);
    }
}
