<?php
namespace App\Controllers;

use App\Core\Controller;

class ApiController extends Controller
{
    private ?array $apiUser = null;

    private function bearerUser(): ?array
    {
        if ($this->apiUser !== null) return $this->apiUser;
        $hdr = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
        if (!$hdr && function_exists('apache_request_headers')) {
            $headers = apache_request_headers();
            $hdr = $headers['Authorization'] ?? '';
        }
        if (!preg_match('/^Bearer\s+(\S+)/i', $hdr, $m)) {
            return null;
        }
        $token = $m[1];
        $stmt = $this->pdo->prepare('SELECT t.user_id, u.id, u.name, u.email, u.role FROM api_tokens t JOIN users u ON u.id=t.user_id WHERE t.token=:t LIMIT 1');
        $stmt->execute(['t' => $token]);
        $user = $stmt->fetch();
        if ($user) $this->apiUser = $user;
        return $user ?: null;
    }

    private function json($data, int $code = 200): void
    {
        http_response_code($code);
        header('Content-Type: application/json');
        echo json_encode($data);
    }

    private function requireAuth(): ?array
    {
        $u = $this->bearerUser();
        if (!$u) {
            $this->json(['error' => 'Unauthorized'], 401);
            exit;
        }
        return $u;
    }

    public function walletBalance(): void
    {
        $user = $this->requireAuth();
        $stmt = $this->pdo->prepare('SELECT w.balance FROM wallets w WHERE w.user_id = :uid LIMIT 1');
        $stmt->execute(['uid' => $user['id']]);
        $row = $stmt->fetch();
        $balance = $row ? (float)$row['balance'] : 0.0;
        $this->json(['balance' => $balance]);
    }

    public function createTicket(): void
    {
        $user = $this->requireAuth();
        $json = json_decode(file_get_contents('php://input'), true) ?: [];
        $subject = trim($json['subject'] ?? '');
        $message = trim($json['message'] ?? '');
        if ($subject === '' || $message === '') {
            $this->json(['error' => 'subject and message required'], 422);
            return;
        }
        $stmt = $this->pdo->prepare('INSERT INTO tickets (user_id, subject, message) VALUES (:user_id, :subject, :message)');
        $stmt->execute(['user_id' => $user['id'], 'subject' => $subject, 'message' => $message]);
        $this->json(['status' => 'ok', 'ticket_id' => (int)$this->pdo->lastInsertId()], 201);
    }

    public function createBooking(): void
    {
        $user = $this->requireAuth();
        $json = json_decode(file_get_contents('php://input'), true) ?: [];
        $module = $json['module'] ?? '';
        $item_id = (int)($json['item_id'] ?? 0);
        $pax = max(1, (int)($json['pax'] ?? 1));
        if (!in_array($module, ['hotel','activity','taxi','evisa'], true) || $item_id <= 0) {
            $this->json(['error' => 'Invalid module or item_id'], 422);
            return;
        }
        // Minimal pricing: vendor base price from respective table
        $table = $module . 's';
        $priceCol = 'base_price';
        $stmt = $this->pdo->prepare("SELECT {$priceCol} as base FROM {$table} WHERE id=:id LIMIT 1");
        $stmt->execute(['id' => $item_id]);
        $row = $stmt->fetch();
        if (!$row) { $this->json(['error' => 'Item not found'], 404); return; }
        $base = (float)$row['base'];

        // TODO: Apply pricing engine steps (group pricing, admin markup, corporate, promo, taxes)
        $total = $base * $pax;

        $stmt = $this->pdo->prepare('INSERT INTO bookings (user_id, module, item_id, pax, price, status) VALUES (:user_id, :module, :item_id, :pax, :price, :status)');
        $stmt->execute([
            'user_id' => $user['id'],
            'module' => $module,
            'item_id' => $item_id,
            'pax' => $pax,
            'price' => $total,
            'status' => 'pending',
        ]);
        $this->json(['status' => 'ok', 'booking_id' => (int)$this->pdo->lastInsertId(), 'price' => $total], 201);
    }
}
