<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;

class AirportsAdminController extends Controller
{
    public function unlock(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $_SESSION['admin_airports_unlocked'] = true;
        $this->redirect('/admin/airports');
    }

    public function index(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $unlocked = !empty($_SESSION['admin_airports_unlocked']);
        $rows = [];
        if ($unlocked) {
            try {
                $rows = $this->pdo->query('SELECT id, code, name, city, country FROM airports ORDER BY code')->fetchAll() ?: [];
            } catch (\Throwable $_) { $rows = []; }
        }
        $this->view('admin/airports_index', compact('csrf','unlocked','rows'));
    }

    public function create(): void
    {
        Auth::requireRole(['Admin']);
        if (empty($_SESSION['admin_airports_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock airports with the update password first.';
            $this->redirect('/admin/airports');
            return;
        }
        $csrf = Security::csrfToken();
        $airport = ['id'=>null,'code'=>'','name'=>'','city'=>'','country'=>''];
        $this->view('admin/airports_form', compact('csrf','airport'));
    }

    public function edit(): void
    {
        Auth::requireRole(['Admin']);
        if (empty($_SESSION['admin_airports_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock airports with the update password first.';
            $this->redirect('/admin/airports');
            return;
        }
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/airports'); return; }
        $row = null;
        try {
            $st = $this->pdo->prepare('SELECT * FROM airports WHERE id = :id');
            $st->execute([':id'=>$id]);
            $row = $st->fetch(\PDO::FETCH_ASSOC) ?: null;
        } catch (\Throwable $_) { $row = null; }
        if (!$row) { $this->redirect('/admin/airports'); return; }
        $csrf = Security::csrfToken();
        $airport = $row;
        $this->view('admin/airports_form', compact('csrf','airport'));
    }

    public function store(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['admin_airports_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock airports with the update password first.';
            $this->redirect('/admin/airports');
            return;
        }
        $code = strtoupper(trim((string)($_POST['code'] ?? '')));
        $name = trim((string)($_POST['name'] ?? ''));
        $city = trim((string)($_POST['city'] ?? ''));
        $country = trim((string)($_POST['country'] ?? ''));
        $errors = [];
        if ($code === '') $errors[] = 'Code is required';
        if ($name === '') $errors[] = 'Name is required';
        if ($errors) { $_SESSION['errors']=$errors; $this->redirect('/admin/airports/create'); return; }
        try {
            $st = $this->pdo->prepare('INSERT INTO airports (code, name, city, country) VALUES (:code,:name,:city,:country)');
            $st->execute([':code'=>$code, ':name'=>$name, ':city'=>($city!==''?$city:null), ':country'=>($country!==''?$country:null)]);
            $_SESSION['flash'] = 'Airport created.';
        } catch (\Throwable $e) {
            $_SESSION['errors'] = ['Failed to save airport (maybe duplicate code).'];
        }
        $this->redirect('/admin/airports');
    }

    public function update(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['admin_airports_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock airports with the update password first.';
            $this->redirect('/admin/airports');
            return;
        }
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/airports'); return; }
        $code = strtoupper(trim((string)($_POST['code'] ?? '')));
        $name = trim((string)($_POST['name'] ?? ''));
        $city = trim((string)($_POST['city'] ?? ''));
        $country = trim((string)($_POST['country'] ?? ''));
        $errors = [];
        if ($code === '') $errors[] = 'Code is required';
        if ($name === '') $errors[] = 'Name is required';
        if ($errors) { $_SESSION['errors']=$errors; $this->redirect('/admin/airports/edit?id='.$id); return; }
        try {
            $st = $this->pdo->prepare('UPDATE airports SET code=:code, name=:name, city=:city, country=:country WHERE id=:id');
            $st->execute([':id'=>$id, ':code'=>$code, ':name'=>$name, ':city'=>($city!==''?$city:null), ':country'=>($country!==''?$country:null)]);
            $_SESSION['flash'] = 'Airport updated.';
        } catch (\Throwable $e) {
            $_SESSION['errors'] = ['Failed to update airport.'];
        }
        $this->redirect('/admin/airports');
    }

    public function delete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['admin_airports_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock airports with the update password first.';
            $this->redirect('/admin/airports');
            return;
        }
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            try {
                $st = $this->pdo->prepare('DELETE FROM airports WHERE id = :id');
                $st->execute([':id'=>$id]);
                $_SESSION['flash'] = 'Airport deleted.';
            } catch (\Throwable $e) {
                $_SESSION['errors'] = ['Failed to delete airport.'];
            }
        }
        $this->redirect('/admin/airports');
    }
}
