<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Middleware\AgentGuard;
use App\Services\WalletService;
use App\Core\Security;

class AgentWalletDepositsController extends Controller
{
    public function new(): void
    {
        AgentGuard::requireLogin();
        $agent = $_SESSION['agent'] ?? null;
        if (!$agent || empty($agent['id'])) { $this->redirect('/b2b/agent/login'); return; }

        $this->view('agent/wallet_deposit_request', [
            'title' => 'Wallet Deposit Request',
        ]);
    }

    public function create(): void
    {
        AgentGuard::requireLogin();
        Security::requireCsrf();
        $agent = $_SESSION['agent'] ?? null;
        if (!$agent || empty($agent['id'])) { $this->redirect('/b2b/agent/login'); return; }
        $uid = (int)$agent['id'];

        $amount = (float)($_POST['amount'] ?? 0);
        $mode = (string)($_POST['payment_mode'] ?? '');
        if ($amount <= 0 || !in_array($mode, ['bank_transfer','cash_deposit'], true)) {
            $_SESSION['flash_error'] = 'Please enter a valid amount and payment mode.';
            $this->redirect('/b2b/agent/wallet/deposit');
            return;
        }

        $ref = strtoupper(bin2hex(random_bytes(6))); // 12 hex chars
        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '';

        $stmt = $this->pdo->prepare('INSERT INTO wallet_deposit_requests (agent_user_id, amount, payment_mode, status, request_ip, reference_code, requested_at) VALUES (:u,:a,:m, "pending", :ip, :ref, NOW())');
        $stmt->execute([':u'=>$uid, ':a'=>$amount, ':m'=>$mode, ':ip'=>$ip, ':ref'=>$ref]);
        $id = (int)$this->pdo->lastInsertId();

        // Audit
        $this->pdo->prepare('INSERT INTO wallet_deposit_audit (request_id, action, actor_user_id, actor_role, ip, meta) VALUES (:rid, "created", :uid, "agent", :ip, JSON_OBJECT("amount", :amt, "payment_mode", :pm))')
            ->execute([':rid'=>$id, ':uid'=>$uid, ':ip'=>$ip, ':amt'=>$amount, ':pm'=>$mode]);

        $_SESSION['flash_success'] = 'Request submitted. Admin will assign bank account shortly. Reference: '.$ref;
        $this->redirect('/b2b/agent/wallet/deposit/show?id=' . $id);
    }

    public function show(): void
    {
        AgentGuard::requireLogin();
        $agent = $_SESSION['agent'] ?? null;
        if (!$agent || empty($agent['id'])) { $this->redirect('/b2b/agent/login'); return; }
        $uid = (int)$agent['id'];

        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { http_response_code(404); echo 'Not found'; return; }
        $stmt = $this->pdo->prepare('SELECT r.*, cba.bank_name, cba.account_name, cba.account_number, cba.branch, cba.swift_code, cba.currency FROM wallet_deposit_requests r LEFT JOIN company_bank_accounts cba ON cba.id = r.bank_account_id WHERE r.id = :id AND r.agent_user_id = :u LIMIT 1');
        $stmt->execute([':id'=>$id, ':u'=>$uid]);
        $req = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$req) { http_response_code(404); echo 'Not found'; return; }

        // Load audit trail
        $aud = $this->pdo->prepare('SELECT action, actor_role, actor_user_id, ip, meta, created_at FROM wallet_deposit_audit WHERE request_id = :rid ORDER BY id ASC');
        $aud->execute([':rid'=>$id]);
        $audit = $aud->fetchAll(\PDO::FETCH_ASSOC) ?: [];

        $this->view('agent/wallet_deposit_request', [
            'title' => 'Wallet Deposit Request',
            'request' => $req,
            'audit' => $audit,
        ]);
    }

    public function uploadProof(): void
    {
        AgentGuard::requireLogin();
        Security::requireCsrf();
        $agent = $_SESSION['agent'] ?? null;
        if (!$agent || empty($agent['id'])) { $this->redirect('/b2b/agent/login'); return; }
        $uid = (int)$agent['id'];

        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $_SESSION['flash_error'] = 'Invalid request.'; $this->redirect('/b2b/agent/wallet/deposit'); return; }
        // Ownership check and status
        $stmt = $this->pdo->prepare('SELECT * FROM wallet_deposit_requests WHERE id = :id AND agent_user_id = :u LIMIT 1');
        $stmt->execute([':id'=>$id, ':u'=>$uid]);
        $req = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$req) { http_response_code(404); echo 'Not found'; return; }

        $txRef = trim((string)($_POST['transaction_ref'] ?? ''));
        if ($txRef === '') {
            $_SESSION['flash_error'] = 'Transaction reference is required.';
            $this->redirect('/b2b/agent/wallet/deposit/'.$id);
            return;
        }

        // Handle upload
        $path = null;
        if (isset($_FILES['proof_file']) && is_uploaded_file($_FILES['proof_file']['tmp_name'])) {
            $dir = dirname(__DIR__, 2).'/storage/wallet_deposits/';
            if (!is_dir($dir)) { @mkdir($dir, 0775, true); }
            $ext = pathinfo($_FILES['proof_file']['name'], PATHINFO_EXTENSION);
            $safe = 'proof_'.$id.'_'.time().'.'.preg_replace('/[^a-zA-Z0-9]/','', $ext);
            $full = $dir.$safe;
            if (!move_uploaded_file($_FILES['proof_file']['tmp_name'], $full)) {
                $_SESSION['flash_error'] = 'Failed to upload file.';
                $this->redirect('/b2b/agent/wallet/deposit/'.$id);
                return;
            }
            // Public relative path if needed for admin view
            $path = 'storage/wallet_deposits/'.$safe;
        }

        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
        $this->pdo->prepare('UPDATE wallet_deposit_requests SET proof_file = COALESCE(:p, proof_file), transaction_ref = :tr, proof_uploaded_at = NOW(), status = "under_verification" WHERE id = :id AND agent_user_id = :u')
            ->execute([':p'=>$path, ':tr'=>$txRef, ':id'=>$id, ':u'=>$uid]);
        $this->pdo->prepare('INSERT INTO wallet_deposit_audit (request_id, action, actor_user_id, actor_role, ip, meta) VALUES (:rid, "proof_uploaded", :uid, "agent", :ip, JSON_OBJECT("transaction_ref", :tr, "proof_file", :pf))')
            ->execute([':rid'=>$id, ':uid'=>$uid, ':ip'=>$ip, ':tr'=>$txRef, ':pf'=>$path]);

        $_SESSION['flash_success'] = 'Proof uploaded. Awaiting verification.';
        $this->redirect('/b2b/agent/wallet/deposit/show?id=' . $id);
    }

    public function history(): void
    {
        AgentGuard::requireLogin();
        $agent = $_SESSION['agent'] ?? null;
        if (!$agent || empty($agent['id'])) { $this->redirect('/b2b/agent/login'); return; }
        $uid = (int)$agent['id'];

        $stmt = $this->pdo->prepare('SELECT id, reference_code, amount, payment_mode, status, requested_at, updated_at FROM wallet_deposit_requests WHERE agent_user_id = :u ORDER BY id DESC');
        $stmt->execute([':u'=>$uid]);
        $rows = $stmt->fetchAll(\PDO::FETCH_ASSOC) ?: [];

        $this->view('agent/wallet_deposit_history', [ 'title' => 'Wallet Deposit History', 'rows' => $rows ]);
    }
}
