<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Security;
use App\Core\Mail\MailService;
use App\Core\Mail\Envelope;
use DateTimeImmutable;
use DateInterval;

class AgentAuthController extends Controller
{
    // GET /agent/register
    public function register(): void
    {
        // Load app config to get brand/app name for the view (no global helper available here)
        $appCfgPath = dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'app.php';
        $appCfg = file_exists($appCfgPath) ? (require $appCfgPath) : [];
        $appName = (string)($appCfg['name'] ?? 'B2B Travel');
        $this->view('agent/auth/register', [
            'title' => 'Agent Registration',
            'csrf' => Security::csrfToken(),
            'appName' => $appName,
        ]);
    }

    // POST /agent/register
    public function registerPost(): void
    {
        if (!Security::verifyCsrf((string)($_POST['csrf'] ?? ''))) {
            http_response_code(400); echo 'Invalid CSRF'; return;
        }
        $name = trim((string)($_POST['name'] ?? ''));
        $email = strtolower(trim((string)($_POST['email'] ?? '')));
        $phone = trim((string)($_POST['phone'] ?? ($_POST['agent_phone'] ?? '')));
        $country = strtoupper(trim((string)($_POST['country'] ?? '')));
        $businessType = strtolower(trim((string)($_POST['business_type'] ?? 'freelancer')));
        // We no longer collect agency_name at registration; use legal company later in KYC
        $agencyName = '';
        $company = trim((string)($_POST['agent_company'] ?? ''));
        $password = (string)($_POST['password'] ?? '');
        $password2 = (string)($_POST['password_confirm'] ?? '');

        $errors = [];
        if ($name === '') $errors[] = 'Full name is required';
        if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = 'Valid email is required';
        if ($password === '' || strlen($password) < 8) $errors[] = 'Password must be at least 8 characters';
        if ($password !== $password2) $errors[] = 'Passwords do not match';
        $allowedTypes = ['freelancer','sole_proprietor','partnership','company'];
        if (!in_array($businessType, $allowedTypes, true)) { $businessType = 'freelancer'; }
        $needsCompany = ($businessType !== 'freelancer');
        if ($needsCompany && $company === '') { $errors[] = 'Company/Agency is required for the selected business type.'; }
        if ($errors) { $_SESSION['errors'] = $errors; $this->redirect('/agent/register'); return; }

        // Check if email already exists
        $st = $this->pdo->prepare('SELECT id FROM users WHERE email = :e LIMIT 1');
        $st->execute([':e' => $email]);
        if ($st->fetch()) { $_SESSION['errors'] = ['Email already registered']; $this->redirect('/agent/register'); return; }

        // Create user as B2B Agent (inactive until email verified)
        $hash = password_hash($password, PASSWORD_DEFAULT);
        $role = 'B2B Agent';
        $status = 'Active';
        $ins = $this->pdo->prepare('INSERT INTO users (name, email, password, role, status, email_verified) VALUES (:n,:e,:h,:r,:s,0)');
        try {
            $ins->execute([':n'=>$name, ':e'=>$email, ':h'=>$hash, ':r'=>$role, ':s'=>$status]);
        } catch (\Throwable $e) {
            $_SESSION['errors'] = ['Could not create user: '.$e->getMessage()]; $this->redirect('/agent/register'); return;
        }
        $userId = (int)$this->pdo->lastInsertId();

        // Create minimal agent profile and store phone there
        try {
            // Persist selected country into agent profile for downstream workflows
            $stmt = $this->pdo->prepare('INSERT INTO agent_profiles (user_id, phone, business_type, agency_name, company, country, commission_percent, credit_limit, created_at) VALUES (:uid,:phone,:bt,:agency,:company,:country,0,0,NOW())');
            $stmt->execute([':uid'=>$userId, ':phone'=>$phone, ':bt'=>$businessType, ':agency'=>null, ':company'=>$company, ':country' => ($country !== '' ? $country : null)]);
        } catch (\Throwable $e) {
            // non-fatal: continue, but record a flash message
            $_SESSION['flash'] = 'Account created. Note: profile initialization deferred.';
        }

        // Issue email OTP (6 digits, 10 minutes). Let DB compute expiry to avoid PHP/DB timezone drift.
        $otp = (string)random_int(100000, 999999);
        $otpHash = password_hash($otp, PASSWORD_DEFAULT);
        $ins2 = $this->pdo->prepare("INSERT INTO verification_tokens (user_id, type, token_hash, otp_code, expires_at) VALUES (:u,'email_otp',:th,:oc, DATE_ADD(NOW(), INTERVAL 10 MINUTE))");
        $ins2->execute([':u'=>$userId, ':th'=>$otpHash, ':oc'=>$otp]);

        // Send email via MailService (respects active provider + admin settings)
        $brand = 'BookMyThai';
        $subject = "Verify your email - $brand";
        $html = '<p>Hello '.htmlspecialchars($name).',</p>'
              . '<p>Your one-time code is: <strong style="font-size:18px">'.$otp.'</strong></p>'
              . '<p>This code will expire in 10 minutes.</p>'
              . '<p>If you did not request this, you can ignore this email.</p>';
        $ms = new MailService($this->pdo);
        $env = new Envelope($email, $subject, $html, '', 'admin@bookmythai.com', 'BookMyThai');
        $ms->send($env);

        $_SESSION['flash'] = 'We sent a 6-digit code to your email. Please verify to continue.';
        $_SESSION['pending_verify_user_id'] = $userId;
        $this->redirect('/agent/verify-email');
    }

    // GET /agent/verify-email
    public function verifyEmail(): void
    {
        // Only allow access if a pending verification exists
        // If agent is already logged in, go to dashboard; else require pending user id
        if (!empty($_SESSION['agent']) && !empty($_SESSION['agent']['id'])) {
            $this->redirect('/b2b/agent');
            return;
        }
        $userId = (int)($_SESSION['pending_verify_user_id'] ?? 0);
        if ($userId <= 0) {
            $this->redirect('/b2b/agent/login');
            return;
        }
        $sentAt = null; $expiresAt = null;
        if ($userId > 0) {
            try {
                $st = $this->pdo->prepare("SELECT created_at, expires_at FROM verification_tokens WHERE user_id=:u AND type='email_otp' AND consumed_at IS NULL ORDER BY id DESC LIMIT 1");
                $st->execute([':u' => $userId]);
                if ($row = $st->fetch(\PDO::FETCH_ASSOC)) {
                    $sentAt = (string)($row['created_at'] ?? null);
                    $expiresAt = (string)($row['expires_at'] ?? null);
                }
            } catch (\Throwable $_) { /* ignore */ }
        }
        $this->view('agent/auth/verify_email', [
            'title' => 'Verify Email',
            'csrf' => Security::csrfToken(),
            'sent_at' => $sentAt,
            'expires_at' => $expiresAt,
        ]);
    }

    // POST /agent/verify-email
    public function verifyEmailPost(): void
    {
        if (!Security::verifyCsrf((string)($_POST['csrf'] ?? ''))) {
            http_response_code(400); echo 'Invalid CSRF'; return;
        }
        $userId = (int)($_SESSION['pending_verify_user_id'] ?? 0);
        $code = trim((string)($_POST['otp'] ?? ''));
        if ($userId <= 0 || $code === '') { $_SESSION['errors'] = ['Invalid request']; $this->redirect('/agent/verify-email'); return; }
        // Fetch latest non-expired token (let DB decide expiry using its own NOW())
        $st = $this->pdo->prepare("SELECT * FROM verification_tokens WHERE user_id=:u AND type='email_otp' AND consumed_at IS NULL AND expires_at > NOW() ORDER BY id DESC LIMIT 1");
        $st->execute([':u'=>$userId]);
        $row = $st->fetch();
        if (!$row) { $_SESSION['errors'] = ['Code expired. Please resend.']; $this->redirect('/agent/verify-email'); return; }
        // Validate OTP by timing-safe compare (we stored hash as token_hash but also saved otp_code for audits)
        $ok = password_verify($code, (string)$row['token_hash']);
        if (!$ok) { $_SESSION['errors'] = ['Invalid code']; $this->redirect('/agent/verify-email'); return; }
        // Mark verified
        $this->pdo->prepare('UPDATE users SET email_verified=1 WHERE id=:id')->execute([':id'=>$userId]);
        $this->pdo->prepare('UPDATE verification_tokens SET consumed_at=NOW() WHERE id=:id')->execute([':id'=>(int)$row['id']]);
        unset($_SESSION['pending_verify_user_id']);
        // Send user to agent login with a clear success message
        $_SESSION['agent_login_success'] = 'Verification successful. You can now sign in and complete your KYC.';
        $this->redirect('/b2b/agent/login');
    }

    // POST /agent/resend-otp
    public function resendOtp(): void
    {
        if (!Security::verifyCsrf((string)($_POST['csrf'] ?? ''))) {
            http_response_code(400); echo 'Invalid CSRF'; return;
        }
        $userId = (int)($_SESSION['pending_verify_user_id'] ?? 0);
        if ($userId <= 0) { $_SESSION['errors'] = ['No pending verification']; $this->redirect('/agent/verify-email'); return; }
        // Cooldown: 60s
        $st = $this->pdo->prepare("SELECT created_at FROM verification_tokens WHERE user_id=:u AND type='email_otp' ORDER BY id DESC LIMIT 1");
        $st->execute([':u'=>$userId]);
        $last = $st->fetchColumn();
        if ($last) {
            $lastTs = strtotime((string)$last);
            if (time() - $lastTs < 60) { $_SESSION['errors'] = ['Please wait before requesting another code']; $this->redirect('/agent/verify-email'); return; }
        }
        // Re-issue code
        $otp = (string)random_int(100000, 999999);
        $otpHash = password_hash($otp, PASSWORD_DEFAULT);
        $this->pdo->prepare("INSERT INTO verification_tokens (user_id, type, token_hash, otp_code, expires_at) VALUES (:u,'email_otp',:th,:oc, DATE_ADD(NOW(), INTERVAL 10 MINUTE))")
            ->execute([':u'=>$userId, ':th'=>$otpHash, ':oc'=>$otp]);
        // Fetch email
        $st2 = $this->pdo->prepare('SELECT name, email FROM users WHERE id=:id');
        $st2->execute([':id'=>$userId]);
        $u = $st2->fetch();
        $ms = new MailService($this->pdo);
        $subject = 'Your OTP code';
        $html = '<p>Hello '.htmlspecialchars((string)($u['name'] ?? 'Agent')).',</p>'
              . '<p>Your one-time code is: <strong style="font-size:18px">'.$otp.'</strong></p>'
              . '<p>This code will expire in 10 minutes.</p>';
        $env = new Envelope((string)$u['email'], $subject, $html, '', 'admin@bookmythai.com', 'BookMyThai');
        $ms->send($env);
        $_SESSION['flash'] = 'A new code was sent.';
        $this->redirect('/agent/verify-email');
    }
}
