<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Mail\Mailer;
use App\Core\Security;
use App\Middleware\AuthGuard; // assuming admin auth uses AuthGuard roles

class AdminWalletDepositsController extends Controller
{
    private function ensureAdmin(): void
    {
        $role = $_SESSION['user']['role'] ?? '';
        if (empty($_SESSION['user']) || !in_array($role, ['admin','Admin'], true)) {
            header('Location: ' . rtrim(\APP_URL, '/') . '/login/admin');
            exit;
        }
    }

    public function index(): void
    {
        $this->ensureAdmin();
        $status = (string)($_GET['status'] ?? 'pending');
        $allowed = ['pending','bank_assigned','under_verification','approved','rejected'];
        if (!in_array($status, $allowed, true)) { $status = 'pending'; }
        $stmt = $this->pdo->prepare("SELECT r.*, u.email AS agent_email FROM wallet_deposit_requests r LEFT JOIN users u ON u.id = r.agent_user_id WHERE r.status = :s ORDER BY r.created_at DESC");
        $stmt->execute([':s'=>$status]);
        $rows = $stmt->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        $this->view('admin/wallet_deposits_list', [ 'title' => 'Wallet Deposits', 'rows' => $rows, 'status' => $status ]);
    }

    public function show(): void
    {
        $this->ensureAdmin();
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/wallet/deposits'); return; }
        $stmt = $this->pdo->prepare('SELECT r.*, u.email as agent_email, ap.company AS agent_company, cba.bank_name, cba.account_name, cba.account_number, cba.branch, cba.swift_code, cba.currency FROM wallet_deposit_requests r LEFT JOIN users u ON u.id = r.agent_user_id LEFT JOIN agent_profiles ap ON ap.user_id = r.agent_user_id LEFT JOIN company_bank_accounts cba ON cba.id = r.bank_account_id WHERE r.id = :id');
        $stmt->execute([':id'=>$id]);
        $req = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$req) { http_response_code(404); echo 'Not found'; return; }
        // bank accounts for assignment
        $banks = $this->pdo->query('SELECT id, bank_name, account_name, account_number, currency FROM company_bank_accounts WHERE active = 1 ORDER BY bank_name')->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        $this->view('admin/wallet_deposit_detail', [ 'title' => 'Wallet Deposit Detail', 'request' => $req, 'banks' => $banks ]);
    }

    public function assign(): void
    {
        $this->ensureAdmin();
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $_SESSION['flash_error'] = 'Invalid request'; $this->redirect('/admin/wallet/deposits'); return; }
        $bankId = (int)($_POST['bank_account_id'] ?? 0);
        if ($bankId <= 0) { $_SESSION['flash_error'] = 'Select a bank account.'; $this->redirect('/admin/wallet/deposits/'.$id); return; }
        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
        $adminId = (int)($_SESSION['user']['id'] ?? 0);
        $this->pdo->prepare('UPDATE wallet_deposit_requests SET bank_account_id = :b, status = "bank_assigned", assigned_admin_id = :aid, assigned_ip = :ip, assigned_at = NOW() WHERE id = :id')
            ->execute([':b'=>$bankId, ':aid'=>$adminId, ':ip'=>$ip, ':id'=>$id]);
        $this->pdo->prepare('INSERT INTO wallet_deposit_audit (request_id, action, actor_user_id, actor_role, ip, meta) VALUES (:rid, "assigned", :uid, "admin", :ip, JSON_OBJECT("bank_account_id", :bid))')
            ->execute([':rid'=>$id, ':uid'=>$adminId, ':ip'=>$ip, ':bid'=>$bankId]);

        // Notify agent by email with assigned account details
        try {
            $st = $this->pdo->prepare('SELECT r.reference_code, r.amount, u.email, c.bank_name, c.account_name, c.account_number, c.branch, c.swift_code, c.currency FROM wallet_deposit_requests r JOIN users u ON u.id=r.agent_user_id LEFT JOIN company_bank_accounts c ON c.id = r.bank_account_id WHERE r.id = :id');
            $st->execute([':id'=>$id]);
            if ($row = $st->fetch(\PDO::FETCH_ASSOC)) {
                $mailCfg = require dirname(__DIR__,2) . '/config/mail.php';
                $mailer = new Mailer($mailCfg);
                $html = '<p>Your wallet deposit request <strong>'.$row['reference_code'].'</strong> has been updated. Please transfer ฿'.number_format((float)$row['amount'],2).' to:</p>'
                      . '<ul>'
                      . '<li><strong>Bank:</strong> '.htmlspecialchars((string)$row['bank_name']).'</li>'
                      . '<li><strong>Account Name:</strong> '.htmlspecialchars((string)$row['account_name']).'</li>'
                      . '<li><strong>Account Number:</strong> '.htmlspecialchars((string)$row['account_number']).'</li>'
                      . '<li><strong>Currency:</strong> '.htmlspecialchars((string)$row['currency']).'</li>'
                      . '</ul>'
                      . '<p>Use Reference Code: <strong>'.$row['reference_code'].'</strong> in remarks if possible and upload the slip in the portal.</p>';
                $mailer->sendHtml((string)$row['email'], 'Bank Account Assigned for Deposit '.$row['reference_code'], $html);
            }
        } catch (\Throwable $_) { /* ignore email failures */ }
        $_SESSION['flash_success'] = 'Bank account assigned.';
        $this->redirect('/admin/wallet/deposits/view?id='.$id);
    }

    public function approve(): void
    {
        $this->ensureAdmin();
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $_SESSION['flash_error'] = 'Invalid request.'; $this->redirect('/admin/wallet/deposits'); return; }
        // Load request
        $stmt = $this->pdo->prepare('SELECT * FROM wallet_deposit_requests WHERE id = :id LIMIT 1');
        $stmt->execute([':id'=>$id]);
        $req = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$req) { http_response_code(404); echo 'Not found'; return; }
        if (!in_array($req['status'], ['under_verification','bank_assigned'], true)) { $_SESSION['flash_error'] = 'Invalid status for approval.'; $this->redirect('/admin/wallet/deposits/'.$id); return; }
        $adminId = (int)($_SESSION['user']['id'] ?? 0);
        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '';

        // Credit wallet and insert wallet_transactions via helper function if present, else via WalletService
        $amount = (float)$req['amount'];
        if ($amount <= 0) { $_SESSION['flash_error'] = 'Invalid amount.'; $this->redirect('/admin/wallet/deposits/'.$id); return; }

        // Try SQL function insert_wallet_transaction if exists
        try {
            $this->pdo->beginTransaction();
            // Ensure a wallet row exists for this user (one per user). Do NOT update balance directly here.
            // Triggers on wallet_transactions will adjust wallets.balance.
            $this->pdo->prepare('INSERT IGNORE INTO wallets (user_id, balance) VALUES (:u, 0)')->execute([':u'=>(int)$req['agent_user_id']]);
            // Log wallet_transactions if table exists
            $hasWt = $this->pdo->query("SHOW TABLES LIKE 'wallet_transactions'")->fetch();
            if ($hasWt) {
                $st = $this->pdo->prepare('INSERT INTO wallet_transactions (user_id, transaction_type, category, amount, balance_before, balance_after, currency, description, reference_id, reference_type, admin_user_id, metadata, status, created_at) VALUES (:u, "credit", "topup", :a, 0, 0, "THB", :desc, :ref, "admin_action", :aid, JSON_OBJECT("wdr_id", :rid), "completed", NOW())');
                $st->execute([':u'=>(int)$req['agent_user_id'], ':a'=>$amount, ':desc'=>'Wallet top-up approved', ':ref'=>'WDR-'.$id, ':aid'=>$adminId, ':rid'=>$id]);
            }
            $this->pdo->prepare('UPDATE wallet_deposit_requests SET status = "approved", verified_admin_id = :aid, verified_ip = :ip, verified_at = NOW() WHERE id = :id')
                ->execute([':aid'=>$adminId, ':ip'=>$ip, ':id'=>$id]);
            $this->pdo->prepare('INSERT INTO wallet_deposit_audit (request_id, action, actor_user_id, actor_role, ip, meta) VALUES (:rid, "approved", :uid, "admin", :ip, JSON_OBJECT("amount", :amt))')
                ->execute([':rid'=>$id, ':uid'=>$adminId, ':ip'=>$ip, ':amt'=>$amount]);
            // notify agent
            try {
                $st = $this->pdo->prepare('SELECT u.email, r.reference_code FROM wallet_deposit_requests r JOIN users u ON u.id = r.agent_user_id WHERE r.id = :id');
                $st->execute([':id'=>$id]);
                if ($row = $st->fetch(\PDO::FETCH_ASSOC)) {
                    $mailCfg = require dirname(__DIR__,2) . '/config/mail.php';
                    $mailer = new Mailer($mailCfg);
                    $mailer->sendHtml((string)$row['email'], 'Deposit Approved '.$row['reference_code'], '<p>Your wallet deposit request <strong>'.$row['reference_code'].'</strong> has been approved and credited.</p>');
                }
            } catch (\Throwable $_) { /* ignore */ }
            $this->pdo->commit();
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) $this->pdo->rollBack();
            $_SESSION['flash_error'] = 'Approval failed: '.$e->getMessage();
            $this->redirect('/admin/wallet/deposits/'.$id);
            return;
        }
        $_SESSION['flash_success'] = 'Deposit approved and wallet credited.';
        $this->redirect('/admin/wallet/deposits/view?id='.$id);
    }

    public function reject(): void
    {
        $this->ensureAdmin();
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $_SESSION['flash_error'] = 'Invalid request.'; $this->redirect('/admin/wallet/deposits'); return; }
        $reason = trim((string)($_POST['rejection_reason'] ?? 'Invalid proof'));
        $adminId = (int)($_SESSION['user']['id'] ?? 0);
        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
        $this->pdo->prepare('UPDATE wallet_deposit_requests SET status = "rejected", verified_admin_id = :aid, verified_ip = :ip, verified_at = NOW(), rejection_reason = :r WHERE id = :id')
            ->execute([':aid'=>$adminId, ':ip'=>$ip, ':r'=>$reason, ':id'=>$id]);
        $this->pdo->prepare('INSERT INTO wallet_deposit_audit (request_id, action, actor_user_id, actor_role, ip, meta) VALUES (:rid, "rejected", :uid, "admin", :ip, JSON_OBJECT("reason", :r))')
            ->execute([':rid'=>$id, ':uid'=>$adminId, ':ip'=>$ip, ':r'=>$reason]);
        // notify agent
        try {
            $st = $this->pdo->prepare('SELECT u.email, r.reference_code FROM wallet_deposit_requests r JOIN users u ON u.id = r.agent_user_id WHERE r.id = :id');
            $st->execute([':id'=>$id]);
            if ($row = $st->fetch(\PDO::FETCH_ASSOC)) {
                $mailCfg = require dirname(__DIR__,2) . '/config/mail.php';
                $mailer = new Mailer($mailCfg);
                $mailer->sendHtml((string)$row['email'], 'Deposit Rejected '.$row['reference_code'], '<p>Your wallet deposit request <strong>'.$row['reference_code'].'</strong> has been rejected.</p><p>Reason: '.htmlspecialchars($reason).'</p>');
            }
        } catch (\Throwable $_) { /* ignore */ }
        $_SESSION['flash_success'] = 'Deposit rejected.';
        $this->redirect('/admin/wallet/deposits/'.$id);
    }

    public function exportCsv(): void
    {
        $this->ensureAdmin();
        $status = (string)($_GET['status'] ?? '');
        $where = '';
        $params = [];
        if ($status !== '') { $where = 'WHERE r.status = :s'; $params[':s'] = $status; }
        $sql = 'SELECT r.id, r.reference_code, u.email AS agent_email, r.amount, r.payment_mode, r.status, r.requested_at, r.updated_at FROM wallet_deposit_requests r LEFT JOIN users u ON u.id = r.agent_user_id ' . $where . ' ORDER BY r.id DESC';
        $st = $this->pdo->prepare($sql);
        $st->execute($params);
        $rows = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename=wallet_deposits_'.date('Ymd_His').'.csv');
        $out = fopen('php://output', 'w');
        fputcsv($out, array_keys($rows[0] ?? ['id','reference_code','agent_email','amount','payment_mode','status','requested_at','updated_at']));
        foreach ($rows as $r) { fputcsv($out, $r); }
        fclose($out);
        exit;
    }
}
