<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Middleware\AgentGuard;

class ActivityAgentController extends Controller
{
    // GET/POST /b2b/agent/activity/checkout
    public function checkout(): void
    {
        AgentGuard::requireLogin();
        $agent = $_SESSION['agent'] ?? null;
        if (!$agent || empty($agent['id'])) { $this->redirect('/b2b/agent/login'); return; }

        if (strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST') {
            $csrf = (string)($_POST['csrf_token'] ?? '');
            $valid = AgentGuard::validateCsrf('activity_checkout', $csrf) || AgentGuard::validateCsrf('cart_add', $csrf);
            if (!$valid) { $this->redirect('/b2b/agent/activity/checkout?err=csrf'); return; }

            // Distinguish stage by presence of lead_name (details submit) vs. items[] (cart submit)
            $isDetailsSubmit = isset($_POST['lead_name']) || isset($_POST['lead_email']) || isset($_POST['lead_phone']);
            if ($isDetailsSubmit) {
                // Persist guest payload from checkout form
                $payload = [
                    'lead' => [
                        'name' => trim((string)($_POST['lead_name'] ?? '')),
                        'email' => trim((string)($_POST['lead_email'] ?? '')),
                        'phone' => trim((string)($_POST['lead_phone'] ?? '')),
                        'nationality' => trim((string)($_POST['lead_nationality'] ?? '')),
                        'passport_no' => trim((string)($_POST['lead_passport_no'] ?? '')),
                        'passport_expiry' => trim((string)($_POST['lead_passport_expiry'] ?? '')),
                    ],
                    'date' => trim((string)($_POST['booking_date'] ?? '')),
                    'show_time' => trim((string)($_POST['show_time'] ?? '')),
                    'transport' => [
                        'option_id' => (int)($_POST['transport_option_id'] ?? 0),
                        'pickup_notes' => trim((string)($_POST['pickup_notes'] ?? '')),
                        'hotel_name' => trim((string)($_POST['pickup_hotel_name'] ?? '')),
                        'hotel_address' => trim((string)($_POST['pickup_hotel_address'] ?? '')),
                    ],
                    'notes' => trim((string)($_POST['special_requests'] ?? '')),
                ];
                // Preserve previously stored cart and date/time if empty
                $existing = $_SESSION['activity_guest'] ?? [];
                if (!($payload['date'] ?? '') && isset($existing['date'])) $payload['date'] = $existing['date'];
                if (!($payload['show_time'] ?? '') && isset($existing['show_time'])) $payload['show_time'] = $existing['show_time'];
                $_SESSION['activity_guest'] = $payload;

                // Create a pending booking row now (pre-issue), similar to Taxi
                $this->activityLog('checkout:start_details_submit', [
                    'agent_id' => (int)$agent['id'],
                    'date' => (string)($payload['date'] ?? ''),
                    'show_time' => (string)($payload['show_time'] ?? ''),
                ]);
                try {
                    $agentId = (int)$agent['id'];
                    $packageId = (int)($_SESSION['activity_package_id'] ?? 0);
                    $guest = $_SESSION['activity_guest'] ?? [];
                    $cart = $_SESSION['activity_cart'] ?? [];
                    $bookingCode = 'ACT-' . date('Ymd') . '-' . substr((string)time(), -5);

                    // compute qty_total and amount_total from cart
                    $qtyTotal = 0; $amount = 0.0; $itemsArr = [];
                    if (!empty($cart)) {
                        // map price_id -> qty
                        $priceIdQty = [];
                        foreach ($cart as $it) { $pid=(int)($it['price_id'] ?? 0); $qty=max(0,(int)($it['qty'] ?? 0)); if($pid>0 && $qty>0){ $priceIdQty[$pid]=($priceIdQty[$pid] ?? 0)+$qty; $qtyTotal += $qty; } }
                        if (!empty($priceIdQty)) {
                            $ids = array_keys($priceIdQty);
                            $ph = implode(',', array_fill(0, count($ids), '?'));
                            // Fetch detailed price and variant information
                            $st = $this->pdo->prepare('SELECT vpp.id, vpp.agent_price, vpp.variant_id, vpp.pax_type, vpp.price_type,
                                vpv.name as variant_name, vpv.package_id, vp.name as package_name
                                FROM vendor_package_prices vpp 
                                LEFT JOIN vendor_package_variants vpv ON vpp.variant_id = vpv.id
                                LEFT JOIN vendor_packages vp ON vpv.package_id = vp.id
                                WHERE vpp.id IN (' . $ph . ')'
                            );
                            $st->execute($ids);
                            $map = [];
                            foreach ($st->fetchAll(\PDO::FETCH_ASSOC) ?: [] as $r) { 
                                $map[(int)$r['id']] = $r; 
                            }
                            
                            foreach ($priceIdQty as $pid=>$qty) {
                                $unit = isset($map[$pid]['agent_price']) ? (float)$map[$pid]['agent_price'] : 0.0;
                                $vidLocal = isset($map[$pid]['variant_id']) ? (int)$map[$pid]['variant_id'] : null;
                                $line = round($unit * $qty, 2);
                                $amount += $line;
                                
                                // Derive human-readable price_name from pax_type/price_type
                                $paxType = (string)($map[$pid]['pax_type'] ?? '');
                                $priceType = (string)($map[$pid]['price_type'] ?? '');
                                $label = $paxType;
                                if ($paxType === 'adult') { $label = 'Adult'; }
                                elseif ($paxType === 'child') { $label = 'Child'; }
                                elseif ($paxType === 'flat') { $label = 'Flat'; }
                                else { $label = strtoupper($paxType); }
                                if ($priceType === 'group_tier') { $label .= ' (Group Tier)'; }

                                // Enhanced item data with package and variant details
                                $itemData = [
                                    'variant_id' => $vidLocal,
                                    'variant_name' => $map[$pid]['variant_name'] ?? '',
                                    'package_id' => $map[$pid]['package_id'] ?? null,
                                    'package_name' => $map[$pid]['package_name'] ?? '',
                                    'price_id' => $pid,
                                    'price_name' => $label,
                                    'qty' => $qty,
                                    'unit' => $unit,
                                    'line_total' => $line
                                ];
                                $itemsArr[] = $itemData;
                            }
                            // Derive package_id if unknown from first variant
                            if ($packageId === 0) {
                                try {
                                    $firstPid = (int)array_key_first($priceIdQty);
                                    $q = $this->pdo->prepare('SELECT vpv.package_id FROM vendor_package_prices vpp JOIN vendor_package_variants vpv ON vpp.variant_id = vpv.id WHERE vpp.id = ? LIMIT 1');
                                    $q->execute([$firstPid]);
                                    $pk = (int)($q->fetchColumn() ?: 0);
                                    if ($pk > 0) { $packageId = $pk; $_SESSION['activity_package_id'] = $pk; }
                                } catch (\Throwable $_) { /* ignore */ }
                            }
                        }
                    }
                    $itemsJson = json_encode($itemsArr, JSON_UNESCAPED_UNICODE);

                    // resolve vendor_id and summary name
                    $vendorId = null;
                    try {
                        if ($packageId > 0) {
                            $sp = $this->pdo->prepare('SELECT vendor_id FROM vendor_packages WHERE id = ?');
                            $sp->execute([$packageId]);
                            if ($row = $sp->fetch(\PDO::FETCH_ASSOC)) { $vendorId = isset($row['vendor_id']) ? (int)$row['vendor_id'] : null; }
                        }
                    } catch (\Throwable $_) { /* ignore */ }

                    $userAgent = (string)($_SERVER['HTTP_USER_AGENT'] ?? '');
                    $ipAddr = (string)($_SERVER['REMOTE_ADDR'] ?? '');
                    $pickup = (array)($payload['transport'] ?? []);

                    // Validate required fields for schema: booking_date, package_id and qty_total
                    $bdateRaw = (string)($payload['date'] ?? '');
                    if ($bdateRaw === '') { $this->activityLog('checkout:error_missing_date'); $this->redirect('/b2b/agent/activity/checkout?err=missing_date'); return; }
                    $ts = @strtotime($bdateRaw);
                    if ($ts === false) { $this->activityLog('checkout:error_invalid_date_format', ['input'=>$bdateRaw]); $this->redirect('/b2b/agent/activity/checkout?err=missing_date'); return; }
                    $bdate = date('Y-m-d', $ts);
                    if ($qtyTotal <= 0) { $this->redirect('/b2b/agent/activity/checkout?err=no_items'); return; }
                    if ($packageId <= 0) { $this->redirect('/b2b/agent/activity/checkout?err=no_package'); return; }

                    $this->activityLog('checkout:creating_pending_booking', [
                        'agent_id' => $agentId,
                        'package_id' => $packageId,
                        'qty_total' => $qtyTotal,
                        'amount' => $amount,
                        'booking_date' => $bdate,
                    ]);
                    
                    if ($bdate !== '' && $packageId > 0 && $qtyTotal > 0) {
                        try {
                            $insSql = "INSERT INTO activity_bookings (
                                booking_code, agent_id, created_by, updated_by, vendor_id, package_id, 
                                booking_date, show_time, items_json, qty_total, amount_total, currency, 
                                channel, status, payment_method, gateway_name, payment_status, payment_txn_id, 
                                ip_address, user_agent, lead_name, lead_phone, lead_email, 
                                pickup_required, pickup_hotel_name, pickup_address, pickup_city, pickup_notes, 
                                created_at, updated_at
                            ) VALUES (
                                :code, :uid, :uid, :uid, :vid, :pkg, :bdate, :stime, :items, :qty, :amt, :cur, 
                                'portal', 'pending', NULL, NULL, 'pending', NULL, 
                                :ip, :ua, :lname, :lphone, :lemail, 
                                :p_req, :p_hotel, :p_addr, :p_city, :p_notes, 
                                NOW(), NOW()
                            )";
                            
                            $this->pdo->beginTransaction();
                            
                            $ins = $this->pdo->prepare($insSql);
                            $ins->execute([
                                ':code' => $bookingCode,
                                ':uid' => $agentId,
                                ':vid' => $vendorId,
                                ':pkg' => $packageId,
                                ':bdate' => $bdate,
                                ':stime' => !empty($payload['show_time']) ? (string)$payload['show_time'] : null,
                                ':items' => $itemsJson,
                                ':qty' => $qtyTotal,
                                ':amt' => $amount,
                                ':cur' => 'THB',
                                ':ip' => $ipAddr,
                                ':ua' => $userAgent,
                                ':lname' => (string)($payload['lead']['name'] ?? ''),
                                ':lphone' => (string)($payload['lead']['phone'] ?? ''),
                                ':lemail' => (string)($payload['lead']['email'] ?? ''),
                                ':p_req' => (int)(!empty($pickup['option_id'])),
                                ':p_hotel' => (string)($pickup['hotel_name'] ?? ''),
                                ':p_addr' => (string)($pickup['hotel_address'] ?? ''),
                                ':p_city' => '',
                                ':p_notes' => (string)($pickup['pickup_notes'] ?? ($pickup['notes'] ?? '')),
                            ]);
                            
                            $bid = (int)$this->pdo->lastInsertId();
                            
                            // Add booking event
                            $ev = $this->pdo->prepare('INSERT INTO activity_booking_events (booking_id, event_type, note, data_json, created_at) 
                                VALUES (:bid, :type, :note, :data, NOW())');
                            $ev->execute([
                                ':bid' => $bid,
                                ':type' => 'created',
                                ':note' => 'Pending booking created',
                                ':data' => json_encode([
                                    'qty_total' => $qtyTotal,
                                    'amount' => $amount,
                                    'package_id' => $packageId
                                ])
                            ]);
                            
                            $this->pdo->commit();
                            
                            // Store booking ID and code in session
                            $_SESSION['activity_booking_id'] = $bid;
                            $_SESSION['activity_booking_code'] = $bookingCode;
                            
                            // Debug log (can be removed in production)
                            $this->activityLog('checkout:pending_booking_created', [
                                'booking_id' => $bid,
                                'booking_code' => $bookingCode,
                                'amount' => $amount,
                                'qty_total' => $qtyTotal,
                            ]);
                            
                        } catch (\Throwable $e) {
                            if ($this->pdo->inTransaction()) {
                                $this->pdo->rollBack();
                            }
                            error_log("Error creating pending booking: " . $e->getMessage());
                            $this->activityLog('checkout:pending_booking_error', [
                                'error' => $e->getMessage(),
                                'agent_id' => $agentId,
                                'package_id' => $packageId,
                                'qty_total' => $qtyTotal,
                                'amount' => $amount,
                            ]);
                            $bid = 0;
                        }
                    } else {
                        $bid = 0; // skip creating pending row due to missing required data
                        error_log("Skipping booking creation - missing required data: date={$bdate}, pkg={$packageId}, qty={$qtyTotal}");
                        $this->activityLog('checkout:skip_pending_booking_missing_required', [
                            'booking_date' => $bdate,
                            'package_id' => $packageId,
                            'qty_total' => $qtyTotal,
                        ]);
                    }
                    if ($bid > 0) {
                        $_SESSION['activity_booking_id'] = $bid;
                        $_SESSION['activity_booking_code'] = $bookingCode;
                        try {
                            $ev = $this->pdo->prepare('INSERT INTO activity_booking_events (booking_id, event_type, note, data_json, created_at) VALUES (:bid,:t,:n,:d,NOW())');
                            $ev->execute([':bid'=>$bid, ':t'=>'created', ':n'=>'Pending booking created', ':d'=>json_encode(['qty_total'=>$qtyTotal,'amount'=>$amount])]);
                        } catch (\Throwable $_) {}
                    }
                } catch (\Throwable $_) { /* allow proceed without blocking */ }

                $this->activityLog('checkout:redirect_payment', [ 'agent_id' => (int)$agent['id'] ]);
                $this->redirect('/b2b/agent/activity/payment');
                return;
            } else {
                // Cart submit from activity detail page: store items and prelim info, then show checkout form (GET)
                $items = $_POST['items'] ?? [];
                // Remember package for back-navigation
                $pkgId = (int)($_POST['package_id'] ?? 0);
                $cart = [];
                foreach ((array)$items as $it) {
                    $pid = (int)($it['price_id'] ?? 0);
                    $qty = max(0, (int)($it['qty'] ?? 0));
                    if ($pid > 0 && $qty > 0) {
                        $cart[] = [
                            'price_id' => $pid,
                            'qty' => $qty,
                            // Keep a client subtotal for display; server will verify later
                            'subtotal' => (float)($it['subtotal'] ?? 0),
                        ];
                    }
                }
                $_SESSION['activity_cart'] = $cart;
                if ($pkgId > 0) { $_SESSION['activity_package_id'] = $pkgId; }
                // Store prelim guest date/time and transport to prefill form
                $pre = $_SESSION['activity_guest'] ?? [];
                $pre['date'] = trim((string)($_POST['booking_date'] ?? ($pre['date'] ?? '')));
                $pre['show_time'] = trim((string)($_POST['show_time'] ?? ($pre['show_time'] ?? '')));
                $pre['transport']['option_id'] = (int)($_POST['transport_option_id'] ?? ($pre['transport']['option_id'] ?? 0));
                $_SESSION['activity_guest'] = $pre;
                $this->redirect('/b2b/agent/activity/checkout');
                return;
            }
        }

        $csrf = AgentGuard::generateCsrfToken('activity_checkout');
        $guest = $_SESSION['activity_guest'] ?? [];
        $cart = $_SESSION['activity_cart'] ?? [];
        $packageId = (int)($_SESSION['activity_package_id'] ?? 0);
        $displayTotal = 0.0; try { foreach ((array)$cart as $it) { $displayTotal += (float)($it['subtotal'] ?? 0); } } catch (\Throwable $_) {}
        // Resolve line items for summary (best-effort, tolerate schema diffs)
        $items = [];
        try {
            if (!empty($cart)) {
                $ids = array_values(array_unique(array_map(fn($x)=> (int)($x['price_id'] ?? 0), $cart)));
                $ids = array_filter($ids, fn($v)=> $v>0);
                if ($ids) {
                    $ph = implode(',', array_fill(0, count($ids), '?'));
                    $rows = [];
                    try {
                        $st = $this->pdo->prepare('SELECT id, agent_price, pax_type, price_type, min_quantity, variant_id FROM vendor_package_prices WHERE id IN (' . $ph . ')');
                        $st->execute($ids);
                        $rows = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
                    } catch (\Throwable $_) {
                        // fallback minimal
                        $st = $this->pdo->prepare('SELECT id, agent_price FROM vendor_package_prices WHERE id IN (' . $ph . ')');
                        $st->execute($ids);
                        $rows = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
                    }
                    $byId = [];
                    $variantIds = [];
                    foreach ($rows as $r) {
                        $rid = (int)($r['id'] ?? 0);
                        $byId[$rid] = $r;
                        if (!empty($r['variant_id'])) { $variantIds[] = (int)$r['variant_id']; }
                    }
                    $variantNames = [];
                    $variantIds = array_values(array_unique(array_filter($variantIds)));
                    if ($variantIds) {
                        try {
                            $phv = implode(',', array_fill(0, count($variantIds), '?'));
                            $sv = $this->pdo->prepare('SELECT id, name FROM vendor_package_variants WHERE id IN (' . $phv . ')');
                            $sv->execute($variantIds);
                            foreach ($sv->fetchAll(\PDO::FETCH_ASSOC) ?: [] as $vr) { $variantNames[(int)$vr['id']] = (string)($vr['name'] ?? ''); }
                        } catch (\Throwable $_) { /* ignore */ }
                    }
                    foreach ($cart as $it) {
                        $pid = (int)($it['price_id'] ?? 0); $qty = max(0,(int)($it['qty'] ?? 0)); if ($pid<=0 || $qty<=0) continue;
                        $pr = $byId[$pid] ?? [];
                        $unit = isset($pr['agent_price']) ? (float)$pr['agent_price'] : (float)($it['subtotal'] ?? 0) / max(1,$qty);
                        $vname = '';
                        $vid = (int)($pr['variant_id'] ?? 0);
                        if ($vid && isset($variantNames[$vid])) { $vname = $variantNames[$vid]; }
                        $pax = (string)($pr['pax_type'] ?? ''); if ($pax==='') $pax = 'ticket';
                        $label = trim(($vname!=='' ? ($vname . ' • ') : '') . ucfirst($pax));
                        $sub = $unit * $qty;
                        $items[] = [ 'label' => $label, 'qty' => $qty, 'unit' => $unit, 'subtotal' => $sub ];
                    }
                    // Optional transport line
                    $transportId = (int)($guest['transport']['option_id'] ?? 0);
                    if ($transportId > 0) {
                        try {
                            $stT = $this->pdo->prepare('SELECT vehicle_type, price, is_active FROM vendor_package_transport_options WHERE id = ?');
                            $stT->execute([$transportId]);
                            if ($rowT = $stT->fetch(\PDO::FETCH_ASSOC)) {
                                if ((int)($rowT['is_active'] ?? 1) === 1) {
                                    $unit = (float)($rowT['price'] ?? 0);
                                    $label = 'Transport' . (!empty($rowT['vehicle_type']) ? (' • ' . (string)$rowT['vehicle_type']) : '');
                                    $items[] = [ 'label'=>$label, 'qty'=>1, 'unit'=>$unit, 'subtotal'=>$unit ];
                                }
                            }
                        } catch (\Throwable $_) { /* ignore */ }
                    }
                    if ($items) { $displayTotal = array_reduce($items, fn($c,$x)=> $c + (float)$x['subtotal'], 0.0); }
                }
            }
        } catch (\Throwable $_) { /* ignore and keep minimal */ }
        // Fetch package meta similar to Taxi UI (best-effort)
        $package = ['name' => '', 'vendor_name' => ''];
        if ($packageId > 0) {
            try {
                // Package name
                $sp = $this->pdo->prepare('SELECT name, vendor_id FROM vendor_packages WHERE id = ?');
                $sp->execute([$packageId]);
                if ($prow = $sp->fetch(\PDO::FETCH_ASSOC)) {
                    $package['name'] = (string)($prow['name'] ?? '');
                    $vId = isset($prow['vendor_id']) ? (int)$prow['vendor_id'] : 0;
                    if ($vId > 0) {
                        try {
                            $sv = $this->pdo->prepare('SELECT name FROM vendors WHERE id = ?');
                            $sv->execute([$vId]);
                            $package['vendor_name'] = (string)($sv->fetchColumn() ?: '');
                        } catch (\Throwable $_) { /* ignore */ }
                    }
                }
            } catch (\Throwable $_) { /* ignore */ }
        }

        $this->view('agent/activity_checkout', [
            'title' => 'Activity Checkout',
            'csrf' => $csrf,
            'guest' => $guest,
            'cart' => $cart,
            'display_total' => $displayTotal,
            'items' => $items,
            'package_id' => $packageId,
            'package' => $package,
        ]);
    }

    // GET /activity/verify?t=
    // Public verification endpoint similar to TaxiController@verify
    public function verify(): void
    {
        $t = trim((string)($_GET['t'] ?? ''));
        if ($t === '' || strpos($t, '.') === false) {
            http_response_code(400);
            echo 'Invalid verification token';
            return;
        }
        [$payload, $sigB64] = explode('.', $t, 2);
        $code = base64_decode(strtr($payload, '-_', '+/')); // raw code used for signing
        if ($code === false || $code === '') {
            http_response_code(400);
            echo 'Invalid token payload';
            return;
        }
        $secret = (string)($_ENV['VERIFY_SECRET'] ?? ($_ENV['APP_KEY'] ?? 'dev-secret-change'));
        $calc = hash_hmac('sha256', $code, $secret, true);
        $calcB64 = rtrim(strtr(base64_encode($calc), '+/', '-_'), '=');
        if (!hash_equals($calcB64, $sigB64)) {
            http_response_code(403);
            echo 'Verification failed';
            return;
        }

        // Lookup booking by booking_code
        $booking = null; $result = null; $pkgName = '';
        try {
            $st = $this->pdo->prepare('SELECT * FROM activity_bookings WHERE booking_code = :code LIMIT 1');
            $st->execute([':code' => $code]);
            $booking = $st->fetch(\PDO::FETCH_ASSOC) ?: null;
            if ($booking) {
                // Try to resolve package name
                try {
                    $pk = (int)($booking['package_id'] ?? 0);
                    if ($pk > 0) {
                        $sp = $this->pdo->prepare('SELECT name FROM vendor_packages WHERE id = ?');
                        $sp->execute([$pk]);
                        $pkgName = (string)($sp->fetchColumn() ?: '');
                    }
                } catch (\Throwable $_) { /* ignore */ }

                // Build result map similar to Taxi verify
                $result = [
                    'booking_code' => (string)$booking['booking_code'],
                    'status' => (string)($booking['status'] ?? ''),
                    'payment_status' => (string)($booking['payment_status'] ?? ''),
                    'activity_date' => (string)($booking['booking_date'] ?? ''),
                    'show_time' => (string)($booking['show_time'] ?? ''),
                    'customer_name' => (string)($booking['lead_name'] ?? ''),
                    'package_name' => $pkgName,
                    'amount_total' => (float)($booking['amount_total'] ?? 0),
                ];
            }
        } catch (\Throwable $_) { $booking = null; }

        // Render public verification page (like Taxi)
        $this->view('agent/activity_verify', [
            'title' => 'Activity Booking Verification',
            'result' => $result,
            'code' => $code,
            'token' => $t,
        ]);
        return;
    }

    // GET /b2b/agent/activity/payment
    public function payment(): void
    {
        AgentGuard::requireLogin();
        $agent = $_SESSION['agent'] ?? null;
        if (!$agent || empty($agent['id'])) { $this->redirect('/b2b/agent/login'); return; }

        $guest = $_SESSION['activity_guest'] ?? null;
        $cart = $_SESSION['activity_cart'] ?? [];
        $packageId = (int)($_SESSION['activity_package_id'] ?? 0);
        // Strict derivation like pay(): sum agent_price*qty + optional transport
        $amount = 0.0; $currency = 'THB';
        $paxTotal = 0; $summaryTitle = 'Activity Booking';
        try {
            if (is_array($cart) && !empty($cart)) {
                $priceIdQty = [];
                foreach ($cart as $it) {
                    $pid = (int)($it['price_id'] ?? 0);
                    $qty = max(0, (int)($it['qty'] ?? 0));
                    $paxTotal += $qty;
                    if ($pid > 0 && $qty > 0) { $priceIdQty[$pid] = ($priceIdQty[$pid] ?? 0) + $qty; }
                }
                if (!empty($priceIdQty)) {
                    $ids = array_keys($priceIdQty);
                    $ph = implode(',', array_fill(0, count($ids), '?'));
                    $stmt = $this->pdo->prepare('SELECT id, agent_price, active FROM vendor_package_prices WHERE id IN (' . $ph . ')');
                    $stmt->execute($ids);
                    $map = [];
                    foreach ($stmt->fetchAll(\PDO::FETCH_ASSOC) ?: [] as $r) { $map[(int)$r['id']] = $r; }
                    foreach ($priceIdQty as $pid => $qty) {
                        if (!isset($map[$pid])) { continue; }
                        $pr = (float)($map[$pid]['agent_price'] ?? 0.0);
                        if ($pr > 0) { $amount += $pr * $qty; }
                    }
                }
                $transportId = (int)($guest['transport']['option_id'] ?? 0);
                if ($transportId > 0) {
                    try {
                        $stT = $this->pdo->prepare('SELECT price, is_active FROM vendor_package_transport_options WHERE id = ?');
                        $stT->execute([$transportId]);
                        if ($rowT = $stT->fetch(\PDO::FETCH_ASSOC)) {
                            if ((int)($rowT['is_active'] ?? 1) === 1) { $amount += (float)($rowT['price'] ?? 0.0); }
                        }
                    } catch (\Throwable $_) { /* ignore */ }
                }
            }
            // Try to resolve activity/package title
            if ($packageId > 0) {
                try {
                    $sp = $this->pdo->prepare('SELECT name FROM vendor_packages WHERE id = ?');
                    $sp->execute([$packageId]);
                    $summaryTitle = (string)($sp->fetchColumn() ?: $summaryTitle);
                } catch (\Throwable $_) { /* ignore */ }
            }
        } catch (\Throwable $_) { $amount = 0.0; }

        if ($amount <= 0) { $this->redirect('/b2b/agent/activity/checkout?err=amount'); return; }
        $csrfPay = AgentGuard::generateCsrfToken('activity_pay');
        // Booking code preview (issued after payment); show a friendly preview
        $bookingPreview = 'ACT-' . date('Ymd') . '-' . substr((string)time(), -2);

        // Fetch wallet balance to display like Taxi
        $walletBalance = null; try {
            $svc = new \App\Services\WalletService($this->pdo);
            $wid = $svc->getOrCreateWallet((int)$agent['id']);
            $stb = $this->pdo->prepare('SELECT balance FROM wallets WHERE id = :id');
            $stb->execute([':id' => $wid]);
            $walletBalance = (float)($stb->fetchColumn() ?: 0.0);
        } catch (\Throwable $_) { $walletBalance = null; }

        $this->view('agent/activity_payment', [
            'title' => 'Activity Payment',
            'guest' => $guest,
            'amount' => $amount,
            'currency' => $currency,
            'csrf_pay' => $csrfPay,
            'pax_total' => $paxTotal,
            'summary_title' => $summaryTitle,
            'booking_preview' => $bookingPreview,
            'wallet_balance' => $walletBalance,
        ]);
    }

    // POST /b2b/agent/activity/pay
    public function pay(): void
    {
        AgentGuard::requireLogin();
        $csrf = (string)($_POST['csrf_token'] ?? '');
        if (!AgentGuard::validateCsrf('activity_pay', $csrf)) { $this->redirect('/b2b/agent/activity/payment?err=csrf'); return; }

        $agent = $_SESSION['agent'] ?? null;
        if (!$agent || empty($agent['id'])) { $this->redirect('/b2b/agent/login'); return; }

        $method = (string)($_POST['method'] ?? 'wallet');
        $guest = $_SESSION['activity_guest'] ?? [];
        $cart = $_SESSION['activity_cart'] ?? [];
        $amount = 0.0; $currency = 'THB';
        // Strict server-side pricing: sum agent_price * qty from DB
        try {
            if (!is_array($cart) || empty($cart)) { $this->redirect('/b2b/agent/activity/payment?err=amount'); return; }
            // Collect price ids
            $priceIdQty = [];
            foreach ($cart as $it) {
                $pid = (int)($it['price_id'] ?? 0);
                $qty = max(0, (int)($it['qty'] ?? 0));
                if ($pid > 0 && $qty > 0) { $priceIdQty[$pid] = ($priceIdQty[$pid] ?? 0) + $qty; }
            }
            if (empty($priceIdQty)) { $this->redirect('/b2b/agent/activity/payment?err=amount'); return; }
            $ids = array_keys($priceIdQty);
            $ph = implode(',', array_fill(0, count($ids), '?'));
            $stmt = $this->pdo->prepare('SELECT id, agent_price, active FROM vendor_package_prices WHERE id IN (' . $ph . ')');
            $stmt->execute($ids);
            $map = [];
            foreach ($stmt->fetchAll(\PDO::FETCH_ASSOC) ?: [] as $r) { $map[(int)$r['id']] = $r; }
            foreach ($priceIdQty as $pid => $qty) {
                if (!isset($map[$pid])) { $this->redirect('/b2b/agent/activity/payment?err=verify_mismatch'); return; }
                $pr = (float)($map[$pid]['agent_price'] ?? 0.0);
                if ($pr <= 0) { $this->redirect('/b2b/agent/activity/payment?err=verify_mismatch'); return; }
                $amount += $pr * $qty;
            }
            // Optional transport price
            $transportId = (int)($guest['transport']['option_id'] ?? 0);
            if ($transportId > 0) {
                try {
                    $stT = $this->pdo->prepare('SELECT price, is_active FROM vendor_package_transport_options WHERE id = ?');
                    $stT->execute([$transportId]);
                    if ($rowT = $stT->fetch(\PDO::FETCH_ASSOC)) {
                        if ((int)($rowT['is_active'] ?? 1) === 1) { $amount += (float)($rowT['price'] ?? 0.0); }
                    }
                } catch (\Throwable $_) { /* tolerate absence */ }
            }
        } catch (\Throwable $_) { $this->activityLog('pay:error_verify_mismatch'); $this->redirect('/b2b/agent/activity/payment?err=verify_mismatch'); return; }
        if ($amount <= 0) { $this->activityLog('pay:error_amount_zero'); $this->redirect('/b2b/agent/activity/payment?err=amount'); return; }

        // Log the start of payment processing
        $this->activityLog('pay:start', [
            'agent_id' => (int)$agent['id'],
            'method' => $method,
            'amount' => $amount,
            'currency' => $currency,
            'cart_items' => is_array($cart) ? count($cart) : 0,
        ]);

        if ($method === 'wallet') {
            // PIN validation & lockout (basic)
            $pin = trim((string)($_POST['pin'] ?? ''));
            if (!preg_match('/^\d{4,6}$/', $pin)) { $this->redirect('/b2b/agent/activity/payment?err=pin'); return; }
            try {
                $qu = $this->pdo->prepare('SELECT tx_pin_hash, tx_pin_locked_until, tx_pin_attempts FROM users WHERE id = :id');
                $qu->execute([':id' => (int)$agent['id']]);
                $u = $qu->fetch(\PDO::FETCH_ASSOC) ?: null;
                if (!$u || empty($u['tx_pin_hash'])) { $this->redirect('/b2b/agent/activity/payment?err=pin_not_set'); return; }
                $lockedUntil = (string)($u['tx_pin_locked_until'] ?? '');
                if ($lockedUntil !== '' && strtotime($lockedUntil) > time()) { $this->redirect('/b2b/agent/activity/payment?err=wallet_locked'); return; }
                $okPin = password_verify($pin, (string)$u['tx_pin_hash']);
                if (!$okPin) {
                    $attempts = (int)($u['tx_pin_attempts'] ?? 0) + 1;
                    if ($attempts >= 5) {
                        $this->pdo->prepare('UPDATE users SET tx_pin_attempts = 5, tx_pin_locked_until = DATE_ADD(NOW(), INTERVAL 24 HOUR) WHERE id = :id')->execute([':id'=>(int)$agent['id']]);
                        $this->redirect('/b2b/agent/activity/payment?err=pin_locked'); return;
                    } else {
                        $this->pdo->prepare('UPDATE users SET tx_pin_attempts = :a WHERE id = :id')->execute([':a'=>$attempts, ':id'=>(int)$agent['id']]);
                        $left = max(0, 5 - $attempts);
                        $this->redirect('/b2b/agent/activity/payment?err=pin_invalid&left='.$left); return;
                    }
                }
                // Reset attempts on success
                $this->pdo->prepare('UPDATE users SET tx_pin_attempts = 0, tx_pin_locked_until = NULL WHERE id = :id')->execute([':id'=>(int)$agent['id']]);
            } catch (\Throwable $_) {
                // Soft-pass for environments where PIN columns may not exist or query fails
                // Proceed without blocking on pin_check to avoid poor UX
            }

            try {
                $svc = new \App\Services\WalletService($this->pdo);
                $agentIdI = (int)$agent['id'];
                $walletId = $svc->getOrCreateWallet($agentIdI);
                // Step 1: Wallet balance check (insufficient funds guard)
                try {
                    $stBal = $this->pdo->prepare('SELECT balance FROM wallets WHERE id = :id');
                    $stBal->execute([':id' => $walletId]);
                    $balance = (float)($stBal->fetchColumn() ?: 0.0);
                    if ($balance < $amount) {
                        $this->activityLog('pay:error_insufficient_funds', ['wallet_id'=>$walletId, 'balance'=>$balance, 'amount'=>$amount]);
                        $this->redirect('/b2b/agent/activity/payment?err=insufficient_funds'); return;
                    }
                } catch (\Throwable $_) { /* if balance not available, proceed but log */ $this->activityLog('pay:balance_check_skipped'); }

                // Create pending debit and approve (reserve then commit)
                $meta = [
                    'flow' => 'checkout',
                    'booking_type' => 'activity',
                    'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
                    'at' => date('c'),
                ];
                $ledgerId = $svc->ledgerEntry($walletId, 'debit', (float)$amount, 'manual', 'pending', $meta);
                $ok = $svc->approveLedger($ledgerId);
                if (!$ok) { $this->redirect('/b2b/agent/activity/payment?err=ledger'); return; }
                
                // Save booking to database before processing payment
                try {
                    // Diagnostics similar to Taxi logging
                    $cartCount = is_array($cart) ? count($cart) : 0;
                    $leadName = (string)($guest['lead']['name'] ?? '');
                    $leadPhone = (string)($guest['lead']['phone'] ?? '');
                    $leadEmail = (string)($guest['lead']['email'] ?? '');
                    $bDate = (string)($guest['date'] ?? '');
                    $sTime = (string)($guest['show_time'] ?? '');
                    error_log(sprintf('ACT pay: begin save booking | agent_id=%d amount=%.2f currency=%s cart_items=%d date=%s time=%s lead=%s/%s/%s',
                        (int)$agent['id'], (float)$amount, (string)$currency, (int)$cartCount, $bDate, $sTime, $leadName, $leadPhone, $leadEmail));

                    $this->activityLog('pay:save_begin', [
                        'agent_id' => (int)$agent['id'],
                        'amount' => $amount,
                        'currency' => $currency,
                        'cart_items' => $cartCount,
                        'date' => $bDate,
                        'time' => $sTime,
                    ]);
                    $bookingId = $this->saveBookingToDatabase($agent['id'], $guest, $cart, $amount, $currency);
                    if (!$bookingId) {
                        throw new \Exception('Failed to save booking (NULL ID)');
                    }
                    error_log('ACT pay: booking saved id=' . (int)$bookingId);
                    $this->activityLog('pay:saved', [ 'booking_id' => (int)$bookingId ]);
                    
                    // Get booking code from database
                    $stmt = $this->pdo->prepare('SELECT booking_code FROM activity_bookings WHERE id = ?');
                    $stmt->execute([$bookingId]);
                    $bookingCode = (string)$stmt->fetchColumn();
                    error_log('ACT pay: fetched booking_code=' . $bookingCode);
                    $this->activityLog('pay:code_fetched', [ 'booking_id' => (int)$bookingId, 'booking_code' => $bookingCode ]);
                    
                    if ($bookingCode === '' || $bookingCode === null) {
                        throw new \Exception('Failed to retrieve booking code');
                    }
                    
                    // Update ledger meta with booking reference (schema does not have reference_id/reference_type columns)
                    try {
                        $stMeta = $this->pdo->prepare('SELECT meta FROM wallet_ledger WHERE id = ?');
                        $stMeta->execute([$ledgerId]);
                        $metaJson = (string)($stMeta->fetchColumn() ?: '');
                        $metaArr = [];
                        if ($metaJson !== '') { $metaArr = json_decode($metaJson, true) ?: []; }
                        $metaArr['booking_reference'] = [
                            'id' => (int)$bookingId,
                            'type' => 'activity_booking',
                            'code' => (string)$bookingCode,
                        ];
                        $newMeta = json_encode($metaArr, JSON_UNESCAPED_UNICODE);
                        $upMeta = $this->pdo->prepare('UPDATE wallet_ledger SET meta = :m WHERE id = :id');
                        $upMeta->execute([':m' => $newMeta, ':id' => $ledgerId]);
                        error_log('ACT pay: wallet_ledger meta updated ledger_id=' . (int)$ledgerId);
                        $this->activityLog('pay:ledger_meta_updated', [ 'ledger_id' => (int)$ledgerId, 'booking_id' => (int)$bookingId ]);
                    } catch (\Throwable $eMeta) {
                        error_log('ACT pay: wallet_ledger meta update failed: ' . $eMeta->getMessage());
                        $this->activityLog('pay:ledger_meta_update_failed', [ 'error' => $eMeta->getMessage() ]);
                    }
                    
                    // Link ledger meta already done above; now record wallet_transactions entry similar to Taxi
                    try {
                        // Get balances to populate before/after (approximate)
                        $stWB = $this->pdo->prepare('SELECT balance FROM wallets WHERE id = :id');
                        $stWB->execute([':id' => $walletId]);
                        $balanceAfter = (float)($stWB->fetchColumn() ?: 0.0);
                        $balanceBefore = $balanceAfter + (float)$amount; // debit applied already
                        $metaTx = [
                            'flow' => 'activity_payment',
                            'booking_id' => (int)$bookingId,
                            'ledger_id' => (int)$ledgerId,
                            'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
                            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null,
                        ];
                        $this->pdo->prepare('INSERT INTO wallet_transactions (user_id, transaction_type, category, amount, balance_before, balance_after, currency, description, reference_id, reference_type, metadata, status, created_at) VALUES (:u, "debit", :cat, :amt, :bb, :ba, :cur, :desc, :rid, :rtype, :meta, "completed", NOW())')
                            ->execute([
                                ':u' => $agentIdI,
                                // DB enum wallet_transactions.category does not include 'activity_payment'. Use 'booking_payment'.
                                ':cat' => 'booking_payment',
                                ':amt' => $amount,
                                ':bb' => $balanceBefore,
                                ':ba' => $balanceAfter,
                                ':cur' => $currency,
                                ':desc' => 'Activity Booking Payment',
                                ':rid' => (string)$bookingId,
                                // DB enum wallet_transactions.reference_type does not include 'activity_booking'. Use generic 'order'.
                                ':rtype' => 'order',
                                ':meta' => json_encode($metaTx, JSON_UNESCAPED_UNICODE),
                            ]);
                        $this->activityLog('pay:wallet_transaction_recorded', ['booking_id'=>$bookingId]);
                    } catch (\Throwable $eWT) {
                        error_log('wallet_transactions insert failed: ' . $eWT->getMessage());
                        $this->activityLog('pay:wallet_tx_insert_failed', ['error'=>$eWT->getMessage()]);
                    }

                    // Generate secure token for voucher access
                    $token = $this->generateSecureToken($bookingId, $bookingCode);
                    
                    // Clear session data
                    unset($_SESSION['activity_guest'], $_SESSION['activity_cart'], $_SESSION['activity_booking_id'], $_SESSION['activity_booking_code']);
                    
                    // Redirect to success interstitial page with links to voucher and booking details
                    $redirectUrl = '/b2b/agent/activity/success?id=' . $bookingId . '&code=' . urlencode($bookingCode) . '&token=' . $token;
                    error_log('ACT pay: redirect success ' . $redirectUrl);
                    $this->activityLog('pay:redirect_success', [ 'url' => $redirectUrl ]);
                    $this->redirect($redirectUrl);
                    return;
                    
                } catch (\Throwable $e) {
                    error_log('ACT pay: booking save exception: ' . $e->getMessage());
                    if (method_exists($e, 'getTraceAsString')) { error_log('ACT pay: stack ' . $e->getTraceAsString()); }
                    // Attempt to refund the debit to wallet to avoid funds stuck
                    try {
                        $svc->creditApproved($agentIdI, (float)$amount, 'manual', ['flow'=>'activity_refund_on_failure','reason'=>$e->getMessage()]);
                        $this->activityLog('pay:refund_on_failure', ['amount'=>$amount]);
                    } catch (\Throwable $_r) { $this->activityLog('pay:refund_on_failure_failed'); }
                    // Surface a consistent error code to UI
                    $this->activityLog('pay:exception_booking_failed', [ 'error' => $e->getMessage() ]);
                    $this->redirect('/b2b/agent/activity/payment?err=booking_failed');
                    return;
                }
            } catch (\Throwable $e) {
                $this->redirect('/b2b/agent/activity/payment?err=exception'); return;
            }
        } else {
            // Stripe gateway: create checkout session and redirect
            try {
                $pgCfg = @require __DIR__ . '/../../config/payment_gateways.php';
                if (empty($pgCfg['stripe']['enabled'])) { $this->redirect('/b2b/agent/activity/payment?err=gateway_not_configured'); return; }
                $secret = (string)($pgCfg['stripe']['secret_key'] ?? '');
                if ($secret === '') { $this->redirect('/b2b/agent/activity/payment?err=gateway_not_configured'); return; }
                \Stripe\Stripe::setApiKey($secret);
                $appUrl = (string)($_ENV['APP_URL'] ?? 'http://127.0.0.1');
                $successUrl = rtrim($appUrl, '/') . '/b2b/agent/activity/gateway-success?sid={CHECKOUT_SESSION_ID}';
                $cancelUrl = rtrim($appUrl, '/') . '/b2b/agent/activity/payment?err=cancelled';
                $amountMinor = (int)round($amount * 100);
                $session = \Stripe\Checkout\Session::create([
                    'mode' => 'payment',
                    'payment_method_types' => ['card'],
                    'line_items' => [[
                        'price_data' => [
                            'currency' => strtolower($currency),
                            'product_data' => [ 'name' => 'Activity booking' ],
                            'unit_amount' => $amountMinor,
                        ],
                        'quantity' => 1,
                    ]],
                    'success_url' => $successUrl,
                    'cancel_url' => $cancelUrl,
                    'metadata' => [ 'user_id' => (string)((int)$agent['id']), 'module' => 'activity' ],
                ]);
                header('Location: ' . $session->url);
                return;
            } catch (\Throwable $e) {
                $this->redirect('/b2b/agent/activity/payment?err=exception'); return;
            }
        }

        // Persist booking (best-effort, tolerate schema differences)
        $bookingId = null; $bookingCode = 'ACT-' . date('Ymd') . '-' . substr((string)time(), -5);
        try {
            $lead = (array)($guest['lead'] ?? []);
            $date = (string)($guest['date'] ?? '');
            $showTime = (string)($guest['show_time'] ?? '');
            $pickup = (array)($guest['transport'] ?? []);
            // Resolve pax and activity title
            $paxTotal = 0; foreach (($cart ?? []) as $it) { $q = (int)($it['qty'] ?? 0); if ($q>0) { $paxTotal += $q; } }
            $summaryTitle = 'Activity Booking';
            $packageId = (int)($_SESSION['activity_package_id'] ?? 0);
            $vendorId = null;
            if ($packageId > 0) {
                try {
                    $sp = $this->pdo->prepare('SELECT name, vendor_id FROM vendor_packages WHERE id = ?');
                    $sp->execute([$packageId]);
                    if ($row = $sp->fetch(\PDO::FETCH_ASSOC)) {
                        $name = (string)($row['name'] ?? ''); if ($name !== '') { $summaryTitle = $name; }
                        $vendorId = isset($row['vendor_id']) ? (int)$row['vendor_id'] : null;
                    }
                } catch (\Throwable $_) { /* ignore */ }
            }
            // Build items_json array from resolved items
            // Build items and pax count from current cart to avoid NULL qty_total
            $paxTotal = 0;
            $itemsArr = [];
            try {
                $cartNow = $_SESSION['activity_cart'] ?? [];
                // map price_id -> qty
                $priceIdQty = [];
                foreach ((array)$cartNow as $it) {
                    $pid = (int)($it['price_id'] ?? 0);
                    $qty = max(0, (int)($it['qty'] ?? 0));
                    if ($pid > 0 && $qty > 0) { $priceIdQty[$pid] = ($priceIdQty[$pid] ?? 0) + $qty; $paxTotal += $qty; }
                }
                if (!empty($priceIdQty)) {
                    $ids = array_keys($priceIdQty);
                    $ph = implode(',', array_fill(0, count($ids), '?'));
                    $st = $this->pdo->prepare('SELECT id, agent_price, variant_id FROM vendor_package_prices WHERE id IN (' . $ph . ')');
                    $st->execute($ids);
                    $map = [];
                    foreach ($st->fetchAll(\PDO::FETCH_ASSOC) ?: [] as $r) { $map[(int)$r['id']] = $r; }
                    foreach ($priceIdQty as $pid=>$qty) {
                        $unit = isset($map[$pid]['agent_price']) ? (float)$map[$pid]['agent_price'] : 0.0;
                        $vidLocal = isset($map[$pid]['variant_id']) ? (int)$map[$pid]['variant_id'] : null;
                        $itemsArr[] = ['variant_id'=>$vidLocal,'price_id'=>$pid,'qty'=>$qty,'unit'=>$unit,'line_total'=>round($unit*$qty,2)];
                    }
                }
            } catch (\Throwable $_) { /* best-effort */ }
            $itemsJson = json_encode($itemsArr, JSON_UNESCAPED_UNICODE);
            $userAgent = (string)($_SERVER['HTTP_USER_AGENT'] ?? '');
            $ipAddr = (string)($_SERVER['REMOTE_ADDR'] ?? '');
            $txnId = 'ledger:' . ((int)($_SESSION['activity_payment']['ledger_id'] ?? 0));

            $bookingId = (int)($_SESSION['activity_booking_id'] ?? 0);
            if ($bookingId > 0) {
                // Update existing pending booking to paid
                $upd = $this->pdo->prepare("UPDATE activity_bookings SET status='confirmed', payment_method='wallet', gateway_name='wallet', payment_status='paid', payment_txn_id=:tx, items_json=:items, qty_total=:qty, amount_total=:amt, currency=:cur, paid_at=NOW(), updated_at=NOW() WHERE id=:id AND agent_id=:uid");
                $upd->execute([':tx'=>$txnId, ':items'=>$itemsJson, ':qty'=>max(1,(int)$paxTotal), ':amt'=>(float)$amount, ':cur'=>$currency, ':id'=>$bookingId, ':uid'=>(int)$agent['id']]);
                $this->activityLog('pay:booking_mark_paid', ['booking_id'=>$bookingId]);
            } else {
                // Fallback: insert paid booking
                $insSql = "INSERT INTO activity_bookings (booking_code, agent_id, created_by, updated_by, vendor_id, package_id, booking_date, show_time, items_json, qty_total, amount_total, currency, channel, status, payment_method, gateway_name, payment_status, payment_txn_id, ip_address, user_agent, paid_at, lead_name, lead_phone, lead_email, pickup_required, pickup_hotel_name, pickup_address, pickup_city, pickup_notes, created_at, updated_at)
                            VALUES (:code, :uid, :uid, :uid, :vid, :pkg, :bdate, :stime, :items, :qty, :amt, :cur, 'portal', 'confirmed', 'wallet', 'wallet', 'paid', :tx, :ip, :ua, NOW(), :lname, :lphone, :lemail, :p_req, :p_hotel, :p_addr, :p_city, :p_notes, NOW(), NOW())";
                $ins = $this->pdo->prepare($insSql);
                $ins->execute([
                    ':code'=>$bookingCode,
                    ':uid'=>(int)$agent['id'],
                    ':vid'=>$vendorId,
                    ':pkg'=>$packageId,
                    ':bdate'=>$date,
                    ':stime'=>$showTime ?: null,
                    ':items'=>$itemsJson,
                    ':qty'=>$paxTotal,
                    ':amt'=>(float)$amount,
                    ':cur'=>$currency,
                    ':tx'=>$txnId,
                    ':ip'=>$ipAddr,
                    ':ua'=>$userAgent,
                    ':lname'=>(string)($lead['name'] ?? ''),
                    ':lphone'=>(string)($lead['phone'] ?? ''),
                    ':lemail'=>(string)($lead['email'] ?? ''),
                    ':p_req'=> (int) (!empty($pickup['option_id'])),
                    ':p_hotel'=>(string)($pickup['hotel_name'] ?? ''),
                    ':p_addr'=>(string)($pickup['hotel_address'] ?? ''),
                    ':p_city'=>'',
                    ':p_notes'=>(string)($pickup['pickup_notes'] ?? ($pickup['notes'] ?? '')),
                ]);
                $bookingId = (int)$this->pdo->lastInsertId();
                $_SESSION['activity_booking_id'] = $bookingId;
                $_SESSION['activity_booking_code'] = $bookingCode;
                // Insert created event for fallback insert
                try { $ev=$this->pdo->prepare('INSERT INTO activity_booking_events (booking_id, event_type, note, data_json, created_at) VALUES (:bid,\'created\',\'Activity booking created\',:d,NOW())'); $ev->execute([':bid'=>$bookingId, ':d'=>json_encode(['qty_total'=>$paxTotal,'amount'=>$amount])]); } catch (\Throwable $_) {}
            }

            // Event: payment_captured
            try {
                $ev = $this->pdo->prepare('INSERT INTO activity_booking_events (booking_id, event_type, note, data_json, created_at) VALUES (:bid,\'payment_captured\',\'Payment captured via wallet\',:data,NOW())');
                $ev->execute([':bid'=>$bookingId, ':data'=>json_encode(['method'=>$method,'ledger_id'=>($_SESSION['activity_payment']['ledger_id'] ?? null),'amount'=>$amount,'currency'=>$currency])]);
            } catch (\Throwable $_) { /* best-effort */ }
        } catch (\Throwable $_) { $bookingId = null; }

        if ($bookingId) {
            // Ensure the booking ID is in the session for the voucher page
            $_SESSION['activity_booking_id'] = $bookingId;
            if (!empty($bookingCode)) {
                $_SESSION['activity_booking_code'] = $bookingCode;
            }
            // Include code+token so voucher can validate without relying on session only
            $code = (string)($_SESSION['activity_booking_code'] ?? $bookingCode);
            $token = $code !== '' ? $this->generateSecureToken($bookingId, $code) : '';
            $url = '/b2b/agent/activity/voucher?id=' . $bookingId;
            if ($code !== '' && $token !== '') { $url .= '&code=' . urlencode($code) . '&token=' . urlencode($token); }
            $this->redirect($url);
            return;
        }
        
        // If we get here, try to redirect with the session booking ID as a fallback
        $sessionBookingId = (int)($_SESSION['activity_booking_id'] ?? 0);
        if ($sessionBookingId > 0) {
            $code = (string)($_SESSION['activity_booking_code'] ?? '');
            $token = $code !== '' ? $this->generateSecureToken($sessionBookingId, $code) : '';
            $url = '/b2b/agent/activity/voucher?id=' . $sessionBookingId;
            if ($code !== '' && $token !== '') { $url .= '&code=' . urlencode($code) . '&token=' . urlencode($token); }
            $this->redirect($url);
            return;
        }
        
        // Last resort: redirect to voucher page and let it handle the error
        $this->redirect('/b2b/agent/activity/voucher');
    }

    /**
     * Get booking from database with security checks
     */
    private function getBookingFromDatabase(int $bookingId, string $bookingCode, int $agentId): ?array 
    {
        try {
            $stmt = $this->pdo->prepare('SELECT * FROM activity_bookings WHERE id = ? AND booking_code = ? AND agent_id = ? LIMIT 1');
            $stmt->execute([$bookingId, $bookingCode, $agentId]);
            return $stmt->fetch(\PDO::FETCH_ASSOC) ?: null;
        } catch (\PDOException $e) {
            error_log('Database error in getBookingFromDatabase: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Generate a secure token for URL verification
     */
    private function generateSecureToken(int $bookingId, string $bookingCode): string
    {
        // Stable token derived from booking id and code
        $secret = getenv('APP_SECRET') ?: 'default-secret-key';
        $data = $bookingId . '|' . $bookingCode;
        return hash_hmac('sha256', $data, $secret);
    }
    
    /**
     * Verify secure token
     */
    private function verifySecureToken(string $token, int $bookingId, string $bookingCode, int $maxAge = 3600): bool
    {
        // Stable token without time component to avoid skew issues
        $secret = getenv('APP_SECRET') ?: 'default-secret-key';
        $data = $bookingId . '|' . $bookingCode;
        $expected = hash_hmac('sha256', $data, $secret);
        $ok = hash_equals($expected, $token);
        if (!$ok) {
            $this->activityLog('voucher:token_mismatch', [ 'booking_id' => $bookingId, 'code' => $bookingCode ]);
        }
        return $ok;
    }
    
    /**
     * Write activity booking flow logs to storage/logs/activity_payments/YYYY-MM-DD.log
     */
    private function activityLog(string $message, array $context = []): void
    {
        try {
            $base = __DIR__ . '/../../storage/logs/activity_payments';
            if (!is_dir($base)) { @mkdir($base, 0777, true); }
            $file = $base . '/' . date('Y-m-d') . '.log';
            $line = '[' . date('Y-m-d H:i:s') . '] ' . $message;
            if (!empty($context)) { $line .= ' ' . json_encode($context, JSON_UNESCAPED_UNICODE); }
            $line .= PHP_EOL;
            @file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
        } catch (\Throwable $_) { /* ignore */ }
    }
    
    /**
     * Minimal error page renderer for Activity flows
     */
    private function showErrorPage(string $message, int $status = 400): void
    {
        try { http_response_code($status); } catch (\Throwable $_) {}
        $this->activityLog('voucher:error_page', [ 'status' => $status, 'message' => $message ]);
        // Try to render a dedicated view if available; otherwise echo plain text
        try {
            $this->view('agent/activity_error', [
                'title' => 'Activity Error',
                'message' => $message,
                'status' => $status,
            ]);
            return;
        } catch (\Throwable $_) {
            // Fallback
        }
        echo htmlspecialchars($message);
    }
    
    /**
     * Save booking to database
     */
    private function saveBookingToDatabase(int $agentId, array $guest, array $cart, float $amount, string $currency): ?int
    {
        // Log the start of the save process
        $this->activityLog('save_booking:start', [
            'agent_id' => $agentId,
            'cart_count' => is_countable($cart) ? count($cart) : 0,
            'amount' => $amount,
            'currency' => $currency
        ]);

        // Validate cart
        if (!is_array($cart) || empty($cart)) {
            $this->activityLog('save_booking:error', ['error' => 'Empty or invalid cart']);
            return null;
        }

        // Log cart contents for debugging
        $this->activityLog('save_booking:cart_contents', ['cart' => $cart]);

        try {
            $this->pdo->beginTransaction();
            
            // Generate booking code
            $bookingCode = 'ACT-' . date('Ymd') . '-' . strtoupper(substr(md5(uniqid()), 0, 8));
            
            // Prepare booking data
            $bookingDate = !empty($guest['date']) ? date('Y-m-d', strtotime($guest['date'])) : date('Y-m-d');
            $showTime = !empty($guest['show_time']) ? $guest['show_time'] : null;
            $pickupRequired = !empty($guest['transport']['option_id']) ? 1 : 0;
            
            // Resolve package_id and vendor_id from first cart item's price_id
            $packageId = null;
            $vendorId = null;
            try {
                $firstPriceId = null;
                foreach ($cart as $ci) { $pid = (int)($ci['price_id'] ?? 0); if ($pid > 0) { $firstPriceId = $pid; break; } }
                if ($firstPriceId) {
                    $q = $this->pdo->prepare(
                        'SELECT vp.id AS package_id, vp.vendor_id AS vendor_id
                         FROM vendor_package_prices p
                         JOIN vendor_package_variants v ON p.variant_id = v.id
                         JOIN vendor_packages vp ON v.package_id = vp.id
                         WHERE p.id = ? LIMIT 1'
                    );
                    $q->execute([$firstPriceId]);
                    if ($row = $q->fetch(\PDO::FETCH_ASSOC)) {
                        $packageId = (int)$row['package_id'];
                        $vendorId = isset($row['vendor_id']) ? (int)$row['vendor_id'] : null;
                    }
                }
            } catch (\Throwable $e) {
                error_log('saveBookingToDatabase: failed to resolve package/vendor from price_id: ' . $e->getMessage());
            }
            if (!$packageId) {
                // Cannot insert without a valid package_id (NOT NULL)
                throw new \RuntimeException('package_id could not be resolved from cart');
            }
            
            // Calculate total quantity and prepare items with full details
            $totalQty = 0;
            $items = [];
            
            // First, get all price IDs from cart
            $priceIds = [];
            foreach ($cart as $item) {
                $pid = (int)($item['price_id'] ?? 0);
                $qty = (int)($item['qty'] ?? 0);
                if ($pid > 0 && $qty > 0) {
                    $priceIds[$pid] = $qty;
                    $totalQty += $qty;
                }
            }
            
            // If we have price IDs, fetch their full details
            if (empty($priceIds)) {
                $this->activityLog('save_booking:error', ['error' => 'No valid price IDs found in cart', 'cart' => $cart]);
                throw new \RuntimeException('No valid price IDs found in cart');
            }

            $placeholders = str_repeat('?,', count($priceIds) - 1) . '?';
            
            try {
                $stmt = $this->pdo->prepare("
                    SELECT 
                        vpp.id as price_id,
                        vpp.agent_price as price,
                        vpp.pax_type,
                        vpp.price_type,
                        vpp.variant_id,
                        vpv.name as variant_name,
                        vpv.package_id,
                        vp.name as package_name
                    FROM vendor_package_prices vpp
                    LEFT JOIN vendor_package_variants vpv ON vpp.variant_id = vpv.id
                    LEFT JOIN vendor_packages vp ON vpv.package_id = vp.id
                    WHERE vpp.id IN ($placeholders)
                ");
                
                $stmt->execute(array_keys($priceIds));
                $priceDetails = $stmt->fetchAll(\PDO::FETCH_ASSOC);
                
                if (empty($priceDetails)) {
                    $this->activityLog('save_booking:error', [
                        'error' => 'No price details found for the given price IDs',
                        'price_ids' => array_keys($priceIds)
                    ]);
                    throw new \RuntimeException('No price details found for the given price IDs');
                }
                
                // Log the price details for debugging
                $this->activityLog('save_booking:price_details', ['price_details' => $priceDetails]);
                
                // Map price details to items array with quantities
                foreach ($priceDetails as $detail) {
                    $pid = (int)$detail['price_id'];
                    $qty = $priceIds[$pid] ?? 0;
                    if ($qty > 0) {
                        $paxType = (string)($detail['pax_type'] ?? '');
                        $priceType = (string)($detail['price_type'] ?? '');
                        $label = $paxType;
                        if ($paxType === 'adult') { $label = 'Adult'; }
                        elseif ($paxType === 'child') { $label = 'Child'; }
                        elseif ($paxType === 'flat') { $label = 'Flat'; }
                        else { $label = strtoupper($paxType); }
                        if ($priceType === 'group_tier') { $label .= ' (Group Tier)'; }
                        $items[] = [
                            'price_id' => $pid,
                            'qty' => $qty,
                            'price' => (float)($detail['price'] ?? 0),
                            'price_name' => $label,
                            'variant_id' => !empty($detail['variant_id']) ? (int)$detail['variant_id'] : null,
                            'variant_name' => $detail['variant_name'] ?? '',
                            'package_id' => !empty($detail['package_id']) ? (int)$detail['package_id'] : null,
                            'package_name' => $detail['package_name'] ?? ''
                        ];
                    }
                }
                
                // Log the final items array for debugging
                $this->activityLog('save_booking:final_items', ['items' => $items]);
                
            } catch (\PDOException $e) {
                $this->activityLog('save_booking:database_error', [
                    'error' => $e->getMessage(),
                    'sql' => $e->getCode() . ': ' . $e->getMessage()
                ]);
                throw $e; // Re-throw to be caught by the outer try-catch
            }
            
            // Insert booking (columns aligned to schema in database/patches/b2b_travel.sql)
            $stmt = $this->pdo->prepare("
                INSERT INTO activity_bookings (
                    booking_code, agent_id, created_by, updated_by, vendor_id, package_id,
                    booking_date, show_time, items_json, qty_total, amount_total, currency,
                    channel, status, payment_method, payment_status, ip_address, user_agent,
                    lead_name, lead_phone, lead_email, pickup_required, pickup_hotel_name,
                    pickup_address, pickup_notes, created_at, updated_at
                ) VALUES (
                    :booking_code, :agent_id, :agent_id, :agent_id, :vendor_id, :package_id,
                    :booking_date, :show_time, :items_json, :qty_total, :amount_total, :currency,
                    'portal', 'pending', NULL, 'pending', :ip_address, :user_agent,
                    :lead_name, :lead_phone, :lead_email, :pickup_required, :pickup_hotel_name,
                    :pickup_address, :pickup_notes, NOW(), NOW()
                )
            ");
            
            $stmt->execute([
                ':booking_code' => $bookingCode,
                ':agent_id' => $agentId,
                ':vendor_id' => $vendorId,
                ':package_id' => $packageId,
                ':booking_date' => $bookingDate,
                ':show_time' => $showTime,
                ':items_json' => json_encode($items, JSON_UNESCAPED_UNICODE),
                ':qty_total' => $totalQty,
                ':amount_total' => $amount,
                ':currency' => $currency,
                ':ip_address' => $_SERVER['REMOTE_ADDR'] ?? '',
                ':user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                ':lead_name' => $guest['lead']['name'] ?? '',
                ':lead_phone' => $guest['lead']['phone'] ?? '',
                ':lead_email' => $guest['lead']['email'] ?? '',
                ':pickup_required' => $pickupRequired,
                ':pickup_hotel_name' => $guest['transport']['hotel_name'] ?? '',
                ':pickup_address' => $guest['transport']['hotel_address'] ?? '',
                ':pickup_notes' => $guest['transport']['pickup_notes'] ?? '',
            ]);
            
            $bookingId = (int)$this->pdo->lastInsertId();
            
            // Add booking event
            $eventStmt = $this->pdo->prepare("
                INSERT INTO activity_booking_events (
                    booking_id, event_type, note, data_json, created_at
                ) VALUES (
                    :booking_id, 'created', 'Booking created', :data_json, NOW()
                )
            ");
            
            $eventStmt->execute([
                ':booking_id' => $bookingId,
                ':data_json' => json_encode([
                    'amount' => $amount,
                    'currency' => $currency,
                    'items' => $items
                ])
            ]);
            
            $this->pdo->commit();
            
            // Log successful booking save
            $this->activityLog('save_booking:success', [
                'booking_id' => $bookingId,
                'booking_code' => $bookingCode,
                'items_count' => count($items)
            ]);
            
            return $bookingId;
            
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) {
                try {
                    $this->pdo->rollBack();
                    $this->activityLog('save_booking:transaction_rolled_back', ['error' => $e->getMessage()]);
                } catch (\Exception $rollbackEx) {
                    $this->activityLog('save_booking:rollback_failed', [
                        'original_error' => $e->getMessage(),
                        'rollback_error' => $rollbackEx->getMessage()
                    ]);
                }
            }
            
            // Log detailed error information
            $this->activityLog('save_booking:error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'context' => [
                    'agent_id' => $agentId,
                    'booking_date' => $bookingDate ?? null,
                    'show_time' => $showTime ?? null,
                    'package_id' => $packageId ?? null,
                    'vendor_id' => $vendorId ?? null,
                    'total_qty' => $totalQty ?? null,
                    'amount' => $amount,
                ],
            ]);
            return null;
        }
    }
    
    // GET /b2b/agent/activity/voucher
    public function voucher(): void
    {
        AgentGuard::requireLogin();
        
        // Get authenticated agent
        $agent = $_SESSION['agent'] ?? null;
        if (!$agent || empty($agent['id'])) { 
            $this->redirect('/b2b/agent/login');
            return;
        }
        
        // Get booking ID and code from URL parameters
        $bookingId = (int)($_GET['id'] ?? 0);
        $bookingCode = trim((string)($_GET['code'] ?? ''));
        $token = trim((string)($_GET['token'] ?? ''));
        
        // Log access attempt
        error_log(sprintf(
            'Voucher access - Agent ID: %d, Booking ID: %d, Code: %s, IP: %s',
            $agent['id'],
            $bookingId,
            $bookingCode,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown'
        ));
        
        // Validate parameters: id is required; code is optional (will be resolved)
        if ($bookingId <= 0) {
            $this->showErrorPage('Invalid booking reference. Please check the URL and try again.');
            return;
        }
        
        try {
            // Get booking from database with security check
            if ($bookingCode === '') {
                // Resolve booking_code using id and agent
                $stmt = $this->pdo->prepare('SELECT * FROM activity_bookings WHERE id = ? AND agent_id = ? LIMIT 1');
                $stmt->execute([$bookingId, (int)$agent['id']]);
                $booking = $stmt->fetch(\PDO::FETCH_ASSOC) ?: null;
                if ($booking) { $bookingCode = (string)($booking['booking_code'] ?? ''); }
            } else {
                $booking = $this->getBookingFromDatabase($bookingId, $bookingCode, $agent['id']);
            }
            if (!$booking) {
                throw new \Exception('Booking not found or access denied');
            }
            
            // Fetch package name from database if package_id exists
            if (!empty($booking['package_id'])) {
                $stmt = $this->pdo->prepare('SELECT name FROM vendor_packages WHERE id = ? LIMIT 1');
                $stmt->execute([(int)$booking['package_id']]);
                $package = $stmt->fetch(\PDO::FETCH_ASSOC);
                if ($package) {
                    $booking['package_name'] = $package['name'];
                }
            }
            
            // Verify secure token if provided
            if (!empty($token) && !$this->verifySecureToken($token, $bookingId, $bookingCode)) {
                throw new \Exception('Invalid or expired security token');
            }
            
            // Auto-reconcile: if booking is still pending but a completed wallet transaction exists, mark as paid
            try {
                $wasPending = (strtolower((string)($booking['payment_status'] ?? '')) !== 'paid');
                if ($wasPending) {
                    // Look for a completed booking_payment referencing this booking
                    $stWT = $this->pdo->prepare("SELECT id FROM wallet_transactions WHERE user_id = :uid AND category = 'booking_payment' AND status = 'completed' AND reference_id = :rid LIMIT 1");
                    $stWT->execute([':uid' => (int)$agent['id'], ':rid' => (string)$bookingId]);
                    if ($stWT->fetchColumn()) {
                        $this->pdo->prepare("UPDATE activity_bookings SET status='confirmed', payment_method=COALESCE(NULLIF(payment_method,''),'wallet'), gateway_name=COALESCE(NULLIF(gateway_name,''),'wallet'), payment_status='paid', paid_at=COALESCE(paid_at, NOW()), updated_at=NOW() WHERE id=:id AND agent_id=:uid")
                            ->execute([':id'=>$bookingId, ':uid'=>(int)$agent['id']]);
                        try {
                            $ev = $this->pdo->prepare('INSERT INTO activity_booking_events (booking_id, event_type, note, data_json, created_at) VALUES (:bid,\'payment_captured\',\'Payment reconciled via wallet\',:data,NOW())');
                            $ev->execute([':bid'=>$bookingId, ':data'=>json_encode(['method'=>'wallet','reconciled'=>true])]);
                        } catch (\Throwable $_) { /* ignore */ }
                        // Reload booking
                        $booking = $this->getBookingFromDatabase($bookingId, $bookingCode, $agent['id']);
                    }
                }
            } catch (\Throwable $_) { /* best-effort */ }

            // Log successful access
            error_log(sprintf(
                'Voucher displayed - Booking ID: %d, Status: %s',
                $bookingId,
                $booking['status']
            ));

            // Get package details
            $package = null;
            if (!empty($booking['package_id'])) {
                try {
                    $stmt = $this->pdo->prepare('SELECT * FROM activity_packages WHERE id = ?');
                    $stmt->execute([$booking['package_id']]);
                    $package = $stmt->fetch(\PDO::FETCH_ASSOC);
                } catch (\PDOException $e) {
                    error_log('Error fetching package details: ' . $e->getMessage());
                }
            }
            
            // Get booking events
            $events = [];
            try {
                $stmt = $this->pdo->prepare('SELECT * FROM activity_booking_events WHERE booking_id = ? ORDER BY created_at DESC');
                $stmt->execute([$bookingId]);
                $events = $stmt->fetchAll(\PDO::FETCH_ASSOC);
            } catch (\PDOException $e) {
                error_log('Error fetching booking events: ' . $e->getMessage());
            }
            
            // Update last viewed timestamp
            try {
                $this->pdo->prepare('UPDATE activity_bookings SET last_viewed_at = NOW() WHERE id = ?')
                    ->execute([$bookingId]);
            } catch (\PDOException $e) {
                error_log('Error updating last viewed timestamp: ' . $e->getMessage());
            }
            
            // Map DB row to boarding view-friendly keys
            $bookingView = $booking;
            $bookingView['customer_name'] = (string)($booking['lead_name'] ?? ($booking['customer_name'] ?? ''));
            $bookingView['customer_email'] = (string)($booking['lead_email'] ?? ($booking['customer_email'] ?? ''));
            $bookingView['customer_phone'] = (string)($booking['lead_phone'] ?? ($booking['customer_phone'] ?? ''));
            $bookingView['activity_date'] = (string)($booking['booking_date'] ?? ($booking['activity_date'] ?? ''));
            $bookingView['payment_status'] = (string)($booking['payment_status'] ?? (in_array($booking['status'] ?? '', ['confirmed','paid'], true) ? 'paid' : 'unpaid'));
            // Best-effort activity_name from package
            if (empty($bookingView['activity_name']) && !empty($package['name'])) { $bookingView['activity_name'] = (string)$package['name']; }

            // Fetch vendor information using existing PDO (resolve vendor_id robustly)
            $vendor = [];
            try {
                // Prefer vendor_id on booking row; fallback to vendor_packages
                $vendorId = (int)($bookingView['vendor_id'] ?? 0);
                if ($vendorId <= 0) {
                    $pkgId = (int)($bookingView['package_id'] ?? 0);
                    if ($pkgId > 0) {
                        try {
                            $spv = $this->pdo->prepare('SELECT vendor_id FROM vendor_packages WHERE id = ?');
                            $spv->execute([$pkgId]);
                            $vendorId = (int)($spv->fetchColumn() ?: 0);
                        } catch (\Throwable $_) { $vendorId = 0; }
                    }
                }
                if ($vendorId > 0) {
                    $this->activityLog('voucher:vendor_lookup', ['vendor_id' => $vendorId]);
                    $sv = $this->pdo->prepare('SELECT name, address, city, country FROM vendors WHERE id = ?');
                    $sv->execute([$vendorId]);
                    $vendor = $sv->fetch(\PDO::FETCH_ASSOC) ?: [];
                    $this->activityLog('voucher:vendor_result', ['found' => !empty($vendor), 'vendor' => array_intersect_key((array)$vendor, array_flip(['name','city','country']))]);
                }
            } catch (\Throwable $_) { $vendor = []; }

            // Prepare view data for voucher
            $viewData = [
                'title' => 'Activity Voucher',
                'booking' => array_merge($bookingView, [
                    'vendor_name' => $vendor['name'] ?? ($bookingView['vendor_name'] ?? ''),
                    'vendor_address' => $vendor['address'] ?? ($bookingView['vendor_address'] ?? ''),
                    'vendor_city' => $vendor['city'] ?? ($bookingView['vendor_city'] ?? ''),
                    'vendor_country' => $vendor['country'] ?? ($bookingView['vendor_country'] ?? ''),
                    'vendor_phone' => $vendor['calling_number'] ?? ($vendor['contact_person_phone'] ?? ($bookingView['vendor_phone'] ?? '')),
                    'vendor_email' => $vendor['contact_email'] ?? ($vendor['contact_person_email'] ?? ($bookingView['vendor_email'] ?? '')),
                    'vendor_logo' => $bookingView['vendor_logo'] ?? ''
                ]),
                'guest' => $_SESSION['activity_guest'] ?? [],
                'paid' => in_array(strtolower((string)$bookingView['payment_status']), ['paid','completed','success'], true),
                'pdo' => $this->pdo, // Pass the PDO instance to the view
            ];
            // If not paid yet, render pending view (no Booking Code or QR)
            if (!$viewData['paid']) {
                $this->view('agent/activity_voucher_pending', $viewData);
                return;
            }
            // Render the boarding voucher view (paid)
            $this->view('agent/activity_voucher_boarding', $viewData);
            return;
            
        } catch (\Exception $e) {
            error_log('Voucher error: ' . $e->getMessage());
            $this->showErrorPage($e->getMessage());
            return;
        }
        
        // If we get here, there was an error
        $this->showErrorPage('Unable to process your request. Please try again.');
    }
    

    // GET /b2b/agent/activity/gateway-success?sid=
    public function gatewaySuccess(): void
    {
        AgentGuard::requireLogin();
        $sid = trim((string)($_GET['sid'] ?? ''));
        // Fallback: legacy wallet flow uses id, code, token. If no sid, try that shape.
        if ($sid === '') {
            $bookingId = (int)($_GET['id'] ?? 0);
            $bookingCode = trim((string)($_GET['code'] ?? ''));
            $token = trim((string)($_GET['token'] ?? ''));
            if ($bookingId > 0 && $bookingCode !== '' && $token !== '') {
                // Verify token and booking belong to logged-in agent
                $agent = $_SESSION['agent'] ?? null;
                if (!$agent || empty($agent['id'])) { $this->redirect('/b2b/agent/login'); return; }
                if ($this->verifySecureToken($token, $bookingId, $bookingCode)) {
                    $bk = $this->getBookingFromDatabase($bookingId, $bookingCode, (int)$agent['id']);
                    if ($bk) {
                        // All good: send to voucher
                        $this->redirect('/b2b/agent/activity/voucher?id=' . $bookingId . '&code=' . urlencode($bookingCode) . '&token=' . urlencode($token));
                        return;
                    }
                    $this->showErrorPage('Booking not found for your account.', 404);
                    return;
                }
                $this->showErrorPage('Invalid or expired token.', 403);
                return;
            }
            // Neither sid nor wallet success params present
            $this->redirect('/b2b/agent/activity/payment?err=exception');
            return;
        }
        $agent = $_SESSION['agent'] ?? null;
        if (!$agent || empty($agent['id'])) { $this->redirect('/b2b/agent/login'); return; }

        // Recompute amount (strict) before persisting
        $guest = $_SESSION['activity_guest'] ?? [];
        $cart = $_SESSION['activity_cart'] ?? [];
        $amount = 0.0; $currency = 'THB';
        try {
            if (!is_array($cart) || empty($cart)) { $this->redirect('/b2b/agent/activity/payment?err=amount'); return; }
            $priceIdQty = [];
            foreach ($cart as $it) {
                $pid = (int)($it['price_id'] ?? 0);
                $qty = max(0, (int)($it['qty'] ?? 0));
                if ($pid > 0 && $qty > 0) { $priceIdQty[$pid] = ($priceIdQty[$pid] ?? 0) + $qty; }
            }
            if (empty($priceIdQty)) { $this->redirect('/b2b/agent/activity/payment?err=amount'); return; }
            $ids = array_keys($priceIdQty);
            $ph = implode(',', array_fill(0, count($ids), '?'));
            $stmt = $this->pdo->prepare('SELECT id, agent_price, active FROM vendor_package_prices WHERE id IN (' . $ph . ')');
            $stmt->execute($ids);
            $map = [];
            foreach ($stmt->fetchAll(\PDO::FETCH_ASSOC) ?: [] as $r) { $map[(int)$r['id']] = $r; }
            foreach ($priceIdQty as $pid => $qty) {
                if (!isset($map[$pid])) { $this->redirect('/b2b/agent/activity/payment?err=verify_mismatch'); return; }
                $pr = (float)($map[$pid]['agent_price'] ?? 0.0); if ($pr <= 0) { $this->redirect('/b2b/agent/activity/payment?err=verify_mismatch'); return; }
                $amount += $pr * $qty;
            }
            $transportId = (int)($guest['transport']['option_id'] ?? 0);
            if ($transportId > 0) {
                try { $stT = $this->pdo->prepare('SELECT price, is_active FROM vendor_package_transport_options WHERE id = ?'); $stT->execute([$transportId]); if ($rowT = $stT->fetch(\PDO::FETCH_ASSOC)) { if ((int)($rowT['is_active'] ?? 1) === 1) { $amount += (float)($rowT['price'] ?? 0.0); } } } catch (\Throwable $_) {}
            }
        } catch (\Throwable $_) { $this->redirect('/b2b/agent/activity/payment?err=verify_mismatch'); return; }

        // Retrieve Stripe session and verify status
        try {
            $pgCfg = @require __DIR__ . '/../../config/payment_gateways.php';
            $secret = (string)($pgCfg['stripe']['secret_key'] ?? '');
            if ($secret === '') { $this->redirect('/b2b/agent/activity/payment?err=gateway_not_configured'); return; }
            \Stripe\Stripe::setApiKey($secret);
            $session = \Stripe\Checkout\Session::retrieve($sid);
            if (!in_array((string)($session->payment_status ?? ''), ['paid','no_payment_required'], true)) {
                $this->redirect('/b2b/agent/activity/payment?err=exception'); return;
            }
        } catch (\Throwable $e) {
            $this->redirect('/b2b/agent/activity/payment?err=exception'); return;
        }

        // Persist booking and payment
        $bookingId = null; $bookingCode = 'ACT-' . date('Ymd') . '-' . substr((string)time(), -5);
        $lead = (array)($guest['lead'] ?? []);
        $date = (string)($guest['date'] ?? '');
        $showTime = (string)($guest['show_time'] ?? '');
        $pickup = (array)($guest['transport'] ?? []);
        try {
            $sp = $this->pdo->prepare('SELECT name, vendor_id FROM vendor_packages WHERE id = ?');
            $sp->execute([$packageId]);
            if ($row = $sp->fetch(\PDO::FETCH_ASSOC)) {
                $name = (string)($row['name'] ?? ''); if ($name !== '') { $summaryTitle = $name; }
                $vendorId = isset($row['vendor_id']) ? (int)$row['vendor_id'] : null;
            }
        } catch (\Throwable $_) { /* ignore */ }
        // Build items_json from cart for gateway flow as well
        $itemsArr = [];
        foreach (($cart ?? []) as $it) {
            $qty = (int)($it['qty'] ?? 0);
            $pid = (int)($it['price_id'] ?? 0);
            $unit = 0.0;
            try {
                $q = $this->pdo->prepare('SELECT agent_price, variant_id FROM vendor_package_prices WHERE id = ?');
                $q->execute([$pid]);
                if ($r = $q->fetch(\PDO::FETCH_ASSOC)) {
                    $unit = (float)($r['agent_price'] ?? 0.0);
                    $vidLocal = isset($r['variant_id']) ? (int)$r['variant_id'] : null;
                } else { $vidLocal = null; }
            } catch (\Throwable $_) { $vidLocal = null; }
            $itemsArr[] = [ 'variant_id'=>$vidLocal, 'price_id'=>$pid, 'qty'=>$qty, 'unit'=>$unit, 'line_total'=>round($unit*$qty,2) ];
        }
        $itemsJson = json_encode($itemsArr, JSON_UNESCAPED_UNICODE);
        $userAgent = (string)($_SERVER['HTTP_USER_AGENT'] ?? '');
        $ipAddr = (string)($_SERVER['REMOTE_ADDR'] ?? '');
        $txnId = 'gateway:' . $sid;

        $bookingId = (int)($_SESSION['activity_booking_id'] ?? 0);
        if ($bookingId > 0) {
            try {
                $upd = $this->pdo->prepare("UPDATE activity_bookings SET status='confirmed', payment_method='gateway', gateway_name='stripe', payment_status='paid', payment_txn_id=:tx, items_json=:items, qty_total=:qty, amount_total=:amt, currency='THB', paid_at=NOW(), updated_at=NOW() WHERE id=:id AND agent_id=:uid");
                $upd->execute([':tx'=>$txnId, ':items'=>$itemsJson, ':qty'=>$paxTotal, ':amt'=>(float)$amount, ':id'=>$bookingId, ':uid'=>(int)$agent['id']]);
                try {
                    $ev = $this->pdo->prepare('INSERT INTO activity_booking_events (booking_id, event_type, note, data_json, created_at) VALUES (:bid,\'payment_captured\',\'Payment captured via gateway\',:data,NOW())');
                    $ev->execute([':bid'=>$bookingId, ':data'=>json_encode(['method'=>'gateway','sid'=>$sid,'amount'=>$amount,'currency'=>'THB'])]);
                } catch (\Throwable $_) {}
                $this->redirect('/b2b/agent/activity/voucher?id=' . $bookingId); return; 
            } catch (\Throwable $_) { /* fall through to insert */ }
        }

        try {
            $insSql = "INSERT INTO activity_bookings (booking_code, agent_id, created_by, updated_by, vendor_id, package_id, booking_date, show_time, items_json, qty_total, amount_total, currency, channel, status, payment_method, gateway_name, payment_status, payment_txn_id, ip_address, user_agent, paid_at, lead_name, lead_phone, lead_email, pickup_required, pickup_hotel_name, pickup_address, pickup_city, pickup_notes, created_at, updated_at)
                        VALUES (:code, :uid, :uid, :uid, :vid, :pkg, :bdate, :stime, :items, :qty, :amt, :cur, 'portal', 'confirmed', 'gateway', 'stripe', 'paid', :tx, :ip, :ua, NOW(), :lname, :lphone, :lemail, :p_req, :p_hotel, :p_addr, :p_city, :p_notes, NOW(), NOW())";
            $ins = $this->pdo->prepare($insSql);
            $ins->execute([
                ':code'=>$bookingCode,
                ':uid'=>(int)$agent['id'],
                ':vid'=>$vendorId,
                ':pkg'=>$packageId,
                ':bdate'=>$date,
                ':stime'=>$showTime ?: null,
                ':items'=>$itemsJson,
                ':qty'=>$paxTotal,
                ':amt'=>(float)$amount,
                ':cur'=>'THB',
                ':tx'=>$txnId,
                ':ip'=>$ipAddr,
                ':ua'=>$userAgent,
                ':lname'=>(string)($lead['name'] ?? ''),
                ':lphone'=>(string)($lead['phone'] ?? ''),
                ':lemail'=>(string)($lead['email'] ?? ''),
                ':p_req'=>0,
                ':p_hotel'=>'',
                ':p_addr'=>'',
                ':p_city'=>'',
                ':p_notes'=>'',
            ]);
            $bookingId = (int)$this->pdo->lastInsertId();
            try {
                $ev = $this->pdo->prepare('INSERT INTO activity_booking_events (booking_id, event_type, note, data_json, created_at) VALUES (:bid,:type,:note,:data,NOW())');
                $ev->execute([':bid'=>$bookingId, ':type'=>'created', ':note'=>'Activity booking created', ':data'=>json_encode(['pax'=>$paxTotal,'amount'=>$amount,'currency'=>'THB'])]);
                $ev->execute([':bid'=>$bookingId, ':type'=>'payment_captured', ':note'=>'Payment captured via gateway', ':data'=>json_encode(['method'=>'gateway','sid'=>$sid,'amount'=>$amount,'currency'=>'THB'])]);
            } catch (\Throwable $_) { /* ignore */ }
            $this->redirect('/b2b/agent/activity/voucher?id=' . $bookingId); return; 
        } catch (\Throwable $_) { /* insert failed */ }

        $this->redirect('/b2b/agent/activity/payment?err=exception');
        $this->redirect('/b2b/agent/activity/payment?err=exception');
    }
}
